package gov.vha.ctt.ntrt;

import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.jira.bc.projectroles.ProjectRoleService;
import com.atlassian.jira.exception.CreateException;
import com.atlassian.jira.junit.rules.AvailableInContainer;
import com.atlassian.jira.junit.rules.MockitoContainer;
import com.atlassian.jira.junit.rules.MockitoMocksInContainer;
import com.atlassian.jira.mock.MockProjectRoleManager;
import com.atlassian.jira.mock.component.MockComponentWorker;
import com.atlassian.jira.mock.ofbiz.MockGenericValue;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectImpl;
import com.atlassian.jira.project.ProjectManager;
import com.atlassian.jira.security.groups.GroupManager;
import com.atlassian.jira.security.login.LoginManager;
import com.atlassian.jira.security.roles.ProjectRole;
import com.atlassian.jira.security.roles.ProjectRoleImpl;
import com.atlassian.jira.security.roles.ProjectRoleManager;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.user.MockApplicationUser;
import com.atlassian.jira.user.MockGroup;
import com.atlassian.jira.user.UserDetails;
import com.atlassian.jira.user.util.UserManager;
import com.atlassian.jira.util.ErrorCollection;
import com.atlassian.seraph.config.SecurityConfig;
import com.atlassian.seraph.elevatedsecurity.ElevatedSecurityGuard;
import com.atlassian.seraph.service.rememberme.RememberMeService;
import com.google.common.collect.ImmutableMap;
import org.apache.log4j.Appender;
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.log4j.spi.LoggingEvent;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;
import org.springframework.mock.web.MockHttpSession;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.security.Principal;
import java.util.*;

import static com.atlassian.jira.security.roles.ProjectRoleActor.USER_ROLE_ACTOR_TYPE;
import static com.google.common.collect.ImmutableMap.of;
import static gov.vha.ctt.ntrt.SsoiAuthenticator.*;
import static gov.vha.ctt.ntrt.SsoiAuthenticator.Role.*;
import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.isA;
import static org.mockito.ArgumentMatchers.matches;
import static org.mockito.BDDMockito.given;
import static org.mockito.Mockito.*;
import static org.springframework.test.util.ReflectionTestUtils.setField;

public class SsoiAuthenticatorTest {

    private SsoiAuthenticator authenticator;

    private static final String ADMIN_ROLE = NTRT_ADMIN.toString().toLowerCase();
    private static final String STAFF_ROLE = NTRT_STAFF.toString().toLowerCase();
    private static final String USER_ROLE = NTRT_USER.toString().toLowerCase();
    private static final String TEST_USER_NAME = "testuser";

    @Rule
    public MockitoContainer mockitoContainer = MockitoMocksInContainer.rule(this);

    @Mock
    @AvailableInContainer
    private RememberMeService rememberMeService;

    @Mock
    @AvailableInContainer
    private SecurityConfig securityConfig;

    @Mock
    @AvailableInContainer
    private UserManager userManager;

    @Mock
    @AvailableInContainer
    private ElevatedSecurityGuard elevatedSecurityGuard;

    @Mock
    @AvailableInContainer
    private LoginManager loginManager;

    @Mock
    @AvailableInContainer
    private GroupManager groupManager;

    @Mock
    @AvailableInContainer
    private ProjectRoleService projectRoleService;

    @Mock
    @AvailableInContainer
    private ProjectManager projectManager;

    @Mock
    @AvailableInContainer
    private ProjectRoleManager projectRoleManager;

    @Mock
    private Appender appender;

    @Captor
    private ArgumentCaptor<LoggingEvent> captor;

    private final Project ntrtProject = new ProjectImpl(new MockGenericValue("NTRT", ImmutableMap.of("key", "NTRT")));
    private final Group jiraServiceDeskUsersGroup = new MockGroup(JIRA_GROUP_SERVICEDESK_USERS);

    @Before
    public void setUp() throws Exception {
        authenticator = new SsoiAuthenticator();
        new MockComponentWorker()
                .addMock(UserManager.class, userManager)
                .addMock(RememberMeService.class, rememberMeService)
                .addMock(SecurityConfig.class, securityConfig)
                .addMock(ElevatedSecurityGuard.class, elevatedSecurityGuard)
                .addMock(LoginManager.class, loginManager)
                .addMock(GroupManager.class, groupManager)
                .addMock(ProjectRoleService.class, projectRoleService)
                .addMock(ProjectManager.class, projectManager)
                .addMock(ProjectRoleManager.class, projectRoleManager)
                .init();

        Logger rootLogger = LogManager.getRootLogger();
        rootLogger.addAppender(appender);
        rootLogger.setLevel(Level.DEBUG);
    }

    @Test
    public void findNtrtRolesShouldFindAllRoles() throws Exception {
        EnumSet<Role> roles = authenticator.findNtrtRoles(mockRequest(of(
                HEADER_PRISME_ROLES, format("foo,%s,%s,%s,bar", ADMIN_ROLE, STAFF_ROLE, USER_ROLE))));
        assertThat(roles).containsExactlyInAnyOrder(NTRT_ADMIN, NTRT_STAFF, NTRT_USER);
    }

    @Test
    public void findNtrtRolesShouldFindOneRole() throws Exception {
        EnumSet<Role> roles = authenticator.findNtrtRoles(mockRequest(of(
                HEADER_PRISME_ROLES, format("foo,%s,bar", ADMIN_ROLE))));
        assertThat(roles).containsExactly(NTRT_ADMIN);
    }

    @Test
    public void findNtrtRolesShouldFindNoRoles() throws Exception {
        EnumSet<Role> roles = authenticator.findNtrtRoles(mockRequest(of(
                HEADER_PRISME_ROLES, "foo,bar")));
        assertThat(roles).isEmpty();
    }

    @Test
    public void getDisplayNameShouldPreferHeadersToEmailAddress() {
        HttpServletRequest request = mockRequest(of(
                HEADER_FIRSTNAME, "HOMER",
                HEADER_LASTNAME, "SIMPSON",
                HEADER_ADEMAIL, "PII                 "));
        assertThat(authenticator.getDisplayName(request)).isEqualTo("Homer Simpson");
    }

    @Test
    public void getDisplayNameShouldUseAccountNameIfEmailInWrongFormat() {
        HttpServletRequest request = mockRequest(of(
                HEADER_ADSAMACCOUNTNAME, "homersimpson",
                HEADER_ADEMAIL, "PII                "));
        assertThat(authenticator.getDisplayName(request)).isEqualTo("homersimpson");
    }

    @Test
    public void getDisplayNameShouldUseEmailWhenNamesAbsent() {
        HttpServletRequest request = mockRequest(of(HEADER_ADEMAIL, "PII                 "));
        assertThat(authenticator.getDisplayName(request)).isEqualTo("Homer Simpson");
    }

    @Test
    public void getDisplayNameShouldUseEmailWhenFirstNameAbsent() {
        HttpServletRequest request = mockRequest(of(
                HEADER_LASTNAME, "SIMPSON",
                HEADER_ADEMAIL, "PII                 "));
        assertThat(authenticator.getDisplayName(request)).isEqualTo("Homer Simpson");
    }

    @Test
    public void getDisplayNameShouldUseEmailWhenLastNameAbsent() {
        HttpServletRequest request = mockRequest(of(
                HEADER_FIRSTNAME, "HOMER",
                HEADER_ADEMAIL, "PII                 "));
        assertThat(authenticator.getDisplayName(request)).isEqualTo("Homer Simpson");
    }

    @Test
    public void getUserReturnsNullWhenNtrtRoleAbsent() {
        given(userManager.getUserByName(TEST_USER_NAME)).willReturn(null);

        HttpServletRequest request = mockRequest(of(
                HEADER_ADSAMACCOUNTNAME, TEST_USER_NAME,
                HEADER_PRISME_ROLES, "foo,bar"));
        assertThat(authenticator.getUser(request, null)).isNull();
    }

    @Test
    public void getUserUsesCachedRoles() {

        MockHttpSession session = mockSessionWithCachedRoles(NTRT_USER, NTRT_STAFF, NTRT_ADMIN);

        MockHttpServletRequest request = mockRequest(of(
                HEADER_ADSAMACCOUNTNAME, TEST_USER_NAME,
                HEADER_PRISME_ROLES, format("%s,%s,%s", USER_ROLE, STAFF_ROLE, ADMIN_ROLE)));
        request.setSession(session);

        givenUserManagerWillReturnTestUser();

        Principal user = authenticator.getUser(request, new MockHttpServletResponse());
        assertThat(user).isNotNull();
        assertThat(user.getName()).isEqualTo(TEST_USER_NAME);

        // if the cached roles were used, we won't look for roles in the database
        verify(projectRoleManager, never()).getProjectRoles(isA(ApplicationUser.class), isA(Project.class));
    }

    @Test
    @SuppressWarnings("deprecation")
    public void getUserAssignsRolesToUserWhenCachedRolesNotFound() {
        stubTestAndAdminUsersAndUsersGroupAndProject();
        givenProjectRoleServiceWillReturnProjectRole(JIRA_ROLE_CUSTOMER);
        givenProjectRoleServiceWillReturnProjectRole(JIRA_ROLE_TEAM);
        givenProjectRoleServiceWillReturnProjectRole(JIRA_ROLE_ADMIN);
        givenProjectRoleManagerWillReturnProjectRoles(
                projectRole(JIRA_ROLE_CUSTOMER), projectRole(JIRA_ROLE_TEAM), projectRole(JIRA_ROLE_ADMIN));

        MockHttpServletRequest request = mockRequest(of(
                HEADER_ADSAMACCOUNTNAME, TEST_USER_NAME,
                HEADER_PRISME_ROLES, format("%s,%s,%s", USER_ROLE, STAFF_ROLE, ADMIN_ROLE)));

        Principal user = authenticator.getUser(request, new MockHttpServletResponse());
        assertThat(user).isNotNull();
        assertThat(user.getName()).isEqualTo(TEST_USER_NAME);

        // verify the roles were cached in the session
        Set<String> cachedRoles = authenticator.getCachedRoles(request.getSession());
        assertThat(cachedRoles).isNotNull();
        assertThat(cachedRoles.size()).isEqualTo(3);
        assertThat(cachedRoles).containsExactlyInAnyOrder(JIRA_ROLE_CUSTOMER, JIRA_ROLE_TEAM, JIRA_ROLE_ADMIN);

        Set<String> userKeys = Collections.singleton(TEST_USER_NAME);
        ApplicationUser adminUser = authenticator.getAdminUser();

        // user should have been added to three roles on the back end
        verify(projectRoleService, times(3)).addActorsToProjectRole(
                eq(adminUser),
                eq(userKeys),
                isA(ProjectRole.class),
                isA(Project.class),
                eq(USER_ROLE_ACTOR_TYPE),
                isA(ErrorCollection.class)
        );
    }

    @Test
    public void getUserReturnsNullWhenNoRolesAreCachedAndUserCantBeAssignedToProjectRole() {
        stubTestAndAdminUsersAndUsersGroupAndProject();
        givenProjectRoleServiceWillReturnProjectRole(JIRA_ROLE_ADMIN);

        MockHttpServletRequest request = mockRequest(of(
                HEADER_ADSAMACCOUNTNAME, TEST_USER_NAME,
                HEADER_PRISME_ROLES, format("%s,%s,%s", USER_ROLE, STAFF_ROLE, ADMIN_ROLE)));

        doFailAddActorsToProjectRole();

        Principal user = authenticator.getUser(request, new MockHttpServletResponse());
        assertThat(user).isNull();
    }

    @Test
    public void getUserReturnsNullWhenCachedRolesAreInvalidAndUserCantBeAssignedToProjectRole() {
        stubTestAndAdminUsersAndUsersGroupAndProject();
        givenProjectRoleServiceWillReturnProjectRole(JIRA_ROLE_CUSTOMER);

        MockHttpServletRequest request = mockRequest(of(
                HEADER_ADSAMACCOUNTNAME, TEST_USER_NAME,
                HEADER_PRISME_ROLES, USER_ROLE));
        MockHttpSession session = mockSessionWithCachedRoles(NTRT_ADMIN);
        request.setSession(session);

        doFailAddActorsToProjectRole();

        Principal user = authenticator.getUser(request, new MockHttpServletResponse());
        assertThat(user).isNull();
    }

    @Test
    public void getUserDelegatesToParentWhenAccountHeaderNotPresent() {
        HttpServletRequest request = mockRequest(of(HEADER_PRISME_ROLES, "ntrt_user"));
        final String username = "nonNtrtUser";

        setField(authenticator, "config", securityConfig);
        given(securityConfig.getRememberMeService()).willReturn(rememberMeService);
        given(securityConfig.getElevatedSecurityGuard()).willReturn(elevatedSecurityGuard);
        given(rememberMeService.getRememberMeCookieAuthenticatedUsername(
                isA(HttpServletRequest.class), isA(HttpServletResponse.class))).willReturn(username);
        given(userManager.getUserByName(username)).willReturn(new MockApplicationUser(username));
        given(elevatedSecurityGuard.performElevatedSecurityCheck(
                isA(HttpServletRequest.class), matches(username))).willReturn(true);
        given(loginManager.authoriseForLogin(
                isA(ApplicationUser.class), isA(HttpServletRequest.class))).willReturn(true);
        assertThat(authenticator.getUser(request, mockResponse()).getName()).isEqualTo(username);
    }

    @Test
    public void createNewUserReturnsNullWhenAccountNameIsPresentButEmailAddressIsAbsent() {
        final String accountName = TEST_USER_NAME;
        HttpServletRequest request = mockRequest(of(HEADER_ADSAMACCOUNTNAME, accountName, HEADER_PRISME_ROLES, "ntrt_user"));
        assertThat(authenticator.createNewUser(request, accountName)).isNull();
    }

    @Test
    @SuppressWarnings("deprecation")
    public void getUserReturnsPrincipalOnHappyPath() throws Exception {

        givenGroupManagerWillReturnServiceDeskUsersGroup();

        ProjectRole teamRole = projectRole(JIRA_ROLE_TEAM);
        given(projectRoleService.getProjectRoleByName(isA(String.class), isA(ErrorCollection.class))).willReturn(teamRole);

        Project project = mock(Project.class);
        given(projectManager.getProjectByCurrentKey(JIRA_PROJECT_NAME_NTRT)).willReturn(project);
        givenProjectRoleManagerWillReturnProjectRoles(teamRole);

        ApplicationUser testUser = new MockApplicationUser(TEST_USER_NAME);
        given(userManager.createUser(isA(UserDetails.class))).willReturn(testUser);
        given(userManager.getUserByName(TEST_USER_NAME)).willReturn(testUser);

        ApplicationUser adminUser = new MockApplicationUser("admin");
        given(userManager.getUserObject(JIRA_USER_NAME_ADMIN)).willReturn(adminUser);

        HttpServletRequest request = mockRequest(of(
                HEADER_ADSAMACCOUNTNAME, TEST_USER_NAME,
                HEADER_ADEMAIL, "test@user.com",
                HEADER_PRISME_ROLES, "ntrt_user"));

        Principal expected = this.authenticator.getUser(request, null);
        assertThat(expected).isInstanceOf(Principal.class);
        assertThat(expected.getName()).isEqualTo(TEST_USER_NAME);
    }

    @Test
    public void getUserReturnsNullOnCreateUserFailure() throws Exception {
        given(userManager.createUser(isA(UserDetails.class))).willThrow(CreateException.class);

        HttpServletRequest request = mockRequest(of(
                HEADER_ADSAMACCOUNTNAME, TEST_USER_NAME,
                HEADER_ADEMAIL, "test@user.com",
                HEADER_PRISME_ROLES, "ntrt_user"));

        assertThat(authenticator.getUser(request, null)).isNull();
    }

    @Test
    public void getHeaderHandlesIncorrectCase() throws Exception {
        final String expected = "bar";

        HttpServletRequest request = mockRequest(of("foo", expected));
        assertThat(authenticator.getHeader(request, "FOO")).isEqualTo(expected);

        request = mockRequest(of("FOO", expected));
        assertThat(authenticator.getHeader(request, "foo")).isEqualTo(expected);

        request = mockRequest(of("FoO", expected));
        assertThat(authenticator.getHeader(request, "foo")).isEqualTo(expected);
        assertThat(authenticator.getHeader(request, "FOO")).isEqualTo(expected);
    }

    @Test
    public void getHeaderLogsMissingHeader() throws Exception {

        MockHttpSession session = new MockHttpSession();
        MockHttpServletRequest request = mockRequest(of("baz", "quux", "doodle", "dee"));
        request.setSession(session);

        authenticator.getHeader(request, "foo");
        authenticator.getHeader(request, "bar");

        verify(appender, times(2)).doAppend(captor.capture());

        @SuppressWarnings("unchecked")
        List<String> actualMissedHeaders = (List<String>) session.getAttribute(NTRT_AUTHENTICATOR_MISSING_HEADERS_LOGGED_KEY);
        assertThat(actualMissedHeaders).containsExactly("foo", "bar");

        String messageTemplate = LOG_MESSAGE_MISSING_HEADER.replace("{}", "%s");
        String expectedFoundHeaders = Arrays.asList("baz", "doodle").toString();
        String bazMessage = format(messageTemplate, "foo", session.getId(), expectedFoundHeaders);
        String doodleMessage = format(messageTemplate, "bar", session.getId(), expectedFoundHeaders);

        List<LoggingEvent> logs = captor.getAllValues();
        assertThat(logs.get(0).getMessage()).isEqualTo(bazMessage);
        assertThat(logs.get(1).getMessage()).isEqualTo(doodleMessage);
    }

    @Test
    public void validateCachedRolesIdentifiesMatches() {
        assertValidateCachedRoles(ntrtRoles(NTRT_USER), projectRolesFor(NTRT_USER), true);
        assertValidateCachedRoles(ntrtRoles(NTRT_STAFF), projectRolesFor(NTRT_STAFF), true);
        assertValidateCachedRoles(ntrtRoles(NTRT_ADMIN), projectRolesFor(NTRT_ADMIN), true);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_STAFF), projectRolesFor(NTRT_USER, NTRT_STAFF), true);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_ADMIN), projectRolesFor(NTRT_USER, NTRT_ADMIN), true);
        assertValidateCachedRoles(ntrtRoles(NTRT_STAFF, NTRT_ADMIN), projectRolesFor(NTRT_STAFF, NTRT_ADMIN), true);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_STAFF, NTRT_ADMIN), projectRolesFor(NTRT_USER, NTRT_STAFF, NTRT_ADMIN), true);
    }

    @Test
    public void validateCachedRolesIdentifiesMismatches() {

        // user has wrong project role
        assertValidateCachedRoles(ntrtRoles(NTRT_USER), projectRolesFor(NTRT_STAFF), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER), projectRolesFor(NTRT_ADMIN), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_STAFF), projectRolesFor(NTRT_USER), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_STAFF), projectRolesFor(NTRT_ADMIN), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_ADMIN), projectRolesFor(NTRT_USER), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_ADMIN), projectRolesFor(NTRT_STAFF), false);

        // user has one too many project roles
        assertValidateCachedRoles(ntrtRoles(NTRT_USER), projectRolesFor(NTRT_USER, NTRT_STAFF), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER), projectRolesFor(NTRT_USER, NTRT_ADMIN), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_STAFF), projectRolesFor(NTRT_STAFF, NTRT_USER), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_STAFF), projectRolesFor(NTRT_STAFF, NTRT_USER), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_ADMIN), projectRolesFor(NTRT_ADMIN, NTRT_USER), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_ADMIN), projectRolesFor(NTRT_ADMIN, NTRT_STAFF), false);

        // user has two too many project roles
        assertValidateCachedRoles(ntrtRoles(NTRT_USER), projectRolesFor(NTRT_USER, NTRT_STAFF, NTRT_ADMIN), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_STAFF), projectRolesFor(NTRT_STAFF, NTRT_USER, NTRT_ADMIN), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_ADMIN), projectRolesFor(NTRT_ADMIN, NTRT_STAFF, NTRT_USER), false);

        // user has one too few project roles
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_STAFF), projectRolesFor(NTRT_USER), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_STAFF), projectRolesFor(NTRT_STAFF), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_ADMIN), projectRolesFor(NTRT_USER), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_ADMIN), projectRolesFor(NTRT_ADMIN), false);

        // user has two too few project roles
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_STAFF, NTRT_ADMIN), projectRolesFor(NTRT_USER), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_STAFF, NTRT_ADMIN), projectRolesFor(NTRT_STAFF), false);
        assertValidateCachedRoles(ntrtRoles(NTRT_USER, NTRT_STAFF, NTRT_ADMIN), projectRolesFor(NTRT_ADMIN), false);
    }

    @Test
    public void cacheProjectRolesCachesProjectRoles() {
        MockApplicationUser user = new MockApplicationUser(TEST_USER_NAME, "Test User", "test@user.com");
        MockHttpSession session = new MockHttpSession();

        givenProjectManagerWillReturnNtrtProject();
        given(projectRoleManager.getProjectRoles(eq(user), isA(Project.class))).willReturn(Arrays.asList(
                new ProjectRoleImpl(JIRA_ROLE_CUSTOMER, ""),
                new ProjectRoleImpl(JIRA_ROLE_TEAM, ""),
                new ProjectRoleImpl(JIRA_ROLE_ADMIN, "")
        ));

        authenticator.cacheProjectRoles(user, session);
        Set<String> cachedRoles = authenticator.getCachedRoles(session);

        assertThat(cachedRoles).isNotNull();
        assertThat(cachedRoles.size()).isEqualTo(3);
        assertThat(cachedRoles).containsExactlyInAnyOrder(JIRA_ROLE_CUSTOMER, JIRA_ROLE_TEAM, JIRA_ROLE_ADMIN);
    }

    @Test
    public void cacheProjectRolesDoesNotCacheZeroRoles() {
        MockApplicationUser user = new MockApplicationUser(TEST_USER_NAME, "Test User", "test@user.com");
        MockHttpSession session = new MockHttpSession();

        givenProjectManagerWillReturnNtrtProject();
        given(projectRoleManager.getProjectRoles(eq(user), isA(Project.class))).willReturn(emptyList());

        authenticator.cacheProjectRoles(user, session);
        Set<String> cachedRoles = authenticator.getCachedRoles(session);

        assertThat(cachedRoles).isNull();
    }

    @Test(expected = IllegalArgumentException.class)
    @SuppressWarnings("deprecation")
    public void getAdminUserThrowsWhenUserNotFound() {
        given(userManager.getUserObject(JIRA_USER_NAME_ADMIN)).willReturn(null);
        authenticator.getAdminUser();
    }

    @Test
    public void addUserToGroupReturnsFalseWhenAddOperationFails() throws Exception {
        givenGroupManagerWillReturnServiceDeskUsersGroup();
        doThrow(OperationFailedException.class).when(groupManager).addUserToGroup(isA(ApplicationUser.class), isA(Group.class));
        assertThat(authenticator.addUserToGroup(testUser(), JIRA_GROUP_SERVICEDESK_USERS)).isFalse();
    }

    @Test
    public void assignUserToProjectRolesReturnsFalseIfUserCannotBeAddedToServiceDeskUsersGroup() throws Exception {
        givenGroupManagerWillReturnServiceDeskUsersGroup();
        doThrow(OperationFailedException.class).when(groupManager).addUserToGroup(isA(ApplicationUser.class), isA(Group.class));
        assertThat(authenticator.assignUserToProjectRoles(testUser(), null)).isFalse();
    }

    @Test
    public void assignUserToProjectRolesReturnsFalseIfUserCannotBeAssignedToProjectRole() {
        stubTestAndAdminUsersAndUsersGroupAndProject();
        givenProjectRoleServiceWillReturnProjectRole(JIRA_ROLE_TEAM);
        doFailAddActorsToProjectRole();

        assertThat(authenticator.assignUserToProjectRoles(testUser(), EnumSet.of(NTRT_STAFF))).isFalse();
    }

    @Test
    public void removeUserFromProjectRoleTrapsError() {
        stubTestAndAdminUsersAndUsersGroupAndProject();
        givenProjectRoleManagerWillReturnProjectRoles(projectRole(JIRA_ROLE_CUSTOMER));
        doFailRemoveActorsFromProjectRole();

        String messageTemplate = LOG_MESSAGE_COULD_NOT_REMOVE_PROJECT_ROLE.replace("{}", "%s");
        String expectedMessage = format(messageTemplate,
                format("%s(%s)", TEST_USER_NAME, TEST_USER_NAME),
                format("Project Role: %s(%d)", JIRA_ROLE_CUSTOMER, 1000),
                "Project: NTRT", "[]", "[]");

        authenticator.removeUserFromProjectRole(testUser(), projectRole(JIRA_ROLE_CUSTOMER));

        verify(appender, times(1)).doAppend(captor.capture());
        List<LoggingEvent> logs = captor.getAllValues();
        assertThat(logs.size()).isEqualTo(1);
        assertThat(logs.get(0).getMessage()).isEqualTo(expectedMessage);
    }

    @Test
    public void getProjectRoleTrapsError() throws Exception {
        stubTestAndAdminUsersAndUsersGroupAndProject();
        givenProjectRoleServiceWillReturnProjectRole(JIRA_ROLE_CUSTOMER);

        String messageTemplate = LOG_MESSAGE_GET_PROJECT_ROLE_ERROR.replace("{}", "%s");
        String expectedMessage = format(messageTemplate, JIRA_ROLE_CUSTOMER, "[]", "[]");

        addErrorToErrorCollectionWhen(projectRoleService, 1).getProjectRoleByName(
                isA(String.class),
                isA(ErrorCollection.class));

        authenticator.getProjectRole(JIRA_ROLE_CUSTOMER);

        verify(appender, times(1)).doAppend(captor.capture());
        List<LoggingEvent> logs = captor.getAllValues();
        assertThat(logs.size()).isEqualTo(1);
        assertThat(logs.get(0).getMessage()).isEqualTo(expectedMessage);
    }

    /*
        Utilities
     */

    private ProjectRole projectRole(String projectRole) {
        return new MockProjectRoleManager.MockProjectRole(1000, projectRole, projectRole);
    }

    private void givenProjectRoleManagerWillReturnProjectRoles(ProjectRole... projectRoles) {
        given(projectRoleManager.getProjectRoles(isA(ApplicationUser.class), isA(Project.class)))
                .willReturn(Arrays.asList(projectRoles));
    }

    private void stubTestAndAdminUsersAndUsersGroupAndProject() {
        givenUserManagerWillReturnAdminUser();
        givenUserManagerWillReturnTestUser();
        givenGroupManagerWillReturnServiceDeskUsersGroup();
        givenProjectManagerWillReturnNtrtProject();
    }

    @SuppressWarnings("deprecation")
    private void doFailAddActorsToProjectRole() {
        Set<String> userKeys = Collections.singleton(TEST_USER_NAME);
        ApplicationUser adminUser = authenticator.getAdminUser();

        addErrorToErrorCollectionWhen(projectRoleService, 5).addActorsToProjectRole(
                eq(adminUser),
                eq(userKeys),
                isA(ProjectRole.class),
                isA(Project.class),
                eq(USER_ROLE_ACTOR_TYPE),
                isA(ErrorCollection.class));
    }

    @SuppressWarnings("deprecation")
    private void doFailRemoveActorsFromProjectRole() {
        Set<String> userKeys = Collections.singleton(TEST_USER_NAME);
        ApplicationUser adminUser = authenticator.getAdminUser();

        addErrorToErrorCollectionWhen(projectRoleService, 5).removeActorsFromProjectRole(
                eq(adminUser),
                eq(userKeys),
                isA(ProjectRole.class),
                isA(Project.class),
                eq(USER_ROLE_ACTOR_TYPE),
                isA(ErrorCollection.class));
    }

    private <T> T addErrorToErrorCollectionWhen(T t, int errorCollectionArgumentIndex) {
        return doAnswer(invocation -> {
            // populating the errors collection will cause addUserToProjectRole() to return false
            ErrorCollection errors1 = invocation.getArgument(errorCollectionArgumentIndex);
            errors1.addError("Induced error", "");
            return null;
        }).when(t);
    }

    private MockHttpSession mockSessionWithCachedRoles(Role... ntrtRoles) {
        MockHttpSession session = new MockHttpSession();
        Set<String> cachedRoles = projectRolesFor(ntrtRoles);
        session.setAttribute(NTRT_SSOI_AUTH_ROLES_KEY, cachedRoles);
        return session;
    }

    private void givenProjectRoleServiceWillReturnProjectRole(String projectRoleName) {
        ProjectRole projectRole = new MockProjectRoleManager.MockProjectRole(0, projectRoleName, "");
        given(projectRoleService.getProjectRoleByName(eq(projectRoleName), isA(ErrorCollection.class))).willReturn(projectRole);
    }

    private void givenProjectManagerWillReturnNtrtProject() {
        given(projectManager.getProjectByCurrentKey(eq(JIRA_PROJECT_NAME_NTRT))).willReturn(ntrtProject);
    }

    private void givenGroupManagerWillReturnServiceDeskUsersGroup() {
        given(groupManager.getGroup(JIRA_GROUP_SERVICEDESK_USERS)).willReturn(jiraServiceDeskUsersGroup);
    }

    @SuppressWarnings("deprecation")
    private void givenUserManagerWillReturnTestUser() {
        given(userManager.getUserObject(TEST_USER_NAME)).willReturn(testUser());
    }

    private MockApplicationUser testUser() {
        return new MockApplicationUser(TEST_USER_NAME, TEST_USER_NAME);
    }

    @SuppressWarnings("deprecation")
    private void givenUserManagerWillReturnAdminUser() {
        given(userManager.getUserObject(JIRA_USER_NAME_ADMIN)).willReturn(new MockApplicationUser(JIRA_USER_NAME_ADMIN, JIRA_USER_NAME_ADMIN));
    }

    private void assertValidateCachedRoles(EnumSet<Role> ntrtRoles, Set<String> projectRoles, boolean expected) {
        assertThat(authenticator.validateCachedRoles(ntrtRoles, projectRoles)).isEqualTo(expected);
    }

    private Set<String> projectRolesFor(Role... ntrtRoles) {
        HashSet<String> projectRoles = new HashSet<>(3);
        for (Role ntrtRole : ntrtRoles) {
            projectRoles.add(projectRoleFor(ntrtRole));
        }
        return projectRoles;
    }

    private EnumSet<Role> ntrtRoles(Role... ntrtRoles) {
        EnumSet<Role> roles = EnumSet.noneOf(Role.class);
        roles.addAll(Arrays.asList(ntrtRoles));
        return roles;
    }


    private String projectRoleFor(Role ntrtRole) {
        return PROJECT_ROLES.get(ntrtRole);
    }

    private MockHttpServletRequest mockRequest(Map<String, String> values) {
        MockHttpServletRequest request = new MockHttpServletRequest();
        values.forEach(request::addHeader);
        return request;
    }

    private MockHttpServletResponse mockResponse() {
        return new MockHttpServletResponse();
    }

}