package gov.vha.ctt.ntrt;


import com.atlassian.jira.util.json.JSONArray;
import com.atlassian.jira.util.json.JSONException;
import com.atlassian.jira.util.json.JSONObject;
import com.codeborne.selenide.SelenideElement;
import com.google.common.collect.ImmutableMap;
import okhttp3.Request;
import okhttp3.Response;
import org.junit.*;
import org.junit.runners.MethodSorters;
import org.openqa.selenium.logging.LogType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.*;

import static com.codeborne.selenide.Condition.text;
import static com.codeborne.selenide.Selenide.$;
import static com.codeborne.selenide.Selenide.open;
import static com.codeborne.selenide.WebDriverRunner.getWebDriver;
import static gov.vha.ctt.ntrt.OkHttp.*;
import static gov.vha.ctt.ntrt.SelenideTests.*;
import static gov.vha.ctt.ntrt.SsoiAuthenticator.*;
import static gov.vha.ctt.ntrt.SsoiAuthenticator.Role.*;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.Assert.fail;

/**
 * Note: tests which access a non-chromeProfile view immediately after user creation rely on the disabling of JIRA's
 * onboarding facility. To disable this feature, the property '-Datlassian.darkfeature.jira.onboarding.feature.disabled=true'
 * is passed to JIRA on startup. See src/test/docker/conf/setenv.sh.
 */
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public class SsoiAuthenticatorIT {

    private static class TestUser {
        String username;
        String emailAddress;

        TestUser() {
            username = UUID.randomUUID().toString();
            emailAddress = username + "@test.com";
        }

        TestUser(String username, String emailAddress) {
            this.username = username == null ? UUID.randomUUID().toString() : username;
            this.emailAddress = emailAddress;
        }
    }

    private static final Logger LOG = LoggerFactory.getLogger(SsoiAuthenticatorTest.class);

    private static final String BASE_URL = "https://DNS                     /ntrt";
    private static final String DASHBOARD_URI = "/secure/Dashboard.jspa";
    private static final String PROFILE_URI = "/secure/ViewProfile.jspa";
    private static final String REST_URI = "/rest/api/2";

    private final TestUser customerUser = new TestUser("ntrt_customer", "ntrt_customer@test.com");
    private final TestUser teamUser = new TestUser("ntrt_team", "ntrt_team@test.com");
    private final TestUser adminUser = new TestUser("ntrt_admin", "ntrt_admin@test.com");


    @BeforeClass
    public static void setUpOnce() throws Exception {
        initDriverService();
    }

    @Before
    public void setUp() throws Exception {
        startChromeDriver();
    }

    @After
    public void tearDown() {
        dumpSeleniumLogs(LOG, LogType.BROWSER);
        dumpSeleniumLogs(LOG, LogType.CLIENT);
        stopChromeDriver();
    }

    @AfterClass
    public static void tearDownOnce() {
        stopDriverService();
    }

    @Test
    public void _chromeDriverSmokeCheck() throws JSONException {
        setHeadersFor(customerUser, NTRT_USER);
        open(BASE_URL + PROFILE_URI);
        $("#up-user-title-name").shouldHave(text("NTRT Customer"));
    }

    @Test
    public void newUsersAreCreatedWithRoleAndVisibleInProfileView() throws Exception {
        TestUser user = new TestUser();
        assertUserCreatedOnFirstAccessWithRole(user, NTRT_USER);
        assertUserVisibleInProfileView(user, NTRT_USER);

        user = new TestUser();
        assertUserCreatedOnFirstAccessWithRole(user, NTRT_STAFF);
        assertUserVisibleInProfileView(user, NTRT_STAFF);

        user = new TestUser();
        assertUserCreatedOnFirstAccessWithRole(user, NTRT_ADMIN);
        assertUserVisibleInProfileView(user, NTRT_ADMIN);
    }

    @Test
    public void existingUsersCanSeeDashboardView() throws Exception {
        assertUserCanSeeDashboardView(customerUser, NTRT_USER);
        assertUserCanSeeDashboardView(teamUser, NTRT_STAFF);
        assertUserCanSeeDashboardView(adminUser, NTRT_ADMIN);
    }

    @Test
    public void existingUsersCanSeeProfileView() throws Exception {
        assertUserVisibleInProfileView(customerUser, NTRT_USER);
        assertUserVisibleInProfileView(teamUser, NTRT_STAFF);
        assertUserVisibleInProfileView(adminUser, NTRT_ADMIN);
    }

    @Test
    public void modifiedRoleShouldBeUpdated() throws Exception {
        TestUser user = new TestUser();
        assertUserCreatedOnFirstAccessWithRole(user, NTRT_USER);
        assertUserRoleIsUpdatedAfterExternalChange(user, NTRT_ADMIN);
        assertUserRoleIsUpdatedAfterExternalChange(user, NTRT_STAFF);
        assertUserRoleIsUpdatedAfterExternalChange(user, NTRT_USER);
    }

    @Test
    public void newUserIsCreatedWithMultipleProjectRoles() throws Exception {
        TestUser threeRolesUser = new TestUser();
        send(newRequest(BASE_URL, threeRolesUser, NTRT_ADMIN, NTRT_STAFF, NTRT_USER));
        assertUserHasProjectRole(threeRolesUser, projectRoleFor(NTRT_ADMIN));
        assertUserHasProjectRole(threeRolesUser, projectRoleFor(NTRT_STAFF));
        assertUserHasProjectRole(threeRolesUser, projectRoleFor(NTRT_USER));

        TestUser twoRolesUser = new TestUser();
        send(newRequest(BASE_URL, twoRolesUser, NTRT_STAFF, NTRT_USER));
        assertUserHasProjectRole(twoRolesUser, projectRoleFor(NTRT_STAFF));
        assertUserHasProjectRole(twoRolesUser, projectRoleFor(NTRT_USER));
        assertUserDoesNotHaveProjectRole(twoRolesUser, projectRoleFor(NTRT_ADMIN));
    }

    @Test
    public void multipleNtrtRoleChangesAreReflectedInProjectRoles() throws Exception {
        TestUser user = new TestUser();

        // create the user as a customer
        send(newRequest(BASE_URL, user, NTRT_USER));
        assertUserHasProjectRole(user, projectRoleFor(NTRT_USER));
        assertUserDoesNotHaveProjectRole(user, projectRoleFor(NTRT_STAFF));
        assertUserDoesNotHaveProjectRole(user, projectRoleFor(NTRT_ADMIN));

        // replace customer with team and admin in one request
        send(newRequest(BASE_URL, user, NTRT_ADMIN, NTRT_STAFF));
        assertUserHasProjectRole(user, projectRoleFor(NTRT_ADMIN));
        assertUserHasProjectRole(user, projectRoleFor(NTRT_STAFF));
        assertUserDoesNotHaveProjectRole(user, projectRoleFor(NTRT_USER));
    }

    /**
     * This test is meant to capture the case where a user's username has changed after creation, and thus differs
     * from their key. There was a bug which would prevent adding a project role for such a user, for which this
     * serves as a regression test.
     */
    @Test
    public void usernameChangeDoesNotPreventRoleChange() throws Exception {

        // create the user with the admin role
        TestUser user = new TestUser();
        assertUserCreatedOnFirstAccessWithRole(user, NTRT_ADMIN);

        String originalName = user.username;
        user.username = user.username + "-2";

        JSONObject json = new JSONObject();
        json.put("name", user.username);

        Response response = sendRestPut(BASE_URL + REST_URI + "/user?username=" + originalName, json);
        assertThat(response.isSuccessful()).isTrue();
        assertUserExists(user);

        // change role to user
        assertUserRoleIsUpdatedAfterExternalChange(user, NTRT_USER);
        assertUserDoesNotHaveProjectRole(user, projectRoleFor(NTRT_ADMIN));
        assertUserDoesNotHaveProjectRole(user, projectRoleFor(NTRT_STAFF));

        // verify user views
        assertUserVisibleInProfileView(user, NTRT_USER);
        assertUserCanSeeDashboardView(user, NTRT_USER);
    }

    @Test
    public void customLogoutUrlIsHonored() throws Exception {
        setHeadersFor(customerUser, NTRT_USER);
        open(BASE_URL + PROFILE_URI);

        // click the user menu to expose the log out option
        $("#header-details-user-fullname > span > span > img").click();

        // now click the log out button
        $("#log_out").click();

        assertThat(getWebDriver().getCurrentUrl()).startsWith("https://DNS       /centrallogin/centrallanding.aspx");
    }

    private void assertUserRoleIsUpdatedAfterExternalChange(TestUser user, Role newNtrtRole) throws Exception {

        // simulate the first request after a role change in PRISME
        send(newRequest(BASE_URL, user, newNtrtRole));

        // verify the user's new role role has been added and old role(s) removed
        for (Map.Entry<Role, String> entry : PROJECT_ROLES.entrySet()) {
            Role ntrtRole = entry.getKey();
            String projectRole = entry.getValue();

            if (newNtrtRole == ntrtRole)
                assertUserHasProjectRole(user, projectRole);
            else
                assertUserDoesNotHaveProjectRole(user, projectRole);
        }
    }

    private void assertUserCanSeeDashboardView(TestUser user, Role... ntrtRoles) throws Exception {
        setHeadersFor(user, ntrtRoles);
        open(BASE_URL + DASHBOARD_URI);

        // page header
        $("#dashboard-content > div.aui-page-header > div > div.aui-page-header-main > h1")
                .shouldHave(text("System Dashboard"));

        // introduction section
        $("#gadget-10000 > div > div > h3").shouldHave(text("Welcome to NTRT"));

        assertThat($("#footer > fieldset > input[type=\"hidden\"]:nth-child(1)")
                .attr("value")).isEqualTo(user.username);
    }

    private void assertUserVisibleInProfileView(TestUser user, Role ntrtRole) throws Exception {
        setHeadersFor(user, ntrtRole);
        open(BASE_URL + PROFILE_URI);

        // user name
        assertThat($("#footer > fieldset > input[type=\"hidden\"]:nth-child(1)").attr("value")).isEqualTo(user.username);
        $("#up-d-username").shouldHave(text(user.username));

        // email address
        SelenideElement email = $("#up-d-email > a");
        email.shouldHave(text(user.emailAddress));
        assertThat(email.attr("href")).isEqualTo("mailto:" + user.emailAddress);

        // user should be a member of the service desk user's group
        assertThat($("#details-profile-fragment > div.mod-content > ul > li > dl:nth-child(7) > dd").text())
                .contains("jira-servicedesk-users");
    }

    private void assertUserCreatedOnFirstAccessWithRole(TestUser user, Role ntrtRole) throws Exception {
        assertUserNotFound(user.username);
        send(newRequest(BASE_URL, user, ntrtRole));
        assertUserExists(user);
        assertUserHasProjectRole(user, projectRoleFor(ntrtRole));
    }

    private void assertUserDoesNotHaveProjectRole(TestUser user, String projectRole) throws Exception {
        assertProjectRoleActor(user, projectRole, false);
    }

    private void assertUserHasProjectRole(TestUser user, String projectRole) throws Exception {
        assertProjectRoleActor(user, projectRole, true);
    }

    private void assertProjectRoleActor(TestUser user, String projectRole, boolean expectUserIsActor) throws Exception {
        JSONArray actors = getProjectRoleActors(projectRole);
        boolean didFindActor = false;

        for (int i = 0; i < actors.length(); i++) {
            JSONObject actor = actors.getJSONObject(i);
            if (user.username.equals(actor.getString("name"))) {
                didFindActor = true;
                break;
            }
        }

        if (expectUserIsActor && !didFindActor)
            fail(format("The user '%s' was expected but was not found among the actors for project role '%s', found actors: %s", user.username, projectRole, actors));
        else if (!expectUserIsActor && didFindActor)
            fail(format("The user '%s' was not expected but was found among the actors for project role '%s', found actors: %s", user.username, projectRole, actors));
    }

    private JSONArray getProjectRoleActors(String projectRole) throws Exception {
        Response projectRoleResponse = sendRestGet(getProjectRoleUri(projectRole));
        JSONObject role = new JSONObject(assertResponseNotEmpty(projectRoleResponse));
        return role.getJSONArray("actors");
    }

    private Map<String, String> projectRoleUris = new HashMap<>(3);

    private String getProjectRoleUri(String projectRole) throws Exception {
        String roleUri = projectRoleUris.get(projectRole);

        if (roleUri == null) {
            Response response = sendRestGet(BASE_URL + "/rest/api/2/project/NTRT/role");
            JSONObject roles = new JSONObject(assertResponseNotEmpty(response));

            roleUri = roles.getString(projectRole);
            assertThat(roleUri).isNotEmpty();
            projectRoleUris.put(projectRole, roleUri);
        }

        return roleUri;
    }

    private String assertResponseNotEmpty(Response response) throws IOException {
        String body = response.body().string();
        assertThat(body).isNotEmpty();
        return body;
    }

    private void assertUserNotFound(String username) throws Exception {
        Response restResponse = sendRestGet(BASE_URL + "/rest/api/2/user?username=" + username);
        JSONObject json = new JSONObject(assertResponseNotEmpty(restResponse));
        JSONArray errorMessages = json.getJSONArray("errorMessages");
        assertThat(errorMessages.length()).isEqualTo(1);
        assertThat(errorMessages.getString(0)).isEqualTo(format("The user named '%s' does not exist", username));
    }

    private void assertUserExists(TestUser user) throws Exception {
        Response response = sendRestGet(BASE_URL + "/rest/api/2/user?username=" + user.username);
        JSONObject json = new JSONObject(assertResponseNotEmpty(response));

        assertThat(json.getString("name")).isEqualTo(user.username);
        assertThat(json.getString("emailAddress")).isEqualTo(user.emailAddress);
    }

    private void send(Request request) throws Exception {
        Response response = unsafeClient().newCall(request).execute();

        if (!response.isSuccessful()) {
            fail(format("Request for url '%s' failed with status %d and body %s",
                    request.url(), response.code(), response.body().string()));
        }
    }

    private Request newRequest(String url, TestUser user, Role... roles) {
        return new Request.Builder()
                .url(url)
                .header(HEADER_PRISME_ROLES, rolesString(roles))
                .header(HEADER_ADEMAIL, user.emailAddress)
                .header(HEADER_ADSAMACCOUNTNAME, user.username)
                .build();
    }

    private String rolesString(Role... roles) {
        String roleString;

        if (roles.length == 1) {
            roleString = normalizeRoleString(roles[0]);

        } else {
            StringBuilder builder = new StringBuilder();
            Iterator<Role> it = asList(roles).iterator();

            while (it.hasNext()) {
                Role role = it.next();
                builder.append(normalizeRoleString(role));

                if (it.hasNext())
                    builder.append(",");
            }

            roleString = builder.toString();
        }
        return roleString;
    }

    private String normalizeRoleString(Role role) {
        return role.toString().toLowerCase();
    }

    private String projectRoleFor(Role ntrtRole) {
        return PROJECT_ROLES.get(ntrtRole);
    }

    private String profilesStringFor(TestUser user, Role... ntrtRoles) throws JSONException {

        JSONObject profile = new JSONObject(ImmutableMap.of(
                "title", "Selenium",
                "appendMode", "",
                "hideComment", true,
                "respHeaders", new JSONArray(),
                "filters", new JSONArray()));

        JSONArray headers = new JSONArray();
        headers.put(header(HEADER_PRISME_ROLES, rolesString(ntrtRoles)));
        headers.put(header(HEADER_ADEMAIL, user.emailAddress));
        headers.put(header(HEADER_ADSAMACCOUNTNAME, user.username));
        profile.put("headers", headers);

        return new JSONArray(Collections.singleton(profile)).toString();
    }

    private JSONObject header(String name, String value) throws JSONException {
        JSONObject header = new JSONObject();
        header.put("enabled", true);
        header.put("name", name);
        header.put("value", value);
        header.put("comment", "");
        return header;
    }

    private void setHeadersFor(TestUser user, Role... ntrtRoles) throws JSONException {
        setModHeaderProfiles(profilesStringFor(user, ntrtRoles));
    }
}
