package gov.vha.ctt.ntrt;

import com.atlassian.jira.util.json.JSONException;
import com.codeborne.selenide.Configuration;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.Resources;
import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.chrome.ChromeDriverService;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.slf4j.Logger;

import java.io.File;

import static com.codeborne.selenide.WebDriverRunner.getWebDriver;
import static com.codeborne.selenide.WebDriverRunner.setWebDriver;
import static java.lang.String.format;
import static java.util.logging.Level.*;

@SuppressWarnings("WeakerAccess")
public class SelenideTests {

    private static ChromeDriverService chromeDriverService;

    static void initDriverService() throws Exception {
        chromeDriverService = new ChromeDriverService.Builder()
                .usingDriverExecutable(new File("/opt/chromedriver"))
                .withEnvironment(ImmutableMap.of("DISPLAY", ":99"))
                .usingAnyFreePort()
                .build();
        chromeDriverService.start();
    }

    public static void startChromeDriver() throws Exception {
        Configuration.reportsFolder = "target/selenide-reports";

        File modHeader = new File(Resources.getResource("ModHeader_2_1_2.crx").toURI());
        File testClassesDir = modHeader.getParentFile();
        File userDataDir = new File(testClassesDir, "user-data-dir");

        ChromeOptions options = new ChromeOptions();
        options.addExtensions(modHeader);
        options.addArguments(
                "--disable-gpu",
                "--window-size=1600x900",
                "--ignore-certificate-errors",
                "--disable-cache",
                "--host-resolver-rules=MAP *.va.gov localhost",
                "--user-data-dir=" + userDataDir.getAbsolutePath());

        DesiredCapabilities capabilities = DesiredCapabilities.chrome();
        capabilities.setCapability(ChromeOptions.CAPABILITY, options);

        setWebDriver(new RemoteWebDriver(chromeDriverService.getUrl(), capabilities));
    }

    public static void stopChromeDriver() {
        getWebDriver().quit();
    }

    public static void stopDriverService() {
        if (chromeDriverService != null)
            chromeDriverService.stop();
    }

    public static void setModHeaderProfiles(String profiles) throws JSONException {
        WebDriver driver = getWebDriver();
        driver.get("chrome-extension://idgpnmonknjnojddfkpgkljpfnnfcklj/icon_128.png");
        ((JavascriptExecutor) driver).executeScript(
                format("localStorage.setItem('profiles', JSON.stringify(%s));", profiles));
    }

    public static void dumpSeleniumLogs(Logger logger, String logType) {
        String messageFormat = logType.toUpperCase() + " {}";

        getWebDriver().manage().logs().get(logType).forEach(entry -> {
            String message = entry.getMessage();

            if (WARNING == entry.getLevel())
                logger.warn(messageFormat, message);

            else if (SEVERE == entry.getLevel())
                logger.error(messageFormat, message);

            else if (INFO == entry.getLevel())
                logger.info(messageFormat, message);

            else
                logger.debug(messageFormat, message);
        });
    }
}
