package gov.vha.ctt.ntrt;

import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.jira.bc.projectroles.ProjectRoleService;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.exception.CreateException;
import com.atlassian.jira.exception.PermissionException;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectManager;
import com.atlassian.jira.security.groups.GroupManager;
import com.atlassian.jira.security.login.JiraSeraphAuthenticator;
import com.atlassian.jira.security.roles.ProjectRole;
import com.atlassian.jira.security.roles.ProjectRoleManager;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.user.UserDetails;
import com.atlassian.jira.user.UserUtils;
import com.atlassian.jira.util.SimpleErrorCollection;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.security.Principal;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.atlassian.jira.component.ComponentAccessor.getComponent;
import static com.atlassian.jira.security.roles.ProjectRoleActor.USER_ROLE_ACTOR_TYPE;
import static gov.vha.ctt.ntrt.SsoiAuthenticator.Role.*;
import static java.util.Collections.list;
import static org.apache.commons.lang3.StringUtils.*;

@SuppressWarnings("WeakerAccess")
public class SsoiAuthenticator extends JiraSeraphAuthenticator {

    private static final Logger LOG = LoggerFactory.getLogger(SsoiAuthenticator.class);

    /*
        Example of actual header names:

        [host, accept, referer, accept-language, user-agent, accept-encoding, dnt, cookie, sm_transactionid, sm_sdomain, sm_realm, sm_realmoid, sm_authtype, sm_authreason, sm_universalid, sm_authdiroid, sm_authdirname, sm_authdirserver, sm_authdirnamespace, sm_user, sm_userdn, sm_serversessionid, sm_serversessionspec, sm_timetoexpire, sm_serveridentityspec, adsamaccountname, sessionscope, vauid, accessroles, dodedipnid, issueinstant, firstname, ademail, ssoi-loggedout-url, ssoi-landing-url, organization, assurlevel, proofingauth, role, authntype, organizationid, adupn, transactionid, addomain, lastname, secid, prisme-roles, x-forwarded-for, x-forwarded-host, x-forwarded-server, connection]
     */

    // header field names
    static final String HEADER_PRISME_ROLES = "prisme-roles";
    static final String HEADER_ADEMAIL = "ademail";
    static final String HEADER_FIRSTNAME = "firstname";
    static final String HEADER_LASTNAME = "lastname";
    static final String HEADER_ADSAMACCOUNTNAME = "adsamaccountname";

    static final String JIRA_GROUP_SERVICEDESK_USERS = "jira-servicedesk-users";

    static final String JIRA_ROLE_ADMIN = "Administrators";
    static final String JIRA_ROLE_TEAM = "Service Desk Team";
    static final String JIRA_ROLE_CUSTOMER = "Service Desk Customers";

    static final String JIRA_PROJECT_NAME_NTRT = "NTRT";
    static final String JIRA_USER_NAME_ADMIN = "admin";

    static final String LOG_MESSAGE_MISSING_HEADER = "Unable to find header '{}' for session {}, found header names: {}";
    static final String LOG_MESSAGE_COULD_NOT_REMOVE_PROJECT_ROLE = "Could not remove user: {} from role: {} on project {} with reasons: {} and messages: {}";
    static final String LOG_MESSAGE_GET_PROJECT_ROLE_ERROR = "Could not retrieve project role: '{}' with reasons: {} and messages: {}";

    static final String NTRT_AUTHENTICATOR_MISSING_HEADERS_LOGGED_KEY = "ntrt.authenticator.missing.headers.logged";
    static final String NTRT_SSOI_AUTH_ROLES_KEY = "ntrt.ssoi.auth.roles";

    private static final Pattern NTRT_ROLE_PATTERN = Pattern.compile("(ntrt_[a-z]+)");
    private static final Pattern EMAIL_NAME_PATTERN = Pattern.compile("([a-zA-Z]+)\\.([a-zA-Z]+)@.*");

    static final Map<Role, String> PROJECT_ROLES = ImmutableMap.of(
            NTRT_ADMIN, JIRA_ROLE_ADMIN,
            NTRT_STAFF, JIRA_ROLE_TEAM,
            NTRT_USER, JIRA_ROLE_CUSTOMER
    );

    enum Role {
        NTRT_ADMIN, NTRT_STAFF, NTRT_USER
    }

    /**
     * Here we inspect all incoming requests for header fields added to the request by the reverse proxy.
     * One header contains the PRISME roles for the user. If the user has an 'ntrt_' value in the PRISME roles
     * header field, and the user has an existing account matching the provided account name, we return a Principal
     * for that user. If the user does not exist, we create one. If we have insufficient information to create a
     * user (I'm looking at you, SSOi) we return null.
     */
    @Override
    public Principal getUser(HttpServletRequest request, HttpServletResponse response) {

        String accountName = getHeader(request, HEADER_ADSAMACCOUNTNAME);

        if (isBlank(accountName))
            return super.getUser(request, response);

        EnumSet<Role> ntrtRoles = findNtrtRoles(request);

        if (!validateRoles(ntrtRoles))
            return null;

        ApplicationUser applicationUser = UserUtils.getUser(accountName);

        if (applicationUser == null) {
            applicationUser = createNewUser(request, accountName);

            if (!validateNewUser(applicationUser, accountName))
                return null;
        }

        Set<String> cachedRoles = getCachedRoles(request.getSession());

        if (cachedRoles == null) {

            // no roles are cached, so we attempt to assign them, bailing if that fails
            if (!assignUserToProjectRoles(applicationUser, ntrtRoles))
                return null;

            cacheProjectRoles(applicationUser, request.getSession());
        }

        // if cached roles are valid, we proceed
        // if not, and assigning roles fails, we bail
        else if (!validateCachedRoles(ntrtRoles, cachedRoles) &&
                !assignUserToProjectRoles(applicationUser, ntrtRoles)) {
            return null;
        }

        // ApplicationUser implements Principal, so we can return it directly
        return applicationUser;
    }

    @VisibleForTesting
    protected boolean validateCachedRoles(EnumSet<Role> ntrtRoles, Set<String> cachedRoles) {
        Set<String> expectedRoles = new HashSet<>(3);
        ntrtRoles.forEach(role -> expectedRoles.add(PROJECT_ROLES.get(role)));
        return cachedRoles.equals(expectedRoles);
    }

    @VisibleForTesting
    @SuppressWarnings("unchecked")
    protected Set<String> getCachedRoles(HttpSession session) {
        return (Set<String>) session.getAttribute(NTRT_SSOI_AUTH_ROLES_KEY);
    }

    @VisibleForTesting
    protected void cacheProjectRoles(ApplicationUser applicationUser, HttpSession session) {
        Set<String> cachedRoles = new HashSet<>(3);

        for (ProjectRole projectRole : getProjectRoles(applicationUser)) {
            cachedRoles.add(projectRole.getName());
        }

        // don't cache the empty set
        if (cachedRoles.size() > 0)
            session.setAttribute(NTRT_SSOI_AUTH_ROLES_KEY, cachedRoles);
    }

    @VisibleForTesting
    protected ApplicationUser createNewUser(HttpServletRequest request, String accountName) {
        String emailAddress = getHeader(request, HEADER_ADEMAIL);

        if (!validateEmailAddress(emailAddress))
            return null;

        // create the new user
        UserDetails userDetails = new UserDetails(accountName, getDisplayName(request))
                .withEmail(emailAddress)
                .withPassword(UUID.randomUUID().toString());

        return createUser(userDetails);
    }

    @SuppressWarnings("RedundantIfStatement")
    @VisibleForTesting
    protected boolean assignUserToProjectRoles(ApplicationUser applicationUser, Set<Role> ntrtRoles) {

        // each user must be a member of the JSD user's groups
        if (!addUserToGroup(applicationUser, JIRA_GROUP_SERVICEDESK_USERS)) {
            return false;
        }

        Collection<ProjectRole> projectRoles = getProjectRoles(applicationUser);

        if (!updateProjectRoles(applicationUser, NTRT_ADMIN, getProjectRole(JIRA_ROLE_ADMIN), ntrtRoles, projectRoles))
            return false;

        if (!updateProjectRoles(applicationUser, NTRT_STAFF, getProjectRole(JIRA_ROLE_TEAM), ntrtRoles, projectRoles))
            return false;

        if (!updateProjectRoles(applicationUser, NTRT_USER, getProjectRole(JIRA_ROLE_CUSTOMER), ntrtRoles, projectRoles))
            return false;

        return true;
    }

    private Collection<ProjectRole> getProjectRoles(ApplicationUser applicationUser) {
        ProjectRoleManager projectRoleManager = getComponent(ProjectRoleManager.class);
        return projectRoleManager.getProjectRoles(applicationUser, getNtrtProject());
    }

    private boolean updateProjectRoles(ApplicationUser applicationUser, Role ntrtRole, ProjectRole projectRole, Set<Role> ntrtRoles, Collection<ProjectRole> projectRoles) {

        if (ntrtRoles.contains(ntrtRole)) {

            /*
                If user has the NTRT role, and they don't have the project role, try assigning them the project role.
                If that fails, return false.
             */
            if (!projectRoles.contains(projectRole) && !addUserToProjectRole(applicationUser, projectRole))
                return false;

        } else if (projectRoles.contains(projectRole)) {

            /*
                If the user does not have the NTRT role but they do have the project role, remove it.
             */
            removeUserFromProjectRole(applicationUser, projectRole);
        }

        return true;
    }

    @SuppressWarnings("deprecation")
    @VisibleForTesting
    protected void removeUserFromProjectRole(ApplicationUser applicationUser, ProjectRole projectRole) {
        Project project = getNtrtProject();
        SimpleErrorCollection errors = new SimpleErrorCollection();

        getComponent(ProjectRoleService.class).removeActorsFromProjectRole(
                getAdminUser(),
                userKeyToSet(applicationUser),
                projectRole,
                project,
                USER_ROLE_ACTOR_TYPE,
                errors);

        if (errors.hasAnyErrors()) {
            LOG.error(LOG_MESSAGE_COULD_NOT_REMOVE_PROJECT_ROLE,
                    applicationUser, projectRole, project, errors.getReasons(), errors.getErrorMessages());
        }
    }

    @SuppressWarnings("deprecation")
    private boolean addUserToProjectRole(ApplicationUser applicationUser, ProjectRole projectRole) {

        Project project = getNtrtProject();
        SimpleErrorCollection errors = new SimpleErrorCollection();

        getComponent(ProjectRoleService.class).addActorsToProjectRole(
                getAdminUser(),
                userKeyToSet(applicationUser),
                projectRole,
                project,
                USER_ROLE_ACTOR_TYPE,
                errors);

        if (errors.hasAnyErrors()) {
            LOG.error("Could not add user: {} to role: {} on project {} with reasons: {} and messages: {}",
                    applicationUser.getUsername(), projectRole, project, errors.getReasons(), errors.getErrorMessages());
            return false;
        }

        return true;
    }

    private Project getNtrtProject() {
        return getComponent(ProjectManager.class).getProjectByCurrentKey(JIRA_PROJECT_NAME_NTRT);
    }

    @VisibleForTesting
    protected ProjectRole getProjectRole(String jiraRoleName) {
        ProjectRoleService projectRoleService = getComponent(ProjectRoleService.class);

        SimpleErrorCollection errors = new SimpleErrorCollection();
        ProjectRole projectRole = projectRoleService.getProjectRoleByName(jiraRoleName, errors);

        if (errors.hasAnyErrors())
            LOG.error(LOG_MESSAGE_GET_PROJECT_ROLE_ERROR, jiraRoleName, errors.getReasons(), errors.getErrorMessages());

        return projectRole;
    }

    @VisibleForTesting
    protected boolean addUserToGroup(ApplicationUser applicationUser, String group) {
        GroupManager groupManager = ComponentAccessor.getGroupManager();
        Group serviceDeskUsersGroup = groupManager.getGroup(group);

        boolean success = true;

        if (!groupManager.isUserInGroup(applicationUser, group)) {
            try {
                groupManager.addUserToGroup(applicationUser, serviceDeskUsersGroup);
            } catch (Exception e) {
                LOG.error("Unable to add user '{}' to group '{}'", applicationUser, group, e);
                success = false;
            }
        }

        return success;
    }

    private ApplicationUser createUser(UserDetails userDetails) {
        ApplicationUser applicationUser = null;

        try {
            applicationUser = ComponentAccessor.getUserManager().createUser(userDetails);
        } catch (CreateException | PermissionException e) {
            LOG.error("Failed to create user with details: ", userDetails, e);
        }

        return applicationUser;
    }

    private boolean validateNewUser(ApplicationUser user, String accountName) {
        boolean isValid = true;
        if (user == null) {
            LOG.error("Unable to create JIRA user for SSOi user {}", accountName);
            isValid = false;
        }
        return isValid;
    }

    private boolean validateEmailAddress(String emailAddress) {
        boolean isValid = isNotBlank(emailAddress);
        if (!isValid)
            LOG.error("SSOi email address header ({}) is empty, cannot create new user.", HEADER_ADEMAIL);
        return isValid;
    }

    private boolean validateRoles(EnumSet<Role> ntrtRoles) {
        return ntrtRoles.size() > 0;
    }

    @VisibleForTesting
    protected String getHeader(HttpServletRequest request, String headerName) {

        String value = request.getHeader(headerName);

        // try upper case if not found
        if (value == null)
            value = request.getHeader(headerName.toUpperCase());

        // log each missed header once per session
        if (value == null) {

            @SuppressWarnings("unchecked")
            List<String> missing = (List<String>) request.getSession().getAttribute(NTRT_AUTHENTICATOR_MISSING_HEADERS_LOGGED_KEY);

            if (missing == null) {
                missing = new ArrayList<>(5);
                request.getSession().setAttribute(NTRT_AUTHENTICATOR_MISSING_HEADERS_LOGGED_KEY, missing);
            }

            if (missing.add(headerName)) {
                LOG.debug(LOG_MESSAGE_MISSING_HEADER,
                        headerName,
                        request.getSession().getId(),
                        list(request.getHeaderNames()));
            }
        }

        return value;
    }

    /**
     * Extract the roles matching the ntrt_* pattern in the roles getHeader
     */
    @VisibleForTesting
    protected EnumSet<Role> findNtrtRoles(HttpServletRequest request) {
        String rolesHeader = getHeader(request, HEADER_PRISME_ROLES);
        EnumSet<Role> roles = EnumSet.noneOf(SsoiAuthenticator.Role.class);

        if (isNotBlank(rolesHeader)) {
            Matcher matcher = NTRT_ROLE_PATTERN.matcher(rolesHeader);

            while (matcher.find()) {
                for (int i = 0; i < matcher.groupCount(); i++) {
                    roles.add(Role.valueOf(matcher.group(i).toUpperCase()));
                }
            }
        }

        return roles;
    }

    /**
     * Determine the display name using the first/last headers if present. If not present for either,
     * extract them from the email address if it is in the format first.last@dom.tld.
     */
    @VisibleForTesting
    protected String getDisplayName(HttpServletRequest request) {

        String first = getHeader(request, HEADER_FIRSTNAME);
        String last = getHeader(request, HEADER_LASTNAME);

        // if either header is empty, we'll attempt to extract the name from the email address
        if (isBlank(first) || isBlank(last)) {

            // by time we get here, we can be sure there's an email address
            String email = getHeader(request, HEADER_ADEMAIL);
            Matcher matcher = EMAIL_NAME_PATTERN.matcher(email);

            while (matcher.find()) {
                if (matcher.groupCount() == 2) {

                    if (isBlank(first))
                        first = matcher.group(1);

                    if (isBlank(last))
                        last = matcher.group(2);
                }
            }
        }

        // ensure the names are in capital case
        if (isNotBlank(first))
            first = capitalize(first.toLowerCase());

        if (isNotBlank(last))
            last = capitalize(last.toLowerCase());

        // fall back to account name if both first and last are blank
        if (isBlank(first) && isBlank(last))
            return request.getHeader(HEADER_ADSAMACCOUNTNAME);
        else
            return first + " " + last;
    }

    @VisibleForTesting
    protected ApplicationUser getAdminUser() {
        ApplicationUser user = UserUtils.getUser(JIRA_USER_NAME_ADMIN);

        if (user == null) {
            throw new IllegalArgumentException("The admin user '" + JIRA_USER_NAME_ADMIN + "' does not exist");
        }

        return user;
    }

    private Set<String> userKeyToSet(ApplicationUser applicationUser) {
        return Collections.singleton(applicationUser.getKey());
    }

}
