/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.isaac.ochre.impl.utility;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import gov.vha.isaac.ochre.api.util.Interval;

/**
 * Processes command-line arguments supplied to a Java application's
 * <code>main</code> method.
 * <p>
 * Provides methods for specifying which flags and variables are permitted as
 * arguments to the main method of an application, and methods for processing
 * supplied arguments to check for compliance and retrieve variable values and
 * determine the presence of flags.</p>
 * <p>
 * To use <code>MainArgsHandler</code> follow these steps in order:</p>
 * <ol>
 * <li>Acquire the sole instance of <code>MainArgsHandler</code> by calling the
 * static method {@link #getHandler()}.</li>
 * <li>Specify which flag names, if any, are permitted in the command-line
 * arguments for your application by calling the
 * {@link #permitFlag(String, String)} method for each flag.</li>
 * <li>Optionally specify that the character case of flag and variable arguments in the command-line
 * arguments for your application be ignored or enforced by calling the
 * {@link #ignoreCase(boolean)} method.</li>
 * <li>Optionally specify that the default Interval for variable arguments in the command-line
 * arguments for your application by calling the
 * {@link #setDefaultInterval(Interval)} method.</li>
 * <li>Specify which variables, if any, are permitted in the command-line
 * arguments, how many times they must or may appear, and a usage description by calling the
 * {@link #permitVariable(String, Interval, String)} for each variable.</li>
 * <li>Specify which flags, if any, are permitted in the command-line
 * arguments and a usage description by calling the
 * {@link #permitFlag(String, String)} for each variable.</li>
 * <li>Specify which flags and/or arguments are incompatible (mutually exclusive)
 * by calling the
 * {@link #addIncompatibleFlagsOrVariables(String...)} for each set of incompatible values.</li>
 * <li>Specify which flags and/or arguments are dependent (require each other)
 * by calling the
 * {@link #addFlagOrVariableDependency(String...)} for each set of dependent values.</li>
 * <li>Call the {@link #processMainArgs(String[])} method to process the String
 * array received by your application's <code>main(String[])</code> method. Be
 * prepared for an <code>IllegalArgumentException</code> to be thrown if the
 * user calls your application with unrecognised command-line flags or variables
 * or incorrect quantities of permitted variables.</li>
 * <li>Check for the presence of a permitted flag in the command-line arguments
 * by calling {@link #foundFlag(String)}.</li>
 * <li>Check for the presence of a permitted variable in the command-line
 * arguments by calling {@link #foundVariable(String)}.</li>
 * <li>If a variable is found on the command-line, get a
 * <code>List&lt;String&gt;</code> of its values by calling
 * {@link #getValuesFromVariable(String)} for that variable name.</li>
 * <li>If a variable is found on the command-line, get a
 * distinct <code>Set&lt;String&gt;</code> of its values by calling
 * {@link #getDistinctValuesFromVariable(String)} for that variable name.</li>
 * </ol>
 * <p>
 * Each command-line argument can be either a <dfn>flag</dfn> (which has no
 * value) or a <dfn>variable</dfn> (which does have a value).</p>
 * <p>
 * A command-line flag has the form <kbd>--flag-name</kbd> (two hyphens then the
 * flag-name) where the flag-name begins with lowercase a-z and can then contain
 * lowercase and/or uppercase a-zA-Z sequences separated by single hyphens (consecutive hyphens
 * cannot appear within flag names, so <kbd>--flag--name</kbd> would be illegal
 * for instance).</p>
 * <p>
 * A command-line variable may have the form <kbd>--variable-name=value</kbd> (two
 * hyphens then the variable-name then equals-sign then value) where the
 * variable-name must follow the same rules as a flag-name. 
 * A command-line variable may also have the form <kbd>--variable-name value</kbd> (two
 * hyphens then the variable-name then space then value) where the
 * variable-name must follow the same rules as a flag-name. 
 * The value can
 * contain any characters (though this can be validated by specifying a
 * <code>Predicate&lt;String&gt;</code> in the call to <code>permitVariable</code>). (Bear in
 * mind that to pass Java a value which contains spaces the command-line
 * argument must wrap the value in double-quotes, e.g.
 * <kbd>--window-title="Nightly Build v1.740"</kbd> and also note that the
 * double-quotes are automatically stripped away by Java before they reach the
 * main method.)</p>
 * <p>
 * Note that the initial double-hyphen is only used on the command-line, and
 * does not form part of the flag or variable name. The initial double-hyphen
 * must not be included when calling methods such as {@link #permitFlag(String)}
 * or {@link #getValuesFromVariable(String)}.</p>
 * <p>
 * If a flag or variable is encountered which does not fit the permitted syntax
 * then an <code>IllegalArgumentException</code> will be thrown.</p>
 * <p>
 * It is permitted to have a command-line call which provides the same variable
 * name more than once, to provide multiple values for that variable. For
 * instance the following command-line arguments are permitted in a single call:
 * <kbd>--path=/usr/bin --path=/usr/local/bin</kbd> and both of these values
 * will be stored in the map for this variable name "path".</p>
 * <p>
 * It is not possible to use the same name for both a flag and a variable.</p>
 *
 * @author Bobulous
 */
public final class MainArgsHandler {

	/**
	 * Keeps track of the state of this MainArgsHandler object. Most public
	 * methods require the object to be in a certain state in order to run, and
	 * calling them at the wrong time will lead to an IllegalStateException
	 * being thrown.
	 */
	private static enum Status {
		/**
		 * Pre-pre-processing state, before any flags or variables have been permitted.
		 */
		HANDLER_CONFIGURATION,
		/**
		 * Pre-processing state, before processMainArgs has been called.
		 */
		PREPRO,
		/**
		 * Processing state, during processMainArgs method execution.
		 */
		PROCESSING,
		/**
		 * Post-processing state, after processMainArgs has finished
		 * successfully.
		 */
		POPRO
	}

	/**
	 * Represents a flag permitted as a command-line argument.
	 */
	private class Flag {

		/**
		 * The name of this command-line flag.
		 */
		private String name;
		/**
		 * The description intended to advise the application user about this
		 * command-line flag and the effect it has on the application.
		 */
		String usageDescription;
		/**
		 * Will be true if this flag should not be included in the list of
		 * command-line flags supported by the Java application, false
		 * otherwise.
		 */
		boolean forTestUseOnly;

		public Flag(String name, String usageDescription, boolean testUseOnly) {
			this.name = name;
			this.usageDescription = usageDescription;
			this.forTestUseOnly = testUseOnly;
		}

		@Override
		public boolean equals(Object obj) {
			if (!(obj instanceof Flag)) {
				return false;
			}
			Flag that = (Flag) obj;
			return getBasedOnConfiguredCaseSignificance(this.name).equals(getBasedOnConfiguredCaseSignificance(that.name));
		}

		@Override
		public int hashCode() {
			int hash = 7;
			hash = 47 * hash + (getBasedOnConfiguredCaseSignificance(this.name) != null ? getBasedOnConfiguredCaseSignificance(this.name).hashCode() : 0);
			return hash;
		}
	}

	/**
	 * Represents a variable permitted as a command-line argument.
	 */
	private class Variable {

		/**
		 * The name of this command-line variable.
		 */
		String name;
		/**
		 * The number of times this variable must appear on the command-line.
		 */
		Interval permittedCountInterval;
		/**
		 * The description intended to advise the application user about this
		 * command-line variable and the sort of values it can take.
		 */
		String usageDescription;
		/**
		 * A Predicate<String> test which validates this as a valid value for this command-line variable
		 * 
		 * Examples:
		 * 	(arg) -> Pattern.compile("([a-z](?:-[a-zA-Z]|[a-zA-Z])*)").matches(arg)
		 * 	(arg) -> Level.getLevel(arg) != null
		 * 	(arg) -> State.getFromString(arg).isPresent()
		 * 	(arg) -> NumericUtils.getNID(arg).isPresent() || UUIDUtil.isUUID(arg)
		 */
		Predicate<String> validationTest;
		/**
		 * Will be true if this variable should not be included in the list of
		 * command-line variables supported by the Java application, false
		 * otherwise.
		 */
		boolean forTestUseOnly;

		public Variable(String name,
				Interval permittedCountInterval,
				String usageDescription,
				Predicate<String> validationTest,
				boolean testUseOnly) {
			this.name = name;
			this.permittedCountInterval = permittedCountInterval;
			this.usageDescription = usageDescription;
			this.validationTest = validationTest;
			this.forTestUseOnly = testUseOnly;
		}

		/* The command-line variable name is the primary and sole key, so this
		 equals method considers only the variable name when judging equality.
		 */
		@Override
		public boolean equals(Object obj) {
			if (!(obj instanceof Variable)) {
				return false;
			}
			Variable that = (Variable) obj;
			return getBasedOnConfiguredCaseSignificance(this.name).equals(getBasedOnConfiguredCaseSignificance(that.name));
		}

		/* The command-line variable name is the primary and sole key, so this
		 hashCode method conisders only the variable name when calculating a hash
		 value.
		 */
		@Override
		public int hashCode() {
			int hash = 7;
			hash = 83 * hash + (getBasedOnConfiguredCaseSignificance(this.name) != null ? getBasedOnConfiguredCaseSignificance(this.name).hashCode() : 0);
			return hash;
		}
	}

	/**
	 * The sole instance of this class.
	 */
	private static MainArgsHandler singletonInstance;

	/**
	 * Optional application name for display in usage message
	 */
	private String appName = "APP";
	
	/**
	 * The current Status of this MainArgsHandler.
	 */
	private Status objectStatus;
	/**
	 * The set of flags which the client has declared acceptable.
	 */
	private final Map<String, Flag> permittedFlags;
	/**
	 * The set of variables which the client has declared acceptable.
	 */
	private final Map<String, Variable> permittedVariables;
	/**
	 * The set of flags found in the command-line arguments (supplied by the
	 * user).
	 */
	private Set<String> commandLineFlagSet;
	/**
	 * The set of variables found in the command-line arguments (supplied by the
	 * user).
	 */
	private Map<String, List<String>> commandLineVariableValues;
	
	/**
	 * Default interval specifying range of allowed argument instances.
	 * Current default is 0 or 1.
	 */
	private Interval defaultInterval = new Interval("[0,1]");

	/**
	 * Determines whether character case should be ignored in identifying command line flags and variables
	 * Default is true
	 */
	private boolean ignoreCase = true;
	
	/**
	 * String prefix that indicates that a flag or variable argument (and its respective value, if a variable) should be ignored
	 */
	private String argumentCommentPrefix = "#";
	
	/**
	 * Prefix indicating that a command-line argument is a flag or variable
	 */
	private String flagAndVariablePrefix = "-";
	
	/**
	 * Specify whether or not character case should be ignored in identifying command line flags and variables
	 * 
	 * @param ignoreCase
	 */
	public void setIgnoreCase(boolean ignoreCase) {
		if (this.objectStatus != Status.HANDLER_CONFIGURATION) {
			throw new IllegalStateException("Cannot set case significance "
					+ "once flags or variables have been permitted or processing has begun.");
		}
		this.ignoreCase = ignoreCase;
	}

	/**
	 * Specify whether or not character case should be ignored in identifying command line flags and variables
	 * 
	 * @param ignoreCase
	 */
	public void setAppName(String appName) {
		if (this.objectStatus != Status.HANDLER_CONFIGURATION) {
			throw new IllegalStateException("Cannot set application name "
					+ "once flags or variables have been permitted or processing has begun.");
		}
		this.appName = appName;
	}
	private String getBasedOnConfiguredCaseSignificance(String input) {
		if (! ignoreCase) {
			return input;
		} else {
			return input.toLowerCase(Locale.ENGLISH);
		}
	}

	/**
	 * Get the default interval specifying range of allowed argument instances.
	 * 
	 * @return
	 */
	public Interval getDefaultInterval() { return defaultInterval; }

	/**
	 * Set the default argument comment prefix which indicates that a flag or variable argument
	 * (and its respective value, if a variable) should be ignored.
	 * Default value is "#".
	 * 
	 * @param str
	 */
	public void setArgumentCommentPrefix(String str) {
		if (this.objectStatus != Status.HANDLER_CONFIGURATION) {
			throw new IllegalStateException("Cannot set default interval "
					+ "once flags or variables have been permitted or processing has begun.");
		}
		argumentCommentPrefix = str;
	}
	
	public String getFlagAndVariablePrefix() {
		return flagAndVariablePrefix;
	}
	/**
	 * Set prefix indicating that a command-line argument is a flag or variable
	 * 
	 * @param str
	 */
	public void setFlagAndVariablePrefix(String str) {
		if (this.objectStatus != Status.HANDLER_CONFIGURATION) {
			throw new IllegalStateException("Cannot set flag and variable prefix "
					+ "once flags or variables have been permitted or processing has begun.");
		}
		flagAndVariablePrefix = str;
	}
	/**
	 * Set the default interval specifying range of allowed argument instances.
	 * This method may only be called prior to processing of arguments (when objectStatus == Status.PREPRO).
	 * 
	 * @param interval
	 */
	public void setDefaultInterval(Interval interval) {
		if (this.objectStatus != Status.HANDLER_CONFIGURATION) {
			throw new IllegalStateException("Cannot set default interval "
					+ "once flags or variables have been permitted or processing has begun.");
		}
		defaultInterval = interval;
	}

	private final Set<Set<String>> dependentFlagsAndVariables = new HashSet<>();

	private final Set<Set<String>> incompatibleFlagsAndVariables = new HashSet<>();
	
	private final Set<Set<String>> minimallySufficientFlagAndVariableSets = new HashSet<>();

	/**
	 * Add a set of interdependent flags and/or variables for which, if any of the contained flags and/or variables are found
	 * on the command line, then all other flags and/or variables in the set must also be found on the command line.
	 * 
	 * @param depedentVariableOrFlagNames
	 */
	public void addFlagOrVariableDependency(String... depedentVariableOrFlagNames) {
		if (this.objectStatus != Status.PREPRO) {
			throw new IllegalStateException("Cannot add to the set of "
					+ "dependent names once processing of main arguments "
					+ "has already begun.");
		}
		for (String depedentVariableOrFlagName : depedentVariableOrFlagNames) {
			if (! contains(this.permittedVariables.keySet(), depedentVariableOrFlagName)
					&& ! contains(this.permittedFlags.keySet(), depedentVariableOrFlagName)) {
				throw new IllegalArgumentException("Cannot add dependency related to flag or variable with name \""
						+ depedentVariableOrFlagName + "\" because this is not an existing permitted flag or variable name.");
			}
		}
		TreeSet<String> dependentVariableOrFlagNamesSet = new TreeSet<>();
		for (String depedentVariableOrFlagName : depedentVariableOrFlagNames) {
			dependentVariableOrFlagNamesSet.add(depedentVariableOrFlagName);
		}

		dependentFlagsAndVariables.add(Collections.unmodifiableSortedSet(dependentVariableOrFlagNamesSet));
	}

	/**
	 * Add a set of incompatible flags and/or variables for which, if any of the contained flags and/or variables are found
	 * on the command line, then none of the other flags and/or variables in the set may be found on the command line.
	 * 
	 * @param incompatibleVariableOrFlagNames
	 */
	public void addIncompatibleFlagsOrVariables(String... incompatibleVariableOrFlagNames) {
		if (this.objectStatus != Status.PREPRO) {
			throw new IllegalStateException("Cannot add to the set of "
					+ "incompatible names once processing of main arguments "
					+ "has already begun.");
		}
		for (String incompatibleVariableOrFlagName : incompatibleVariableOrFlagNames) {
			if (! contains(this.permittedVariables.keySet(), incompatibleVariableOrFlagName)
					&& ! contains(this.permittedFlags.keySet(), incompatibleVariableOrFlagName)) {
				throw new IllegalArgumentException("Cannot add incompatible related to flag or variable with name \""
						+ incompatibleVariableOrFlagName + "\" because this is not an existing permitted flag or variable name.");
			}
		}
		TreeSet<String> incompatibleVariableOrFlagNamesSet = new TreeSet<>();
		for (String incompatibleVariableOrFlagName : incompatibleVariableOrFlagNames) {
			incompatibleVariableOrFlagNamesSet.add(incompatibleVariableOrFlagName);
		}

		incompatibleFlagsAndVariables.add(Collections.unmodifiableSortedSet(incompatibleVariableOrFlagNamesSet));
	}

	public void addMinimallySufficientFlagOrVariableSet(String... minimallySufficientFlagOrVariableNameSetValues) {
		if (this.objectStatus != Status.PREPRO) {
			throw new IllegalStateException("Cannot add to the set of "
					+ "minimally sufficient names once processing of main arguments "
					+ "has already begun.");
		}
		for (String minimallySufficientFlagOrVariableNameSetValue : minimallySufficientFlagOrVariableNameSetValues) {
			if (! contains(this.permittedVariables.keySet(), minimallySufficientFlagOrVariableNameSetValue)
					&& ! contains(this.permittedFlags.keySet(), minimallySufficientFlagOrVariableNameSetValue)) {
				throw new IllegalArgumentException("Cannot add minimally sufficient flag or variable set value related to flag or variable with name \""
						+ minimallySufficientFlagOrVariableNameSetValue + "\" because this is not an existing permitted flag or variable name.");
			}
		}
		TreeSet<String> minimallySufficientFlagOrVariableNameSet = new TreeSet<>();
		for (String minimallySufficientFlagOrVariableNameSetValue : minimallySufficientFlagOrVariableNameSetValues) {
			minimallySufficientFlagOrVariableNameSet.add(minimallySufficientFlagOrVariableNameSetValue);
		}

		this.minimallySufficientFlagAndVariableSets.add(Collections.unmodifiableSortedSet(minimallySufficientFlagOrVariableNameSet));
	}

	// For reasons beyond me, this first pattern was working absolutely fine in
	// over eighty tests until I added test case
	// "quickbrownfoxjumpsoverthelazyfoxZ" (notice the capital Z at the end
	// which causes the String to become an invalid name. Suddenly the Java
	// Matcher method matcher() was hanging every time it reached this test
	// even though it was fine on every other case, including other simple a-z
	// strings which ended with "Z". Very weird. The alternative regex seems
	// to work fine in all cases, but who knows?
	//	private static final String NAME_REGEX = "([a-z](?:-?[a-z]+)*)";
	/**
	 * The Pattern which defines a valid name for a flag or variable.
	 */
	private static final Pattern NAME_PATTERN = Pattern.compile(
			"([a-z](?:-[a-zA-Z]|[a-zA-Z])*)");

	private MainArgsHandler() {
		this.objectStatus = MainArgsHandler.Status.HANDLER_CONFIGURATION;
//		this.permittedFlagSet = new HashSet<String>();
//		this.permittedVariableNameCounts =
//				new HashMap<String, Interval<Integer>>();
		this.permittedFlags = new HashMap<>();
		this.permittedVariables = new HashMap<>();
	}

	/**
	 * Returns the singleton instance object of the MainArgsHandler class.
	 *
	 * @return the sole MainArgsHandler object.
	 */
	public static MainArgsHandler getHandler() {
		if (MainArgsHandler.singletonInstance == null) {
			MainArgsHandler.singletonInstance = new MainArgsHandler();
		}
		return MainArgsHandler.singletonInstance;
	}

	/**
	 * Package-private method for JUnit testing. Allows the singleton to be
	 * wiped away so that a new handler can be instantiated and tested in JUnit.
	 * Without this method being available to the test classes within this
	 * package, every JUnit test case would be working on the same handler,
	 * which would only allow one use before hurling exceptions like cluster
	 * bombs. Which would be undesirable.
	 */
	static void nukeHandler() {
		MainArgsHandler.singletonInstance = null;
	}

	/**
	 * Reports on whether the specified string would be accepted by
	 * MainArgsHandler as a valid name for a command-line flag or variable.
	 *
	 * @param name the string to test.
	 * @return true if the specified string would be a valid name for a flag or
	 * variable.
	 */
	private boolean validName(String name) {
		return NAME_PATTERN.matcher(name).matches();
	}

	/**
	 * Instruct this handler to accept a command-line flag with the specified
	 * name. The specified flag can occur zero or one times in the command-line
	 * arguments for them to be accepted as valid by this handler.
	 * <p>
	 * <strong>Note:</strong> do not include the initial double-hyphen in the
	 * flag name when calling <code>permitFlag</code>.</p>
	 * <p>
	 * A flag name must begin with lowercase a-z and can then contain lowercase
	 * and/or uppercase a-zA-Z terms separated by single hyphens. Consecutive hyphens cannot appear
	 * within flag names, so <kbd>flag--name</kbd> would be illegal, for
	 * example.</p>
	 *
	 * @param flagName the name of a flag permitted by this handler.
	 * @throws IllegalStateException if this method is called after the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if this method is provided with a flag
	 * name which does not fit the permitted pattern.
	 */
	public void permitFlag(String flagName) {
		this.permitFlag(flagName, null, false);
	}

	/**
	 * Instruct this handler to accept a command-line flag with the specified
	 * name and usage description text. The specified flag can occur zero or one
	 * times in the command-line arguments for them to be accepted as valid by
	 * this handler.
	 * <p>
	 * <strong>Note:</strong> do not include the initial double-hyphen in the
	 * flag name when calling <code>permitFlag</code>.</p>
	 * <p>
	 * A flag name must begin with lowercase a-z and can then contain lowercase
	 * a-z terms separated by single hyphens. Consecutive hyphens cannot appear
	 * within flag names, so <kbd>flag--name</kbd> would be illegal, for
	 * example.</p>
	 *
	 * @param flagName the name of a flag permitted by this handler.
	 * @param usageDescription a brief description of when to use this flag, for
	 * display to a user who needs help on calling the client application from
	 * the command-line.
	 * @throws IllegalStateException if this method is called after the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if this method is provided with a flag
	 * name which does not fit the permitted pattern.
	 */
	public void permitFlag(String flagName, String usageDescription) {
		this.permitFlag(flagName, usageDescription, false);
	}

	/**
	 * Instruct this handler to accept a command-line flag which is intended
	 * only for test purposes. This works in exactly the same way as the
	 * {@link #permitFlag(String, String)} method, but marks the flag as being
	 * for test use only, so that this flag is not displayed to end users if
	 * they are shown a list of permitted command-line flags. This method should
	 * be used to permit flags which should only ever be used by the developer
	 * of an application, and should not be commonly known to end users.
	 *
	 * @param flagName the name of the test flag to permit.
	 * @throws IllegalStateException if this method is called after the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if this method is provided with a flag
	 * name which does not fit the permitted pattern.
	 */
	public void permitTestFlag(String flagName) {
		this.permitFlag(flagName, null, true);
	}

	private void permitFlag(String flagName, String usageDescription,
			boolean testUseOnly) {
		if (this.objectStatus == Status.HANDLER_CONFIGURATION) {
			this.objectStatus = Status.PREPRO;
		}
		if (this.objectStatus != Status.PREPRO) {
			throw new IllegalStateException("Cannot add to the set of "
					+ "permitted flag names once processing of main arguments "
					+ "has already begun.");
		}
		if (!this.validName(flagName)) {
			throw new IllegalArgumentException("Cannot permit flag with name \""
					+ flagName + "\" because this name does not fit the "
					+ "accepted pattern.");
		}
		if (contains(this.permittedVariables.keySet(), flagName)
				|| contains(this.permittedFlags.keySet(), flagName)) {
			throw new IllegalArgumentException("Cannot permit flag with name \""
					+ flagName + "\" because this is already a permitted flag "
					+ "or a permitted or required variable name.");
		}
		Flag newFlag = new Flag(flagName, usageDescription, testUseOnly);
		this.permittedFlags.put(flagName, newFlag);
	}

	/**
	 * Instruct this handler to accept command-line variables with the specified
	 * name, in any quantity which is permitted by the specified interval. Any
	 * String is permitted for the variable value.
	 * <p>
	 * <strong>Note:</strong> do not include the initial double-hyphen in the
	 * variable name when calling <code>permitVariable</code>.</p>
	 * <p>
	 * A variable name must begin with lowercase a-z and can then contain
	 * lowercase and/or uppercase a-zA-Z terms separated by single hyphens. Consecutive hyphens
	 * cannot appear within variable names, so <kbd>variable--name</kbd> would
	 * be illegal, for example.</p>
	 *
	 * @param variableName the name of a variable permitted by this handler.
	 * @param permittedCountInterval an <code>Interval</code> of
	 * <code>Integer</code> values whose lower endpoint must be at least zero,
	 * and whose upper endpoint must be at least one. The lower endpoint cannot
	 * be <code>null</code> (equivalent to negative infinity) but the upper
	 * endpoint can be <code>null</code> (equivalent to positive infinity).
	 * @throws IllegalStateException if this method is called after the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if this method is provided with a
	 * variable name which does not fit the permitted pattern, or if the
	 * provided <code>Interval</code> has a <code>null</code> or negative value
	 * for its lower endpoint, or a value of less than one for its upper
	 * endpoint.
	 */
	public void permitVariable(String variableName,
			Interval permittedCountInterval) {
		this.permitVariable(variableName, permittedCountInterval, null, null,
				false);
	}
	public void permitVariable(String variableName) {
		this.permitVariable(variableName, getDefaultInterval(), null, null,
				false);
	}

	/**
	 * Instruct this handler to accept command-line variables with the specified
	 * name, in any quantity permitted by the specified interval, and attach the
	 * provided usage description text. Any String is accepted as the value of
	 * the variable.
	 * <p>
	 * <strong>Note:</strong> do not include the initial double-hyphen in the
	 * variable name when calling <code>permitVariable</code>.</p>
	 * <p>
	 * A variable name must begin with lowercase a-z and can then contain
	 * lowercase and/or uppercase a-zA-Z terms separated by single hyphens. Consecutive hyphens
	 * cannot appear within variable names, so <kbd>variable--name</kbd> would
	 * be illegal, for example.</p>
	 *
	 * @param variableName the name of a variable permitted by this handler.
	 * @param permittedCountInterval an <code>Interval</code> of
	 * <code>Integer</code> values whose lower endpoint must be at least zero,
	 * and whose upper endpoint must be at least one. The lower endpoint cannot
	 * be <code>null</code> (equivalent to negative infinity) but the upper
	 * endpoint can be <code>null</code> (equivalent to positive infinity).
	 * @param usageDescription a brief description of how to use this variable,
	 * for display to a user who needs help on calling the client application
	 * from the command-line.
	 * @throws IllegalStateException if this method is called after the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if this method is provided with a
	 * variable name which does not fit the permitted pattern, or if the
	 * provided <code>Interval</code> has a <code>null</code> or negative value
	 * for its lower endpoint, or a value of less than one for its upper
	 * endpoint.
	 */
	public void permitVariable(String variableName,
			Interval permittedCountInterval, String usageDescription) {
		this.permitVariable(variableName, permittedCountInterval,
				usageDescription, null, false);
	}
	public void permitVariable(String variableName,
			String usageDescription) {
		this.permitVariable(variableName, getDefaultInterval(),
				usageDescription, null, false);
	}

	/**
	 * Instruct this handler to accept command-line variables with the specified
	 * name, in any quantity permitted by the supplied interval, attach the
	 * provided usage description text, and instruct the variable to reject any
	 * value which does not entirely match the specified <code>Pattern</code>.
	 * <p>
	 * <strong>Note:</strong> do not include the initial double-hyphen in the
	 * variable name when calling <code>permitVariable</code>.</p>
	 * <p>
	 * A variable name must begin with lowercase a-z and can then contain
	 * lowercase and/or uppercase a-zA-Z terms separated by single hyphens. Consecutive hyphens
	 * cannot appear within variable names, so <kbd>variable--name</kbd> would
	 * be illegal, for example.</p>
	 *
	 * @param variableName the name of a variable permitted by this handler.
	 * @param permittedCountInterval an <code>Interval</code> of
	 * <code>Integer</code> values whose lower endpoint must be at least zero,
	 * and whose upper endpoint must be at least one. The lower endpoint cannot
	 * be <code>null</code> (equivalent to negative infinity) but the upper
	 * endpoint can be <code>null</code> (equivalent to positive infinity).
	 * @param usageDescription a brief description of how to use this variable,
	 * for display to a user who needs help on calling the client application
	 * from the command-line.
	 * @param Predicate<String> a Predicate<String> functional interface which
	 * determines whether or not a value
	 * provided to this command-line variable is valid. Can be <code>null</code>
	 * if any String value is acceptable as a value for this command-line
	 * variable.
	 * @throws IllegalStateException if this method is called after the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if this method is provided with a
	 * variable name which does not fit the permitted pattern, or if the
	 * provided <code>Interval</code> has a <code>null</code> or negative value
	 * for its lower endpoint, or a value of less than one for its upper
	 * endpoint.
	 */
	public void permitVariable(String variableName,
			Interval permittedCountInterval, String usageDescription,
			Predicate<String> validationTest) {
		this.permitVariable(variableName, permittedCountInterval,
				usageDescription, validationTest, false);
	}
	public void permitVariable(String variableName, String usageDescription,
			Predicate<String> validationTest) {
		this.permitVariable(variableName, getDefaultInterval(),
				usageDescription, validationTest, false);
	}

	/**
	 * Instruct this handler to accept a command-line variable which is intended
	 * only for test purposes. This works in exactly the same way as the
	 * {@link #permitVariable(String, Interval)} method, but marks the variable
	 * as being for test use only, so that this command-line variable is not
	 * displayed to end-users if they are shown a list of permitted command-line
	 * variables. This method should be used to permit variables which should
	 * only ever be used by the developer of an application, and should not be
	 * commonly known to end users.
	 *
	 * @param variableName the name of a test variable permitted by this
	 * handler.
	 * @param permittedCountInterval an <code>Interval</code> of
	 * <code>Integer</code> values whose lower endpoint must be zero (because
	 * test variables must be optional), and whose upper endpoint must be at
	 * least one and can be <code>null</code> (equivalent to positive infinity).
	 * @throws IllegalStateException if this method is called after the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if this method is provided with a
	 * variable name which does not fit the permitted pattern, or if the
	 * provided interval has a non-zero value for its lower endpoint, or a value
	 * of less than one for its upper endpoint.
	 */
	public void permitTestVariable(String variableName,
			Interval permittedCountInterval) {
		if (permittedCountInterval.getLeft().intValue() != 0) {
			throw new IllegalArgumentException("The lower endpoint of the "
					+ "permitted count interval for a test variable must be "
					+ "zero, because a test variable must be optional.");
		}
		this.permitVariable(variableName, permittedCountInterval, null, null,
				true);
	}
	public void permitTestVariable(String variableName) {
		permitTestVariable(variableName, getDefaultInterval());
	}

	/**
	 * Instruct this handler to accept command-line variables intended for test
	 * purposes, which accept only values that match the specified
	 * <code>Pattern</code>. This works in exactly the same way as the
	 * {@link #permitVariable(String, Interval)} method, but marks the variable
	 * as being for test use only, so that this command-line variable is not
	 * displayed to end-users if they are shown a list of permitted command-line
	 * variables. This method should be used to permit variables which should
	 * only ever be used by the developer of an application, and should not be
	 * commonly known to end users.
	 *
	 * @param variableName the name of a test variable permitted by this
	 * handler.
	 * @param permittedCountInterval an <code>Interval</code> of
	 * <code>Integer</code> values whose lower endpoint must be zero (because
	 * test variables must be optional), and whose upper endpoint must be at
	 * least one and can be <code>null</code> (equivalent to positive infinity).
	 * @param Predicate<String> a Predicate<String> functional interface which
	 * determines whether or not a value
	 * provided to this command-line variable is valid. Can be <code>null</code>
	 * if any String value is acceptable as a value for this command-line
	 * variable.
	 * @throws IllegalStateException if this method is called after the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if this method is provided with a
	 * variable name which does not fit the permitted pattern, or if the
	 * provided interval has a non-zero value for its lower endpoint, or a value
	 * of less than one for its upper endpoint.
	 */
	public void permitTestVariable(String variableName, Predicate<String> validationTest) {
		permitTestVariable(variableName, getDefaultInterval(), validationTest);
	}
	public void permitTestVariable(String variableName,
			Interval permittedCountInterval, Predicate<String> validationTest) {
		if (permittedCountInterval.getLeft().intValue() != 0) {
			throw new IllegalArgumentException("The lower endpoint of the "
					+ "permitted count interval for a test variable must be "
					+ "zero, because a test variable must be optional.");
		}
		this.permitVariable(variableName, permittedCountInterval, null,
				validationTest,
				true);
	}

	/**
	 * Declares a permitted command-line variable with the specified name,
	 * quantity, and usage description, which accepts values of the specified
	 * Pattern, and specifies whether or not the variable is for test usage.
	 *
	 * @param variableName the name of the command-line variable.
	 * @param permittedCountInterval an <code>Interval</code> which specifies
	 * the number of times this command-line variable must occur when starting
	 * the client Java application.
	 * @param usageDescription the text which should be used to describe this
	 * command-line variable, or <code>null</code> if no description is to be
	 * provided to the application user. Note that the value of
	 * <code>usageDescription</code> is ignored if <code>testUseOnly</code> is
	 * set to <code>true</code>.
	 * @param Predicate<String> a Predicate<String> functional interface which
	 * determines whether or not a value
	 * provided to this command-line variable is valid. Can be <code>null</code>
	 * if any String value is acceptable as a value for this command-line
	 * variable.
	 * @param testUseOnly should be <code>true</code> if this command-line
	 * variable is for developer test purposes only, and should not be listed in
	 * the usage description text for the Java application; <code>false</code>
	 * otherwise.
	 */
	private void permitVariable(String variableName, String usageDescription,
			Predicate<String> validationTest, boolean testUseOnly) {
		permitVariable(
				variableName, getDefaultInterval(), usageDescription,
				validationTest, testUseOnly);
	}
	private void permitVariable(String variableName,
			Interval permittedCountInterval, String usageDescription,
			Predicate<String> validationTest, boolean testUseOnly) {
		if (this.objectStatus == Status.HANDLER_CONFIGURATION) {
			this.objectStatus = Status.PREPRO;
		}
		if (this.objectStatus != Status.PREPRO) {
			throw new IllegalStateException("Cannot add to the list of "
					+ "permitted variables once processing of main arguments "
					+ "has already begun.");
		}
		if (permittedCountInterval.getLeft() == null
				|| permittedCountInterval.getLeft().intValue() < 0) {
			throw new IllegalArgumentException("Cannot specify a minimum "
					+ "variable count of less than zero.");
		}
		if (permittedCountInterval.getRight() != null
				&& permittedCountInterval.getRight().intValue() < 1) {
			throw new IllegalArgumentException("Cannot specify a maximum "
					+ "variable count of less than one.");
		}
		if (!this.validName(variableName)) {
			throw new IllegalArgumentException("Cannot permit variable with "
					+ "name \"" + variableName + "\" because this name does not "
					+ "fit the accepted pattern.");
		}
		if (contains(this.permittedFlags.keySet(), variableName)
				|| contains(this.permittedVariables.keySet(), variableName)) {
			throw new IllegalArgumentException("Cannot permit variable with "
					+ "name \"" + variableName + "\" because this is already a "
					+ "permitted flag name or permitted or required variable "
					+ "name (NOTE: name character case is " + (this.ignoreCase ? "" : "NOT ") + "ignored)");
		}
		Variable newVariable
				= new Variable(variableName, permittedCountInterval,
						usageDescription, validationTest, testUseOnly);
		this.permittedVariables.put(variableName, newVariable);
	}

	/**
	 * Process the command-line arguments passed to the <code>main</code> method
	 * of your Java application.
	 * <p>
	 * Call this method only after you have specified all permitted flag names
	 * by calling <code>permitFlag</code>, and all permitted variable names and
	 * counts by calling <code>permitVariable</code>.</p>
	 * <p>
	 * If any of the arguments found on the command-line are not found in the
	 * set of permitted flags or the set of permitted variable names then an
	 * IllegalArgumentException will be thrown.</p>
	 *
	 * @param args the array of strings passed to the main method of your
	 * application.
	 * @throws IllegalStateException if this method is called more than once.
	 * @throws IllegalArgumentException if a command-line argument is found
	 * which is a flag not found in the set of permitted flag names, or which is
	 * a variable not found in the set of permitted variable names.
	 */
	public void processMainArgs(String[] args) {
		Objects.requireNonNull(args, "The args parameter cannot be null.");
		if (this.objectStatus == Status.PROCESSING || this.objectStatus == Status.POPRO) {
			throw new IllegalStateException("Cannot process main arguments "
					+ "twice.");
		}
		this.objectStatus = Status.PROCESSING; //switch to processing state

		// Dice the command-line arguments up into a set of flags and a set of
		// variables.
		this.diceArgs(args);
		
		this.objectStatus = Status.POPRO;
		
		// Check that variable counts are all within permitted intervals.
		this.checkVariableCounts();

		// Check that all flags and variables are set for which dependent flags or variables are set
		this.checkDependentFlagsAndVariables();

		// Check that all flags and variables are not set for which incompatible flags or variables are set
		this.checkIncompatibleFlagsAndVariables();
		
		// Chack that any minimally sufficient requirements are satisfied
		this.checkMinimallySufficientFlagAndVariableSets();
	}

	/* 
	 * Gather the command-line arguments into a set of flag names and a Map
	 * which maps each variable name to a List of value strings.
	 */
	private void diceArgs(String[] args) {
		this.commandLineFlagSet = new HashSet<>();
		this.commandLineVariableValues = new HashMap<>();
		Pattern nameValuePattern = Pattern.compile(
				"^" + flagAndVariablePrefix + NAME_PATTERN.pattern() + "(?:=(.+))?$");
		// If argumentCommentPrefix is set (non-null) then test if the argument is commented-out
		Pattern commentedOutNameValuePattern =
				argumentCommentPrefix != null
				? Pattern.compile("^" + argumentCommentPrefix + flagAndVariablePrefix + NAME_PATTERN.pattern() + "(?:=(.+))?$")
				: null;
		for (int i = 0; i < args.length; ++i) {
			String arg = args[i];
			Matcher nameValueMatcher = null;
			Matcher commentedOutNameValueMatcher = (commentedOutNameValuePattern == null)? null : commentedOutNameValuePattern.matcher(arg);
			
			if (argumentCommentPrefix != null && commentedOutNameValueMatcher != null && commentedOutNameValueMatcher.matches()) {
				nameValueMatcher = commentedOutNameValueMatcher;
			} else {
				nameValueMatcher = nameValuePattern.matcher(arg);
			}
			if (nameValueMatcher.matches()) {
				int groupCount = nameValueMatcher.groupCount();
				if (groupCount != 2) {
					throw new RuntimeException("Regex pattern should return "
							+ "two subcapture groups, but another quantity "
							+ "was found when processing \"" + arg + "\"");
				}
				String key = nameValueMatcher.group(1);
				String value = nameValueMatcher.group(2);
				// Support --var {arg} syntax in addition to --var=arg syntax
				if (value == null && i + 1 < args.length && args[i + 1] != null && ! (args[i + 1].startsWith(flagAndVariablePrefix) || (argumentCommentPrefix != null && args[i + 1].startsWith(argumentCommentPrefix + flagAndVariablePrefix)))) {
					value = args[i + 1];
					i++;
				}
				// If this arg is commented-out, then skip to next arg, if any
				if (nameValueMatcher == commentedOutNameValueMatcher) {
					continue;
				}
				if (value == null) {
					// No value was found by the regex matcher, so treat this as
					// a simple command-line flag.
					if (!contains(this.permittedFlags.keySet(), key)) {
						// This flag name does not appear in the list of
						// permitted flag names specified by the client.
						throw new IllegalArgumentException("Command-line flag \"" + flagAndVariablePrefix
								+ key + "\" is not permitted by this handler."
								+ " NOTE: the character case of flag arguments is " + (this.ignoreCase ? "" : "NOT ") + "ignored.");
					}
					if (contains(commandLineFlagSet, key)) {
						// This flag has already been provided on the args list
						// but it does not make sense for a flag to occur twice
						// in the same command-line call.
						throw new IllegalArgumentException("Command-line flag \"" + flagAndVariablePrefix
								+ key
								+ "\" appears more than once. Each type of flag can "
								+ "only appear once in a set of command-line "
								+ "arguments."
								+ " NOTE: the character case of flag arguments is " + (this.ignoreCase ? "" : "NOT ") + "ignored.");
					}
					// This flag does not already exist in the map, so add it
					// and map its name to an empty List.
					commandLineFlagSet.add(key);
				} else {
					// A value was found by the regex matcher, so treat this as
					// a command-line variable which maps a variable-name to a
					// value.
					if (!contains(this.permittedVariables.keySet(), key)) {
						// This flag name does not appear in the list of
						// permitted flag names specified by the client.
						throw new IllegalArgumentException("Command-line "
								+ "variable \"" + flagAndVariablePrefix + key + "\" is not "
								+ "permitted by this application."
								+ " NOTE: the character case of flag and variable arguments is " + (this.ignoreCase ? "" : "NOT ") + "ignored.");
					}
					// Check that the supplied value satisfies the regex Pattern
					// specified (if any) for this command-line variable.
					if (get(this.permittedVariables, key).validationTest != null) {
						boolean testResult = false;
						Exception validationException = null;
						try {
							testResult = get(this.permittedVariables, key).validationTest.test(value);
						} catch (Exception e) {
							testResult = false;
							validationException = e;
						}
						if (! testResult) {
							final String msg = "The command-line variable \"" + flagAndVariablePrefix + key
									+ "\" does not permit the value \"" + value
									+ "\".";
							throw validationException != null
									? new IllegalArgumentException(msg, validationException)
									: new IllegalArgumentException(msg);
						}
					}
					if (contains(this.commandLineVariableValues.keySet(), key)) {
						// Mapping already exists, so there must be other values
						// already assigned to this variable argument.
						List<String> valueList = get(this.commandLineVariableValues, key);
						// A list of values already exists, so add this
						// newest value to the list.
						valueList.add(value);
					} else {
						// This variable name is not already found in the map, so
						// create a mapping from this variable name to a list of
						// values and add this first value.
						List<String> newValueList = new ArrayList<>();
						newValueList.add(value);
						this.commandLineVariableValues.put(key, newValueList);
					}
				}
			} else {
				throw new IllegalArgumentException("Command-line argument \""
						+ arg + "\" is not correctly formatted.");
			}
		}
	}

	/*
	 * Check that variables occur as few and as many times as permitted by this
	 * handler, and throw an exception if the actual count is outside of the
	 * permitted interval.
	 */
	private void checkVariableCounts() {
		// By this point, all of the variable names acting as keys in
		// the commandLineVariableValues map must be permitted variable names
		// (because this check is performed by the argsDicer method when each
		// variable name is found in the command-line arguments)
		// but now we need to check that the occurrence count of each variable
		// falls within the permitted interval.
		for (String variableName : permittedVariables.keySet()) {
			// For each variable name in the permitted set . . .
			// Retrieve the permitted interval
			Interval permittedCountInterval = get(permittedVariables, variableName).permittedCountInterval;
			// Determine the actual count provided on the command-line
			List<String> actualVariableValues = get(commandLineVariableValues, variableName);
			int actualVariableCount = (actualVariableValues == null ? 0
					: actualVariableValues.size());
			// Check whether the actual count is permitted by the interval.
			// If the actual count is outside of the permitted interval then
			// throw an exception.
			if (!permittedCountInterval.includes(actualVariableCount)) {
				StringBuilder sb = new StringBuilder("Variable \"");
				sb.append(variableName);
				sb.append("\" appears too ");
				sb.append((actualVariableCount < permittedCountInterval.
						getLeft().intValue() ? "few" : "many"));
				sb.append(" times in command-line arguments. Permitted count ");
				sb.append("must be in ");
				sb.append(permittedCountInterval);
				sb.append(" but actual count was ");
				sb.append(actualVariableCount);
				sb.append('.');
				throw new IllegalArgumentException(sb.toString());
			}
		}
	}

	/**
	 * Validate that for each set of dependent flags and variables,
	 * all flags and variables in each set have been passed
	 * on the command line
	 */
	private void checkDependentFlagsAndVariables() {
		for (Set<String> dependentFlagsAndVariables : this.dependentFlagsAndVariables) {
			String lastFound = null;
			String notFound = null;
			for (String name : dependentFlagsAndVariables) {
				if ((this.isPermittedFlag(name) && this.foundFlag(name)) || (this.isPermittedVariable(name) && this.foundVariable(name))) {
					if (notFound != null) {
						throw new IllegalArgumentException("Variable or flag with "
								+ "name \"" + name + "\" requires that variable or flag \"" + notFound + "\" also be set");
					}
					lastFound = name;
				} else {
					notFound = name;
					if (lastFound != null) {
						throw new IllegalArgumentException("Variable or flag with "
								+ "name \"" + lastFound + "\" requires that variable or flag \"" + notFound + "\" also be set");
					}
				}
			}
		}
	}

	/**
	 * Validate that for each set of incompatible flags and variables,
	 * only one flag or variable in each set has been passed
	 * on the command line
	 */
	private void checkIncompatibleFlagsAndVariables() {
		for (Set<String> incompatibleVariables : this.incompatibleFlagsAndVariables) {
			String firstFound = null;
			String secondFound = null;
			for (String name : incompatibleVariables) {
				if ((this.isPermittedFlag(name) && this.foundFlag(name)) || (this.isPermittedVariable(name) && this.foundVariable(name))) {
					if (firstFound == null) {
						firstFound = name;
					} else {
						secondFound = name;
						throw new IllegalArgumentException("Cannot permit variable or flag with "
								+ "name \"" + secondFound + "\" because there is already an "
								+ "incompatible variable or flag with name \"" + firstFound + "\"");
					}
				}
			}
		}
	}

	private void checkMinimallySufficientFlagAndVariableSets() {
		if (minimallySufficientFlagAndVariableSets.size() > 0) {
			for (Set<String> minimallySufficientVariablesSet : this.minimallySufficientFlagAndVariableSets) {
				boolean requirementSatisfied = true;
				for (String name : minimallySufficientVariablesSet) {
					if (! (this.isPermittedFlag(name) && this.foundFlag(name)) && ! (this.isPermittedVariable(name) && this.foundVariable(name))) {
						requirementSatisfied = false;
						break;
					}
				}
				if (requirementSatisfied) {
					return;
				}
			}

			StringBuilder sb = new StringBuilder();
			for (Set<String> group : this.minimallySufficientFlagAndVariableSets) {
				sb.append("\t" + toString(group) + "\n");
			}
			throw new IllegalArgumentException(
					"Requirement not met that at least one of the "
					+ "minimally sufficient combinations of flags and/or variable arguments "
					+ " be passed on the command line:\n"
					+ sb.toString());
		}
	}

	public boolean isPermittedFlag(String flagName) {
		return contains(this.permittedFlags.keySet(), flagName);
	}

	public boolean isPermittedVariable(String variableName) {
		return contains(this.permittedVariables.keySet(), variableName);
	}

	/**
	 * Reports on whether a specified flag name was found in the command-line
	 * arguments.
	 *
	 * @param flagName the flag name to check.
	 * @return true if the specified flag name was specified in the command-line
	 * arguments.
	 * @throws IllegalStateException if this method is called before the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if the specified flag name is set but not a name
	 * recognised/permitted by this handler instance.
	 */
	public boolean foundFlag(String flagName) {
		if (this.objectStatus != Status.POPRO) {
			throw new IllegalStateException("Cannot query the set of flags "
					+ "found on the command-line before processing has "
					+ "been completed by calling the processMainArgs method.");
		}
		boolean contained = contains(this.commandLineFlagSet, flagName);
		if (/* contained && */ !contains(this.permittedFlags.keySet(), flagName)) {
			throw new IllegalArgumentException("Flag name \"" + flagName + "\""
					+ " is not "
					+ "in the set of permitted flag names, so it does not make "
					+ "sense for it to be on the command-line."
					+ " (NOTE: name character case is " + (this.ignoreCase ? "" : "NOT ") + "ignored)");
		} else {
			return contained;
		}
	}

	/**
	 * Reports on whether a specified variable name was found in the command
	 * line arguments.
	 * <p>
	 * Note that this method does not return any of the values attached to the
	 * variable name. It serves only to confirm the presence of a variable in
	 * the command-line arguments.</p>
	 *
	 * @param variableName the variable name to check.
	 * @return true if the specified variable name was found in the command-line
	 * arguments.
	 * @throws IllegalStateException if this method is called before the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if the specified variable name is set but not a
	 * name recognised/permitted by this handler instance.
	 */
	public boolean foundVariable(String variableName) {
		if (this.objectStatus != Status.POPRO) {
			throw new IllegalStateException("Cannot query the set of variables "
					+ "found on the command-line before processing has "
					+ "been completed by calling the processMainArgs method.");
		}
		boolean contained = contains(this.commandLineVariableValues.keySet(), variableName);
		if (contained && !contains(this.permittedVariables.keySet(), variableName)) {
			throw new IllegalArgumentException("Variable name \"" + variableName + "\""
					+ " is not in the set of permitted variable names, so it "
					+ "does not make sense for it to be on the command-line."
					+ " (NOTE: name character case is " + (this.ignoreCase ? "" : "NOT ") + "ignored)");
		} else {
			return contained;
		}
	}

	/**
	 * Returns the list of command-line values associated with the specified
	 * variable name.
	 *
	 * @param variableName the name of the command-line variable whose values
	 * you want.
	 * @return a <code>List&lt;String&gt;</code> object which contains the
	 * values found with the specified variable name on the command-line. The
	 * <code>List</code> will be empty if the command-line variable is optional
	 * and the application user did not provide a value for it.
	 * @throws IllegalStateException if this method is called before the
	 * {@link #processMainArgs(String[])} method.
	 * @throws IllegalArgumentException if the specified variable name is not a
	 * name recognised by this handler instance.
	 */
	public List<String> getValuesFromVariable(String variableName) {
		if (this.objectStatus != Status.POPRO) {
			throw new IllegalStateException("Cannot query the set of variable "
					+ "values found on the command-line before processing has "
					+ "been completed by calling the processMainArgs method.");
		}
		if (!contains(this.permittedVariables.keySet(), variableName)) {
			throw new IllegalArgumentException("Variable name \"" + variableName + "\""
					+ " is not in the set of permitted variables, so it does "
					+ "not make sense to check for it in the set of variables "
					+ "found on the command-line."
					+ " (NOTE: name character case is " + (this.ignoreCase ? "" : "NOT ") + "ignored)");
		}
		List<String> valueList = get(this.commandLineVariableValues, variableName);
		if (valueList == null) {
			valueList = new ArrayList<>();
		}
		return Collections.unmodifiableList(valueList);
	}
	public Set<String> getDistinctValuesFromVariable(String variableName) {
		List<String> valuesList = getValuesFromVariable(variableName);
		
		Set<String> valuesSet = new HashSet<>();
		
		valuesSet.addAll(valuesList);
		
		return Collections.unmodifiableSet(valuesSet);
	}
	public String getEnforcedSingleValueFromVariable(String variableName) {
		List<String> variableValues = getValuesFromVariable(variableName);
		if (! foundVariable(variableName) || variableValues.size() == 0) {
			throw new IllegalArgumentException("Variable name \"" + variableName + "\""
					+ " is not set");
		} else if (variableValues.size() > 1) {
			throw new IllegalArgumentException("Variable name \"" + variableName + "\""
					+ " has more than 1 (" + variableValues.size() + ") value: " + variableValues);
		} else {
			return variableValues.get(0);
		}
	}

	/**
	 * Generates a <code>String</code> which describes the list of command-line
	 * flags and variables which are permitted or required by the
	 * <code>MainArgsHandler</code> for the running application. It is
	 * recommended to display the content of this <code>String</code> to the
	 * application user if they attempt to call the Java application with
	 * invalid command-line flags or variables.
	 * <p>
	 * Note that the returned <code>String</code> does not contain any mention
	 * of test flags or variables, as those are only intended for use by
	 * developers and the content of this <code>String</code> is intended for
	 * display to application end-users.</p>
	 * <p>
	 * This method cannot be called before the
	 * {@link #processMainArgs(String[])} method has been called (otherwise it
	 * is not guaranteed that all permitted flags and variables have been
	 * declared).</p>
	 *
	 * @throws IllegalStateException if this method is called before the
	 * {@link #processMainArgs(String[])} method.
	 * @return a <code>String</code> which describes the list of flags and
	 * variables which can be supplied to the running application.
	 */
	public String getUsageSummary() {
		if (this.objectStatus == Status.PREPRO) {
			throw new IllegalStateException("Cannot request usage summary text "
					+ "before processing has begun.");
		}

		StringBuilder sb = new StringBuilder();
		sb.append((appName != null ? (appName + " valid") : "Valid") + " command-line flag and variable arguments:\n\n");
		StringBuilder flagUsage = new StringBuilder();
		flagUsage.append("Flags\n"
				+ "\tA flag has the form " + flagAndVariablePrefix + "flag-name (i.e two hyphens "
				+ "and then the flag name).\n"
				+ "\tA flag cannot take a value.\n"
				+ "\tFlags are separated from other flags and variables by a "
				+ "space.\n"
				+ "\tFlags are always optional, and each flag can be supplied "
				+ "at most once.\n"
				+ "\tThe character case of flags is " + (this.ignoreCase ? "" : "NOT ") + "ignored."
				+ "\n\n");

		flagUsage.append(
				"The following command-line flags are accepted by this "
				+ "application:\n\n");
		int nonTestFlagCount = 0;
		for (Flag flag : this.permittedFlags.values()) {
			if (flag.forTestUseOnly) {
				// Do not show test flags as part of the end-user usage guide.
				continue;
			}
			++nonTestFlagCount;
			flagUsage.append("\t" + flagAndVariablePrefix);
			flagUsage.append(flag.name);
			if (flag.usageDescription != null
					&& !flag.usageDescription.isEmpty()) {
				flagUsage.append("\n\t\t");
				flagUsage.append(flag.usageDescription);
			}
			flagUsage.append("\n\n");
		}
		if (nonTestFlagCount == 0) {
			sb.append("Flags\nNo command-line flags are required or accepted "
					+ "by this application.\n\n");
		} else {
			sb.append(flagUsage.toString());
		}
		StringBuilder variableUsage = new StringBuilder();
		variableUsage.append("Variables\n"
				+ "\tA variable may have the form "
				+ flagAndVariablePrefix + "variable-name=VALUE (i.e. two hyphens, then the variable name, "
				+ "then an equals symbol, then the value you "
				+ "want to provide for the variable).\n"
				+ "\tA variable may also have the form "
				+ flagAndVariablePrefix + "variable-name VALUE (two hyphens, then the variable name, "
				+ "then a space, then the value you "
				+ "want to provide for the variable).\n"
				+ "\tVariables are separated from other variables and flags "
				+ "by a space, so if you want to provide a value which "
				+ "contains a space you must wrap the whole value in "
				+ "double-quote symbols.\n"
				+ "\tA variable may be configured to be permitted to be supplied more "
				+ "than once (see the usage information for each variable to "
				+ "check how many times it can occur).\n"
				+ "\tThe character case of variable arguments is " + (this.ignoreCase ? "" : "NOT ") + "ignored."
				+ "\n\n");
		variableUsage.append("The following command-line variables "
				+ "are accepted by this application:\n\n");
		int nonTestVariableCount = 0;
		for (Variable variable : this.permittedVariables.values()) {
			if (variable.forTestUseOnly) {
				// Do not show test variables as part of end-user usage guide.
				continue;
			}
			++nonTestVariableCount;
			variableUsage.append("\t" + flagAndVariablePrefix);
			variableUsage.append(variable.name);
			variableUsage.append("\n\t");
			variableUsage.
					append(intervalAsText(variable.permittedCountInterval));
			if (variable.usageDescription != null && !variable.usageDescription.
					isEmpty()) {
				variableUsage.append("\n\t\t");
				variableUsage.append(variable.usageDescription);
			}
			variableUsage.append("\n\n");
		}
		if (nonTestVariableCount == 0) {
			sb.append("Variables\nNo command-line variables are required or "
					+ "accepted by this application.\n\n");
		} else {
			sb.append(variableUsage.toString());
		}
		
		if (this.dependentFlagsAndVariables.size() > 0) {
			sb.append("The following are dependent Flags/Variables (all in each group must be set):\n");
			for (Set<String> group : this.dependentFlagsAndVariables) {
				sb.append("\t" + toString(group) + "\n");
			}
			sb.append("\n");
		}
		
		if (this.incompatibleFlagsAndVariables.size() > 0) {
			sb.append("The following are incompatible Flags/Variables (only one in each group may be set):\n");
			for (Set<String> group : this.incompatibleFlagsAndVariables) {
				sb.append("\t" + toString(group) + "\n");
			}
			sb.append("\n");
		}
		
		if (this.minimallySufficientFlagAndVariableSets.size() > 0) {
			sb.append("The following are minimally sufficient Flag/Variable requirements (at least one set must be satisfied):\n");
			for (Set<String> group : this.minimallySufficientFlagAndVariableSets) {
				sb.append("\t" + toString(group) + "\n");
			}
			sb.append("\n");
		}
	
		return sb.toString();
	}

	private String intervalAsText(Interval interval) {
		int lower = interval.getLeft().intValue();
		if (interval.getRight() == null) {
			if (lower == 0) {
				return "Optional, can occur ANY number of times.";
			}
			return "Must occur at least " + (lower == 1 ? "once." : lower
					+ " times.");
		}
		int upper = interval.getRight().intValue();
		if (lower == 0) {
			if (upper == 1) {
				return "Optional, can occur at most once.";
			}
			return "Optional, can occur up to " + upper
					+ " times.";
		}
		// Lower endpoint greater than zero.
		if (lower == upper) {
			return "Must occur exactly " + (upper == 1 ? "once." : upper
					+ " times.");
		}
		// Endpoints not equal.
		return "Must occur at least " + (lower == 1 ? "once" : lower + " times")
				+ " and at most " + upper + " times.";
	}
	
	private String toString(Iterable<String> flagsAndVariables) {
		List<String> flagsAndVariablesWithHyphens = new LinkedList<>();
		for (String flagOrVariable : flagsAndVariables) {
			flagsAndVariablesWithHyphens.add(flagAndVariablePrefix + flagOrVariable);
		}
		return Arrays.toString(flagsAndVariablesWithHyphens.toArray(new String[flagsAndVariablesWithHyphens.size()]));
	}

	private boolean contains(Iterable<String> set, String passedKey) {
		for (String key : set) {
			if (ignoreCase) {
				if (key.equalsIgnoreCase(passedKey)) {
					return true;
				}
			} else {
				if (key.equals(passedKey)) {
					return true;
				}
			}
		}

		return false;
	}
	
	private <V> V get(Map<String, V> map, String passedKey) {
		for (Map.Entry<String, V> entry : map.entrySet()) {
			if (ignoreCase) {
				if (entry.getKey().equalsIgnoreCase(passedKey)) {
					return entry.getValue();
				}
			} else {
				if (entry.getKey().equals(passedKey)) {
					return entry.getValue();
				}
			}
		}
		
		return null;
	}
}