package gov.vha.isaac.ochre.utility.export;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Predicate;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.config.Configurator;
import org.apache.mahout.math.Arrays;

import gov.vha.isaac.ochre.api.ConfigurationService;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.LookupService;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.chronicle.ObjectChronology;
import gov.vha.isaac.ochre.api.chronicle.ObjectChronologyType;
import gov.vha.isaac.ochre.api.collections.ConceptSequenceSet;
import gov.vha.isaac.ochre.api.collections.SememeSequenceSet;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.SememeVersion;
import gov.vha.isaac.ochre.api.coordinate.LanguageCoordinate;
import gov.vha.isaac.ochre.api.coordinate.StampCoordinate;
import gov.vha.isaac.ochre.api.coordinate.StampPrecedence;
import gov.vha.isaac.ochre.api.externalizable.OchreExternalizableObjectType;
import gov.vha.isaac.ochre.api.identity.StampedVersion;
import gov.vha.isaac.ochre.api.util.Interval;
import gov.vha.isaac.ochre.api.util.NumericUtils;
import gov.vha.isaac.ochre.api.util.UUIDUtil;
import gov.vha.isaac.ochre.api.util.metainf.MavenArtifactInfo;
import gov.vha.isaac.ochre.api.util.metainf.MetaInfReader;
import gov.vha.isaac.ochre.impl.utility.ComponentDumperDiagnosticUtil;
import gov.vha.isaac.ochre.impl.utility.Frills;
import gov.vha.isaac.ochre.impl.utility.MainArgsHandler;
import gov.vha.isaac.ochre.model.configuration.LanguageCoordinates;
import gov.vha.isaac.ochre.model.configuration.StampCoordinates;
import gov.vha.isaac.ochre.model.sememe.version.DynamicSememeImpl;

/**
 * 
 * {@link IsaacComponentDumper}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 * VM arguments should be something like -DisaacDatabaseLocation=D:\Work\EclipseWorkspaces\isaac\ISAAC-fx-gui-pa\fx-gui-assembly\vhat-2017.09.07-198-1.18-SNAPSHOT-all-SQLIMPORT.data -Xmx14000m
 * 
 * Program arguments should either be a single component NID or UUID
 * i.e. IsaacComponentDumper {NID|UUID}
 * or an IDENTIFIER_SOURCE concept NID or UUID AND a value
 * i.e. IsaacComponentDumper {NID|UUID} {value}
 */
public class IsaacComponentDumper
{
	private static Logger log = LogManager.getLogger(IsaacComponentDumper.class);
	
	private static boolean isaacStarted = false;
	
	@SuppressWarnings("restriction")
	private static void cleanup() {
		if (isaacStarted) {
			isaacStop();
			isaacStarted = false;
		}
		javafx.application.Platform.exit();
		
		System.out.println("Done shutting down ISAAC");
	}

	public static List<File> findDBFoldersNoCreate(File inputFolder)
	{
		List<File> potentialMatches = new LinkedList<>();

		inputFolder = inputFolder.getAbsoluteFile();

		//If it is a folder with a '.data' at the end of the name, just use it.
		if (inputFolder.exists() && inputFolder.getName().endsWith(".data") && inputFolder.isDirectory() && inputFolder.canRead())
		{
			log.info("Found potential data store location at " + inputFolder.getAbsolutePath());
			return Collections.nCopies(1, inputFolder);
		}

		//Otherwise see if we can find a .data folder as a direct child
		if (inputFolder.exists() && inputFolder.isDirectory() && inputFolder.canRead())
		{
			for (File f : inputFolder.listFiles())
			{
				if (f.exists() && f.getName().endsWith(".data") && f.isDirectory() && f.canRead())
				{
					log.info("Found potential data store location at " + f.getAbsolutePath());
					potentialMatches.add(f);
				}
			}
		}

		//Or as a sibling
		if (inputFolder.getParentFile().isDirectory() && inputFolder.canRead())
		{
			for (File f : inputFolder.getParentFile().listFiles())
			{
				//If it is a folder with a '.bdb' at the end of the name, then berkeley-db will be in a sub-folder.
				if (f.exists() && f.getName().endsWith(".data") && f.isDirectory() && inputFolder.canRead())
				{
					log.info("Found potential data store location at " + f.getAbsolutePath());
					potentialMatches.add(f);
				}
			}
		}
		
		return Collections.unmodifiableList(potentialMatches);
	}

	private static File getDataStore(String dataStoreLocationStr) {
		if (StringUtils.isEmpty(dataStoreLocationStr)) {
			throw new IllegalArgumentException("Invalid data store location " + (dataStoreLocationStr != null ? "\"" + dataStoreLocationStr + "\"" : null));
		}

		File dataStoreLocation = new File(dataStoreLocationStr).getAbsoluteFile();

		final String dbDirSuffix = ".data";
		if (!dataStoreLocation.getName().endsWith(dbDirSuffix))
		{
			throw new IllegalArgumentException("Specified data store does not conform to naming convention (name does not end with \"" + dbDirSuffix + "\"): \"" + dataStoreLocation.getAbsoluteFile().getAbsolutePath() + "\"");
		}
		if (!dataStoreLocation.exists())
		{
			throw new IllegalArgumentException("Specified data store does not exist: \"" + dataStoreLocation.getAbsoluteFile().getAbsolutePath() + "\"");
		}
		if (!dataStoreLocation.isDirectory())
		{
			throw new IllegalArgumentException("Specified data store is not a directory or folder: \"" + dataStoreLocation.getAbsolutePath() + "\"");
		}
		if (!dataStoreLocation.canRead())
		{
			throw new IllegalArgumentException("Specified data store is not readable: \"" + dataStoreLocation.getAbsolutePath() + "\"");
		}
		boolean foundObjectChronicles = false;
		boolean foundSearch = false;
		for (File f : dataStoreLocation.listFiles())
		{
			if (f.exists() && f.isDirectory() && f.canRead() && f.getName().equals("object-chronicles"))
			{
				foundObjectChronicles = true;
			}
			else if (f.exists() && f.isDirectory() && f.canRead() && f.getName().equals("search"))
			{
				foundSearch = true;
			}
		}
		if (! foundObjectChronicles) {
			throw new IllegalArgumentException("No valid object-chronicles folder/directory found under specified data store location \"" + dataStoreLocation.getAbsolutePath() + "\"");
		} else if (! foundSearch) {
			throw new IllegalArgumentException("No valid search folder/directory found under specified data store location \"" + dataStoreLocation.getAbsolutePath() + "\"");
		}
		
		return dataStoreLocation;
	}

	private static void isaacInit(File dataStoreLocation)
	{
		System.out.println("Starting up ISAAC");
		log.info("Isaac Init called");
		isaacStarted = true;

		try
		{
			log.info("ISAAC Init thread begins");

			//use the passed in JVM parameter location
			LookupService.getService(ConfigurationService.class).setDataStoreFolderPath(dataStoreLocation.toPath());
			System.out.println("\tSetup AppContext, data store location = " + dataStoreLocation.getAbsolutePath());

			//status_.set("Starting ISAAC");
			LookupService.startupIsaac();

			//status_.set("Ready");
			System.out.println("Done setting up ISAAC");

		}
		catch (Exception e)
		{
			log.error("Failure starting ISAAC", e);
		}
	}

	private static void isaacStop()
	{
		log.info("Stopping ISAAC");
		LookupService.shutdownSystem();
		log.info("ISAAC stopped");
	}
	
	private static ConceptChronology<? extends ConceptVersion<?>> getExistingConcept(String uuidSeqOrNidStr) {
		UUID componentUuidFromString = null;
		
		try {
			componentUuidFromString = UUID.fromString(uuidSeqOrNidStr);
			
			// If componentUuidFromString does not exist in the specified DB
			if (! Get.identifierService().hasUuid(componentUuidFromString)) {
				// FAIL
				throw new IllegalArgumentException("No corresponding component for UUID " + componentUuidFromString);
			} else {
				return Get.conceptService().getConcept(componentUuidFromString);
			}
		} catch (Exception e) {
			// Passed argument not a valid UUID
			
			// Attempt to interpret arg as Sequence or NID
			Optional<Integer> intFromArg = NumericUtils.getInt(uuidSeqOrNidStr);
			if (! intFromArg.isPresent() || intFromArg.get() == 0) {
				throw new IllegalArgumentException("Invalid integer concept sequence or NID value: " + uuidSeqOrNidStr);
			} else {
				return Get.conceptService().getConcept(intFromArg.get());
			}
		}
	}

	private static UUID getUuidForExistingComponent(String uuidOrNidStr) {
		UUID componentUuidFromString = null;
		Optional<UUID> optionalUuidFromString = UUIDUtil.getUUID(uuidOrNidStr);
		if (optionalUuidFromString.isPresent()) {
			componentUuidFromString = optionalUuidFromString.get();
			
			// If componentUuidFromString does not exist in the specified DB
			if (! Get.identifierService().hasUuid(componentUuidFromString)) {
				// FAIL
				throw new IllegalArgumentException("No component for UUID " + uuidOrNidStr);
			}
		} else {
			// Passed argument not a valid UUID
			
			// Attempt to interpret arg as NID
			Optional<Integer> componentNid = NumericUtils.getNID(uuidOrNidStr);
			if (componentNid.isPresent()) {
				Optional<UUID> optionalComponentUuidFromString = Get.identifierService().getUuidPrimordialForNid(componentNid.get());
				if (optionalComponentUuidFromString.isPresent()) {
					componentUuidFromString = optionalComponentUuidFromString.get();
				} else {
					throw new IllegalArgumentException("No component for NID " + uuidOrNidStr);
				}
			} else {
				throw new IllegalArgumentException("Passed String \"" + uuidOrNidStr + "\" is neither a UUID nor a NID");
			}
		}
		
		return componentUuidFromString;
	}

	private static StampCoordinate getDefaultStampCoordinate() {
		return StampCoordinates.getDevelopmentLatest();
	}
	private static LanguageCoordinate getDefaultLanguageCoordinate() {
		return LanguageCoordinates.getUsEnglishLanguageFullySpecifiedNameCoordinate();
	}
	public static void main(String[] args) throws IOException
	{
		// Place in try/finally block to ensure that cleanup() is called
		try {
			final String helpFlagName = "help";
			
			final String tgtUuidOrNidVariableName = "tgtUuidOrNid";
			final String idSrcUuidOrNidVariableName = "idSrcUuidOrNid";
			final String tgtIdVariableName = "tgtId";

			final String moduleVariableName = "module";
			final String dtVariableName = "dt";
			final String stateVariableName = "state";
			final String pathVariableName = "path";
			final String precedenceVariableName = "precedence";

			final String languageVariableName = "language";
			final String descriptionTypeVariableName = "descType";

			final String logLevelVariableName = "logLevel";
			
			final String dbVariableName = "db";
			
			final String requireVersionFlagName = "requireVersion";
			
			final String dbInfoFlagName = "dbInfo";

			final String noRecurseFlagName = "noRecurse";
			final String listChildrenFlagName = "listChildren";

			final String memberSememesFlagName = "memberSememes";
			final String memberConceptsFlagName = "memberConcepts";

			final String countOnlyFlagName = "countOnly";

			final String uuidsOnlyFlagName = "uuidsOnly";
			
			final String dynSememeColsFlagName = "dynSememeCols";
			
			MainArgsHandler argsHandler = MainArgsHandler.getHandler();
			//argsHandler.setIgnoreCase(false);
			argsHandler.permitFlag(helpFlagName, "Specifies that usage message should be displayed");

			argsHandler.permitVariable(tgtUuidOrNidVariableName, new Interval("[0,]"), "A UUID or NID corresponding to a desired target component", (arg) -> NumericUtils.getNID(arg).isPresent() || UUIDUtil.isUUID(arg));
			argsHandler.permitVariable(idSrcUuidOrNidVariableName, "A UUID or NID corresponding to a desired identifier source concept", (arg) -> NumericUtils.getNID(arg).isPresent() || UUIDUtil.isUUID(arg));
			argsHandler.permitVariable(tgtIdVariableName, "A String id, interpreted as a separately-specified identifier source concept, identifying a desired target component");
		
			argsHandler.permitVariable(moduleVariableName, new Interval("[0,]"), "A String id of type sequence, UUID or NID corresponding to a specified module concept for use in constructing the StampCoordinate", (arg) -> NumericUtils.getInt(arg).isPresent() || UUIDUtil.isUUID(arg));
			argsHandler.permitVariable(dtVariableName, "A numeric long representation of the date and time, or a negative value representing latest, for use in constructing the StampCoordinate", (arg) -> NumericUtils.getLong(arg).isPresent());
			argsHandler.permitVariable(stateVariableName, new Interval("[0,]"), "A String representation of a State enum value for use in constructing the StampCoordinate", (arg) -> State.getFromString(arg).isPresent());
			argsHandler.permitVariable(pathVariableName, "A UUID or NID corresponding to a desired stamp coordinate path concept", (arg) -> NumericUtils.getNID(arg).isPresent() || UUIDUtil.isUUID(arg));
			argsHandler.permitVariable(precedenceVariableName, "A StampPrecedence value (one of " + Arrays.toString(StampPrecedence.values()) + ") used for constructing a StampCoordinate", (arg) -> StampPrecedence.fromString(arg).isPresent());

			argsHandler.permitVariable(languageVariableName, "A UUID or NID corresponding to a desired LanguageCoordinate language concept", (arg) -> NumericUtils.getNID(arg).isPresent() || UUIDUtil.isUUID(arg));
			argsHandler.permitVariable(descriptionTypeVariableName, new Interval("[0,]"), "A UUID or NID corresponding to a desired LanguageCoordinate description type concept", (arg) -> NumericUtils.getNID(arg).isPresent() || UUIDUtil.isUUID(arg));

			argsHandler.permitVariable(logLevelVariableName, "A String representation of a Level enumeration class value for use in configuring logging. Default is ERROR.", (arg) -> Level.getLevel(arg) != null);

			Predicate<String> dbVariablePredicate = (arg) -> getDataStore(arg).exists();
			argsHandler.permitVariable(dbVariableName, "The target database directory", dbVariablePredicate);

			argsHandler.permitFlag(requireVersionFlagName, "Only display chronologies with at least one version. Default is to display chronologies without versions.");

			argsHandler.permitFlag(dbInfoFlagName, "Dump database info");
			
			argsHandler.permitFlag(noRecurseFlagName, "Dump only specified target component(s)");
			argsHandler.permitFlag(listChildrenFlagName, "Describe each child of specified concept(s)");

			argsHandler.permitFlag(memberSememesFlagName, "Dump only member sememes of specified assemblage concept");
			argsHandler.permitFlag(memberConceptsFlagName, "Dump only member concepts of specified assemblage concept");
			
			argsHandler.permitFlag(countOnlyFlagName, "Only count object specified and returned");

			argsHandler.permitFlag(uuidsOnlyFlagName, "Only display UUIDs, rather than NIDs or sequences");
			
			argsHandler.permitFlag(dynSememeColsFlagName, "Display details of each dynamic sememe column");

			// Require both idSrcUuidOrNidVariableName and tgtIdVariableName be set if either is set 
			argsHandler.addFlagOrVariableDependency(idSrcUuidOrNidVariableName, tgtIdVariableName);
			
			// Prevent passing tgtUuidOrNidVariableName if either idSrcUuidOrNidVariableName or tgtIdVariableName is set
			argsHandler.addIncompatibleFlagsOrVariables(tgtUuidOrNidVariableName, idSrcUuidOrNidVariableName);
			argsHandler.addIncompatibleFlagsOrVariables(tgtUuidOrNidVariableName, tgtIdVariableName);

			argsHandler.addMinimallySufficientFlagOrVariableSet(idSrcUuidOrNidVariableName, tgtIdVariableName);
			argsHandler.addMinimallySufficientFlagOrVariableSet(tgtUuidOrNidVariableName);
			argsHandler.addMinimallySufficientFlagOrVariableSet(dbInfoFlagName);

			try {
				argsHandler.processMainArgs(args);
			} catch (IllegalArgumentException e) {
				System.err.println(argsHandler.getUsageSummary());
				throw e;
			}

			if (argsHandler.foundFlag(helpFlagName)) {
				System.out.println(argsHandler.getUsageSummary());
				return;
			}

			Level logLevel = Level.ERROR;
			if (argsHandler.foundVariable(logLevelVariableName)) {
				String logLevelString = argsHandler.getEnforcedSingleValueFromVariable(logLevelVariableName).trim().toUpperCase();
				logLevel = Level.getLevel(logLevelString);
				if (logLevel == null) {
					throw new IllegalArgumentException("Invalid log Level specified by " + logLevelVariableName + " argument value: \"" + logLevelString + "\"");
				}
			}

			boolean requireVersion = argsHandler.foundFlag(requireVersionFlagName);
			
			Configurator.setAllLevels(LogManager.getRootLogger().getName(), logLevel);

			String dataStoreLocation = null;
			if (System.getProperty("isaacDatabaseLocation") != null && argsHandler.foundVariable(dbVariableName)
					&& !System.getProperty("isaacDatabaseLocation").equals(argsHandler.getEnforcedSingleValueFromVariable(dbVariableName))) {
				throw new IllegalArgumentException("Cannot specify database using both VM arg \"-DisaacDatabaseLocation=" + System.getProperty("isaacDatabaseLocation") + "\" and commandline arg \"" + argsHandler.getFlagAndVariablePrefix() + dbVariableName + "=" + argsHandler.getEnforcedSingleValueFromVariable(dbVariableName) + "\"");
			} else if (System.getProperty("isaacDatabaseLocation") == null && ! argsHandler.foundVariable(dbVariableName)) {
				throw new IllegalArgumentException("Must specify database using either VM arg \"-DisaacDatabaseLocation={databaseDir}\" or commandline arg \"" + argsHandler.getFlagAndVariablePrefix() + dbVariableName + "={databaseDir}\"");
			} else if (argsHandler.foundVariable(dbVariableName)) {
				dataStoreLocation = argsHandler.getEnforcedSingleValueFromVariable(dbVariableName);
			} else /* if (System.getProperty("isaacDatabaseLocation") != null)  */ {
				dataStoreLocation = System.getProperty("isaacDatabaseLocation");
			}
			isaacInit(getDataStore(dataStoreLocation));

			StampCoordinate stampCoordinate = null;
			LanguageCoordinate languageCoordinate = null;
			Set<UUID> componentUuidsSpecified = new HashSet<>();
			try {
				if (argsHandler.foundVariable(tgtUuidOrNidVariableName)) {
					// If tgtUuidOrNidVariableName passed then interpret variable as UUID or NID
					// Get valid UUID from tgtUuidOrNidVariableName value or throw exception
					for (String tgtUuidOrNidVariableValue : argsHandler.getDistinctValuesFromVariable(tgtUuidOrNidVariableName)) {
						try {
							componentUuidsSpecified.add(getUuidForExistingComponent(tgtUuidOrNidVariableValue));
						} catch (Exception e) {
							throw new IllegalArgumentException("Variable argument \"" + argsHandler.getFlagAndVariablePrefix() + tgtUuidOrNidVariableName + "\" passed invalid argument \"" + tgtUuidOrNidVariableValue + "\"", e);
						}
					}
				} else if (argsHandler.foundVariable(idSrcUuidOrNidVariableName)) {
					// If idSrcUuidOrNidVariableName set then tgtIdVariableName must also be set
					// idSrcUuidOrNidVariableName variable value is IDENTIFIER_SOURCE concept id
					// and
					// tgtIdVariableName variable is an identifier value

					// Get IDENTIFIER_SOURCE concept UUID
					UUID identifierSourceConceptUuid = null;
					ConceptChronology<? extends ConceptVersion<?>> identifierConcept = null;
					try {
						identifierSourceConceptUuid = getExistingConcept(argsHandler.getEnforcedSingleValueFromVariable(idSrcUuidOrNidVariableName)).getPrimordialUuid();
						identifierConcept = Get.conceptService().getConcept(identifierSourceConceptUuid);
					} catch (Exception e) {
						throw new IllegalArgumentException("Variable argument \"" + argsHandler.getFlagAndVariablePrefix() + idSrcUuidOrNidVariableName + "\" passed invalid argument \"" + argsHandler.getEnforcedSingleValueFromVariable(idSrcUuidOrNidVariableName) + "\"", e);
					}
					// Get sequences of concepts matching passed value for passed IDENTIFIER_SOURCE
					Set<Integer> nidsForComponentsMatchingId = new HashSet<>();
					try {
						nidsForComponentsMatchingId.addAll(Frills.getReferencedComponentNidsForIdByIdentifierType(identifierConcept.getConceptSequence(), argsHandler.getEnforcedSingleValueFromVariable(tgtIdVariableName)));
					} catch (Exception e) {
						throw new RuntimeException("No components found matching IDENTIFIER_SOURCE concept " + argsHandler.getEnforcedSingleValueFromVariable(idSrcUuidOrNidVariableName) + " and value \"" + argsHandler.getEnforcedSingleValueFromVariable(tgtIdVariableName) + "\"", e);
					}
					if (nidsForComponentsMatchingId.size() == 0) {
						throw new RuntimeException("No components found matching IDENTIFIER_SOURCE concept " + argsHandler.getEnforcedSingleValueFromVariable(idSrcUuidOrNidVariableName) + " and value \"" + argsHandler.getEnforcedSingleValueFromVariable(tgtIdVariableName) + "\"");
					} else {
						for (int nid : nidsForComponentsMatchingId) {
							componentUuidsSpecified.add(Get.identifierService().getUuidPrimordialForNid(nid).get());
						}
					}
				} else if (argsHandler.foundFlag(dbInfoFlagName)) {
					// This is a minimally sufficient flag. 
				} else {
					throw new IllegalArgumentException("Invalid command line syntax: " + args);
				}
				
				if (argsHandler.foundFlag(dbInfoFlagName)) {
					MavenArtifactInfo info = MetaInfReader.readDbMetadata();
					System.out.println(MavenArtifactInfo.toString(info));
				}

				if (argsHandler.foundVariable(moduleVariableName)) {
					if (componentUuidsSpecified.size() == 0) {
						throw new IllegalArgumentException("Cannot specify \"--" + moduleVariableName + "\" unless \"--" + tgtUuidOrNidVariableName + "\" or (\"--" + idSrcUuidOrNidVariableName + "\" and \"--" + idSrcUuidOrNidVariableName + "\") specified");
					}
					Set<String> moduleIdStrings = argsHandler.getDistinctValuesFromVariable(moduleVariableName);
					if (moduleIdStrings.size() > 0) {
						Set<Integer> existingModules = Frills.getAllModuleSequences();
						Set<Integer> moduleSequences = new HashSet<>();
						for (String moduleIdString : moduleIdStrings) {
							try {
								ConceptChronology<? extends ConceptVersion<?>> moduleConcept = getExistingConcept(moduleIdString);
								if (! existingModules.contains(moduleConcept.getConceptSequence())) {
									throw new IllegalArgumentException("Concept \"" + Get.conceptDescriptionText(moduleConcept.getConceptSequence()) + "\" for id \"" + moduleIdString + "\" is not a valid module concept");
								}
								moduleSequences.add(moduleConcept.getConceptSequence());
							} catch (Exception e) {
								throw new IllegalArgumentException("Failed to get module concept for id \"" + moduleIdString + "\"");
							}
						}

						if (stampCoordinate == null) {
							stampCoordinate = getDefaultStampCoordinate();
						}
						stampCoordinate = stampCoordinate.makeAnalog(ConceptSequenceSet.of(moduleSequences));
					}
				}
				if (argsHandler.foundVariable(dtVariableName)) {
					if (componentUuidsSpecified.size() == 0) {
						throw new IllegalArgumentException("Cannot specify \"--" + dtVariableName + "\" unless \"--" + tgtUuidOrNidVariableName + "\" or (\"--" + idSrcUuidOrNidVariableName + "\" and \"--" + idSrcUuidOrNidVariableName + "\") specified");
					}
					String dateTimeString = null;
					try {
						dateTimeString = argsHandler.getEnforcedSingleValueFromVariable(dtVariableName);

						if (stampCoordinate == null) {
							stampCoordinate = getDefaultStampCoordinate();
						}
						long dateTime = NumberUtils.createLong(dateTimeString).longValue();
						if (dateTime < 0) {
							dateTime = Long.MAX_VALUE;
						}
						stampCoordinate = stampCoordinate.makeAnalog(NumberUtils.createLong(dateTimeString).longValue());
					} catch (Exception e) {
						throw new IllegalArgumentException("Failed to create stamp coordinate with invalid time \"" + dateTimeString + "\"", e);
					}
				}

				if (argsHandler.foundVariable(pathVariableName)) {
					if (componentUuidsSpecified.size() == 0) {
						throw new IllegalArgumentException("Cannot specify \"--" + pathVariableName + "\" unless \"--" + tgtUuidOrNidVariableName + "\" or (\"--" + idSrcUuidOrNidVariableName + "\" and \"--" + idSrcUuidOrNidVariableName + "\") specified");
					}
					try {
						UUID pathConceptUuid = getExistingConcept(argsHandler.getEnforcedSingleValueFromVariable(pathVariableName)).getPrimordialUuid();

						if (stampCoordinate == null) {
							stampCoordinate = getDefaultStampCoordinate();
						}
						
						stampCoordinate = Frills.getStampCoordinateVaryingByPath(stampCoordinate, Get.identifierService().getConceptSequenceForUuids(pathConceptUuid));
					} catch (Exception e) {
						throw new IllegalArgumentException("Failed to create stamp coordinate with invalid PATH concept \"" + argsHandler.getEnforcedSingleValueFromVariable(pathVariableName) + "\"", e);
					}
				}

				if (argsHandler.foundVariable(precedenceVariableName)) {
					if (componentUuidsSpecified.size() == 0) {
						throw new IllegalArgumentException("Cannot specify \"--" + precedenceVariableName + "\" unless \"--" + tgtUuidOrNidVariableName + "\" or (\"--" + idSrcUuidOrNidVariableName + "\" and \"--" + idSrcUuidOrNidVariableName + "\") specified");
					}
					String precedenceString = null;
					try {
						precedenceString = argsHandler.getEnforcedSingleValueFromVariable(precedenceVariableName);

						if (stampCoordinate == null) {
							stampCoordinate = getDefaultStampCoordinate();
						}
						stampCoordinate = Frills.getStampCoordinateVaryingByPrecedence(stampCoordinate, StampPrecedence.fromString(precedenceString).get());
					} catch (Exception e) {
						throw new IllegalArgumentException("Failed to create stamp coordinate with invalid StampPrecedence \"" + precedenceString + "\"", e);
					}
				}

				if (argsHandler.foundVariable(stateVariableName)) {
					if (componentUuidsSpecified.size() == 0) {
						throw new IllegalArgumentException("Cannot specify \"--" + stateVariableName + "\" unless \"--" + tgtUuidOrNidVariableName + "\" or (\"--" + idSrcUuidOrNidVariableName + "\" and \"--" + idSrcUuidOrNidVariableName + "\") specified");
					}
					Set<String> stateStrings = argsHandler.getDistinctValuesFromVariable(stateVariableName);
					if (stateStrings.size() > 0) {
						Set<State> states = new HashSet<>();
						for (String stateString : stateStrings) {
							Optional<State> optionalState = State.getFromString(stateString);
							if (optionalState.isPresent()) {
								states.add(optionalState.get());
							} else {
								throw new IllegalArgumentException("Failed to State value for " + stateVariableName + " argument variable \"" + stateString + "\"");
							}
						}

						if (stampCoordinate == null) {
							stampCoordinate = getDefaultStampCoordinate();
						}
						stampCoordinate = stampCoordinate.makeAnalog(EnumSet.copyOf(states));
					}
				}

				if (argsHandler.foundVariable(languageVariableName)) {
					if (componentUuidsSpecified.size() == 0) {
						throw new IllegalArgumentException("Cannot specify \"--" + languageVariableName + "\" unless \"--" + tgtUuidOrNidVariableName + "\" or (\"--" + idSrcUuidOrNidVariableName + "\" and \"--" + idSrcUuidOrNidVariableName + "\") specified");
					}
					try {
						UUID pathConceptUuid = getExistingConcept(argsHandler.getEnforcedSingleValueFromVariable(languageVariableName)).getPrimordialUuid();

						if (languageCoordinate == null) {
							languageCoordinate = getDefaultLanguageCoordinate();
						}
						
						languageCoordinate = Frills.getLanguageCoordinateVaryingByLanguage(languageCoordinate, Get.identifierService().getConceptSequenceForUuids(pathConceptUuid));
					} catch (Exception e) {
						throw new IllegalArgumentException("Failed to create LanguageCoordinate with invalid LANGUAGE concept \"" + argsHandler.getEnforcedSingleValueFromVariable(languageVariableName) + "\"", e);
					}
				}
				if (argsHandler.foundVariable(descriptionTypeVariableName)) {
					if (componentUuidsSpecified.size() == 0) {
						throw new IllegalArgumentException("Cannot specify \"--" + descriptionTypeVariableName + "\" unless \"--" + tgtUuidOrNidVariableName + "\" or (\"--" + idSrcUuidOrNidVariableName + "\" and \"--" + idSrcUuidOrNidVariableName + "\") specified");
					}
					Set<String> descriptionTypeIdStrings = argsHandler.getDistinctValuesFromVariable(descriptionTypeVariableName);
					if (descriptionTypeIdStrings.size() > 0) {
						List<Integer> descriptionTypeSequences = new LinkedList<>();
						for (String descriptionTypeIdString : descriptionTypeIdStrings) {
							try {								
								descriptionTypeSequences.add(getExistingConcept(descriptionTypeIdString).getConceptSequence());
							} catch (Exception e) {
								throw new IllegalArgumentException("Failed to get description type concept for id \"" + descriptionTypeIdString + "\"");
							}
						}

						if (languageCoordinate == null) {
							languageCoordinate = getDefaultLanguageCoordinate();
						}
						try {
							languageCoordinate = Frills.getLanguageCoordinateVaryingByDescriptionTypePreferenceList(languageCoordinate, descriptionTypeSequences);
						} catch (Exception e) {
							throw new IllegalArgumentException("Failed to create LanguageCoordinate with passed description type preference concept id list \"" + argsHandler.getValuesFromVariable(descriptionTypeVariableName) + "\"", e);
						}
					}
				}
			} catch (Exception e) {
				System.err.println(argsHandler.getUsageSummary());
				throw e;
			}

			if (componentUuidsSpecified.size() > 0) {
				System.out.println("There were " + componentUuidsSpecified.size() + " distinct components specified:");
				for (UUID uuid : componentUuidsSpecified) {
					System.out.println("\t" + Frills.describeExistingComponent(uuid, argsHandler.foundFlag(uuidsOnlyFlagName)));
				}
			}
			
			//TaxonomyCoordinate taxonomyCoordinate = TaxonomyCoordinates.getStatedTaxonomyCoordinate(stampCoordinate != null ? stampCoordinate : StampCoordinates.getDevelopmentLatest(), languageCoordinate != null ? languageCoordinate : LanguageCoordinates.getUsEnglishLanguageFullySpecifiedNameCoordinate());
			List<UUID> componentsToDump = new LinkedList<>();
			for (UUID componentUuidSpecified : componentUuidsSpecified) {
				if (argsHandler.foundFlag(memberConceptsFlagName) || argsHandler.foundFlag(memberSememesFlagName)) {
					// Just validate specified component as a valid assemblage concept
					Optional<? extends ObjectChronology<? extends StampedVersion>> objectSpecified = Get.identifiedObjectService().getIdentifiedObjectChronology(Get.identifierService().getNidForUuids(componentUuidSpecified));
					if (! objectSpecified.isPresent() || objectSpecified.get().getOchreObjectType() != OchreExternalizableObjectType.CONCEPT) {
						throw new IllegalArgumentException(objectSpecified.get().getOchreObjectType() + " specified by " + componentUuidSpecified + " is not a concept");
					}
				} else {
					componentsToDump.add(componentUuidSpecified);
				}
			}

			if (argsHandler.foundFlag(memberConceptsFlagName) || argsHandler.foundFlag(memberSememesFlagName)) {
				// Generate list of components to dump based on flags
				for (UUID componentUuidSpecified : componentUuidsSpecified) {
					ConceptChronology<? extends ConceptVersion<?>> assemblageConcept = Get.conceptService().getConcept(componentUuidSpecified);
					SememeSequenceSet sememesOfAssemblage = Get.sememeService().getSememeSequencesFromAssemblage(assemblageConcept.getConceptSequence());
					if (argsHandler.foundFlag(memberConceptsFlagName)) {
						for (int sememeSeq : sememesOfAssemblage.asArray()) {
							SememeChronology<? extends SememeVersion<?>> sememe = Get.sememeService().getSememe(sememeSeq);
							if (sememe.getSememeType() == SememeType.MEMBER || (sememe.getSememeType() == SememeType.DYNAMIC && ((DynamicSememeImpl)sememe).getData() == null)) {
								componentsToDump.add(Get.identifierService().getUuidPrimordialForNid(sememe.getReferencedComponentNid()).get());
							}
						}
					}
					if (argsHandler.foundFlag(memberSememesFlagName)) {
						for (int sememeSeq : sememesOfAssemblage.asArray()) {
							componentsToDump.add(Get.identifierService().getUuidPrimordialFromSememeId(sememeSeq).get());
						}
					}
				}
			}

			if (componentsToDump.size() > 0) {
				System.out.println("Dumping " + componentsToDump.size() + " distinct components:");
				for (UUID uuid : componentsToDump) {
					System.out.println("\t" + Frills.describeExistingComponent(uuid, argsHandler.foundFlag(uuidsOnlyFlagName)));
				}
			}
			
			if (argsHandler.foundFlag(countOnlyFlagName)) {
				return;
			}

			for (UUID componentUuid : componentsToDump) {
				Integer componentNid = Get.identifierService().getNidForUuids(componentUuid);

				// Get ObjectChronologyType for componentNid
				ObjectChronologyType componentChronologyType = Get.identifierService().getChronologyTypeForNid(componentNid);
				// Switch on componentChronologyType
				switch (componentChronologyType) {
				case CONCEPT:
					// Retrieve concept chronology for passed UUID 
					ConceptChronology<? extends ConceptVersion<?>> conceptChronology = Get.conceptService().getConcept(componentUuid);
					// Recursively dump characteristics of retrieved concept for passed UUID
					ComponentDumperDiagnosticUtil.dumpConceptChronology(
							System.out,
							conceptChronology,
							stampCoordinate,
							languageCoordinate,
							requireVersion,
							! argsHandler.foundFlag(noRecurseFlagName),
							argsHandler.foundFlag(listChildrenFlagName),
							argsHandler.foundFlag(uuidsOnlyFlagName),
							argsHandler.foundFlag(dynSememeColsFlagName));
					break;
				case SEMEME:
					// Retrieve sememe chronology for NID retrieved by passedUUID 
					SememeChronology<? extends SememeVersion<?>> sememeChronology = Get.sememeService().getSememe(componentNid);
					// If sememe chronology specified by passedUUID is a DESCRIPTION
					if (sememeChronology.getSememeType() == SememeType.DESCRIPTION) {
						@SuppressWarnings({ "unchecked" })
						SememeChronology<? extends DescriptionSememe<?>> descriptionChronology = (SememeChronology<? extends DescriptionSememe<?>>)sememeChronology;
						// Use description-specific method to recursively dump sememe characteristics
						ComponentDumperDiagnosticUtil.dumpDescriptionChronology(
								System.out,
								0,
								"Dumping",
								descriptionChronology,
								stampCoordinate,
								requireVersion,
								! argsHandler.foundFlag(noRecurseFlagName),
								argsHandler.foundFlag(uuidsOnlyFlagName),
								argsHandler.foundFlag(dynSememeColsFlagName));
					} else {
						// Use general-purpose method to recursively dump sememe characteristics
						ComponentDumperDiagnosticUtil.dumpSememeChronology(
								System.out,
								sememeChronology,
								stampCoordinate,
								requireVersion,
								! argsHandler.foundFlag(noRecurseFlagName),
								argsHandler.foundFlag(uuidsOnlyFlagName),
								argsHandler.foundFlag(dynSememeColsFlagName));
					}
					break;

				case UNKNOWN_NID:
				default:
					throw new RuntimeException("Unsupported component chronology type " + componentChronologyType);
				}
			}
		}
		finally {
			cleanup();
		}
	}
}