/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.ochre.deployment.publish;

import java.util.ArrayList;
import java.util.List;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.DataTypeException;
import ca.uhn.hl7v2.model.Message;
import ca.uhn.hl7v2.model.v24.message.MFN_M01;
import ca.uhn.hl7v2.model.v24.message.MFR_M01;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;
import ca.uhn.hl7v2.parser.PipeParser;
import gov.vha.isaac.ochre.services.exception.STSException;

public class HL7BaseGenerator
{
	private static final String ACTIVE_FLAG = "1";
	private static final String INACTIVE_FLAG = "0";
	public static final String VERSION_FIELD_NAME = "version";
	public static final String TERMINOLOGY_VERSION_SUBSET = "VERSION";
	public static final String STATUS_FIELD_NAME = "Status";
	public static final String TERMINOLOGY_VERSION_NAME = "GLOBAL VERSION";
	public static final String MAPPINGS_IDENTIFIER = "Mappings";

	/**
	 * The <code>getMessage</code> method converts from a MFN_M01 message object
	 * generated by the HAPI library to a String representation of the message
	 * that may be printed to the console, etc.
	 * 
	 * @param theMessage
	 * @return String HL7 message
	 * @throws STSException
	 */
	public static String getMessage(MFN_M01 theMessage) throws STSException {
		try {
			String messageString = null;

			PipeParser parser = new PipeParser();
			messageString = parser.encode(theMessage);

			return messageString;
		} catch (EncodingNotSupportedException e) {
			throw new STSException(e);
		} catch (HL7Exception e) {
			throw new STSException(e);
		}
	}

	/**
	 * The <code>getMessage</code> method converts from an HL7 message string to
	 * a MFN_M01 message object.
	 *
	 * @param messageString
	 * @return MFN_M01 message object
	 * @throws STSException
	 */
	public static MFN_M01 getMessageMFN(String messageString) throws STSException {
		MFN_M01 messageObject = new MFN_M01();

		try {
			PipeParser parser = new PipeParser();
			Message theMessage = parser.parse(messageString);

			messageObject = (MFN_M01) theMessage;
		} catch (HL7Exception e) {
			throw new STSException(e);
		}

		return messageObject;
	}
	
	/**
	 * The <code>getMessage</code> method converts from a MFR_M01 message object
	 * generated by the HAPI library to a String representation of the message
	 * that may be printed to the console, etc.
	 * 
	 * @param theMessage
	 * @return String HL7 message
	 * @throws STSException
	 */
	public static String getMessage(MFR_M01 theMessage) throws STSException {
		try {
			String messageString = null;

			PipeParser parser = new PipeParser();
			messageString = parser.encode(theMessage);

			return messageString;
		} catch (EncodingNotSupportedException e) {
			throw new STSException(e);
		} catch (HL7Exception e) {
			throw new STSException(e);
		}
	}
	
	/**
	 * The <code>getMessage</code> method converts from an HL7 message string to
	 * a MFR_M01 message object.
	 *
	 * @param messageString
	 * @return MFR_M01 message object
	 * @throws STSException
	 */
	public static MFR_M01 getMessageMFR(String messageString) throws STSException {
		MFR_M01 messageObject = new MFR_M01();

		try {
			PipeParser parser = new PipeParser();
			Message theMessage = parser.parse(messageString);

			messageObject = (MFR_M01) theMessage;
		} catch (HL7Exception e) {
			throw new STSException(e);
		}

		return messageObject;
	}

	/**
	 * The <code>getCombinedMessage</code> method returns a single message in
	 * the form of a String. This single message represents a concatenation of
	 * all messages in the List that was passed to the method. The concatenation
	 * is done in List order. Resulting message has a single HL7 message header.
	 *
	 * @param messages
	 * @return String String message
	 * @return STSException
	 */
	public static String getCombinedMessage(List<String> messages) throws STSException {
		String hl7DateString = HL7DateHelper.getHL7DateFormat(HL7DateHelper.getCurrentDateTime());

		String hl7MergeMessage = "";

		// remove any messages that are zero-length or null and add to new
		// List<String>
		List<String> cleanedMessageList = new ArrayList<String>();
		for (String message : messages) {
			if (message != null && message.length() != 0) {
				cleanedMessageList.add(message);
			}
		}

		for (int i = 0; i < cleanedMessageList.size(); i++) {
			String hl7Message = cleanedMessageList.get(i);

			boolean isFirstMessage = (i == 0);
			int startIndex = isFirstMessage ? 0 : hl7Message.indexOf("MFE", hl7Message.indexOf("MFI"));

			boolean isLastMessage = (i == (cleanedMessageList.size() - 1));
			int endIndex = isLastMessage ? hl7Message.length() : hl7Message.indexOf("MFE^MUP^^^VERSION");

			// now concatenate this message to the existing message
			hl7MergeMessage += hl7Message.substring(startIndex, endIndex);
		}

		if (hl7MergeMessage == null || hl7MergeMessage.length() == 0) {
			return hl7MergeMessage;
		}
		// Convert to MFN_M01 form so we can use HAPI to set date-time fields to
		// current date-time
		MFN_M01 MfnM01Message = getMessageMFN(hl7MergeMessage);
		try {
			MfnM01Message.getMSH().getDateTimeOfMessage().getTimeOfAnEvent().setValue(hl7DateString);
			MfnM01Message.getMFI().getEffectiveDateTime().getTimeOfAnEvent().setValue(hl7DateString);
			MfnM01Message.getMFI().getEnteredDateTime().getTimeOfAnEvent().setValue(hl7DateString);

			// Convert back to String form and return
			return getMessage(MfnM01Message);
		} catch (DataTypeException e) {
			throw new STSException(e);
		}
	}
	
	/**
	 * Converts the boolean status value retrieved from the database and
	 * converts it to a value recognized by VistA
	 * 
	 * @param conceptStatus
	 * @return String
	 */
	public static String getStatusValue(boolean conceptStatus) {
		String statusValue = null;

		if (conceptStatus == true) {
			statusValue = ACTIVE_FLAG;
		} else {
			statusValue = INACTIVE_FLAG;
		}

		return statusValue;
	}

}
