/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.ochre.deployment.hapi.extension.hl7.message;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.TaxonomyService;
import gov.vha.isaac.ochre.api.chronicle.LatestVersion;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptService;
import gov.vha.isaac.ochre.api.component.concept.ConceptSnapshot;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.DynamicSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.SememeVersion;
import gov.vha.isaac.ochre.api.component.sememe.version.StringSememe;
import gov.vha.isaac.ochre.api.coordinate.StampCoordinate;
import gov.vha.isaac.ochre.associations.AssociationInstance;
import gov.vha.isaac.ochre.associations.AssociationUtilities;
import gov.vha.isaac.ochre.deployment.publish.HL7DiscoveryGenerator;
import gov.vha.isaac.ochre.impl.utility.Frills;
import gov.vha.isaac.ochre.model.configuration.LanguageCoordinates;
import gov.vha.isaac.ochre.model.configuration.StampCoordinates;
import gov.vha.isaac.ochre.modules.vhat.VHATConstants;
import gov.vha.isaac.ochre.services.dto.publish.NameValueDTO;
import gov.vha.isaac.ochre.services.dto.publish.PublishConceptDTO;
import gov.vha.isaac.ochre.services.dto.publish.PublishRegionDTO;

/**
 * {@link HL7SiteDiscovery}
 *
 * @author 
 */
public class HL7SiteDiscovery {
	
	private static Logger log = LogManager.getLogger(HL7SiteDiscovery.class);

	private static TaxonomyService ts = Get.taxonomyService();
	
	private static StampCoordinate STAMP_COORDINATES = StampCoordinates.getDevelopmentLatest();	
	
	// ConceptChronology: VHAT Attribute Types <261>
	// uuid:8287530a-b6b0-594d-bf46-252e09434f7e
	// VHAT Metadata -> "Attribute Types"
	private final UUID vhatPropertyTypesUUID = UUID.fromString("8287530a-b6b0-594d-bf46-252e09434f7e");
	private final int vhatPropertyTypesNid = Get.identifierService().getNidForUuids(vhatPropertyTypesUUID);

	// conceptChronology: CODE (ISAAC) <77>
	// uuid:803af596-aea8-5184-b8e1-45f801585d17
	private final UUID codeAssemblageUUID = MetaData.CODE.getPrimordialUuid();
	private final int codeAssemblageConceptSeq = Get.identifierService().getConceptSequenceForUuids(codeAssemblageUUID);
	
	/**
	 * 
	 */
	public HL7SiteDiscovery() {
	}
	
	/**
	 * 
	 * @param vuid
	 * @return
	 * @throws Exception
	 */
	public PublishRegionDTO discovery(Long vuid) throws Exception{
		Optional<Integer> nid = Frills.getNidForVUID(vuid);
		Optional<ConceptSnapshot> conceptSnapshot =  Frills.getConceptSnapshot(nid.get(), null, null);
		
		Set<Integer> childIds = new HashSet<Integer>();
		
		if (conceptSnapshot.isPresent()) {
			childIds = Frills.getAllChildrenOfConcept(conceptSnapshot.get().getConceptSequence(), true, false);
		}
		
		ConceptService conceptService = Get.conceptService();
		List<PublishConceptDTO> publishConceptDTOList = new ArrayList<PublishConceptDTO>();
	
		for (Integer id : childIds) {
			ConceptChronology<? extends ConceptVersion<?>> concept = conceptService.getConcept(id);
			publishConceptDTOList.add(getConceptDetails(concept));
		}
		
		PublishRegionDTO publishRegionDTO = new PublishRegionDTO(conceptSnapshot.get().getConceptDescriptionText() , publishConceptDTOList);
		
		return publishRegionDTO;
	}
	
	/**
	 * 
	 * @param concept
	 * @return
	 * @throws Exception
	 */
	private PublishConceptDTO  getConceptDetails(ConceptChronology<?> concept) throws Exception {
		PublishConceptDTO publishConceptDTO = null;

			try {
				publishConceptDTO = getDesignations(concept);
				publishConceptDTO.setPropertyList(getProperties(concept.getNid()));
				publishConceptDTO.setRelationshipList(getRelationships(concept));
			} catch (Exception ex) {
				String msg = String.format("A system error occured while searching for %s.", concept.getConceptDescriptionText());
				log.error(msg, ex);
				throw new Exception(msg);
			}
		
		return publishConceptDTO;
		
	}
	
	/**
	 * 
	 * @param concept
	 * @return
	 */
	private static List<NameValueDTO> getRelationships(ConceptChronology<?> concept) {

		List<NameValueDTO> relationships = new ArrayList<>();

		for (AssociationInstance ai : AssociationUtilities.getSourceAssociations(concept.getNid(), STAMP_COORDINATES)) {
			String name = null;
			Long code = null;

			try {
		
				if (ai.getTargetComponent().isPresent() && !("has_parent".equalsIgnoreCase(ai.getAssociationType().getDescription()))) {
					name = ai.getAssociationType().getAssociationName();
					code = getCodeFromNid(ai.getTargetComponent().get().getNid());
					
					log.info("Relationship name : {}", name);
					log.info("Relationship code : {}", code);
					
					if (ai.getData().getState() == State.ACTIVE){
						NameValueDTO relationship = new NameValueDTO(name, code.toString());
						relationships.add(relationship);
					}
				}
			} catch (Exception e) {
				log.error("Association build failure");
			}	
		}
		return relationships;
	}
	
	/**
	 * 
	 * @param concept
	 * @return
	 */
	private static PublishConceptDTO getDesignations(ConceptChronology<?> concept) {

		PublishConceptDTO publishConceptDTO = new PublishConceptDTO();

		Get.sememeService().getSememesForComponent(concept.getNid())
				.filter(s -> s.getSememeType() == SememeType.DESCRIPTION).forEach(sememe -> {

					@SuppressWarnings({ "unchecked", "rawtypes" })
					Optional<LatestVersion<DescriptionSememe>> descriptionVersion = ((SememeChronology) sememe)
							.getLatestVersion(DescriptionSememe.class, STAMP_COORDINATES);
					
					if (descriptionVersion.isPresent()) {
						publishConceptDTO.setPublishName(descriptionVersion.get().value().getText());
						publishConceptDTO.setActive(descriptionVersion.get().value().getState() == State.ACTIVE);
						publishConceptDTO.setVuid(getCodeFromNid(sememe.getNid()));
						
						log.debug("Designation Code: {}", publishConceptDTO.getVuid());
						log.debug("Designation Name: {}", publishConceptDTO.getPublishName());
						log.debug("Designation Status: {}", publishConceptDTO.isActive());
						
						List<NameValueDTO> designations = new ArrayList<>();

						Get.sememeService().getSememesForComponent(sememe.getNid()).forEach((nestedSememe) -> {

							// skip code and vuid properties - they are handled already
							if (nestedSememe.getAssemblageSequence() != MetaData.VUID.getConceptSequence()
									&& nestedSememe.getAssemblageSequence() != MetaData.CODE.getConceptSequence()) {
								if (ts.wasEverKindOf(nestedSememe.getAssemblageSequence(),
										VHATConstants.VHAT_ATTRIBUTE_TYPES.getNid())) {
									designations.add(buildProperty(nestedSememe));
								}
							}
						});
						publishConceptDTO.setDesignationList(designations);
						
						designations.forEach(designation -> {
							log.debug("Designation Property name: {}", designation.getName());
							log.debug("Designation Property value: {}", designation.getValue());
						});
					}
				});
		return publishConceptDTO;
	}	
	
	/**
	 * 
	 * @param componentNid
	 * @return
	 */
	private static Long getCodeFromNid(int componentNid) {

		Optional<SememeChronology<? extends SememeVersion<?>>> sc = Get.sememeService()
				.getSememesForComponentFromAssemblage(componentNid, MetaData.CODE.getConceptSequence()).findFirst();

		if (sc.isPresent()) {

			// There was a bug in the older terminology loaders which loaded
			// 'Code' as a static sememe, but marked it as a dynamic sememe.
			// So during edits, new entries would get saves as dynamic sememes,
			// while old entries were static. Handle either....
			if (sc.get().getSememeType() == SememeType.STRING) {
				@SuppressWarnings({ "unchecked", "rawtypes" })
				Optional<LatestVersion<StringSememe<?>>> sv = ((SememeChronology) sc.get())
						.getLatestVersion(StringSememe.class, STAMP_COORDINATES);
				if (sv.isPresent()) {
					return Long.parseLong(sv.get().value().getString());
				}
			}
			// this path will become dead code, after the data is fixed.
			else if (sc.get().getSememeType() == SememeType.DYNAMIC) {
				@SuppressWarnings({ "unchecked", "rawtypes" })
				Optional<LatestVersion<? extends DynamicSememe>> sv = ((SememeChronology) sc.get())
						.getLatestVersion(DynamicSememe.class, STAMP_COORDINATES);
				if (sv.isPresent()) {
					if (sv.get().value().getData() != null && sv.get().value().getData().length == 1) {
						return Long.parseLong(sv.get().value().getData()[0].dataToString());
					}
				}
			} else {
				log.error("Unexpected sememe type for 'Code' sememe on nid " + componentNid);
			}
		}
		return null;
	}
	
	/**
	 * 
	 * @param conceptNid
	 * @return
	 */
	private static String getPreferredNameDescriptionType(int conceptNid) {
		ArrayList<String> descriptions = new ArrayList<>(1);
		ArrayList<String> inActiveDescriptions = new ArrayList<>(1);
		Get.sememeService().getDescriptionsForComponent(conceptNid).forEach(sememeChronology -> {
			@SuppressWarnings({ "rawtypes", "unchecked" })
			Optional<LatestVersion<DescriptionSememe<?>>> latestVersion = ((SememeChronology) sememeChronology)
					.getLatestVersion(DescriptionSememe.class, STAMP_COORDINATES);
			if (latestVersion.isPresent() && VHATConstants.VHAT_PREFERRED_NAME.getPrimordialUuid().equals(
					Frills.getDescriptionExtendedTypeConcept(STAMP_COORDINATES, sememeChronology.getNid(), false).orElse(null))) {
				if (latestVersion.get().value().getState() == State.ACTIVE) {
					descriptions.add(latestVersion.get().value().getText());
				} else {
					inActiveDescriptions.add(latestVersion.get().value().getText());
				}
			}
		});

		if (descriptions.size() == 0) {
			descriptions.addAll(inActiveDescriptions);
		}
		if (descriptions.size() == 0) {
			// This doesn't happen for concept that represent subsets, for
			// example.
			log.debug("Failed to find a description flagged as preferred on concept "
					+ Get.identifierService().getUuidPrimordialForNid(conceptNid));
			String description = Frills.getDescription(conceptNid, STAMP_COORDINATES,
					LanguageCoordinates.getUsEnglishLanguagePreferredTermCoordinate()).orElse("ERROR!");
			if (description.equals("ERROR!")) {
				log.error("Failed to find any description on concept "
						+ Get.identifierService().getUuidPrimordialForNid(conceptNid));
			}
			return description;
		}
		if (descriptions.size() > 1) {
			log.warn("Found " + descriptions.size() + " descriptions flagged as the 'Preferred' vhat type on concept "
					+ Get.identifierService().getUuidPrimordialForNid(conceptNid));
		}
		return descriptions.get(0);
	}
	
	/**
	 * 
	 * @param componentNid
	 * @return
	 */
	private List<NameValueDTO> getProperties(int componentNid) {
		ArrayList<NameValueDTO> properties = new ArrayList<>();
		Get.sememeService().getSememesForComponent(componentNid).forEach((sememe) -> {
			
			// skip code and vuid properties - they have special handling
			if (sememe.getAssemblageSequence() != MetaData.VUID.getConceptSequence()
					&& sememe.getAssemblageSequence() != codeAssemblageConceptSeq
					&& ts.wasEverKindOf(sememe.getAssemblageSequence(), vhatPropertyTypesNid)) {

				properties.add(buildProperty(sememe));
			}
		});

		return properties;
	}	
	
	/**
	 * 
	 * @param sememe
	 * @return
	 */
	private static NameValueDTO buildProperty(SememeChronology<?> sememe) {
		String value = null;
		String name = getPreferredNameDescriptionType(
				Get.identifierService().getConceptNid(sememe.getAssemblageSequence()));

		if (sememe.getSememeType() == SememeType.DYNAMIC) {
			@SuppressWarnings({ "unchecked", "rawtypes" })
			Optional<LatestVersion<? extends DynamicSememe>> sememeVersion = ((SememeChronology) sememe)
					.getLatestVersion(DynamicSememe.class, STAMP_COORDINATES);
			if (sememeVersion.isPresent() && sememeVersion.get().value().getData() != null
					&& sememeVersion.get().value().getData().length > 0) {

				if (!"has_parent".equals(sememeVersion.get().value().getData())) {
					value = sememeVersion.get().value().getData()[0] == null ? null
							: sememeVersion.get().value().getData()[0].dataToString();
				}
			}
		} else if (sememe.getSememeType() == SememeType.STRING) {
			@SuppressWarnings({ "unchecked", "rawtypes" })
			Optional<LatestVersion<? extends StringSememe>> sememeVersion = ((SememeChronology) sememe)
					.getLatestVersion(StringSememe.class, STAMP_COORDINATES);
			if (sememeVersion.isPresent()) {

				if (!"has_parent".equals(sememeVersion.get().value().getString())) {
					value = sememeVersion.get().value().getString();
				}
			}
		} else {
			log.warn("Unexpectedly passed sememe " + sememe + " when we only expected a dynamic or a string type");
			return null;
		}

		return new NameValueDTO(name, value);
	}
		
}
