package gov.vha.isaac.ochre.deployment.hapi.extension.hl7.message;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.access.maint.deployment.dto.PublishChecksumMessage;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.chronicle.LatestVersion;
import gov.vha.isaac.ochre.api.collections.ConceptSequenceSet;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptService;
import gov.vha.isaac.ochre.api.component.concept.ConceptSnapshot;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.DynamicSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.StringSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeUsageDescription;
import gov.vha.isaac.ochre.api.coordinate.StampCoordinate;
import gov.vha.isaac.ochre.api.coordinate.StampPrecedence;
import gov.vha.isaac.ochre.impl.utility.Frills;
import gov.vha.isaac.ochre.model.coordinate.StampCoordinateImpl;
import gov.vha.isaac.ochre.model.coordinate.StampPositionImpl;
import gov.vha.isaac.ochre.services.terminology.DomainConfig;
import gov.vha.isaac.ochre.services.terminology.SubsetConfig;
import gov.vha.isaac.ochre.services.terminology.TerminologyConfigHelper;
import gov.vha.isaac.ochre.services.utility.ChecksumCalculator;


/**
 * Computes an ISAAC based checksum for each of the subsets indicated in the list of PublishChecksumMessages.
 * Each PublishChecksumMessage is expected a provide a VUID of the subset requiring checksum computation
 * in the subset field.
 */
public class ConstructChecksum {


	/** The stamp coordinates. */
	private static StampCoordinate STAMP_COORDINATES = new StampCoordinateImpl(StampPrecedence.PATH,
			new StampPositionImpl(System.currentTimeMillis(), MetaData.DEVELOPMENT_PATH.getConceptSequence()),
			ConceptSequenceSet.EMPTY, State.ANY_STATE_SET);
	
	/** The Constant LOG. */
	private static final Logger LOG = LogManager.getLogger(ConstructChecksum.class);
	
	/**
	 * Checksum.
	 *
	 * @param publishMessages the publish messages
	 * @return the list
	 * @throws Exception the exception
	 */
	public static List<ChecksumComposite> checksum(List<PublishChecksumMessage> publishMessages) throws Exception {
		
		List<ChecksumComposite> checksumCompositeList = new ArrayList<>();
		
		// build map of required properties for each subset
		Map<String, List<String>> subsetToProperties = new HashMap<>();
		Map<String, String> vuidToSubsetName = new HashMap<>();
		
		// use TerminologyConfigHelper to parse TerminologyConfigDefault.xml to
		// get required properties
		List<DomainConfig> domainConfigs = TerminologyConfigHelper.getDomains(false);
		for (DomainConfig domainConfig : domainConfigs) {
		  for (SubsetConfig subsetConfig : domainConfig.getSortedSubsets()) {
			  subsetToProperties.put(subsetConfig.getName(), subsetConfig.getPropertyNameList());
			  vuidToSubsetName.put(subsetConfig.getTreeVuid(), subsetConfig.getName());
		  }
		}
		
		// for each request, compute checksum for the subset
		for (PublishChecksumMessage message : publishMessages) {
		  
		  StringBuffer checksumInput = new StringBuffer();
		  Set<Integer> childIds = new HashSet<>();
			
		  String vuid = message.getSubset();
						
		  // get the subset concept and all of its children
		  ConceptSnapshot result = null;
		  Optional<Integer> nid = Frills.getNidForVUID(Long.parseLong(vuid));
		  Optional<ConceptSnapshot> c = Frills.getConceptSnapshot(nid.get().intValue(), null, null);
		  if (c.isPresent()) {
			result = c.get();
			childIds = Frills.getAllChildrenOfConcept(result.getConceptSequence(), false, false);
		  }
		  ConceptService conceptService = Get.conceptService();
		
		  LinkedHashMap<String, String> idToRowMap = new LinkedHashMap<>();
		  
		  // for each concept in the subset
		  for (Integer id : childIds) {
			// get the concept details
			ConceptChronology<? extends ConceptVersion<?>>  conceptChronology = 
					conceptService.getConcept(id);
			
			HashMap<String, String> conceptProperties = getDesignations(conceptChronology);

			// save details into idToRowMap
			StringBuffer childRow = new StringBuffer();
			String subsetName = vuidToSubsetName.get(vuid);
			List<String> propertyNames = subsetToProperties.get(subsetName);
			childRow.append(conceptProperties.get("Preferred Name VUID/Code (ISAAC)"));
			childRow.append(conceptChronology.getConceptDescriptionList().get(0).getVersionList().get(0).getText());
			for (String propertyName : propertyNames) {
				if (conceptProperties.get(propertyName + " (ISAAC)") != null) {
				  childRow.append(conceptProperties.get(propertyName + " (ISAAC)"));
				}
			}
			idToRowMap.put(conceptProperties.get("Preferred Name VUID/Code (ISAAC)"), childRow.toString());
		  }

		  // sort entries by id
		  Map<String, String> idToRowMapSorted = idToRowMap.entrySet().stream()
					.sorted(Map.Entry.comparingByKey())
					.collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue,
					(oldValue, newValue) -> oldValue, LinkedHashMap::new));
		  
		  // write out details into checksum input one child row at a time
		  for (String childRow : idToRowMapSorted.values()) {
			  checksumInput.append(childRow);

			  LOG.debug(childRow.substring(7));
		  }
		
		  String checksum = ChecksumCalculator.compute(checksumInput.toString());
						
		  // add to checksum list to be returned
		  LOG.info("checksum for " + vuid + " :" + checksum);
		  LOG.info("checksumInput: " + checksumInput);
		  checksumCompositeList.add(new ChecksumComposite(checksumInput.toString(), checksum, vuid));
			
		}
		
		return checksumCompositeList;
		
	}
	

	/**
	 * Gets the designations as a list of concept properties.
	 *
	 * @param concept the concept
	 * @return the designations
	 */
	private static HashMap<String, String> getDesignations(ConceptChronology<?> concept) {
		HashMap<String, String> conceptProperties = new HashMap<>();
		
		Get.sememeService().getSememesForComponent(concept.getNid())
				.filter(s -> s.getSememeType() == SememeType.DESCRIPTION
				).forEach(sememe -> {

					@SuppressWarnings({ "unchecked", "rawtypes" })
					Optional<LatestVersion<DescriptionSememe>> descriptionVersion = ((SememeChronology) sememe)
							.getLatestVersion(DescriptionSememe.class, STAMP_COORDINATES);
					
					if (descriptionVersion.isPresent()) {
						
						Get.sememeService().getSememesForComponent(sememe.getNid()).forEach((nestedSememe) -> {							
								buildProperty(nestedSememe, conceptProperties);
						});
						
					}

				});
		return conceptProperties;
	}
	
	/**
	 * Builds the property.
	 *
	 * @param sememe the sememe
	 * @param conceptProperties the concept properties
	 */
	private static void buildProperty(SememeChronology<?> sememe, HashMap<String, String> conceptProperties) {

		if (sememe.getSememeType() == SememeType.DYNAMIC) {
			@SuppressWarnings({ "unchecked", "rawtypes" })
			Optional<LatestVersion<? extends DynamicSememe>> sememeVersion = ((SememeChronology) sememe)
					.getLatestVersion(DynamicSememe.class, STAMP_COORDINATES);
			if (sememeVersion.isPresent() && sememeVersion.get().value().getData() != null
					&& sememeVersion.get().value().getData().length > 0) {

				if (!"has_parent".equals(sememeVersion.get().value().getData())) {
					
					if (sememeVersion.get().value().getData()[0] != null) {
						DynamicSememeUsageDescription dsud = sememeVersion.get().value().getDynamicSememeUsageDescription();
						String dsn = dsud.getDynamicSememeName();
						conceptProperties.put(dsn, sememeVersion.get().value().getData()[0].dataToString());
						LOG.debug(dsn + ": " + (sememeVersion.get().value().getData()[0] == null ? null
								: sememeVersion.get().value().getData()[0].dataToString()));
					}
				}
			}
		} else if (sememe.getSememeType() == SememeType.STRING) {
			@SuppressWarnings({ "unchecked", "rawtypes" })
			Optional<LatestVersion<? extends StringSememe>> sememeVersion = ((SememeChronology) sememe)
					.getLatestVersion(StringSememe.class, STAMP_COORDINATES);
			if (sememeVersion.isPresent()) {

				if (!"has_parent".equals(sememeVersion.get().value().getString())) {
					LOG.debug("Preferred Name VUID/Code (ISAAC): " + sememeVersion.get().value().getString());
					if (sememeVersion.get().value().getString() != null) {
						conceptProperties.put("Preferred Name VUID/Code (ISAAC)", sememeVersion.get().value().getString());
					}
				}
			}
		} 

		
	}
}
