package gov.va.med.term.vhat.mojo;

import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.And;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.ConceptAssertion;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.NecessarySet;

import java.io.File;
import java.io.FileWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;

import gov.va.med.term.vhat.data.SqlDataReader;
import gov.va.med.term.vhat.data.dto.ConceptImportDTO;
import gov.va.med.term.vhat.data.dto.DesignationExtendedImportDTO;
import gov.va.med.term.vhat.data.dto.DesignationImportDTO;
import gov.va.med.term.vhat.data.dto.MapEntryImportDTO;
import gov.va.med.term.vhat.data.dto.MapSetImportDTO;
import gov.va.med.term.vhat.data.dto.NamedPropertiedItemImportDTO;
import gov.va.med.term.vhat.data.dto.PropertyImportDTO;
import gov.va.med.term.vhat.data.dto.RelationshipImportDTO;
import gov.va.med.term.vhat.data.dto.SubsetImportDTO;
import gov.va.med.term.vhat.data.dto.TypeImportDTO;
import gov.va.med.term.vhat.propertyTypes.PT_Annotations;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Associations;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Descriptions;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Relations;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.Property;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyAssociation;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.ValuePropertyPair;
import gov.va.oia.terminology.converters.sharedUtils.sql.H2DatabaseHandle;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.chronicle.ObjectChronologyType;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.DynamicSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeColumnInfo;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeData;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeDataType;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeValidatorType;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.dataTypes.DynamicSememeString;
import gov.vha.isaac.ochre.api.constants.DynamicSememeConstants;
import gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder;
import gov.vha.isaac.ochre.api.logic.assertions.Assertion;
import gov.vha.isaac.ochre.api.logic.assertions.ConceptAssertion;
import gov.vha.isaac.ochre.api.util.StringUtils;
import gov.vha.isaac.ochre.mapping.constants.IsaacMappingConstants;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeArrayImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeIntegerImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeLongImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeNidImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeStringImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeUUIDImpl;

/**
 * Goal which converts VHAT data into the workbench jbin format
 * 
 * Open the H2 Server and set these indexes _before_ running.
 * 
 * CREATE INDEX IDX_CON_ID ON CONCEPT(ID);
 * CREATE INDEX IDX_CON_EID ON CONCEPT(ENTITY_ID);
 * CREATE INDEX IDX_MEX_MEID ON MAPENTRYEXTENSION (MAPENTRYID);
 * CREATE INDEX IDX_REL_EID ON RELATIONSHIP (ENTITY_ID);
 * CREATE INDEX IDX_REL_SEID ON RELATIONSHIP (SOURCE_ENTITY_ID);
 * CREATE INDEX IDX_REL_TEID ON RELATIONSHIP (TARGET_ENTITY_ID);
 * CREATE INDEX IDX_REL_KIND ON RELATIONSHIP (KIND);
 * CREATE INDEX IDX_REL_VID ON RELATIONSHIP (VERSION_ID);
 * CREATE INDEX IDX_CON_TID ON CONCEPT(TYPE_ID);
 * CREATE INDEX IDX_VER_ID ON VERSION(ID);
 * CREATE INDEX IDX_TYPE_ID ON TYPE(ID);
 */
@Mojo (name = "convert-VHAT-SQL-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class VHATSQLImportMojo extends ConverterBaseMojo
{
	private IBDFCreationUtility importUtil_;

	private HashMap<UUID, String> referencedConcepts = new HashMap<>();
	private HashMap<UUID, String> loadedConcepts = new HashMap<>();
	private UUID rootConceptUUID;

	private PropertyType attributes_, descriptions_, associations_, relationships_;
	private BPT_Refsets refsets_;

	private UUID allVhatConceptsRefset;

	private HashSet<String> conceptsWithNoDesignations = new HashSet<>();
	private int mapEntryCount = 0;
	private int mapSetCount = 0;
	private int conceptCount = 0;
	
	private HashMap<UUID, String> associationOrphanConcepts = new HashMap<>();
	private HashMap<UUID, MapEntryImportDTO> associationOrphanConceptsMapEntries = new HashMap<>();
	
	private SqlDataReader importer = new SqlDataReader();
	
	private HashMap<String, UUID> codeToUUIDs = new HashMap<>();
	private HashMap<Long, UUID> vuidToUUIDs = new HashMap<>(); // Keep track of UUIDs for versions
	private ArrayList<UUID> loadedRefsets = new ArrayList<>();
	private ArrayList<UUID> loadedGraphs = new ArrayList<>();
	private ArrayList<String> descriptionsCodes = new ArrayList<>();
	private ArrayList<Long> descriptionsVUIDs = new ArrayList<>();
	private ArrayList<UUID> loadedMapEntries = new ArrayList<>();
	
	
	@Override
	public void execute() throws MojoExecutionException
	{

		try
		{
			super.execute();

			String temp = converterOutputArtifactVersion.substring(0, 11);

			importUtil_ = new IBDFCreationUtility(Optional.of("VHAT " + converterSourceArtifactVersion), Optional.of(MetaData.VHA_MODULES), outputDirectory, 
					converterOutputArtifactId, converterOutputArtifactVersion, converterOutputArtifactClassifier, true, 
					new SimpleDateFormat("yyyy.MM.dd").parse(temp).getTime());
			
			attributes_ = new PT_Annotations();
			descriptions_ = new BPT_Descriptions("VHAT");
			associations_ = new BPT_Associations();
			relationships_ = new BPT_Relations("VHAT");
			refsets_ = new BPT_Refsets("VHAT");
			refsets_.addProperty("All VHAT Concepts");

			ComponentReference vhatMetadata = ComponentReference.fromConcept(createType(MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid(), 
					"VHAT Metadata" +  IBDFCreationUtility.metadataSemanticTag_));
			
			importUtil_.loadTerminologyMetadataAttributes(vhatMetadata, converterSourceArtifactVersion, 
					Optional.empty(), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);
			
			// TODO: would be nice to automate this
			importUtil_.registerDynamicSememeColumnInfo(IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_NID_EXTENSION.getUUID(), 
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_NID_EXTENSION.getDynamicSememeColumns());
			importUtil_.registerDynamicSememeColumnInfo(IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_SEMEME_TYPE.getUUID(), 
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_SEMEME_TYPE.getDynamicSememeColumns());
			importUtil_.registerDynamicSememeColumnInfo(IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getUUID(), 
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getDynamicSememeColumns());

			importer.process();

			List<TypeImportDTO> dto = new ArrayList<>();
			if (importer.getTypes().isPresent())
			{
				dto = importer.getTypes().get();
			}

			// Read in the dynamic types
			for (TypeImportDTO typeImportDTO : dto)
			{
				if (typeImportDTO.getKind().equals("D")) // DesignationType
				{
					Property p = descriptions_.addProperty(typeImportDTO.getName());
					// Add some rankings for FSN / synonym handling
					if (p.getSourcePropertyNameFSN().equals("Fully Specified Name"))
					{
						p.setPropertySubType(BPT_Descriptions.FSN);
					}
					else if (p.getSourcePropertyNameFSN().equals("Preferred Name"))
					{
						p.setPropertySubType(BPT_Descriptions.SYNONYM);
					}
					else if (p.getSourcePropertyNameFSN().equals("Synonym"))
					{
						p.setPropertySubType(BPT_Descriptions.SYNONYM + 1);
					}
				}
				else if (typeImportDTO.getKind().equals("R")) //RelationshipType
				{
					// Currently loading isA as a graph rel, and an assoiation
					if (typeImportDTO.getName().equals("has_parent"))
					{
						Property p = relationships_.addProperty(typeImportDTO.getName());
						p.setWBPropertyType(MetaData.IS_A.getPrimordialUuid());
					}
					associations_.addProperty(new PropertyAssociation(associations_, typeImportDTO.getName(), typeImportDTO.getName(), null, 
							typeImportDTO.getName(), false));
				}
				else if (typeImportDTO.getKind().equals("P")) //PropertyType
				{
					// GEM_Flags are loaded here so they are available as a column on a mapset
					attributes_.addProperty(typeImportDTO.getName());
				}
				else
				{
					System.err.println("Unexpected Type!");
				}
			}
			
			// Get the refset names
			if (importer.getFullListOfSubsets().isPresent())
			{
				for (SubsetImportDTO subset : importer.getFullListOfSubsets().get())
				{
					refsets_.addProperty(subset.getSubsetName());
				}
			}
			
			importUtil_.loadMetaDataItems(Arrays.asList(descriptions_, attributes_, associations_, relationships_, refsets_), vhatMetadata.getPrimordialUuid());

			ConsoleUtil.println("Metadata load stats");
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(line);
			}
			
			importUtil_.clearLoadStats();

			allVhatConceptsRefset = refsets_.getProperty("All VHAT Concepts").getUUID();
			loadedConcepts.put(allVhatConceptsRefset, "All VHAT Concepts");

			// TODO: Could get rid of this calculation now that things are structured better....
			
			Map<Long, Set<String>> subsetMembershipMap = new HashMap<>();
			
			if (importer.getSubsetsByVuid().isPresent())
			{
				Map<Long, String> _subsetsMap = importer.getSubsetsByVuid().get();
				Map<Long, ArrayList<Long>> subsetMembershipsCodesByVuid = importer.getSubsetMembershipsCodesByVuid().orElse(null);
				
				for (Map.Entry<Long, String> entry : _subsetsMap.entrySet())
				{
					Long subsetVuid = entry.getKey();
					ArrayList<Long> subsetCodes = subsetMembershipsCodesByVuid.get(subsetVuid);
					if (subsetCodes != null)
					{
						for (Long code : subsetCodes)
						{
							Set<String> codes = subsetMembershipMap.get(subsetVuid);
							if (codes == null)
							{
								codes = new HashSet<>();
								subsetMembershipMap.put(subsetVuid, codes);
							}
							codes.add(code.toString());
						}
					}
				}
				
				for (Map.Entry<Long, String> entry : _subsetsMap.entrySet())
				{
					Long subsetVuid = entry.getKey();
					String subsetName = entry.getValue();
					loadRefset(subsetName, subsetVuid, subsetMembershipMap.get(subsetVuid));
				}
			}
			
			//TODO use the codesystem version info?
			//TODO use the Version info?

			if (importer.getConcepts().isPresent())
			{
				for (Map.Entry<Long, ArrayList<ConceptImportDTO>> entry : importer.getConcepts().get().entrySet())
				{
					long conceptEntityId = Long.valueOf(entry.getKey()).longValue();
					for (ConceptImportDTO item : entry.getValue())
					{
						conceptCount++; 
						writeEConcept(item, conceptEntityId);
						if (conceptCount % 100 == 0)
						{
							ConsoleUtil.showProgress();
						}
						if (conceptCount % 3000 == 0)
						{
							ConsoleUtil.println("Processed " + conceptCount + " concepts");
						}
					}
				}
			}
			
			// Not needed any more, let the GC reclaim memory to speed up processing
			importer.clearRelationships();
			ConsoleUtil.println("Processed " + conceptCount + " concepts");
			ConsoleUtil.println("Starting mapsets");
			
			if (importer.getMapSets().isPresent())
			{
				for (Map.Entry<Long, ArrayList<MapSetImportDTO>> entry : importer.getMapSets().get().entrySet())
				{
					long conceptEntityId = Long.valueOf(entry.getKey().longValue());
					for (MapSetImportDTO item : entry.getValue())
					{
						conceptCount++;
						writeEConcept(item, conceptEntityId);
						ConsoleUtil.println("Processed " + mapSetCount + " mapsets with " + mapEntryCount + " members");
					}
				}
			}
			ConsoleUtil.println("Finished mapsets");
			
			// Put in names instead of IDs so the load stats print nicer:
			Hashtable<String, String> stringsToSwap = new Hashtable<String, String>();
			if (importer.getFullListOfSubsets().isPresent())
			{
				for (SubsetImportDTO subset : importer.getFullListOfSubsets().get())
				{
					stringsToSwap.put(subset.getVuid() + "", subset.getSubsetName());
				}
			}
			
			if (importer.getFullListOfMapSets().isPresent())
			{
				for (MapSetImportDTO mapSet : importer.getFullListOfMapSets().get())
				{
					stringsToSwap.put(mapSet.getVuid() + "", mapSet.getName());
				}
			}
			
			// Not needed any more, let the GC reclaim memory to speed up processing
			importer = null;

			ArrayList<UUID> missingConcepts = new ArrayList<>();

			for (UUID refUUID : referencedConcepts.keySet())
			{
				if (loadedConcepts.get(refUUID) == null)
				{
					missingConcepts.add(refUUID);
					ConsoleUtil.printErrorln("Data error - The concept " + refUUID + " - " + referencedConcepts.get(refUUID)
							+ " was referenced, but not loaded - will be created as '-MISSING-'");
				}
			}

			if (missingConcepts.size() > 0)
			{
				ConceptChronology<? extends ConceptVersion<?>> missingParent = importUtil_.createConcept("Missing Concepts", true);
				importUtil_.addParent(ComponentReference.fromChronology(missingParent), rootConceptUUID);
				for (UUID refUUID : missingConcepts)
				{
					ComponentReference c = ComponentReference.fromChronology(importUtil_.createConcept(refUUID, "-MISSING-", true));
					importUtil_.addParent(c, missingParent.getPrimordialUuid());
				}
			}
			
			// Handle missing association sources and targets
			ConsoleUtil.println("Creating placeholder concepts for " + associationOrphanConcepts.size() + " association orphans");
			// We currently don't have these association targets, so need to invent placeholder concepts.
			ComponentReference missingSDORefset = ComponentReference.fromConcept(importUtil_.createConcept(null, "Missing SDO Code System Concepts", 
					null, null, null, refsets_.getPropertyTypeUUID(), refsets_.getSecondParentUUID()));
			importUtil_.configureConceptAsDynamicRefex(missingSDORefset, "A simple refset to store the missing concepts we have to create during import because"
					+ " we don't yet have the SDO code systems in place",
					null, ObjectChronologyType.CONCEPT, null);
			
			int orphanCount = 0;
			for (Entry<UUID, String> item : associationOrphanConcepts.entrySet())
			{
				if (loadedConcepts.get(item.getKey()) == null)
				{
					importUtil_.addRefsetMembership(ComponentReference.fromConcept(importUtil_.createConcept(item.getKey(), item.getValue(), true)), 
							missingSDORefset.getPrimordialUuid(), State.ACTIVE, associationOrphanConceptsMapEntries.get(item.getKey()).getTime());
				}
				
				orphanCount++;
				if (orphanCount % 50000 == 0)
				{
					ConsoleUtil.println("Processed " + orphanCount + " association orphans");
				}
			}
			ConsoleUtil.println("Processed " + associationOrphanConcepts.size() + " association orphans");
			
			ConsoleUtil.println("Load Statistics");
			// Swap out vuids with names to make it more readable...
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				Enumeration<String> e = stringsToSwap.keys();
				while (e.hasMoreElements())
				{
					String current = e.nextElement();
					line = line.replaceAll(current, stringsToSwap.get(current));
				}
				ConsoleUtil.println(line);
			}

			// This could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "vhatUuid");

			if (conceptsWithNoDesignations.size() > 0)
			{
				ConsoleUtil.printErrorln(conceptsWithNoDesignations.size() + " concepts were found with no descriptions at all.  These were assigned '-MISSING-'");
				FileWriter fw = new FileWriter(new File(outputDirectory, "NoDesignations.txt"));
				for (String s : conceptsWithNoDesignations)
				{
					fw.write(s);
					fw.write(System.getProperty("line.separator"));
				}
				fw.close();
			}
			importUtil_.shutdown();
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		}
		catch (Exception ex)
		{
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}

	}

	private void writeEConcept(NamedPropertiedItemImportDTO conceptOrMapSet, Long conceptEntityId) throws Exception
	{
		boolean isMapSet = false;
		if (conceptOrMapSet instanceof MapSetImportDTO)
		{
			isMapSet = true;
			mapSetCount++;
		}

		ComponentReference concept = ComponentReference.fromConcept(importUtil_.createConcept(getConceptUuid(conceptOrMapSet.getCode()), conceptOrMapSet.getTime(), 
				conceptOrMapSet.isActive() ? State.ACTIVE : State.INACTIVE, null));
		
		// Only add Code to the chronology if it changed
		if (!codeToUUIDs.containsKey(conceptOrMapSet.getCode()))
		{
			// For use with Relationships later
			codeToUUIDs.put(conceptOrMapSet.getCode(), getConceptUuid(conceptOrMapSet.getCode()));
			importUtil_.addStaticStringAnnotation(concept, conceptOrMapSet.getCode(), attributes_.getProperty("Code").getUUID(), State.ACTIVE);
			loadedConcepts.put(concept.getPrimordialUuid(), conceptOrMapSet.getCode());
		}
		
		// Only add VUID to the chronology if it changed
		if (!vuidToUUIDs.containsKey(conceptOrMapSet.getVuid()))
		{
			vuidToUUIDs.put(conceptOrMapSet.getVuid(), getConceptUuid(conceptOrMapSet.getVuid().toString()));
			importUtil_.addStaticStringAnnotation(concept, conceptOrMapSet.getVuid().toString(), attributes_.getProperty("VUID").getUUID(), State.ACTIVE);
		}
		
		ArrayList<ValuePropertyPairExtended> descriptionHolder = new ArrayList<>();
		
		// All designations using DesignationExtendedImportDTO class
		if (importer.getDesignationsForEntity(conceptEntityId).isPresent())
		{
			for (DesignationExtendedImportDTO didto : importer.getDesignationsForEntity(conceptEntityId).get())
			{
				descriptionHolder.add(new ValuePropertyPairExtended(didto.getValueNew(), getDescriptionUuid(didto.getCode().toString()),
						descriptions_.getProperty(didto.getTypeName()), didto, !didto.isActive()));
			}
		}
		
		if (importer.getPropertiesForEntity(conceptEntityId).isPresent())
		{
			for (PropertyImportDTO property : importer.getPropertiesForEntity(conceptEntityId).get())
			{
				ConverterUUID.disableUUIDMap_ = true;
				importUtil_.addStringAnnotation(concept, property.getValueNew(), attributes_.getProperty(property.getTypeName()).getUUID(), 
						property.isActive() ? State.ACTIVE : State.INACTIVE);
				ConverterUUID.disableUUIDMap_ = false;
			}
		}
		
		// Per Dan, this is a valid reason to create duplicate UUIDs, as 
		// only the active/inactive values have changed
		ConverterUUID.disableUUIDMap_ = true;
		List<SememeChronology<DescriptionSememe<?>>> wbDescriptions = importUtil_.addDescriptions(concept, descriptionHolder);
		ConverterUUID.disableUUIDMap_ = false;
		
		//Descriptions have now all been added to the concepts - now we need to process the rest of the ugly bits of vhat
		//and place them on the descriptions.
		for (int i = 0; i < descriptionHolder.size(); i++)
		{
			ValuePropertyPairExtended vpp = descriptionHolder.get(i);
			SememeChronology<DescriptionSememe<?>> desc = wbDescriptions.get(i);
			
			if (vpp.getValue().equals("VHAT"))
			{
				// On the root node, we need to add some extra attributes
				importUtil_.addDescription(concept, "VHAT", DescriptionType.SYNONYM, true, null, State.ACTIVE);
				importUtil_.addDescription(concept, "VHA Terminology", DescriptionType.SYNONYM, false, null, State.ACTIVE);
				ConsoleUtil.println("Root concept FSN is 'VHAT' and the UUID is " + concept.getPrimordialUuid());
				importUtil_.addParent(concept, MetaData.ISAAC_ROOT.getPrimordialUuid());
				rootConceptUUID = concept.getPrimordialUuid();
			}
			
			if (!descriptionsVUIDs.contains(vpp.getDesignationImportDTO().getVuid()))
			{
				importUtil_.addStaticStringAnnotation(ComponentReference.fromChronology(desc, () -> "Description"), vpp.getDesignationImportDTO().getVuid() + "", 
						attributes_.getProperty("VUID").getUUID(), State.ACTIVE);
				descriptionsVUIDs.add(vpp.getDesignationImportDTO().getVuid());
			}
			if (!descriptionsCodes.contains(vpp.getDesignationImportDTO().getCode()))
			{
				importUtil_.addStaticStringAnnotation(ComponentReference.fromChronology(desc, () -> "Description"), vpp.getDesignationImportDTO().getCode(), 
						attributes_.getProperty("Code").getUUID(), State.ACTIVE);
				descriptionsCodes.add(vpp.getDesignationImportDTO().getCode());
			}

			// VHAT is kind of odd, in that the attributes are attached to the description, 
			// rather than the concept.
			if (!isMapSet && importer.getPropertiesForVuid(vpp.getDesignationImportDTO().getVuid()).isPresent())
			{
				for (PropertyImportDTO property : importer.getPropertiesForVuid(vpp.getDesignationImportDTO().getVuid()).get())
				{
					ConverterUUID.disableUUIDMap_ = true;
					importUtil_.addStringAnnotation(ComponentReference.fromChronology(desc, () -> "Description"), property.getValueNew(), 
							attributes_.getProperty(property.getTypeName()).getUUID(), property.isActive() ? State.ACTIVE : State.INACTIVE);
					ConverterUUID.disableUUIDMap_ = false;
				}
			}
		}
		
		if (descriptionHolder.size() == 0)
		{
			// Seems like a data error - but it is happening... no descriptions at all.....
			if (!conceptsWithNoDesignations.contains(conceptOrMapSet.getCode()))
			{
				conceptsWithNoDesignations.add(conceptOrMapSet.getCode());
				// The workbench implodes if you don't have a fully specified name....
				importUtil_.addDescription(concept, null, "-MISSING-", DescriptionType.FSN, true, null, null, null, null, null, 
						State.ACTIVE, conceptOrMapSet.getTime());
			}
		}
		
		LogicalExpressionBuilder leb = Get.logicalExpressionBuilderService().getLogicalExpressionBuilder();
		ArrayList<ConceptAssertion> assertions = new ArrayList<>();
		
		if (importer.getRelationshipsForEntity(conceptEntityId).isPresent())
		{
			for (RelationshipImportDTO relationshipImportDTO : importer.getRelationshipsForEntity(conceptEntityId).get())
			{
				UUID sourceUuid = getConceptUuid(conceptOrMapSet.getCode());
				UUID targetUuid = getConceptUuid(relationshipImportDTO.getNewTargetCode());

				referencedConcepts.put(targetUuid, relationshipImportDTO.getNewTargetCode());

				if (!sourceUuid.equals(concept.getPrimordialUuid()))
				{
					//throw new MojoExecutionException("Design failure!");
				}

				// TODO: I'm assuming this is also a valid area to create duplicate UUIDs, 
				// only the active/inactive values have changed
				ConverterUUID.disableUUIDMap_ = true;
				importUtil_.addAssociation(concept, null, targetUuid, associations_.getProperty(relationshipImportDTO.getTypeName()).getUUID(),
						relationshipImportDTO.isActive() ? State.ACTIVE : State.INACTIVE, relationshipImportDTO.getTime(), null);
				ConverterUUID.disableUUIDMap_ = false;
				
				//If it is an isA rel, also create it as a rel.
				if (relationships_.getProperty(relationshipImportDTO.getTypeName()) != null)
				{
					if (relationships_.getProperty(relationshipImportDTO.getTypeName()).getWBTypeUUID().equals(MetaData.IS_A.getPrimordialUuid()))
					{
						assertions.add(ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(targetUuid), leb));
					}
					else
					{
						throw new RuntimeException("Only is_a is handled so far as a relationship");
					}
				}
			}
			if (assertions.size() > 0)
			{
				// TODO: A gragh can only be loaded once, correct?
				if (!loadedGraphs.contains(concept.getPrimordialUuid()))
				{
					NecessarySet(And(assertions.toArray(new Assertion[assertions.size()])));
					importUtil_.addRelationshipGraph(concept, null, leb.build(), true, conceptOrMapSet.getTime(), null);  //TODO handle inactive
					loadedGraphs.add(concept.getPrimordialUuid());
				}
			}
		}

		if (!loadedRefsets.contains(concept.getPrimordialUuid()))
		{
			importUtil_.addRefsetMembership(concept, allVhatConceptsRefset, State.ACTIVE, conceptOrMapSet.getTime());
			loadedRefsets.add(concept.getPrimordialUuid());
		}
		
		if (isMapSet)
		{
			// We are only fetching relationships of concepts, so ignoring this
			// Add a relationship to the subsets metadata concept.
			/*if (relationshipImports != null && relationshipImports.size() > 0)
			{
				throw new RuntimeException("Didn't expect mapsets to have their own relationships!");
			}*/
			
			// Add it as an association too
			// TODO: Validate this is ok for duplicates
			ConverterUUID.disableUUIDMap_ = true;
			importUtil_.addAssociation(concept, null, refsets_.getSecondParentUUID(), associations_.getProperty("has_parent").getUUID(),
					State.ACTIVE, concept.getTime(), null);
			ConverterUUID.disableUUIDMap_ = false;
			
			// Place it in three places - refsets under VHAT Metadata, vhat refsets under SOLOR Refsets, and the dynamic sememe mapping sememe type.
			NecessarySet(And(new Assertion[] {
					ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(refsets_.getSecondParentUUID()), leb),
					ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(refsets_.getPropertyTypeUUID()), leb),
					ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_SEMEME_TYPE.getPrimordialUuid()), leb)}));

			// TODO: Validate
			if (!loadedGraphs.contains(concept.getPrimordialUuid()))
			{
				importUtil_.addRelationshipGraph(concept, null, leb.build(), true, concept.getTime(), null);
				loadedGraphs.add(concept.getPrimordialUuid());
			}
			
			MapSetImportDTO mapSet = ((MapSetImportDTO) conceptOrMapSet);
			
			// Before defining the columns, we need to determine if this mapset makes use of gem flags
			boolean mapSetDefinitionHasGemFlag = false;
			if (importer.getMapEntriesForMapSet(conceptOrMapSet.getVuid()).isPresent())
			{
				for (MapEntryImportDTO mapItem : importer.getMapEntriesForMapSet(conceptOrMapSet.getVuid()).get())
				{
					if (mapSetDefinitionHasGemFlag)
					{
						break;
					}

					if (importer.getPropertiesForVuid(mapItem.getVuid()).isPresent())
					{
						for (PropertyImportDTO mapItemProperty : importer.getPropertiesForVuid(mapItem.getVuid()).get()) 
						{
							if (mapItemProperty.getTypeName().equals("GEM_Flags"))
							{
								mapSetDefinitionHasGemFlag = true;
								break;
							}
						}
					}
				}
			}
			
			DynamicSememeColumnInfo[] columns = new DynamicSememeColumnInfo[mapSetDefinitionHasGemFlag ? 6 : 5];
			int col = 0;
			columns[col] = new DynamicSememeColumnInfo(col++, DynamicSememeConstants.get().DYNAMIC_SEMEME_COLUMN_ASSOCIATION_TARGET_COMPONENT.getUUID(),
					DynamicSememeDataType.UUID, null, false, DynamicSememeValidatorType.COMPONENT_TYPE,
					new DynamicSememeArrayImpl<>(new DynamicSememeString[] { new DynamicSememeStringImpl(ObjectChronologyType.CONCEPT.name()) }), true);
			if (mapSetDefinitionHasGemFlag)
			{
				columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_GEM_FLAGS.getUUID(),
						DynamicSememeDataType.STRING, null, false, true);
			}
			columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_EQUIVALENCE_TYPE.getUUID(), DynamicSememeDataType.UUID,
					null, false, DynamicSememeValidatorType.IS_KIND_OF, new DynamicSememeUUIDImpl(IsaacMappingConstants.get().MAPPING_EQUIVALENCE_TYPES.getUUID()), true);
			columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_SEQUENCE.getUUID(), DynamicSememeDataType.INTEGER,
					null, false, true);
			columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_GROUPING.getUUID(), DynamicSememeDataType.LONG,
					null, false, true);
			columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_EFFECTIVE_DATE.getUUID(),
					DynamicSememeDataType.LONG, null, false, true);
			
			// TODO: Not sure this is right
			ConverterUUID.disableUUIDMap_ = true;
			importUtil_.configureConceptAsDynamicRefex(concept, mapSet.getName(), columns, ObjectChronologyType.CONCEPT, null);
			
			// Annotate this concept as a mapset definition concept.
			importUtil_.addAnnotation(concept, null, null, IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_SEMEME_TYPE.getUUID(), State.ACTIVE, conceptOrMapSet.getTime());
						
			// Now that we have defined the map sememe, add the other annotations onto the map set definition.
			if (StringUtils.isNotBlank(mapSet.getSourceCodeSystemName()))
			{
				importUtil_.addAnnotation(concept, null, 
					new DynamicSememeData[] {
							new DynamicSememeNidImpl(IsaacMappingConstants.get().MAPPING_SOURCE_CODE_SYSTEM.getNid()),
							new DynamicSememeStringImpl(mapSet.getSourceCodeSystemName())},
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getPrimordialUuid(), State.ACTIVE, null, null);
			}
			
			if (StringUtils.isNotBlank(mapSet.getSourceVersionName()))
			{
				importUtil_.addAnnotation(concept, null, 
					new DynamicSememeData[] {
							new DynamicSememeNidImpl(IsaacMappingConstants.get().MAPPING_SOURCE_CODE_SYSTEM_VERSION.getNid()),
							new DynamicSememeStringImpl(mapSet.getSourceVersionName())},
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getPrimordialUuid(), State.ACTIVE, null, null);
			}
			
			if (StringUtils.isNotBlank(mapSet.getTargetCodeSystemName()))
			{
				importUtil_.addAnnotation(concept, null, 
					new DynamicSememeData[] {
							new DynamicSememeNidImpl(IsaacMappingConstants.get().MAPPING_TARGET_CODE_SYSTEM.getNid()),
							new DynamicSememeStringImpl(mapSet.getTargetCodeSystemName())},
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getPrimordialUuid(), State.ACTIVE, null, null);
			}
			
			if (StringUtils.isNotBlank(mapSet.getTargetVersionName()))
			{
				importUtil_.addAnnotation(concept, null, 
					new DynamicSememeData[] {
							new DynamicSememeNidImpl(IsaacMappingConstants.get().MAPPING_TARGET_CODE_SYSTEM_VERSION.getNid()),
							new DynamicSememeStringImpl(mapSet.getTargetVersionName())},
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getPrimordialUuid(), State.ACTIVE, null, null);
			}
			ConverterUUID.disableUUIDMap_ = false;
			
			if (importer.getMapEntriesForMapSet(conceptOrMapSet.getVuid()).isPresent())
			{
				for (MapEntryImportDTO mapItem : importer.getMapEntriesForMapSet(conceptOrMapSet.getVuid()).get())
				{
					ComponentReference sourceConcept = ComponentReference.fromConcept(mapSet.getSourceCodeSystemName().equals("VHAT") ?
							getConceptUuid(mapItem.getSourceCode()) : getAssociationOrphanUuid(mapItem.getSourceCode()));

					if (!loadedConcepts.containsKey(sourceConcept.getPrimordialUuid()))
					{
						if (mapSet.getSourceCodeSystemName().equals("VHAT")) // TODO: check this
						{
							ConsoleUtil.printErrorln("Missing VHAT association source concept! " + mapItem.getSourceCode());
						}
						associationOrphanConcepts.put(sourceConcept.getPrimordialUuid(), mapItem.getSourceCode());
						// TODO: I hate doing this, but works for now
						associationOrphanConceptsMapEntries.put(sourceConcept.getPrimordialUuid(), mapItem);
					}
					
					UUID targetConcept =  mapSet.getTargetCodeSystemName().equals("VHAT") ?
							getConceptUuid(mapItem.getTargetCode()) : getAssociationOrphanUuid(mapItem.getTargetCode());
					
					if (!loadedConcepts.containsKey(targetConcept))
					{
						if (mapSet.getTargetCodeSystemName().equals("VHAT"))
						{
							ConsoleUtil.printErrorln("Missing VHAT association target concept! " + mapItem.getTargetCode());
						}
						associationOrphanConcepts.put(targetConcept, mapItem.getTargetCode());
						// TODO: I hate doing this, but works for now
						associationOrphanConceptsMapEntries.put(targetConcept, mapItem);
					}
					
					ArrayList<PropertyImportDTO> mapItemProperties = importer.getPropertiesForVuid(mapItem.getVuid())
							.orElse(new ArrayList<PropertyImportDTO>());
					String gemFlag = null;
					for (PropertyImportDTO property : mapItemProperties)
					{
						if (property.getTypeName().equals("GEM_Flags"))
						{
							if (gemFlag != null)
							{
								throw new RuntimeException("Didn't expect multiple gem flags on a single mapItem!");
							}
							gemFlag = property.getValueNew();
						}
					}
					
					DynamicSememeData[] columnData = new DynamicSememeData[mapSetDefinitionHasGemFlag ? 6 : 5];
					col = 0;
					columnData[col++] = new DynamicSememeUUIDImpl(targetConcept);
					if (mapSetDefinitionHasGemFlag )
					{
						columnData[col++] = gemFlag == null ? null : new DynamicSememeStringImpl(gemFlag);
					}
					columnData[col++] = null;  //qualifier column
					columnData[col++] = new DynamicSememeIntegerImpl(mapItem.getSequence()); //sequence column
					columnData[col++] = mapItem.getGrouping() != null ? new DynamicSememeLongImpl(mapItem.getGrouping()) : null; //grouping column
					columnData[col++] = mapItem.getEffectiveDate() != null ? new DynamicSememeLongImpl(mapItem.getEffectiveDate().getTime()) : null; //effectiveDate
					
					// TODO: Assuming these can have duplicates?
					ConverterUUID.disableUUIDMap_ = true;
					UUID mapItemUUID = getMapItemUUID(mapSet.getVuid().toString(), mapItem.getVuid().toString());
					SememeChronology<DynamicSememe<?>> association = importUtil_.addAnnotation(sourceConcept, mapItemUUID, columnData, concept.getPrimordialUuid(), mapItem.isActive() ? State.ACTIVE : State.INACTIVE, mapSet.getTime(), null);
					ConverterUUID.disableUUIDMap_ = false;
									
					if (!loadedMapEntries.contains(mapItemUUID))
					{
						loadedMapEntries.add(mapItemUUID);
						importUtil_.addStaticStringAnnotation(ComponentReference.fromChronology(association, () -> "Association"), mapItem.getVuid().toString(), 
								attributes_.getProperty("VUID").getUUID(), State.ACTIVE);
					}
					
					for (PropertyImportDTO property : mapItemProperties)
					{
						if (property.getTypeName().equals("GEM_Flags"))
						{
							// Already handled above
						}
						else 
						{
							throw new RuntimeException("Properties on map set items not yet handled"); 
							// This code is correct, but our gui doesn't expect this, so throw an error for now, so we know if any show up.
							//importUtil_.addStringAnnotation(ComponentReference.fromChronology(association), property.getValueNew(), 
							//	attributes_.getProperty(property.getTypeName()).getUUID(), property.isActive() ? State.ACTIVE : State.INACTIVE);
						}
					}
					/* We just won't add these, as we control the data queried (if they were present)
					if (mapItem.getDesignations() != null && mapItem.getDesignations().size() > 0)
					{
						throw new RuntimeException("Designations on map set items not yet handled");
					}
					if (mapItem.getRelationships() != null && mapItem.getRelationships().size() > 0)
					{
						throw new RuntimeException("Relationships on map set items not yet handled");
					}*/
					mapEntryCount++;
				}
			}
		}
	}
	
	private ConceptChronology<? extends ConceptVersion<?>> createType(UUID parentUuid, String typeName) throws Exception
	{
		ConceptChronology<? extends ConceptVersion<?>> concept = importUtil_.createConcept(typeName, true);
		loadedConcepts.put(concept.getPrimordialUuid(), typeName);
		importUtil_.addParent(ComponentReference.fromConcept(concept), parentUuid);
		return concept;
	}

	private void loadRefset(String typeName, Long vuid, Set<String> refsetMembership) throws Exception
	{
		Property p = refsets_.getProperty(typeName);
		if (p != null)
		{
			UUID concept = refsets_.getProperty(typeName).getUUID();
			loadedConcepts.put(concept, typeName);
			if (vuid != null)
			{
				importUtil_.addStaticStringAnnotation(ComponentReference.fromConcept(concept), vuid.toString(), 
					attributes_.getProperty("VUID").getUUID(), State.ACTIVE);
			}
	
			if (refsetMembership != null)
			{
				for (String memberCode : refsetMembership)
				{
					importUtil_.addRefsetMembership(ComponentReference.fromSememe(getDescriptionUuid(memberCode)), concept, State.ACTIVE, null);
				}
			}
		}
	}
	
	private UUID getAssociationOrphanUuid(String code)
	{
		return ConverterUUID.createNamespaceUUIDFromString("associationOrphan:" + code, true);
	}
	
	private UUID getMapItemUUID(String mapSetVuid, String vuid)
	{
		return ConverterUUID.createNamespaceUUIDFromString("mapSetVuid:" + mapSetVuid + "mapItemVuid:" + vuid, false);
	}

	private UUID getConceptUuid(String codeId)
	{
		return ConverterUUID.createNamespaceUUIDFromString("code:" + codeId, true);
	}

	private UUID getDescriptionUuid(String descriptionId)
	{
		return ConverterUUID.createNamespaceUUIDFromString("description:" + descriptionId, true);
	}

	public static void main(String[] args) throws MojoExecutionException
	{
		VHATSQLImportMojo i = new VHATSQLImportMojo();
		// Change here if not using the default in vhat-mojo/src/main/resources/config.properties
		i.setupH2(new File("c:\\va\\tmp\\vts-import"));
		i.outputDirectory = new File("../vhat-ibdf-sql/target");
		i.inputFileLocation = new File("../vhat-ibdf-sql/target/generated-resources/src/");
		i.converterOutputArtifactVersion = "2017.01.24.foo";
		i.converterOutputArtifactId = "vhat-ibdf-sql";
		i.converterOutputArtifactClassifier = "";
		i.converterVersion = "SNAPSHOT";
		i.converterSourceArtifactVersion = "fre";
		i.execute();
		javafx.application.Platform.exit();
	}
	
	private void setupH2(File h2File) throws MojoExecutionException
	{
		try
		{
			H2DatabaseHandle _h2 = new H2DatabaseHandle();
			_h2.createOrOpenDatabase(h2File);
			importer.setDatabaseConnection(_h2.getConnection());
		}
		catch (Exception e)
		{
			throw new MojoExecutionException(e.getLocalizedMessage(), e);
		}
	}
	
	private class ValuePropertyPairExtended extends ValuePropertyPair
	{
		private DesignationImportDTO didto_;
		public ValuePropertyPairExtended(String value, UUID descriptionUUID, Property property, DesignationImportDTO didto, boolean disabled)
		{
			super(value, descriptionUUID, property);
			didto_ = didto;
			setDisabled(disabled);
		}
		
		public DesignationImportDTO getDesignationImportDTO()
		{
			return didto_;
		}
	}
}