package gov.va.med.term.vhat.mojo;

import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.And;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.ConceptAssertion;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.NecessarySet;
import java.io.File;
import java.io.FileWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import gov.va.med.term.vhat.data.TerminologyDataReader;
import gov.va.med.term.vhat.data.dto.ConceptImportDTO;
import gov.va.med.term.vhat.data.dto.DesignationExtendedImportDTO;
import gov.va.med.term.vhat.data.dto.DesignationImportDTO;
import gov.va.med.term.vhat.data.dto.MapEntryImportDTO;
import gov.va.med.term.vhat.data.dto.MapSetImportDTO;
import gov.va.med.term.vhat.data.dto.NamedPropertiedItemImportDTO;
import gov.va.med.term.vhat.data.dto.PropertyImportDTO;
import gov.va.med.term.vhat.data.dto.RelationshipImportDTO;
import gov.va.med.term.vhat.data.dto.SubsetImportDTO;
import gov.va.med.term.vhat.data.dto.SubsetMembershipImportDTO;
import gov.va.med.term.vhat.data.dto.TerminologyDTO;
import gov.va.med.term.vhat.data.dto.TypeImportDTO;
import gov.va.med.term.vhat.propertyTypes.PT_Annotations;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Associations;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Descriptions;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Relations;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.Property;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyAssociation;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.ValuePropertyPair;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.chronicle.ObjectChronologyType;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.DynamicSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeColumnInfo;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeData;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeDataType;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeValidatorType;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.dataTypes.DynamicSememeString;
import gov.vha.isaac.ochre.api.constants.DynamicSememeConstants;
import gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder;
import gov.vha.isaac.ochre.api.logic.assertions.Assertion;
import gov.vha.isaac.ochre.api.logic.assertions.ConceptAssertion;
import gov.vha.isaac.ochre.mapping.constants.IsaacMappingConstants;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeArrayImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeIntegerImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeLongImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeNidImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeStringImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeUUIDImpl;

/**
 * Goal which converts VHAT data into the workbench jbin format
 */
@Mojo (name = "convert-VHAT-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class VHATImportMojo extends ConverterBaseMojo
{
	private IBDFCreationUtility importUtil_;

	private HashMap<UUID, String> referencedConcepts = new HashMap<>();
	private HashMap<UUID, String> loadedConcepts = new HashMap<>();
	private UUID rootConceptUUID;

	private PropertyType attributes_, descriptions_, associations_, relationships_;
	private BPT_Refsets refsets_;

	private UUID allVhatConceptsRefset;

	private HashSet<String> conceptsWithNoDesignations = new HashSet<>();
	private int mapEntryCount = 0;
	private int mapSetCount = 0;
	private int conceptCount = 0;
	
	private HashMap<UUID, String> associationOrphanConcepts = new HashMap<>();//ID to designation

	@Override
	public void execute() throws MojoExecutionException
	{

		try
		{
			super.execute();

			String temp = converterOutputArtifactVersion.substring(0, 11);

			importUtil_ = new IBDFCreationUtility(Optional.of("VHAT " + converterSourceArtifactVersion), Optional.of(MetaData.VHA_MODULES), outputDirectory, 
					converterOutputArtifactId, converterOutputArtifactVersion, converterOutputArtifactClassifier, false, new SimpleDateFormat("yyyy.MM.dd").parse(temp).getTime());
			
			attributes_ = new PT_Annotations();
			descriptions_ = new BPT_Descriptions("VHAT");
			associations_ = new BPT_Associations();
			relationships_ = new BPT_Relations("VHAT");
			refsets_ = new BPT_Refsets("VHAT");
			refsets_.addProperty("All VHAT Concepts");

			TerminologyDataReader importer = new TerminologyDataReader(inputFileLocation);
			TerminologyDTO terminology = importer.process();

			List<TypeImportDTO> dto = terminology.getTypes();

			ComponentReference vhatMetadata = ComponentReference.fromConcept(createType(MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid(), 
					"VHAT Metadata" + IBDFCreationUtility.metadataSemanticTag_));
			
			importUtil_.loadTerminologyMetadataAttributes(vhatMetadata, converterSourceArtifactVersion, 
					Optional.empty(), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);
			
			//TODO would be nice to automate this
			importUtil_.registerDynamicSememeColumnInfo(IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_NID_EXTENSION.getUUID(), 
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_NID_EXTENSION.getDynamicSememeColumns());
			importUtil_.registerDynamicSememeColumnInfo(IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_SEMEME_TYPE.getUUID(), 
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_SEMEME_TYPE.getDynamicSememeColumns());
			importUtil_.registerDynamicSememeColumnInfo(IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getUUID(), 
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getDynamicSememeColumns());

			// read in the dynamic types
			for (TypeImportDTO typeImportDTO : dto)
			{
				if (typeImportDTO.getKind().equals("DesignationType"))
				{
					Property p = descriptions_.addProperty(typeImportDTO.getName());
					//Add some rankings for FSN / synonym handling
					if (p.getSourcePropertyNameFSN().equals("Fully Specified Name"))
					{
						p.setPropertySubType(BPT_Descriptions.FSN);
					}
					else if (p.getSourcePropertyNameFSN().equals("Preferred Name"))
					{
						p.setPropertySubType(BPT_Descriptions.SYNONYM);
					}
					else if (p.getSourcePropertyNameFSN().equals("Synonym"))
					{
						p.setPropertySubType(BPT_Descriptions.SYNONYM + 1);
					}
				}
				else if (typeImportDTO.getKind().equals("RelationshipType"))
				{
					//currently loading isA as a graph rel, and an assoiation
					if (typeImportDTO.getName().equals("has_parent"))
					{
						Property p = relationships_.addProperty(typeImportDTO.getName());
						p.setWBPropertyType(MetaData.IS_A.getPrimordialUuid());
					}
					associations_.addProperty(new PropertyAssociation(associations_, typeImportDTO.getName(), typeImportDTO.getName(), null, 
							typeImportDTO.getName(), false));
				}
				else if (typeImportDTO.getKind().equals("PropertyType"))
				{
					if (typeImportDTO.getName().equals("GEM_Flags"))
					{
						//skip - GEM_Flags are loaded as a column on a mapset instead.
					}
					else
					{
						attributes_.addProperty(typeImportDTO.getName());
					}
				}
				else
				{
					System.err.println("Unexpected Type!");
				}
			}
			
			//get the refset names
			for (SubsetImportDTO subset : terminology.getSubsets())
			{
				refsets_.addProperty(subset.getSubsetName());
			}
			
			importUtil_.loadMetaDataItems(Arrays.asList(descriptions_, attributes_, associations_, relationships_, refsets_), vhatMetadata.getPrimordialUuid());

			ConsoleUtil.println("Metadata load stats");
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(line);
			}
			
			importUtil_.clearLoadStats();

			allVhatConceptsRefset = refsets_.getProperty("All VHAT Concepts").getUUID();
			loadedConcepts.put(allVhatConceptsRefset, "All VHAT Concepts");

			//TODO Could get rid of this calculation now that things are structured better....
			
			Map<Long, Set<String>> subsetMembershipMap = new HashMap<>();
			// get the subset memberships to build the refset for each subset
			for (ConceptImportDTO item : terminology.getCodeSystem().getVersion().getConcepts())
			{
				for (DesignationExtendedImportDTO designation : item.getDesignations())
				{
					for (SubsetMembershipImportDTO subsetMembership : designation.getSubsets())
					{
						Set<String> codes = subsetMembershipMap.get(subsetMembership.getVuid());
						if (codes == null)
						{
							codes = new HashSet<>();
							subsetMembershipMap.put(subsetMembership.getVuid(), codes);
						}
						codes.add(designation.getCode());
					}
				}
			}
			
			for (SubsetImportDTO subset : terminology.getSubsets())
			{
				loadRefset(subset.getSubsetName(), subset.getVuid(), subsetMembershipMap.get(subset.getVuid()));
			}

			//TODO use the codesystem version info?
			//TODO use the Version info?
			//

			for (ConceptImportDTO item : terminology.getCodeSystem().getVersion().getConcepts())
			{
				conceptCount++;
				writeEConcept(item);
				if (conceptCount % 500 == 0)
				{
					ConsoleUtil.showProgress();
				}
				if (conceptCount % 10000 == 0)
				{
					ConsoleUtil.println("Processed " + conceptCount + " concepts");
				}
			}
			
			ConsoleUtil.println("Processed " + conceptCount + " concepts");
			ConsoleUtil.println("Starting mapsets");
			

			for (MapSetImportDTO item : terminology.getCodeSystem().getVersion().getMapsets())
			{
				mapSetCount++;
				writeEConcept(item);
				if (mapEntryCount % 100 == 0)
				{
					ConsoleUtil.showProgress();
				}
				if (mapEntryCount % 500 == 0)
				{
					ConsoleUtil.println("Processed " + mapSetCount + " mapsets with " + mapEntryCount + " members");
				}
			}
			
			ConsoleUtil.println("Processed " + mapSetCount + " mapsets with " + mapEntryCount + " members");

			ArrayList<UUID> missingConcepts = new ArrayList<>();

			for (UUID refUUID : referencedConcepts.keySet())
			{
				if (loadedConcepts.get(refUUID) == null)
				{
					missingConcepts.add(refUUID);
					ConsoleUtil.printErrorln("Data error - The concept " + refUUID + " - " + referencedConcepts.get(refUUID)
							+ " was referenced, but not loaded - will be created as '-MISSING-'");
				}
			}

			if (missingConcepts.size() > 0)
			{
				ConceptChronology<? extends ConceptVersion<?>> missingParent = importUtil_.createConcept("Missing Concepts", true);
				importUtil_.addParent(ComponentReference.fromChronology(missingParent), rootConceptUUID);
				for (UUID refUUID : missingConcepts)
				{
					ComponentReference c = ComponentReference.fromChronology(importUtil_.createConcept(refUUID, "-MISSING-", true));
					importUtil_.addParent(c, missingParent.getPrimordialUuid());
				}
			}
			
			//Handle missing association sources and targets
			ConsoleUtil.println("Creating placeholder concepts for " + associationOrphanConcepts.size() + " association orphans");
			//We currently don't have these association targets, so need to invent placeholder concepts.
			ComponentReference missingSDORefset = ComponentReference.fromConcept(importUtil_.createConcept(null, "Missing SDO Code System Concepts", 
					null, null, null, refsets_.getPropertyTypeUUID(), refsets_.getSecondParentUUID()));
			importUtil_.configureConceptAsDynamicRefex(missingSDORefset, "A simple refset to store the missing concepts we have to create during import because"
					+ " we don't yet have the SDO code systems in place",
					null, ObjectChronologyType.CONCEPT, null);
			for (Entry<UUID, String> item : associationOrphanConcepts.entrySet())
			{
				if (loadedConcepts.get(item.getKey()) == null)
				{
					importUtil_.addRefsetMembership(ComponentReference.fromConcept(importUtil_.createConcept(item.getKey(), item.getValue(), true)), 
							missingSDORefset.getPrimordialUuid(), State.ACTIVE, null);
				}
			}
			

			// Put in names instead of IDs so the load stats print nicer:
			Hashtable<String, String> stringsToSwap = new Hashtable<String, String>();
			for (SubsetImportDTO subset : terminology.getSubsets())
			{
				stringsToSwap.put(subset.getVuid() + "", subset.getSubsetName());
			}
			
			for (MapSetImportDTO mapSet : terminology.getCodeSystem().getVersion().getMapsets())
			{
				stringsToSwap.put(mapSet.getVuid() + "", mapSet.getName());
			}
			

			ConsoleUtil.println("Load Statistics");
			// swap out vuids with names to make it more readable...
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				Enumeration<String> e = stringsToSwap.keys();
				while (e.hasMoreElements())
				{
					String current = e.nextElement();
					line = line.replaceAll(current, stringsToSwap.get(current));
				}
				ConsoleUtil.println(line);
			}

			// this could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "vhatUuid");

			if (conceptsWithNoDesignations.size() > 0)
			{
				ConsoleUtil.printErrorln(conceptsWithNoDesignations.size() + " concepts were found with no descriptions at all.  These were assigned '-MISSING-'");
				FileWriter fw = new FileWriter(new File(outputDirectory, "NoDesignations.txt"));
				for (String s : conceptsWithNoDesignations)
				{
					fw.write(s);
					fw.write(System.getProperty("line.separator"));
				}
				fw.close();
			}
			importUtil_.shutdown();
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		}
		catch (Exception ex)
		{
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}

	}

	private void writeEConcept(NamedPropertiedItemImportDTO conceptOrMapSet) throws Exception
	{
		boolean isMapSet = false;
		if (conceptOrMapSet instanceof MapSetImportDTO)
		{
			isMapSet = true;
			mapSetCount++;
		}

		ComponentReference concept = ComponentReference.fromConcept(importUtil_.createConcept(getConceptUuid(conceptOrMapSet.getCode()), null, 
				conceptOrMapSet.isActive() ? State.ACTIVE : State.INACTIVE, null));
		loadedConcepts.put(concept.getPrimordialUuid(), conceptOrMapSet.getCode());
		importUtil_.addStaticStringAnnotation(concept, conceptOrMapSet.getVuid().toString(), attributes_.getProperty("VUID").getUUID(), State.ACTIVE);
		importUtil_.addStaticStringAnnotation(concept, conceptOrMapSet.getCode(), attributes_.getProperty("Code").getUUID(), State.ACTIVE);

		ArrayList<ValuePropertyPairExtended> descriptionHolder = new ArrayList<>();
		if (isMapSet)
		{
			for (DesignationImportDTO didto : ((MapSetImportDTO)conceptOrMapSet).getDesignations())
			{
				descriptionHolder.add(new ValuePropertyPairExtended(didto.getValueNew(), getDescriptionUuid(didto.getCode().toString()),
						descriptions_.getProperty(didto.getTypeName()), didto, !didto.isActive()));
			}
		}
		else
		{
			for (DesignationExtendedImportDTO didto : ((ConceptImportDTO)conceptOrMapSet).getDesignations())
			{
				descriptionHolder.add(new ValuePropertyPairExtended(didto.getValueNew(), getDescriptionUuid(didto.getCode().toString()),
						descriptions_.getProperty(didto.getTypeName()), didto, !didto.isActive()));
			}
		}
		
		for (PropertyImportDTO property : conceptOrMapSet.getProperties())
		{
			importUtil_.addStringAnnotation(concept, property.getValueNew(), attributes_.getProperty(property.getTypeName()).getUUID(), 
					property.isActive() ? State.ACTIVE : State.INACTIVE);
		}

		List<SememeChronology<DescriptionSememe<?>>> wbDescriptions = importUtil_.addDescriptions(concept, descriptionHolder);
		
		//Descriptions have now all been added to the concepts - now we need to process the rest of the ugly bits of vhat
		//and place them on the descriptions.
		for (int i = 0; i < descriptionHolder.size(); i++)
		{
			ValuePropertyPairExtended vpp = descriptionHolder.get(i);
			SememeChronology<DescriptionSememe<?>> desc = wbDescriptions.get(i);
			
			if (vpp.getValue().equals("VHAT"))
			{
				// On the root node, we need to add some extra attributes
				importUtil_.addDescription(concept, "VHAT", DescriptionType.SYNONYM, true, null, State.ACTIVE);
				importUtil_.addDescription(concept, "VHA Terminology", DescriptionType.SYNONYM, false, null, State.ACTIVE);
				ConsoleUtil.println("Root concept FSN is 'VHAT' and the UUID is " + concept.getPrimordialUuid());
				importUtil_.addParent(concept, MetaData.ISAAC_ROOT.getPrimordialUuid());
				rootConceptUUID = concept.getPrimordialUuid();
			}
			importUtil_.addStaticStringAnnotation(ComponentReference.fromChronology(desc, () -> "Description"), vpp.getDesignationImportDTO().getVuid() + "", 
				attributes_.getProperty("VUID").getUUID(), State.ACTIVE);
			importUtil_.addStaticStringAnnotation(ComponentReference.fromChronology(desc, () -> "Description"), vpp.getDesignationImportDTO().getCode(), 
					attributes_.getProperty("Code").getUUID(), State.ACTIVE);

			// VHAT is kind of odd, in that the attributes are attached to the description, rather than the concept.
			if (!isMapSet)
			{
				for (PropertyImportDTO property : ((DesignationExtendedImportDTO)vpp.getDesignationImportDTO()).getProperties())
				{
					importUtil_.addStringAnnotation(ComponentReference.fromChronology(desc, () -> "Description"), property.getValueNew(), 
							attributes_.getProperty(property.getTypeName()).getUUID(), property.isActive() ? State.ACTIVE : State.INACTIVE);
				}
			}
		}
		
		if (descriptionHolder.size() == 0)
		{
			// Seems like a data error - but it is happening... no descriptions at all.....
			conceptsWithNoDesignations.add(conceptOrMapSet.getCode());
			// The workbench implodes if you don't have a fully specified name....
			importUtil_.addDescription(concept, "-MISSING-", DescriptionType.FSN, true, null, State.ACTIVE);
		}

		List<RelationshipImportDTO> relationshipImports = conceptOrMapSet.getRelationships();
		LogicalExpressionBuilder leb = Get.logicalExpressionBuilderService().getLogicalExpressionBuilder();
		ArrayList<ConceptAssertion> assertions = new ArrayList<>();
		if (relationshipImports != null)
		{
			for (RelationshipImportDTO relationshipImportDTO : relationshipImports)
			{
				UUID sourceUuid = getConceptUuid(conceptOrMapSet.getCode());
				UUID targetUuid = getConceptUuid(relationshipImportDTO.getNewTargetCode());

				referencedConcepts.put(targetUuid, relationshipImportDTO.getNewTargetCode());

				if (!sourceUuid.equals(concept.getPrimordialUuid()))
				{
					throw new MojoExecutionException("Design failure!");
				}

				importUtil_.addAssociation(concept, null, targetUuid, associations_.getProperty(relationshipImportDTO.getTypeName()).getUUID(),
						relationshipImportDTO.isActive() ? State.ACTIVE : State.INACTIVE, null, null);
				
				//If it is an isA rel, also create it as a rel.
				
				if (relationships_.getProperty(relationshipImportDTO.getTypeName()) != null)
				{
					if (relationships_.getProperty(relationshipImportDTO.getTypeName()).getWBTypeUUID().equals(MetaData.IS_A.getPrimordialUuid()))
					{
						assertions.add(ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(targetUuid), leb));
					}
					else
					{
						throw new RuntimeException("Only is_a is handled so far as a relationship");
					}
				}
			}
			if (assertions.size() > 0)
			{
				NecessarySet(And(assertions.toArray(new Assertion[assertions.size()])));
				importUtil_.addRelationshipGraph(concept, null, leb.build(), true, null, null);  //TODO handle inactive
			}
		}

		importUtil_.addRefsetMembership(concept, allVhatConceptsRefset, State.ACTIVE, null);

		if (isMapSet)
		{
			//Add a relationship to the subsets metadata concept.
			if (relationshipImports != null && relationshipImports.size() > 0)
			{
				throw new RuntimeException("Didn't expect mapsets to have their own relationships!");
			}
			
			//add it as an association too
			importUtil_.addAssociation(concept, null, refsets_.getSecondParentUUID(), associations_.getProperty("has_parent").getUUID(),
					State.ACTIVE, null, null);
			
			//place it in three places - refsets under VHAT Metadata, vhat refsets under SOLOR Refsets, and the dynamic sememe mapping sememe type.
			NecessarySet(And(new Assertion[] {
					ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(refsets_.getSecondParentUUID()), leb),
					ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(refsets_.getPropertyTypeUUID()), leb),
					ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_SEMEME_TYPE.getPrimordialUuid()), leb)}));
			importUtil_.addRelationshipGraph(concept, null, leb.build(), true, null, null);

			MapSetImportDTO mapSet = ((MapSetImportDTO)conceptOrMapSet);
			
			//before defining the columns, we need to determine if this mapset makes use of gem flags
			boolean mapSetDefinitionHasGemFlag = false;
			for (MapEntryImportDTO mapItem : mapSet.getMapEntries())
			{
				if (mapSetDefinitionHasGemFlag)
				{
					break;
				}
				for (PropertyImportDTO mapItemProperty : mapItem.getProperties())
				{
					if (mapItemProperty.getTypeName().equals("GEM_Flags"))
					{
						mapSetDefinitionHasGemFlag = true;
						break;
					}
				}
			}
			
			
			DynamicSememeColumnInfo[] columns = new DynamicSememeColumnInfo[mapSetDefinitionHasGemFlag ? 6 : 5];
			int col = 0;
			columns[col] = new DynamicSememeColumnInfo(col++, DynamicSememeConstants.get().DYNAMIC_SEMEME_COLUMN_ASSOCIATION_TARGET_COMPONENT.getUUID(),
					DynamicSememeDataType.UUID, null, false, DynamicSememeValidatorType.COMPONENT_TYPE,
					new DynamicSememeArrayImpl<>(new DynamicSememeString[] { new DynamicSememeStringImpl(ObjectChronologyType.CONCEPT.name()) }), true);
			columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_EQUIVALENCE_TYPE.getUUID(), DynamicSememeDataType.UUID,
					null, false, DynamicSememeValidatorType.IS_KIND_OF, new DynamicSememeUUIDImpl(IsaacMappingConstants.get().MAPPING_EQUIVALENCE_TYPES.getUUID()), true);
			columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_SEQUENCE.getUUID(), DynamicSememeDataType.INTEGER,
					null, false, true);
			columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_GROUPING.getUUID(), DynamicSememeDataType.LONG,
					null, false, true);
			columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_EFFECTIVE_DATE.getUUID(),
					DynamicSememeDataType.LONG, null, false, true);
			//moved to end - make it more convenient for GUI where target and qualifier are extracted, and used elsewhere - its convenient not to have the order change.
			if (mapSetDefinitionHasGemFlag)
			{
				columns[col] = new DynamicSememeColumnInfo(col++, IsaacMappingConstants.get().DYNAMIC_SEMEME_COLUMN_MAPPING_GEM_FLAGS.getUUID(),
						DynamicSememeDataType.STRING, null, false, true);
			}
			
			importUtil_.configureConceptAsDynamicRefex(concept, mapSet.getName(), columns, ObjectChronologyType.CONCEPT, null);
			
			//Annotate this concept as a mapset definition concept.
			importUtil_.addAnnotation(concept, null, null, IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_SEMEME_TYPE.getUUID(), State.ACTIVE, null);
			
			//Now that we have defined the map sememe, add the other annotations onto the map set definition.
			if (StringUtils.isNotBlank(mapSet.getSourceCodeSystemName()))
			{
				importUtil_.addAnnotation(concept, null, 
					new DynamicSememeData[] {
							new DynamicSememeNidImpl(IsaacMappingConstants.get().MAPPING_SOURCE_CODE_SYSTEM.getNid()),
							new DynamicSememeStringImpl(mapSet.getSourceCodeSystemName())},
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getPrimordialUuid(), State.ACTIVE, null, null);
			}
			
			if (StringUtils.isNotBlank(mapSet.getSourceVersionName()))
			{
				importUtil_.addAnnotation(concept, null, 
					new DynamicSememeData[] {
							new DynamicSememeNidImpl(IsaacMappingConstants.get().MAPPING_SOURCE_CODE_SYSTEM_VERSION.getNid()),
							new DynamicSememeStringImpl(mapSet.getSourceVersionName())},
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getPrimordialUuid(), State.ACTIVE, null, null);
			}
			
			if (StringUtils.isNotBlank(mapSet.getTargetCodeSystemName()))
			{
				importUtil_.addAnnotation(concept, null, 
					new DynamicSememeData[] {
							new DynamicSememeNidImpl(IsaacMappingConstants.get().MAPPING_TARGET_CODE_SYSTEM.getNid()),
							new DynamicSememeStringImpl(mapSet.getTargetCodeSystemName())},
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getPrimordialUuid(), State.ACTIVE, null, null);
			}
			
			if (StringUtils.isNotBlank(mapSet.getTargetVersionName()))
			{
				importUtil_.addAnnotation(concept, null, 
					new DynamicSememeData[] {
							new DynamicSememeNidImpl(IsaacMappingConstants.get().MAPPING_TARGET_CODE_SYSTEM_VERSION.getNid()),
							new DynamicSememeStringImpl(mapSet.getTargetVersionName())},
					IsaacMappingConstants.get().DYNAMIC_SEMEME_MAPPING_STRING_EXTENSION.getPrimordialUuid(), State.ACTIVE, null, null);
			}
			
			for (MapEntryImportDTO mapItem : mapSet.getMapEntries())
			{
				ComponentReference sourceConcept = ComponentReference.fromConcept(mapSet.getSourceCodeSystemName().equals("VHAT") ?
						getConceptUuid(mapItem.getSourceCode()) : getAssociationOrphanUuid(mapItem.getSourceCode()));

				if (!loadedConcepts.containsKey(sourceConcept.getPrimordialUuid()))
				{
					if (mapSet.getSourceCodeSystemName().equals("VHAT"))
					{
						ConsoleUtil.printErrorln("Missing VHAT association source concept! " + mapItem.getSourceCode());
					}
					associationOrphanConcepts.put(sourceConcept.getPrimordialUuid(), mapItem.getSourceCode());
				}
				
				UUID targetConcept =  mapSet.getTargetCodeSystemName().equals("VHAT") ?
						getConceptUuid(mapItem.getTargetCode()) : getAssociationOrphanUuid(mapItem.getTargetCode());
				
				if (!loadedConcepts.containsKey(targetConcept))
				{
					if (mapSet.getTargetCodeSystemName().equals("VHAT"))
					{
						ConsoleUtil.printErrorln("Missing VHAT association target concept! " + mapItem.getTargetCode());
					}
					associationOrphanConcepts.put(targetConcept, mapItem.getTargetCode());
				}
				
				String gemFlag = null;
				if (mapItem.getProperties() != null)
				{
					for (PropertyImportDTO property : mapItem.getProperties())
					{
						if (property.getTypeName().equals("GEM_Flags"))
						{
							if (gemFlag != null)
							{
								throw new RuntimeException("Didn't expect multiple gem flags on a single mapItem!");
							}
							gemFlag = property.getValueNew();
						}
					}
				}
				
				DynamicSememeData[] columnData = new DynamicSememeData[mapSetDefinitionHasGemFlag ? 6 : 5];
				col = 0;
				columnData[col++] = new DynamicSememeUUIDImpl(targetConcept);
				columnData[col++] = null;  //qualifier column
				columnData[col++] = new DynamicSememeIntegerImpl(mapItem.getSequence()); //sequence column
				columnData[col++] = mapItem.getGrouping() != null ? new DynamicSememeLongImpl(mapItem.getGrouping()) : null; //grouping column
				columnData[col++] = mapItem.getEffectiveDate() != null ? new DynamicSememeLongImpl(mapItem.getEffectiveDate().getTime()) : null; //effectiveDate
				if (mapSetDefinitionHasGemFlag )
				{
					columnData[col++] = gemFlag == null ? null : new DynamicSememeStringImpl(gemFlag);
				}
				
				SememeChronology<DynamicSememe<?>> association = importUtil_.addAnnotation(sourceConcept, getMapItemUUID(mapSet.getVuid().toString(),
						mapItem.getVuid().toString()), columnData, concept.getPrimordialUuid(), mapItem.isActive() ? State.ACTIVE : State.INACTIVE, null, null);
				
				importUtil_.addStaticStringAnnotation(ComponentReference.fromChronology(association, () -> "Association"), mapItem.getVuid().toString(), 
						attributes_.getProperty("VUID").getUUID(), State.ACTIVE);
				
				if (mapItem.getProperties() != null)
				{
					for (PropertyImportDTO property : mapItem.getProperties())
					{
						if (property.getTypeName().equals("GEM_Flags"))
						{
							//already handled above
						}
						else 
						{
							throw new RuntimeException("Properties on map set items not yet handled"); 
							//This code is correct, but our gui doesn't expect this, so throw an error for now, so we know if any show up.
//							importUtil_.addStringAnnotation(ComponentReference.fromChronology(association), property.getValueNew(), 
//								attributes_.getProperty(property.getTypeName()).getUUID(), property.isActive() ? State.ACTIVE : State.INACTIVE);
						}
					}
				}
				if (mapItem.getDesignations() != null && mapItem.getDesignations().size() > 0)
				{
					throw new RuntimeException("Designations on map set items not yet handled");
				}
				if (mapItem.getRelationships() != null && mapItem.getRelationships().size() > 0)
				{
					throw new RuntimeException("Relationships on map set items not yet handled");
				}
				mapEntryCount++;
			}
			
		}
	}

	private ConceptChronology<? extends ConceptVersion<?>> createType(UUID parentUuid, String typeName) throws Exception
	{
		ConceptChronology<? extends ConceptVersion<?>> concept = importUtil_.createConcept(typeName, true);
		loadedConcepts.put(concept.getPrimordialUuid(), typeName);
		importUtil_.addParent(ComponentReference.fromConcept(concept), parentUuid);
		return concept;
	}

	private void loadRefset(String typeName, Long vuid, Set<String> refsetMembership) throws Exception
	{
		UUID concept = refsets_.getProperty(typeName).getUUID();
		loadedConcepts.put(concept, typeName);
		if (vuid != null)
		{
			importUtil_.addStaticStringAnnotation(ComponentReference.fromConcept(concept), vuid.toString(), 
				attributes_.getProperty("VUID").getUUID(), State.ACTIVE);
		}

		if (refsetMembership != null)
		{
			for (String memberCode : refsetMembership)
			{
				importUtil_.addRefsetMembership(ComponentReference.fromSememe(getDescriptionUuid(memberCode)), concept, State.ACTIVE, null);
			}
		}
	}
	
	private UUID getAssociationOrphanUuid(String code)
	{
		return ConverterUUID.createNamespaceUUIDFromString("associationOrphan:" + code, true);
	}
	
	private UUID getMapItemUUID(String mapSetVuid, String vuid)
	{
		return ConverterUUID.createNamespaceUUIDFromString("mapSetVuid:" + mapSetVuid + "mapItemVuid:" + vuid, false);
	}

	private UUID getConceptUuid(String codeId)
	{
		return ConverterUUID.createNamespaceUUIDFromString("code:" + codeId, true);
	}

	private UUID getDescriptionUuid(String descriptionId)
	{
		return ConverterUUID.createNamespaceUUIDFromString("description:" + descriptionId, true);
	}

	public static void main(String[] args) throws MojoExecutionException
	{
		VHATImportMojo i = new VHATImportMojo();
		i.outputDirectory = new File("../vhat-ibdf/target");
		i.inputFileLocation = new File("../vhat-ibdf/target/generated-resources/src/");
		i.converterOutputArtifactVersion = "2016.01.07.foo";
		i.converterVersion = "SNAPSHOT";
		i.converterSourceArtifactVersion = "fre";
		i.execute();
	}
	
	private class ValuePropertyPairExtended extends ValuePropertyPair
	{
		private DesignationImportDTO didto_;
		public ValuePropertyPairExtended(String value, UUID descriptionUUID, Property property, DesignationImportDTO didto, boolean disabled)
		{
			super(value, descriptionUUID, property);
			didto_ = didto;
			setDisabled(disabled);
		}
		
		public DesignationImportDTO getDesignationImportDTO()
		{
			return didto_;
		}
	}
}