package gov.va.med.term.vhat.data;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;

import org.apache.maven.plugin.MojoExecutionException;

import gov.va.med.term.vhat.data.dto.CodeSystem;
import gov.va.med.term.vhat.data.dto.ConceptImportDTO;
import gov.va.med.term.vhat.data.dto.DesignationExtendedImportDTO;
import gov.va.med.term.vhat.data.dto.MapEntryImportDTO;
import gov.va.med.term.vhat.data.dto.MapSetImportDTO;
import gov.va.med.term.vhat.data.dto.PropertyImportDTO;
import gov.va.med.term.vhat.data.dto.RelationshipImportDTO;
import gov.va.med.term.vhat.data.dto.SubsetImportDTO;
import gov.va.med.term.vhat.data.dto.SubsetMembershipImportDTO;
import gov.va.med.term.vhat.data.dto.TypeImportDTO;
import gov.va.med.term.vhat.data.dto.Version;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;


public class SqlDataReader {
	
	private Map<Long, ArrayList<PropertyImportDTO>>	_properties = new HashMap<>();
	private Map<Long, ArrayList<PropertyImportDTO>>	_propertiesByVuid = new HashMap<>();
	private Map<Long, ArrayList<RelationshipImportDTO>> _relationships = new HashMap<>();
	private Map<Long, ArrayList<ConceptImportDTO>> _concepts = new HashMap<>();
	private Map<Long, ArrayList<MapSetImportDTO>> _mapsets = new HashMap<>();
	private Map<Long, ArrayList<SubsetImportDTO>> _subsets = new HashMap<>();
	private Map<Long, String> _subsetsByVuid = new HashMap<>();
	private Map<Long, ArrayList<SubsetMembershipImportDTO>>	_subsetmemberships = new HashMap<>();
	private Map<Long, ArrayList<Long>>	_subsetmembershipsByVuid = new HashMap<>();
	private Map<Long, ArrayList<DesignationExtendedImportDTO>> _designations = new HashMap<>();
	private Map<Long, ArrayList<MapEntryImportDTO>> _mapentries = new HashMap<>();
	private List<CodeSystem> _codesystems = new ArrayList<>();
	private List<TypeImportDTO> _types = new ArrayList<>();
	private List<Version> _versions = new ArrayList<>();
	
	private Connection _connection;
	private boolean _isLocal;
	private DateTimeFormatter _dateTimeFormatter;

	
	public static void main(String[] args) throws Exception
	{ }

	
	public SqlDataReader()
	{ }
	
	
	public SqlDataReader(Connection c) throws MojoExecutionException
	{
		_connection = c;
	}

	
	public void setDatabaseConnection(Connection c) throws MojoExecutionException
	{
		_connection = c;
	}

	
	public void setup() throws MojoExecutionException
	{
		try
		{
			// Using properties for now, can change to whatever it needs to be
			// when integrated
			String env = System.getProperty("vaEnv");
			
			if (env == null || env.isEmpty())
			{
				env = "local";
				_isLocal = true;
			}
			
			Map<String, String> props = readProperties(env);
			
			if (_connection == null || _connection.isClosed())
			{
				Class.forName(props.get("driver"));
				String url = props.get("url");
				String username = props.get("username");
				String password = props.get("password");
				
				if (username != null && !username.isEmpty())
				{
					_connection = java.sql.DriverManager.getConnection(url, username, password);
				}
				else
				{
					_connection = java.sql.DriverManager.getConnection(props.get("url"));
				}
			}
			_connection.setReadOnly(true);
			
			if (_isLocal)
			{
				_dateTimeFormatter = new DateTimeFormatterBuilder()
						.parseCaseInsensitive()
						// 16-FEB-11 06.40.36.368000000 PM
						.appendPattern("dd-MMM-yy hh.mm.ss.SSSSSSSSS a")
						.toFormatter(Locale.US);
			}
			else
			{
				_dateTimeFormatter = new DateTimeFormatterBuilder()
						.parseCaseInsensitive()
						// 2011-01-31 00:00:00
						.appendPattern("yyyy-MM-dd KK:mm:ss")
						.parseDefaulting(ChronoField.HOUR_OF_DAY, 0)
						.parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0)
						.parseDefaulting(ChronoField.SECOND_OF_MINUTE, 0)
						.toFormatter(Locale.US);
			}
		}
		catch (Exception e)
		{
			throw new MojoExecutionException("Exception", e);
		}
	}

	
	public void shutdown() throws MojoExecutionException
	{
		try 
		{
			_connection.close();
		} 
		catch (Exception e) 
		{
			throw new MojoExecutionException("Exception", e);
		}
	}
	
	
	private boolean testDatabase() throws SQLException
	{
		boolean ret = false;
		
		if (_connection.isValid(5))
		{
			ResultSet rs = _connection.createStatement().executeQuery("SELECT count(*) FROM concept");
			while (rs.next()) 
			{
				String x = rs.getString(1);
				ret = true;
			}
		}
		
		return ret;
	}

	
	public void process() throws MojoExecutionException
	{
		try 
		{
			this.setup();
			
			if (_connection.isValid(5))
			{
				this.fetchConcepts();
				this.fetchSubsets();
				this.fetchSubsetMemberships();
				this.fetchMapsets();
				this.fetchMapEntries();
				this.fetchRelationships();
				this.fetchTypes();
				//this.fetchCodeSystems(); // TODO
				this.fetchProperties();
				//this.fetchVersions(); // TODO
				this.fetchDesignations();
				this.fetchMapSetDesignations();
				
				this.shutdown();
			}
		} 
		catch (Exception e)
		{
			throw new MojoExecutionException("Exception", e);
		}
	}


	private void fetchVersions() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT DISTINCT v.description, v.name, v.effectivedate, v.releasedate, cs.name, cs.vuid"
					+ " FROM version v"
					+ " INNER JOIN codesystem cs ON cs.id=v.codesystem_id";
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) 
		{
			_versions.add(new Version(
							rs.getString(2), // v.name 
							rs.getString(5), // source, CodeSystem?
							rs.getString(1), // v.description 
							getDateObject(rs.getString(3)).orElse(null), // v.effectiveDate 
							getDateObject(rs.getString(4)).orElse(null), // v.releaseDate 
							Boolean.TRUE.booleanValue() // append - ??
						));
		}
	}
	
	
	public Optional<List<Version>> getVersions()
	{
		return Optional.ofNullable(_versions);
	}
	
	
	private void fetchConcepts() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT c.entity_id, c.name, c.code, c.vuid, c.active, v.id, v.effectivedate"
					+ " FROM concept c"
					+ " INNER JOIN codesystem cs on cs.id=c.codesystem_id AND cs.name='VHAT'"
					+ " INNER JOIN version v ON v.id=c.version_id"
					+ " AND v.id != '9999999999'" // TODO
					+ " WHERE c.kind='C'"
					;
		
		int count = 0;
		printStats(count, 10000, "Concepts", false);
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) 
		{
			ArrayList<ConceptImportDTO> _al = new ArrayList<ConceptImportDTO>();
			// Should never be null according to the schema
			Long entity = getLongValue(rs.getLong(1));
			if (_concepts.containsKey(entity))
			{
				_al = _concepts.get(entity);
			}
			
			ConceptImportDTO ci = new ConceptImportDTO(
											"add", // action
											rs.getString(2), // name
											rs.getString(3), // code
											getLongValue(rs.getLong(4)), // vuid
											rs.getBoolean(5) // active
										);
			ci.setVersion(getLongValue(rs.getLong(6)));
			ci.setTime(getTime(rs, 7));
			_al.add(ci);

			_concepts.put(entity, _al);
			
			printStats(++count, 10000, "Concepts", false);
		}
		
		printStats(count, 10000, "Concepts", true);
	}
	
	
	public Optional<Map<Long, ArrayList<ConceptImportDTO>>> getConcepts()
	{
		return Optional.ofNullable(_concepts);
	} 
	
	
	private void fetchTypes() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT name, kind FROM TYPE";
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			_types.add(new TypeImportDTO(
						rs.getString(2), // kind
						rs.getString(1)) // name
					);
		}
	}
	
	
	public Optional<List<TypeImportDTO>> getTypes()
	{
		return Optional.ofNullable(_types);
	}
	
	
	private void fetchProperties() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT p.conceptentity_id, t.name, p.property_value, c.active, c.vuid"
					+ " FROM property p"
					+ " INNER JOIN concept c ON c.entity_id=p.conceptentity_id"
					+ "     AND c.kind IN ('C','D')"
					+ " INNER JOIN codesystem cs ON cs.id=c.codesystem_id AND cs.name='VHAT'"
					+ " INNER JOIN type t ON t.id=p.propertytype_id";
		
		int count = 0;
		printStats(count, 10000, "Properties", false);
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			ArrayList<PropertyImportDTO> _al = new ArrayList<PropertyImportDTO>();
			Long entity = getLongValue(rs.getLong(1));
			if (_properties.containsKey(entity))
			{
				_al = _properties.get(entity);
			}
			
			ArrayList<PropertyImportDTO> _al2 = new ArrayList<PropertyImportDTO>();
			Long vuid = getLongValue(rs.getLong(5)); // vuid
			if (_propertiesByVuid.containsKey(vuid))
			{
				_al2 = _properties.get(entity);
			}
			
			PropertyImportDTO pi = new PropertyImportDTO(
					"add", // action
					rs.getString(2), // typeName
					null, // valueOld
					rs.getString(3), // valueNew
					rs.getBoolean(4) // active
				);
			_al.add(pi);
			_al2.add(pi);
			_properties.put(entity, _al);
			_propertiesByVuid.put(vuid, _al2);
			
			printStats(++count, 1000, "Properties", false);
		}
		
		printStats(count, 1000, "Properties", true);
	}
	
	
	/*public Optional<Map<Long, ArrayList<PropertyImportDTO>>> getProperties()
	{
		return Optional.ofNullable(_properties);
	}*/
	
	
	public Optional<ArrayList<PropertyImportDTO>> getPropertiesForEntity(final Long conceptEntityId)
	{
		return Optional.ofNullable(_properties.get(conceptEntityId));
	}
	
	
	/*public Optional<Map<Long, ArrayList<PropertyImportDTO>>> getPropertiesByVuid()
	{
		return Optional.ofNullable(_propertiesByVuid);
	}*/
	
	
	public Optional<ArrayList<PropertyImportDTO>> getPropertiesForVuid(final Long vuid)
	{
		//return Optional.of(_propertiesByVuid.getOrDefault(vuid, new ArrayList<PropertyImportDTO>()));
		return Optional.ofNullable(_propertiesByVuid.get(vuid));
	}
	
	
	private void fetchRelationships() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		// Be careful with this. As it does improve performance, JDBC
		// pre-allocates an array of this number of rows, requiring more memory
		s.setFetchSize(1000);
		
		String query = "SELECT t.name, c1.code, c2.code, r.active, c1.entity_id, c1.version_id, v.effectivedate"
					+ " FROM relationship r"
					+ " INNER JOIN concept c1 ON c1.entity_id=r.source_entity_id"
					+ " INNER JOIN concept c2 ON c2.entity_id=r.target_entity_id"
					+ " INNER JOIN type t ON r.type_id=t.id"
					+ " INNER JOIN version v on v.id=c1.version_id"
					+ " AND v.id != '9999999999'"
					+ " WHERE r.kind='C'"
					;
		
		int count = 0;
		printStats(count, 10000, "Relationships", false);
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			ArrayList<RelationshipImportDTO> _al = new ArrayList<RelationshipImportDTO>();
			Long entity = getLongValue(rs.getLong(5));
			if (_relationships.containsKey(entity))
			{
				_al = _relationships.get(entity);
			}
			
			RelationshipImportDTO ri = new RelationshipImportDTO(
							"add", // action 
							rs.getString(1), // typeName
							rs.getString(2), // oldTargetCode
							rs.getString(3), // newTargetCode
							rs.getBoolean(4) // active
							);
			ri.setVersion(getLongValue(rs.getLong(6)));
			ri.setTime(getTime(rs, 7));
			_al.add(ri);
			
			_relationships.put(entity, _al);
			
			printStats(++count, 500000, "Relationships", false);
		}
		
		printStats(count, 500000, "Relationships", true);
	}
	
	
	/*public Optional<Map<Long, ArrayList<RelationshipImportDTO>>> getRelationships()
	{
		return Optional.ofNullable(_relationships);
	}*/
	
	
	public Optional<ArrayList<RelationshipImportDTO>> getRelationshipsForEntity(final Long conceptEntityId)
	{
		//Optional<ArrayList<RelationshipImportDTO>> rels = Optional.empty();
		//if (_relationships.containsKey(conceptEntityId))
		//{
			//rels = Optional.ofNullable(_relationships.get(conceptEntityId));
		//}
		//return rels;
		return Optional.ofNullable(_relationships.get(conceptEntityId));
	}
	
	
	public void clearRelationships()
	{
		_relationships = new HashMap<>();
	}
	
	
	private void fetchCodeSystems() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		// This doesn't account for versions, just a straight fetch from the table 
		String query = "SELECT cs.name, cs.vuid, cs.description, cs.copyright, cs.copyrighturl, t.name as preferred_designation_type"
					+ " FROM CODESYSTEM cs"
					+ " INNER JOIN type t ON t.id=preferred_designation_type_id";
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			_codesystems.add(new CodeSystem(
								rs.getString(1), // codeSystemName 
								getLongValue(rs.getLong(2)), // vuid
								rs.getString(3), // description 
								rs.getString(4), // copyright 
								rs.getString(5), // copyrightURL 
								rs.getString(6), // preferredDesignationType 
								"add"
							));
			// We don't have version in the SQL, or need a way to relate
			//cs.setVersion((Version)elementDataStack.peek().createdChildObjects.remove(0));
		}
	}

	
	public Optional<List<CodeSystem>> getCodeSystems()
	{
		return Optional.ofNullable(_codesystems);
	}

	
	private void fetchMapsets() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT DISTINCT c.name, c.code, c.vuid, c.active, cs1.name, v1.name, cs2.name, v2.name, c.entity_id, v1.id, v1.effectivedate"
					+ " FROM concept c"
					+ " INNER JOIN mapsetextension mse ON mse.mapsetid=c.entity_id"
					+ " INNER JOIN version v1 ON v1.id=mse.sourceversionid"
					+ " INNER JOIN version v2 ON v2.id=mse.targetversionid"
					+ " INNER JOIN codesystem cs1 ON cs1.id=v1.codesystem_id"
					+ " INNER JOIN codesystem cs2 ON cs2.id=v2.codesystem_id"
					+ " WHERE c.kind='M'";
				
		int count = 0;
		printStats(count, 10000, "MapSets", false);
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			ArrayList<MapSetImportDTO> _al = new ArrayList<MapSetImportDTO>();
			Long entity = getLongValue(rs.getLong(9));
			if (_mapsets.containsKey(entity))
			{
				_al = _mapsets.get(entity);
			}
			
			MapSetImportDTO msi = new MapSetImportDTO(
										"add", // action 
										rs.getString(1), // name 
										rs.getString(2), // code
										getLongValue(rs.getLong(3)), // vuid 
										rs.getBoolean(4), // active 
										rs.getString(5), // sourceCodeSystemName 
										rs.getString(6), // sourceVersionName 
										rs.getString(7), // targetCodeSystemName 
										rs.getString(8) // targetVersionName
									);
			msi.setVersion(getLongValue(rs.getLong(10)));
			msi.setTime(getTime(rs, 11));
			_al.add(msi);
			
			_mapsets.put(entity, _al);
			
			printStats(++count, 10000, "MapSets", false);
		}
		
		printStats(count, 10000, "MapSets", true);
	}
	
	
	public Optional<List<MapSetImportDTO>> getFullListOfMapSets()
	{
		List<MapSetImportDTO> returnList = new ArrayList<>();
		_mapsets.values().forEach((al) -> {
			returnList.addAll(al);
		});
		return Optional.ofNullable(returnList);
	}
	
	
	public Optional<Map<Long, ArrayList<MapSetImportDTO>>> getMapSets()
	{
		return Optional.ofNullable(_mapsets);
	}

	
	private void fetchSubsets() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT c.vuid, c.name, c.active, c.entity_id"
					+ " FROM concept c"
					+ " WHERE kind='S'";
		
		int count = 0;
		printStats(count, 10000, "Subsets", false);
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			ArrayList<SubsetImportDTO> _al = new ArrayList<SubsetImportDTO>();
			Long entity = getLongValue(rs.getLong(4));
			if (_subsets.containsKey(entity))
			{
				_al = _subsets.get(entity);
			}
			
			Long vuid = getLongValue(rs.getLong(1)); // vuid
			_al.add(new SubsetImportDTO(
							"add", // action
							vuid,
							rs.getString(2), // subsetName
							rs.getBoolean(3) // active
							));
			
			_subsets.put(entity, _al);
			_subsetsByVuid.put(vuid, rs.getString(2));
			
			printStats(++count, 10000, "Subsets", false);
		}
		
		printStats(count, 10000, "Subsets", true);
	}
	
	
	public Optional<List<SubsetImportDTO>> getFullListOfSubsets()
	{
		List<SubsetImportDTO> returnList = new ArrayList<>();
		_subsets.values().forEach((al) -> {
			returnList.addAll(al);
		});
		return Optional.ofNullable(returnList);
	}

	
	public Optional<Map<Long, ArrayList<SubsetImportDTO>>> getSubsets()
	{
		return Optional.ofNullable(_subsets);
	}
	
	
	public Optional<Map<Long, String>> getSubsetsByVuid()
	{
		return Optional.ofNullable(_subsetsByVuid);
	}
	
	
	private void fetchSubsetMemberships() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT DISTINCT c1.name, c1.vuid, c2.name, c2.vuid, r.active, c2.entity_id, c2.code"
					+ " FROM relationship r"
					+ " INNER JOIN concept c1 ON c1.entity_id=r.source_entity_id"
					+ " INNER JOIN concept c2 ON c2.entity_id=r.target_entity_id"
					+ " WHERE r.kind='S'";

		int count = 0;
		printStats(count, 10000, "SubsetMemberships", false);
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			Long entity = getLongValue(rs.getLong(6));
			Long subsetVuid = getLongValue(rs.getLong(2));
			Long designationCode = getLongValue(rs.getLong(7));
			
			ArrayList<SubsetMembershipImportDTO> _al = new ArrayList<SubsetMembershipImportDTO>();
			if (_subsetmemberships.containsKey(entity))
			{
				_al = _subsetmemberships.get(entity);
			}
			
			_al.add(new SubsetMembershipImportDTO(
					"add", // action 
					subsetVuid, // vuid 
					rs.getBoolean(5) // active
				));
			_subsetmemberships.put(entity, _al);
		
			ArrayList<Long> _al2 = new ArrayList<Long>();
			if (_subsetmembershipsByVuid.containsKey(subsetVuid))
			{
				_al2 = _subsetmembershipsByVuid.get(subsetVuid);
			}
			
			_al2.add(designationCode);
			_subsetmembershipsByVuid.put(subsetVuid, _al2);
			
			printStats(++count, 10000, "SubsetMemberships", false);
		}
		
		printStats(count, 10000, "SubsetMemberships", true);
	}
	
	
	public Map<Long, ArrayList<SubsetMembershipImportDTO>> getSubsetMembershipsMap()
	{
		return _subsetmemberships;
	}
	
	
	public Optional<Map<Long, ArrayList<SubsetMembershipImportDTO>>> getSubsetMemberships()
	{
		return Optional.ofNullable(_subsetmemberships);
	}
	
	
	public Optional<Map<Long, ArrayList<Long>>> getSubsetMembershipsCodesByVuid()
	{
		return Optional.ofNullable(_subsetmembershipsByVuid);
	}
	
	
	private void fetchMapEntries() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT DISTINCT c1.entity_id, c2.entity_id, c2.vuid, r.active, mee.sourcecode, mee.targetcode, r.sequence, r.grouping, mee.effectivedate, c1.vuid, c2.version_id, v.effectivedate"
					+ " FROM relationship r"
					+ " INNER JOIN concept c1 ON c1.entity_id=r.source_entity_id"
					+ " INNER JOIN concept c2 ON c2.entity_id=r.target_entity_id"
					+ " INNER JOIN mapentryextension mee ON mee.mapentryid=c2.entity_id"
					+ " INNER JOIN version v on v.id=c2.version_id"
					+ " WHERE r.kind='M'";
		
		int count = 0;
		printStats(count, 10000, "MapEntries", false);
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			ArrayList<MapEntryImportDTO> _al = new ArrayList<MapEntryImportDTO>();
			// Key'd by the parent MapSet VUID for the Mojo
			Long mapsetVuid = getLongValue(rs.getLong(10));
			Integer sequence = getIntegerValue(rs.getInt(7));

			if (_mapentries.containsKey(mapsetVuid))
			{
				_al = _mapentries.get(mapsetVuid);
			}
			
			MapEntryImportDTO mei = new MapEntryImportDTO(
									"add", // action 
									getLongValue(rs.getLong(3)), // vuid 
									rs.getBoolean(4), // active 
									rs.getString(5), // sourceCode 
									rs.getString(6), // targetCode 
									sequence == null ? 0 : sequence.intValue(), // sequence 
									getLongValue(rs.getLong(8)), // grouping, 
									getDateObject(rs.getString(9)).orElse(null) // effectiveDate
								);
			mei.setVersion(getLongValue(rs.getLong(11)));
			mei.setTime(getTime(rs, 12));
			_al.add(mei);
			
			_mapentries.put(mapsetVuid, _al);
			
			printStats(++count, 100000, "MapEntries", false);
		}
		
		printStats(count, 100000, "MapEntries", true);
	}
	
	
	public Optional<ArrayList<MapEntryImportDTO>> getMapEntriesForMapSet(final Long vuid)
	{
		return Optional.ofNullable(_mapentries.get(vuid));
	}
	
	
	private void fetchDesignations() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT DISTINCT t.name as type_name, c1.code, c1.vuid, c1.name as value, c1.active, v.name, c2.entity_id, v.id, v.effectivedate"
					+ " FROM relationship r"
					+ " INNER JOIN concept c1 ON c1.codesystem_id='1' AND c1.entity_id=r.target_entity_id"
					+ " INNER JOIN concept c2 ON c2.entity_id=r.source_entity_id AND c2.kind='C'"
					+ " INNER JOIN type t ON t.id=c1.type_id"
					+ " INNER JOIN version v ON v.id=r.version_id"
					+ " AND v.id != '9999999999'" // TODO: address this
					+ " WHERE r.kind='D'"
					;
		
		int count = 0;
		printStats(count, 10000, "Designations", false);
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			ArrayList<DesignationExtendedImportDTO> _al = new ArrayList<DesignationExtendedImportDTO>();
			Long entity = getLongValue(rs.getLong(7));
			if (_designations.containsKey(entity))
			{
				_al = _designations.get(entity);
			}
			
			DesignationExtendedImportDTO dei = new DesignationExtendedImportDTO(
									"add", // action 
									rs.getString(1), // typeName 
									rs.getString(2), // code 
									null, // valueOld 
									rs.getString(4), // valueNew 
									getLongValue(rs.getLong(3)), // vuid 
									null, // moveFromConceptCode
									rs.getBoolean(5) // active
								);
			dei.setVersion(getLongValue(rs.getLong(8)));
			dei.setTime(getTime(rs, 9));
			_al.add(dei);
			
			_designations.put(entity, _al);
			
			printStats(++count, 10000, "Designations", false);
		}
		
		printStats(count, 10000, "Designations", true);
	}
	
	
	private void fetchMapSetDesignations() throws SQLException
	{
		Statement s = _connection.createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
		s.setFetchSize(1000);
		
		String query = "SELECT t.name, c1.code, c1.vuid, c1.name, c1.active, v.id, v.effectivedate, r.source_entity_id"
					+ " FROM relationship r"
					+ " INNER JOIN concept c1 ON c1.codesystem_id='1' AND c1.entity_id=r.target_entity_id"
					+ " INNER JOIN type t ON t.id=c1.type_id"
					+ " INNER JOIN version v ON v.id=r.version_id"
					+ " WHERE r.kind='D'"
					+ " AND r.source_entity_id IN (" // Mostly a copy from the Mapset query 
					+ "		SELECT c.entity_id"
					+ "		FROM concept c"
					+ "		INNER JOIN mapsetextension mse ON mse.mapsetid=c.entity_id"
					+ "		INNER JOIN version v1 ON v1.id=mse.sourceversionid"
					+ "		INNER JOIN version v2 ON v2.id=mse.targetversionid"
					+ "		INNER JOIN codesystem cs1 ON cs1.id=v1.codesystem_id"
					+ "		INNER JOIN codesystem cs2 ON cs2.id=v2.codesystem_id"
					+ "		WHERE c.kind='M'"
					+ ")";
		
		int count = 0;
		printStats(count, 10000, "MapSetDesignations", false);
		
		ResultSet rs = s.executeQuery(query);
		while (rs.next())
		{
			ArrayList<DesignationExtendedImportDTO> _al = new ArrayList<DesignationExtendedImportDTO>();
			Long entity = getLongValue(rs.getLong(8));
			if (_designations.containsKey(entity))
			{
				_al = _designations.get(entity);
			}
			
			DesignationExtendedImportDTO dei = new DesignationExtendedImportDTO(
									"add", // action 
									rs.getString(1), // typeName 
									rs.getString(2), // code 
									null, // valueOld 
									rs.getString(4), // valueNew 
									getLongValue(rs.getLong(3)), // vuid 
									null, // moveFromConceptCode
									rs.getBoolean(5) // active
								);
			dei.setVersion(getLongValue(rs.getLong(6)));
			dei.setTime(getTime(rs, 7));
			_al.add(dei);
			
			_designations.put(entity, _al);
				
			printStats(++count, 10, "MapSetDesignations", false);
		}
		
		printStats(count, 10, "MapSetDesignations", true);
	}

	
	public Optional<Map<Long, ArrayList<DesignationExtendedImportDTO>>> getDesignations()
	{
		return Optional.ofNullable(_designations);
	}
	
	
	public Optional<ArrayList<DesignationExtendedImportDTO>> getDesignationsForEntity(final Long conceptEntityId)
	{
		return Optional.ofNullable(_designations.get(conceptEntityId));
	}
	
	
	private Integer getIntegerValue(int value)
	{
		// ResultSet getInt() guaranteed to be not null
		// but we'll return null for any '0' values
		Integer result = null;
		
		if (value > 0)
		{
			result = Integer.valueOf(value);
		}
		
		return result;
	}
	
	
	private Long getLongValue(long value)
	{
		// ResultSet getLong() guaranteed to be not null
		// but we'll return null for any '0' values
		Long result = null;
		
		if (value > 0)
		{
			result = Long.valueOf(value);
		}
		
		return result;
	}
	
	
	private Optional<Date> getDateObject(String d)
	{
		Date dt = null;
		if (d != null)
		{
			try
			{
				// TODO: Using UTC for the moment, need to review
				dt = Date.from(LocalDateTime.parse(d, _dateTimeFormatter)
						.toInstant(ZoneOffset.UTC));
			}
			catch (Exception e)
			{
				// TODO
				System.out.println(e);
			}
		}
		return Optional.ofNullable(dt);
	}

	
	private Long getTimeFromDate(String d)
	{
		Long time = null;
		if (getDateObject(d).isPresent())
		{
			time = Long.valueOf(getDateObject(d).get().getTime());
		}
		return time;
	}
	
	
	private Long getTimeFromTimestamp(java.sql.Timestamp ts)
	{
		Long time = null;
		if (ts != null)
		{
			time = ts.getTime();
		}
		return time;
	}
	
	
	private Long getTime(ResultSet rs, int columnNum) throws SQLException
	{
		Long t = null;
		
		if (_isLocal)
		{
			t = getTimeFromDate(rs.getString(columnNum));
		}
		else
		{
			t = getTimeFromTimestamp(rs.getTimestamp(columnNum));
		}
		
		return t;
	}
	
	
	private void printStats(int count, int interval, String message, boolean finalCount)
	{
		if (count == 0)
		{
			ConsoleUtil.println(message + " : Starting.");
		} 
		else if ((count % interval) == 0)
		{
			ConsoleUtil.println(message + " : Fetched " + count + " rows.");
		} 
		else if (finalCount)
		{
			ConsoleUtil.println(message + " : Finished with " + count + " rows.");
		}
	}
	
	
	private Map<String, String> readProperties(String env) throws IOException
	{
		Properties props = new Properties();
		Map<String, String> envProps = new HashMap<>();
		
		try (InputStream i = SqlDataReader.class.getClassLoader().getResourceAsStream("config.properties"))
		{
			props.load(i);
			
			envProps.put("driver", props.getProperty(env+".jdbc.driver"));
			envProps.put("url", props.getProperty(env+".jdbc.url"));
			envProps.put("username", props.getProperty(env+".jdbc.username"));
			envProps.put("password", props.getProperty(env+".jdbc.password"));
		}
		
		return envProps;
	}
}
