/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.va.rxnorm.standard;

import java.beans.PropertyVetoException;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Annotations;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Associations;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Descriptions;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Relations;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.Property;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyAssociation;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.ValuePropertyPair;
import gov.va.oia.terminology.converters.sharedUtils.sql.TableDefinition;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.AbbreviationExpansion;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.RRFDatabaseHandle;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.Relationship;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.UMLSFileReader;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.ValuePropertyPairWithAttributes;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.propertyTypes.PT_Descriptions;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.propertyTypes.PT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.propertyTypes.PT_Relationship_Metadata;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.propertyTypes.PT_SAB_Metadata;
import gov.va.oia.terminology.converters.sharedUtils.umlsUtils.rrf.REL;
import gov.va.rxnorm.propertyTypes.PT_Annotations;
import gov.va.rxnorm.propertyTypes.ValuePropertyPairWithSAB;
import gov.va.rxnorm.rrf.RXNCONSO;
import gov.va.rxnorm.rrf.RXNSAT;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.chronicle.LatestVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.StringSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeDataType;
import gov.vha.isaac.ochre.api.util.UuidT3Generator;
import gov.vha.isaac.ochre.model.configuration.StampCoordinates;

/**
 * Loader code to convert RxNorm into the workbench.
 */
@Mojo( name = "convert-rxnorm-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES )
public class RxNormMojo extends ConverterBaseMojo
{
	private final String tablePrefix_ = "RXN";
	private final String sctSab_ = "SNOMEDCT_US";
	public static final String rxNormName = "RxNorm";
	
	private HashMap<String, UUID> sTypes_;
	private HashMap<String, UUID> suppress_;
	private HashMap<String, UUID> sourceRestrictionLevels_;
	
	private PropertyType ptUMLSAttributes_;
	private PropertyType ptSABs_;
	private PropertyType ptRelationshipMetadata_;
	private PropertyType ptDescriptions_;//TODO get SAB types for these, annotate
	private BPT_Associations ptAssociations_;//TODO get SAB types for these, annotate
	private BPT_Relations ptRelationships_;//TODO get SAB types for these, annotate
	private PropertyType ptTermAttributes_;//TODO get SAB types for these, annotate
	private PT_Refsets ptRefsets_;

	private HashMap<String, Relationship> nameToRel_ = new HashMap<>();
	private HashMap<String, UUID> semanticTypes_ = new HashMap<>();
	
	private IBDFCreationUtility importUtil_;
	private RRFDatabaseHandle db_;
	
	private ComponentReference metaDataRoot_;

	private HashSet<UUID> loadedRels_ = new HashSet<>();
	private HashSet<UUID> skippedRels_ = new HashSet<>();
	
	private HashMap<String, AbbreviationExpansion> abbreviationExpansions;
	
	private HashMap<String, Boolean> mapToIsa = new HashMap<>();  //FSN, true or false - true for rel only, false for a rel and association representation
	
	private ComponentReference allCUIRefsetConcept_;
	private ComponentReference cpcRefsetConcept_;
	public static final String cpcRefsetConceptKey_ = "Current Prescribable Content";
	
	private PreparedStatement semanticTypeStatement, descSat, cuiRelStatementForward, cuiRelStatementBackward, satRelStatement_, hasTTYType_;
	
	private HashSet<String> allowedCUIsForSABs_;
	private HashMap<Long, UUID> sctIdToUUID_ = new HashMap<>();  //A map of real (found) SCTIDs to their concept UUID
	private HashMap<String, Long> cuiToSCTID_ = new HashMap<>();  //Map CUI to SCTID for the real sctIds to UUIDs found above
	private AtomicInteger skippedRelForNotMatchingCUIFilter = new AtomicInteger();
	
	//Format to parse 01/28/2010
	private SimpleDateFormat dateParse = new SimpleDateFormat("MM/dd/yyyy");
	
	/**
	 * An optional list of TTY types which should be included.  If left blank, we create concepts from all CUI's that are in the
	 * SAB RxNorm.  If provided, we only create concepts where the RxCUI has an entry with a TTY that matches one of the TTY's provided here
	 */
	@Parameter (required = false)
	protected List<String> ttyRestriction;
	
	/**
	 * An optional list of SABs which should be included.  We always include the SAB RXNORM.  Use this parameter to specify others to include.
	 * If SNOMEDCT_US is included, then a snomed CT ibdf file must be present - snomed CT is not loaded from RxNorm, but rather, linked to the provided SCT
	 * IBDF file. 
	 */
	@Parameter (required = false)
	protected List<String> sabsToInclude;
	
	private boolean linkSnomedCT_;
	
	@Override
	public void execute() throws MojoExecutionException
	{
		try
		{
			super.execute();

			init();
			
			allCUIRefsetConcept_ = ComponentReference.fromConcept(ptRefsets_.getProperty(ptRefsets_.CUI_CONCEPTS.getSourcePropertyNameFSN()).getUUID());
			cpcRefsetConcept_ = ComponentReference.fromConcept(ptRefsets_.getProperty(cpcRefsetConceptKey_).getUUID());

			semanticTypeStatement = db_.getConnection().prepareStatement("select TUI, ATUI, CVF from RXNSTY where RXCUI = ?");
			//we always grab the description type NDC if present, even if NDC doesn't come from a SAB we are including.
			descSat = db_.getConnection().prepareStatement("select * from RXNSAT where RXCUI = ? and RXAUI = ? and (" + createSabQueryPart("", false) 
				+ " or ATN='NDC')");

			//UMLS and RXNORM do different things with rels - UMLS never has null CUI's, while RxNorm always has null CUI's (when AUI is specified)
			//Also need to join back to MRCONSO to make sure that the target concept is one that we will load with the SAB filter in place.
			cuiRelStatementForward = db_.getConnection().prepareStatement("SELECT distinct r.RXCUI1, r.RXAUI1, r.STYPE1, r.REL, r.RXCUI2, r.RXAUI2, r.STYPE2, "
					+ "r.RELA, r.RUI, r.SRUI, r.SAB, r.SL, r.DIR, r.RG, r.SUPPRESS, r.CVF from RXNREL as r, RXNCONSO "
					+ "WHERE RXCUI2 = ? and RXAUI2 is null and " + createSabQueryPart("r.", linkSnomedCT_) + " and r.RXCUI1 = RXNCONSO.RXCUI and "
					+ createSabQueryPart("RXNCONSO.", linkSnomedCT_));

			cuiRelStatementBackward = db_.getConnection().prepareStatement("SELECT distinct r.RXCUI1, r.RXAUI1, r.STYPE1, r.REL, r.RXCUI2, r.RXAUI2, r.STYPE2, "
					+ "r.RELA, r.RUI, r.SRUI, r.SAB, r.SL, r.DIR, r.RG, r.SUPPRESS, r.CVF from RXNREL as r, RXNCONSO "
					+ "WHERE RXCUI1 = ? and RXAUI1 is null and " + createSabQueryPart("r.", linkSnomedCT_) + " and r.RXCUI2 = RXNCONSO.RXCUI and "
					+ createSabQueryPart("RXNCONSO.", linkSnomedCT_));
			
			int cuiCounter = 0;

			Statement statement = db_.getConnection().createStatement();
			
			StringBuilder ttyRestrictionQuery = new StringBuilder();
			if (ttyRestriction != null && ttyRestriction.size() > 0)
			{
				ttyRestrictionQuery.append(" and (");
				for (String s : ttyRestriction)
				{
					ttyRestrictionQuery.append("TTY = '");
					ttyRestrictionQuery.append(s);
					ttyRestrictionQuery.append("' or ");
				}
				ttyRestrictionQuery.setLength(ttyRestrictionQuery.length() - " or ".length());
				ttyRestrictionQuery.append(")");
			}
			
			allowedCUIsForSABs_ = new HashSet<>();
			ResultSet rs = statement.executeQuery("select RXCUI from RXNCONSO where " + createSabQueryPart("", linkSnomedCT_) + " " + ttyRestrictionQuery);
			while (rs.next())
			{
				allowedCUIsForSABs_.add(rs.getString("RXCUI"));
			}
			rs.close();
			
//			allowedSCTTargets = new HashMap<>();
//			HashSet<String> bannedSCTTargets = new HashSet<>();
//			if (sctIDToUUID_ != null)
//			{
//				rs = statement.executeQuery("SELECT DISTINCT RXCUI, CODE from RXNCONSO r1 where r1.SAB='" + sctSab_ + "' and not exists "
//						+" (select rxcui from RXNCONSO r2 where r1.rxcui = r2.rxcui and r2.sab = 'RXNORM')");
//				while (rs.next())
//				{
//					String cui = rs.getString("RXCUI");
//					long sctid = Long.parseLong(rs.getString("CODE"));
//					UUID uuid = sctIDToUUID_.get(sctid);
//					if (uuid != null)
//					{
//						//already have a mapping for this CUI to an SCTID - make sure that it maps to the same UUID
//						if (allowedSCTTargets.containsKey(cui) 
//								&& allowedSCTTargets.get(cui) != sctid && !uuid.equals(sctIDToUUID_.get(allowedSCTTargets.get(cui))))
//						{
//							bannedSCTTargets.add(cui);
//							allowedSCTTargets.remove(cui);
//							ConsoleUtil.println("CUI " + cui + " maps to multiple SCT concepts");
//						}
//						else if (!bannedSCTTargets.contains(cui))
//						{
//							//We know there is one and only one mapping from this RxCUI to a concept (that exists) in the version of SCT we are using.
//							allowedSCTTargets.put(cui, sctid);
//						}
//					}
//				}
//				rs.close();
//			}
//			ConsoleUtil.println("Allowing " + allowedSCTTargets.size() + " potential relationships to existing SCT concepts");
//			ConsoleUtil.println("Not allowing " + bannedSCTTargets.size() + " CUIs to link to SCT concepts because they are mapped to multiple concepts");
//			
//			allowedCUIsForRelationships = new HashSet<String>();
//			allowedCUIsForRelationships.addAll(allowedCUIsForRels);
//			allowedCUIsForRelationships.addAll(allowedSCTTargets.keySet());
			
			rs = statement.executeQuery("select RXCUI, LAT, RXAUI, SAUI, SCUI, SAB, TTY, CODE, STR, SUPPRESS, CVF from RXNCONSO " 
					+ "where " + createSabQueryPart("", linkSnomedCT_) + " order by RXCUI" );
			
			HashSet<String> skippedCUIForNotMatchingCUIFilter = new HashSet<String>();
			
			ArrayList<RXNCONSO> conceptData = new ArrayList<>();
			while (rs.next())
			{
				RXNCONSO current = new RXNCONSO(rs);
				if (!allowedCUIsForSABs_.contains(current.rxcui))
				{
					skippedCUIForNotMatchingCUIFilter.add(current.rxcui);
					continue;
				}
				if (conceptData.size() > 0 && !conceptData.get(0).rxcui.equals(current.rxcui))
				{
					processCUIRows(conceptData);
					if (cuiCounter % 100 == 0)
					{
						ConsoleUtil.showProgress();
					}
					cuiCounter++;
					if (cuiCounter % 10000 == 0)
					{
						ConsoleUtil.println("Processed " + cuiCounter + " CUIs creating " + importUtil_.getLoadStats().getConceptCount() + " concepts");
					}
					conceptData.clear();
				}
				
				conceptData.add(current);
			}
			rs.close();
			statement.close();

			// process last
			processCUIRows(conceptData);
			
			ConsoleUtil.println("Processed " + cuiCounter + " CUIs creating " + importUtil_.getLoadStats().getConceptCount() + " concepts");
			ConsoleUtil.println("Skipped " + skippedCUIForNotMatchingCUIFilter.size() + " concepts for not containing the desired TTY");
			ConsoleUtil.println("Skipped " + skippedRelForNotMatchingCUIFilter + " relationships for linking to a concept we didn't include");

			semanticTypeStatement.close();
			descSat.close();

			cuiRelStatementForward.close();
			cuiRelStatementBackward.close();

			finish();
			
			
		}
		catch (Exception e)
		{
			throw new MojoExecutionException("Failure during conversion", e);
		}
		finally
		{
			if (db_ != null)
			{
				try
				{
					db_.shutdown();
				}
				catch (SQLException e)
				{
					throw new RuntimeException(e);
				}
			}
		}
	}
	
	/**
	 * Returns the date portion of the file name - so from 'RxNorm_full_09022014.zip' it returns 09022014
	 */
	private String loadDatabase() throws Exception
	{
		// Set up the DB for loading the temp data
		String toReturn = null;
		
		// Read the RRF file directly from the source zip file - need to find the zip first, to get the date out of the file name.
		ZipFile zf = null;
		for (File f : inputFileLocation.listFiles())
		{
			if (f.getName().toLowerCase().startsWith("rxnorm_full_") && f.getName().toLowerCase().endsWith(".zip"))
			{
				zf = new ZipFile(f);
				toReturn = f.getName().substring("rxnorm_full_".length());
				toReturn = toReturn.substring(0, toReturn.length() - 4); 
				break;
			}
		}
		if (zf == null)
		{
			throw new MojoExecutionException("Can't find source zip file");
		}
		
		db_ = new RRFDatabaseHandle();
		File dbFile = new File(outputDirectory, "rrfDB.h2.db");
		boolean createdNew = db_.createOrOpenDatabase(new File(outputDirectory, "rrfDB"));

		if (!createdNew)
		{
			ConsoleUtil.println("Using existing database.  To load from scratch, delete the file '" + dbFile.getAbsolutePath() + ".*'");
		}
		else
		{
			// RxNorm doesn't give us the UMLS tables that define the table definitions, so I put them into an XML file.
			List<TableDefinition> tables = db_.loadTableDefinitionsFromXML(RxNormMojo.class.getResourceAsStream("/RxNormTableDefinitions.xml"));

			for (TableDefinition td : tables)
			{
				ZipEntry ze = zf.getEntry("rrf/" + td.getTableName() + ".RRF");
				if (ze == null)
				{
					throw new MojoExecutionException("Can't find the file 'rrf/" + td.getTableName() + ".RRF' in the zip file");
				}

				db_.loadDataIntoTable(td, new UMLSFileReader(new BufferedReader(new InputStreamReader(zf.getInputStream(ze), "UTF-8"))), null);
			}
			zf.close();

			// Build some indexes to support the queries we will run

			Statement s = db_.getConnection().createStatement();
			ConsoleUtil.println("Creating indexes");
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX conso_rxcui_index ON RXNCONSO (RXCUI)");
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX conso_rxaui_index ON RXNCONSO (RXAUI)");
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX sat_rxcui_aui_index ON RXNSAT (RXCUI, RXAUI)");
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX sat_aui_index ON RXNSAT (RXAUI)");
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX sty_rxcui_index ON RXNSTY (RXCUI)");
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX sty_tui_index ON RXNSTY (TUI)");
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX rel_rxcui2_index ON RXNREL (RXCUI2, RXAUI2)");
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX rel_rxaui2_index ON RXNREL (RXCUI1, RXAUI1)");
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX rel_rela_rel_index ON RXNREL (RELA, REL)");  //helps with rel metadata
			ConsoleUtil.showProgress();
			s.execute("CREATE INDEX rel_sab_index ON RXNREL (SAB)");  //helps with rel metadata
			s.close();
			ConsoleUtil.println("DB Setup complete");
		}
		return toReturn;
	}

	private void processCUIRows(ArrayList<RXNCONSO> conceptData) throws IOException, SQLException, PropertyVetoException
	{
		String rxCui = conceptData.get(0).rxcui;
		
		HashSet<String> uniqueTTYs = new HashSet<String>();
		HashSet<String> uniqueSABs = new HashSet<String>();
		
		
		//ensure all the same CUI, gather the TTYs involved
		for (RXNCONSO row : conceptData)
		{
			uniqueTTYs.add(row.tty);
			uniqueSABs.add(row.sab);
			if (!row.rxcui.equals(rxCui))
			{
				throw new RuntimeException("Oops");
			}
		}
		
		ComponentReference cuiConcept;
		
		if (uniqueSABs.size() == 1 && uniqueSABs.iterator().next().equals(sctSab_))
		{
			//This is a SCT only concept - we don't want to create it.  But we might need to put some relationships or associations here.
			String sctId = conceptData.get(0).code;
			if (sctId == null)
			{
				throw new RuntimeException("Unexpected");
			}
			cuiConcept = ComponentReference.fromConcept(sctIdToUUID_.get(sctId));
			//Add the RxCUI UUID
			importUtil_.addUUID(cuiConcept.getPrimordialUuid(), createCUIConceptUUID(rxCui));
			//TODO need to look at what else I should be grabbing - the RXCUI for example should be attached.  What else?
		}
		else
		{
			//just creating the reference here, with the UUID - because we don't know if it should be active or inactive yet.
			//create the real concept later.
			cuiConcept = ComponentReference.fromConcept(createCUIConceptUUID(rxCui));
			long conceptTime = Integer.MAX_VALUE;
			//Activate the concept if any description is active
			State conceptState = State.INACTIVE;
			
			importUtil_.addStringAnnotation(cuiConcept, rxCui, ptUMLSAttributes_.getProperty("RXCUI").getUUID(), State.ACTIVE);
	
			ArrayList<ValuePropertyPairWithSAB> cuiDescriptions = new ArrayList<>();
			HashSet<String> sabs = new HashSet<>();
			
			for (RXNCONSO atom : conceptData)
			{
				if (atom.sab.equals(sctSab_))
				{
					continue;
				}
				
				//Add attributes from SAT table
				descSat.clearParameters();
				descSat.setString(1, rxCui);
				descSat.setString(2, atom.rxaui);
				ResultSet rs = descSat.executeQuery();
				
				ArrayList<RXNSAT> satData = new ArrayList<>();
				boolean disableDescription = false;
				Long descriptionTime = null;
				while (rs.next())
				{
					RXNSAT current = new RXNSAT(rs);
					satData.add(current);
					if ("RXN_OBSOLETED".equals(current.atn))
					{
						disableDescription = true;
					}
					if ("RXN_ACTIVATED".equals(current.atn))
					{
						try
						{
							long time = dateParse.parse(current.atv).getTime();
							descriptionTime = time;
							if (time < conceptTime)
							{
								conceptTime = time;
							}
						}
						catch (ParseException e)
						{
							throw new RuntimeException("Can't parse date?");
						}
					}
				}
				rs.close();
	
				ValuePropertyPairWithSAB desc = new ValuePropertyPairWithSAB(atom.str, ptDescriptions_.getProperty(atom.tty), atom.sab, satData);
				if (disableDescription)
				{
					desc.setDisabled(true);
				}
				else
				{
					//if any description is active, concept is still active
					conceptState = State.ACTIVE;
				}
				if (descriptionTime != null)
				{
					desc.setTime(descriptionTime);
				}
				
				desc.setUUID(ConverterUUID.createNamespaceUUIDFromStrings(cuiConcept.getPrimordialUuid().toString(), atom.rxaui));
				
				//used for sorting description to figure out what to use for FSN
				cuiDescriptions.add(desc);
				
				desc.addStringAttribute(ptUMLSAttributes_.getProperty("RXAUI").getUUID(), atom.rxaui);
				desc.addUUIDAttribute(ptUMLSAttributes_.getProperty("SAB").getUUID(), ptSABs_.getProperty(atom.sab).getUUID());
				
				if (StringUtils.isNotBlank(atom.code) && !atom.code.equals("NOCODE"))
				{
					desc.addStringAttribute(ptUMLSAttributes_.getProperty("CODE").getUUID(), atom.code);
				}
					
				if (StringUtils.isNotBlank(atom.saui))
				{
					desc.addStringAttribute(ptUMLSAttributes_.getProperty("SAUI").getUUID(), atom.saui);
				}
				if (StringUtils.isNotBlank(atom.scui))
				{
					desc.addStringAttribute(ptUMLSAttributes_.getProperty("SCUI").getUUID(), atom.scui);
				}
					
				if (StringUtils.isNotBlank(atom.suppress))
				{
					desc.addUUIDAttribute(ptUMLSAttributes_.getProperty("SUPPRESS").getUUID(), suppress_.get(atom.suppress));
				}
					
				if (StringUtils.isNotBlank(atom.cvf))
				{
					if (atom.cvf.equals("4096"))
					{
						desc.addRefsetMembership(cpcRefsetConcept_.getPrimordialUuid());
					}
					else
					{
						throw new RuntimeException("Unexpected value in RXNCONSO cvf column '" + atom.cvf + "'");
					}
				}
				if (!atom.lat.equals("ENG"))
				{
					ConsoleUtil.printErrorln("Non-english lang settings not handled yet!");
				}
				
				//TODO - at this point, sometime in the future, we make make attributes out of the relationships that occur between the AUIs
				//and store them on the descriptions, since OTF doesn't allow relationships between descriptions
	
				//TODO am I supposed to be using sabs?
				sabs.add(atom.sab);
			}
			
			//sanity check on descriptions - make sure we only have one that is of type synonym with the preferred flag
			ArrayList<String> items = new ArrayList<String>();
			for (ValuePropertyPair vpp : cuiDescriptions)
			{
				//Numbers come from the rankings down below in makeDescriptionType(...)
				if (vpp.getProperty().getPropertySubType() >= BPT_Descriptions.SYNONYM && vpp.getProperty().getPropertySubType() <= (BPT_Descriptions.SYNONYM + 20))
				{
					items.add(vpp.getProperty().getSourcePropertyNameFSN() + " " + vpp.getProperty().getPropertySubType());
				}
			}
			
			HashSet<String> ranksLookedAt = new HashSet<>();
			ranksLookedAt.add("204");
			ranksLookedAt.add("206");
			ranksLookedAt.add("210");
			ranksLookedAt.add("208");
			ranksLookedAt.add("212");
			
			boolean oneNotInList = false;
			if (items.size() > 1)
			{
				
				for (String s : items)
				{
					if (!ranksLookedAt.contains(s.substring(s.length() - 3, s.length())))
					{
						oneNotInList = true;
						break;
					}
				}
				
			}
			
			if (oneNotInList)
			{
				ConsoleUtil.printErrorln("Need to rank multiple synonym types that are each marked preferred, determine if ranking is appropriate!");
				for (String s : items)
				{
					ConsoleUtil.printErrorln(s);
				}
			}
			
			List<SememeChronology<DescriptionSememe<?>>> addedDescriptions = importUtil_.addDescriptions(cuiConcept, cuiDescriptions);
			
			if (addedDescriptions.size() != cuiDescriptions.size())
			{
				throw new RuntimeException("oops");
			}
			
			HashSet<String> uniqueUMLSCUI = new HashSet<>();
			
			for (int i = 0; i < cuiDescriptions.size(); i++)
			{
				final SememeChronology<DescriptionSememe<?>> desc = addedDescriptions.get(i);
				ValuePropertyPairWithSAB descPP = cuiDescriptions.get(i);
				
				
				BiFunction<String, String, Boolean> functions = new BiFunction<String, String, Boolean>()
				{
					@Override
					public Boolean apply(String atn, String atv)
					{
						//Pull these up to the concept.
						if ("UMLSCUI".equals(atn))
						{
							uniqueUMLSCUI.add(atv);
							return true;
						}
						return false;
					}
				};
		
				//TODO should I be passing in item code here?
				processSAT(ComponentReference.fromChronology(desc), descPP.getSatData(), null, descPP.getSab(), functions);
			}
			
			
			//pulling up the UMLS CUIs.  
			//uniqueUMLSCUI is populated during processSAT
			for (String umlsCui : uniqueUMLSCUI)
			{
				UUID itemUUID = ConverterUUID.createNamespaceUUIDFromString("UMLSCUI" + umlsCui);
				importUtil_.addStringAnnotation(cuiConcept, itemUUID, umlsCui, ptTermAttributes_.getProperty("UMLSCUI").getUUID(), State.ACTIVE);
			}
			
			ValuePropertyPairWithAttributes.processAttributes(importUtil_, cuiDescriptions, addedDescriptions);
			
			//there are no attributes in rxnorm without an AUI.
			
//			try
//			{
				importUtil_.addRefsetMembership(cuiConcept, allCUIRefsetConcept_.getPrimordialUuid(), State.ACTIVE, null);
//			}
//			catch (RuntimeException e)
//			{
//				if (e.toString().contains("duplicate UUID"))
//				{
//					//ok - this can happen due to multiple merges onto an existing SCT concept
//				}
//				else
//				{
//					throw e;
//				}
//			}
			
			//add semantic types
			semanticTypeStatement.clearParameters();
			semanticTypeStatement.setString(1, rxCui);
			ResultSet rs = semanticTypeStatement.executeQuery();
			processSemanticTypes(cuiConcept, rs);
			
			if (conceptTime < 0)
			{
				throw new RuntimeException("oops");
			}
			importUtil_.createConcept(cuiConcept.getPrimordialUuid(), conceptTime, conceptState, null);
		}

		HashSet<UUID> parents = new HashSet<>();
		cuiRelStatementForward.clearParameters();
		cuiRelStatementForward.setString(1, rxCui);
		parents.addAll(addRelationships(cuiConcept, REL.read(null, cuiRelStatementForward.executeQuery(), true, allowedCUIsForSABs_, 
				skippedRelForNotMatchingCUIFilter, true, (string -> reverseRel(string)))));
		
		cuiRelStatementBackward.clearParameters();
		cuiRelStatementBackward.setString(1, rxCui);
		parents.addAll(addRelationships(cuiConcept, REL.read(null, cuiRelStatementBackward.executeQuery(), false, allowedCUIsForSABs_, 
				skippedRelForNotMatchingCUIFilter, true, (string -> reverseRel(string)))));
		
		//Have to add multiple parents at once, no place to keep all the other details.  Load those as associations for now.
		if (parents.size() > 0)
		{
			ComponentReference.fromChronology(importUtil_.addParent(cuiConcept, null, parents.toArray(new UUID[parents.size()]), null, null));
		}
	}
	

	private Property makeDescriptionType(String fsnName, String altName, String description, final Set<String> tty_classes)
	{
		// The current possible classes are:
		// preferred
		// obsolete
		// entry_term
		// hierarchical
		// synonym
		// attribute
		// abbreviation
		// expanded
		// other

		int descriptionTypeRanking;

		//Note - ValuePropertyPairWithSAB overrides the sorting based on these values to kick RXNORM sabs to the top, where 
		//they will get used as FSN.
		if (fsnName.equals("FN") && tty_classes.contains("preferred"))
		{
			descriptionTypeRanking = BPT_Descriptions.FSN;
		}
		else if (fsnName.equals("FN"))
		{
			descriptionTypeRanking = BPT_Descriptions.FSN + 1;
		}
		// preferred gets applied with others as well, in some cases. Don't want 'preferred' 'obsolete' at the top.
		//Just preferred, and we make it the top synonym.
		else if (tty_classes.contains("preferred") && tty_classes.size() == 1)
		{
			//these sub-rankings are somewhat arbitrary at the moment, and in general, unused.  There is an error check up above which 
			//will fail the conversion if it tries to rely on these sub-rankings to find a preferred term
			int preferredSubRank;
			if (altName.equals("IN"))
			{
				preferredSubRank = 1;
			}
			else if (altName.equals("MIN"))
			{
				preferredSubRank = 2;
			}
			else if (altName.equals("PIN"))
			{
				preferredSubRank = 3;
			}
			else if (altName.equals("SCD"))
			{
				preferredSubRank = 4;
			}
			else if (altName.equals("BN"))
			{
				preferredSubRank = 5;
			}
			else if (altName.equals("SBD"))
			{
				preferredSubRank = 6;
			}
			else if (altName.equals("DF"))
			{
				preferredSubRank = 7;
			}
			else if (altName.equals("BPCK"))
			{
				preferredSubRank = 8;
			}
			else if (altName.equals("GPCK"))
			{
				preferredSubRank = 10;
			}
			else if (altName.equals("DFG"))
			{
				preferredSubRank = 11;
			}
			else if (altName.equals("PSN"))
			{
				preferredSubRank = 12;
			}
			else if (altName.equals("SBDC"))
			{
				preferredSubRank = 13;
			}
			else if (altName.equals("SCDC"))
			{
				preferredSubRank = 14;
			}
			else if (altName.equals("SBDF"))
			{
				preferredSubRank = 15;
			}
			else if (altName.equals("SCDF"))
			{
				preferredSubRank = 16;
			}
			else if (altName.equals("SBDG"))
			{
				preferredSubRank = 17;
			}
			else if (altName.equals("SCDG"))
			{
				preferredSubRank = 18;
			}
			else
			{
				preferredSubRank = 20;
				ConsoleUtil.printErrorln("Unranked preferred TTY type! " + fsnName + " " + altName);
			}
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + preferredSubRank;
		}
		else if (tty_classes.contains("entry_term"))
		{
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + 21;
		}
		else if (tty_classes.contains("synonym"))
		{
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + 22;
		}
		else if (tty_classes.contains("expanded"))
		{
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + 23;
		}
		else if (tty_classes.contains("Prescribable Name"))
		{
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + 24;
		}
		else if (tty_classes.contains("abbreviation"))
		{
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + 25;
		}
		else if (tty_classes.contains("attribute"))
		{
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + 26;
		}
		else if (tty_classes.contains("hierarchical"))
		{
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + 27;
		}
		else if (tty_classes.contains("other"))
		{
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + 28;
		}
		else if (tty_classes.contains("obsolete"))
		{
			descriptionTypeRanking = BPT_Descriptions.SYNONYM + 29;
		}
		else
		{
			throw new RuntimeException("Unexpected class type " + Arrays.toString(tty_classes.toArray()));
		}
		return new Property(null, fsnName, altName, description, false, descriptionTypeRanking, null);
	}

	private void processSAT(ComponentReference itemToAnnotate, List<RXNSAT> satRows, String itemCode, String itemSab, 
			BiFunction<String, String, Boolean> skipCheck) throws SQLException, PropertyVetoException
	{
		for (RXNSAT rxnsat : satRows)
		{
			if (skipCheck != null)
			{
				if (skipCheck.apply(rxnsat.atn, rxnsat.atv))
				{
					continue;
				}
			}
			
			//for some reason, ATUI isn't always provided - don't know why.  must gen differently in those cases...
			UUID stringAttrUUID;
			UUID refsetUUID = ptTermAttributes_.getProperty(rxnsat.atn).getUUID();
			if (rxnsat.atui != null)
			{
				stringAttrUUID = ConverterUUID.createNamespaceUUIDFromString("ATUI" + rxnsat.atui);
			}
			else
			{
				//need to put the aui in here, to keep it unique, as each AUI frequently specs the same CUI
				stringAttrUUID = ConverterUUID.createNamespaceUUIDFromStrings(itemToAnnotate.getPrimordialUuid().toString(), 
					rxnsat.rxaui, rxnsat.atv, refsetUUID.toString());
			}
			
			//You would expect that ptTermAttributes_.get() would be looking up sab, rather than having RxNorm hardcoded... but this is an oddity of 
			//a hack we are doing within the RxNorm load.
			ComponentReference attribute = ComponentReference.fromChronology(importUtil_.addStringAnnotation(itemToAnnotate, stringAttrUUID, 
				rxnsat.atv, refsetUUID, State.ACTIVE), () -> "Attribute");
			
			if (StringUtils.isNotBlank(rxnsat.atui))
			{
				importUtil_.addStringAnnotation(attribute, rxnsat.atui, ptUMLSAttributes_.getProperty("ATUI").getUUID(), null);
			}

			if (StringUtils.isNotBlank(rxnsat.stype))
			{
				importUtil_.addUUIDAnnotation(attribute, sTypes_.get(rxnsat.stype), ptUMLSAttributes_.getProperty("STYPE").getUUID());
			}
			
			if (StringUtils.isNotBlank(rxnsat.code) && StringUtils.isNotBlank(itemCode)&& !rxnsat.code.equals(itemCode))
			{
				throw new RuntimeException("oops");
//				if ()
//				{
//					eConcepts_.addStringAnnotation(attribute, code, ptUMLSAttributes_.getProperty("CODE").getUUID(), State.ACTIVE);
//				}
			}

			if (StringUtils.isNotBlank(rxnsat.satui))
			{
				importUtil_.addStringAnnotation(attribute, rxnsat.satui, ptUMLSAttributes_.getProperty("SATUI").getUUID(), State.ACTIVE);
			}
			
			//only load the sab if it is different than the sab of the item we are putting this attribute on
			if (StringUtils.isNotBlank(rxnsat.sab) && !rxnsat.sab.equals(itemSab))
			{
				throw new RuntimeException("Oops");
				//eConcepts_.addUuidAnnotation(attribute, ptSABs_.getProperty(sab).getUUID(), ptUMLSAttributes_.getProperty("SAB").getUUID());
			}
			if (StringUtils.isNotBlank(rxnsat.suppress))
			{
				importUtil_.addUUIDAnnotation(attribute, suppress_.get(rxnsat.suppress), ptUMLSAttributes_.getProperty("SUPPRESS").getUUID());
			}
			if (StringUtils.isNotBlank(rxnsat.cvf))
			{
				if (rxnsat.cvf.equals("4096"))
				{
					importUtil_.addRefsetMembership(attribute, cpcRefsetConcept_.getPrimordialUuid(), State.ACTIVE, null);
				}
				else
				{
					throw new RuntimeException("Unexpected value in RXNSAT cvf column '" + rxnsat.cvf + "'");
				}
			}
		}
	}
	
	
	/**
	 * If sabList is null or empty, no sab filtering is done. 
	 */
	private void init() throws Exception
	{
		clearTargetFiles();
		
		String fileNameDatePortion = loadDatabase();
		SimpleDateFormat sdf = new SimpleDateFormat("MMddyyyy");
		long defaultTime = sdf.parse(fileNameDatePortion).getTime();
		
		abbreviationExpansions = AbbreviationExpansion.load(getClass().getResourceAsStream("/RxNormAbbreviationsExpansions.txt"));
		
		mapToIsa.put("isa", false);
		mapToIsa.put("inverse_isa", false);
		//not translating this one to isa for now
		//		mapToIsa.add("CHD");
		mapToIsa.put("tradename_of", false);
		mapToIsa.put("has_tradename", false);
		
		//Cleanup the sabsToInclude list
		HashSet<String> temp = new HashSet<>();
		if (sabsToInclude != null)
		{
			for (String s : sabsToInclude)
			{
				temp.add(s.toUpperCase());
			}
		}
		temp.add("RXNORM");
		if (temp.contains(sctSab_))
		{
			linkSnomedCT_ = true;
			temp.remove(sctSab_);
		}
		else
		{
			linkSnomedCT_ = false;
		}
		sabsToInclude = new ArrayList<>();
		sabsToInclude.addAll(temp);
		
		//We need to make queries of an SCT DB for part of this load, pull them in, pre-load.
		File[] ibdfFiles = new File(inputFileLocation, "ibdf").listFiles(new FileFilter()
		{
			@Override
			public boolean accept(File pathname)
			{
				if (linkSnomedCT_ && pathname.isFile() && pathname.getName().toLowerCase().endsWith(".ibdf"))
				{
					return true;
				}
				return false;
			}
		});
		
		importUtil_ = new IBDFCreationUtility(Optional.of("RxNorm " + converterSourceArtifactVersion), Optional.of(MetaData.RXNORM_MODULES), outputDirectory, 
			converterOutputArtifactId, converterOutputArtifactVersion, converterOutputArtifactClassifier, false, defaultTime,
			Arrays.asList(new SememeType[] {SememeType.DESCRIPTION, SememeType.COMPONENT_NID, SememeType.LOGIC_GRAPH}), true, ibdfFiles);

		metaDataRoot_ = ComponentReference.fromConcept(importUtil_.createConcept("RxNorm Metadata" + IBDFCreationUtility.metadataSemanticTag_, true, 
				MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid()));

		loadMetaData();

		importUtil_.loadTerminologyMetadataAttributes(metaDataRoot_, converterSourceArtifactVersion, 
			Optional.of(fileNameDatePortion), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);
		
		ConsoleUtil.println("Metadata Statistics");
		for (String s : importUtil_.getLoadStats().getSummary())
		{
			ConsoleUtil.println(s);
		}

		importUtil_.clearLoadStats();
		satRelStatement_ = db_.getConnection().prepareStatement("select * from " + tablePrefix_ + "SAT where RXAUI" 
				+ "= ? and STYPE='RUI' and " + createSabQueryPart("", linkSnomedCT_));
		
		hasTTYType_ = db_.getConnection().prepareStatement("select count (*) as count from RXNCONSO where rxcui=? and TTY=? and " 
			+ createSabQueryPart("", linkSnomedCT_));
		
		if (linkSnomedCT_)
		{
			prepareSCTMaps();
		}
	}
	
	private void prepareSCTMaps() throws SQLException
	{
		Get.sememeService().getSememeSequencesFromAssemblage(MetaData.SCTID.getConceptSequence()).stream().forEach(sememe ->
		{
			@SuppressWarnings({ "unchecked", "rawtypes" })
			Optional<LatestVersion<StringSememe<?>>> lv = ((SememeChronology)Get.sememeService().getSememe(sememe)).getLatestVersion(StringSememe.class, 
				StampCoordinates.getDevelopmentLatest()); 
			StringSememe<?> ss = lv.get().value();
			Long sctId = Long.parseLong(ss.getString());
			UUID conceptUUID = Get.identifierService().getUuidPrimordialForNid(ss.getReferencedComponentNid()).get();
			sctIdToUUID_.put(sctId, conceptUUID);
		});
		
		ConsoleUtil.println("Read SCTID -> UUID mappings for " + sctIdToUUID_.size() + " items");
		ResultSet rs = db_.getConnection().createStatement().executeQuery("SELECT DISTINCT RXCUI, CODE from RXNCONSO where SAB='" + sctSab_ + "'");
		while (rs.next())
		{
			String cui = rs.getString("RXCUI");
			long sctid = Long.parseLong(rs.getString("CODE"));
			if (sctIdToUUID_.containsKey(sctid))
			{
				cuiToSCTID_.put(cui, sctid);
			}
		}
		rs.close();
		
		ConsoleUtil.println("Read CUI -> SCTID mappings for " + cuiToSCTID_.size() + " items");
	}
	
	private String createSabQueryPart(String tablePrefix, boolean includeSCT)
	{
		StringBuffer sb = new StringBuffer();
		sb.append("(");
		for (String s : sabsToInclude)
		{
			sb.append(tablePrefix + "SAB='" + s + "' OR ");
		}
		if (includeSCT)
		{
			sb.append(tablePrefix + "SAB='" + sctSab_ + "' OR ");
		}
		sb.setLength(sb.length() - 4);
		sb.append(")");
		return sb.toString();
	}
	
	private void finish() throws IOException, SQLException
	{
		checkRelationships();
		satRelStatement_.close();
		hasTTYType_.close();
		
		ConsoleUtil.println("Load Statistics");
		for (String s : importUtil_.getLoadStats().getSummary())
		{
			ConsoleUtil.println(s);
		}

		// this could be removed from final release. Just added to help debug editor problems.
		ConsoleUtil.println("Dumping UUID Debug File");
		ConverterUUID.dump(outputDirectory, "RxNormUUID");

		importUtil_.shutdown();
		ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
	}
	
	private void clearTargetFiles()
	{
		new File(outputDirectory, "RxNormUUIDDebugMap.txt").delete();
		new File(outputDirectory, "ConsoleOutput.txt").delete();
		new File(outputDirectory, "RRF.jbin").delete();
	}
	
	private void loadMetaData() throws Exception
	{
		ptRefsets_ = new PT_Refsets(rxNormName);
		ptRefsets_.addProperty(cpcRefsetConceptKey_);
		final PropertyType sourceMetadata = new PT_SAB_Metadata();
		ptRelationshipMetadata_ = new PT_Relationship_Metadata();

		ptUMLSAttributes_ = new PT_Annotations();
		
		importUtil_.loadMetaDataItems(Arrays.asList(ptRefsets_, sourceMetadata, ptRelationshipMetadata_, ptUMLSAttributes_), 
			metaDataRoot_.getPrimordialUuid());

		//Attributes from MRDoc
		//dynamically add more attributes from *DOC
		{
			ConsoleUtil.println("Creating attribute types");
			ptTermAttributes_ = new BPT_Annotations(rxNormName){};
			ptTermAttributes_.indexByAltNames();
			
			Statement s = db_.getConnection().createStatement();
			//extra logic at the end to keep NDC's from any sab when processing RXNorm
			ResultSet rs = s.executeQuery("SELECT VALUE, TYPE, EXPL from " + tablePrefix_ + "DOC where DOCKEY = 'ATN' and VALUE in (select distinct ATN from " 
					+ tablePrefix_ + "SAT" + " where " + createSabQueryPart("", false) + " or ATN='NDC')");
			while (rs.next())
			{
				String abbreviation = rs.getString("VALUE");
				String type = rs.getString("TYPE");
				String expansion = rs.getString("EXPL");

				if (!type.equals("expanded_form"))
				{
					throw new RuntimeException("Unexpected type in the attribute data within DOC: '" + type + "'");
				}

				String altName = null;
				String description = null;
				if (expansion.length() > 30)
				{
					description = expansion;
				}
				else
				{
					altName = expansion;
				}
				
				AbbreviationExpansion ae = abbreviationExpansions.get(abbreviation);
				if (ae == null)
				{
					ConsoleUtil.printErrorln("No Abbreviation Expansion found for " + abbreviation);
					ptTermAttributes_.addProperty(abbreviation, altName, description);
				}
				else
				{
					ptTermAttributes_.addProperty(ae.getExpansion(), ae.getAbbreviation(), ae.getDescription());
				}
			}

			rs.close();
			s.close();
			
			if (ptTermAttributes_.getProperties().size() > 0)
			{
				importUtil_.loadMetaDataItems(ptTermAttributes_, metaDataRoot_.getPrimordialUuid());
			}
		}
		
		// description types
		{
			ConsoleUtil.println("Creating description_ types");
			ptDescriptions_ = new PT_Descriptions(rxNormName);
			ptDescriptions_.indexByAltNames();
			Statement s = db_.getConnection().createStatement();
			ResultSet usedDescTypes;
			usedDescTypes = s.executeQuery("select distinct TTY from RXNCONSO WHERE " + createSabQueryPart("", false));

			PreparedStatement ps = db_.getConnection().prepareStatement("select TYPE, EXPL from " + tablePrefix_ + "DOC where DOCKEY='TTY' and VALUE=?");

			while (usedDescTypes.next())
			{
				String tty = usedDescTypes.getString(1);
				ps.setString(1, tty);
				ResultSet descInfo = ps.executeQuery();

				String expandedForm = null;
				final HashSet<String> classes = new HashSet<>();

				while (descInfo.next())
				{
					String type = descInfo.getString("TYPE");
					String expl = descInfo.getString("EXPL");
					if (type.equals("expanded_form"))
					{
						if (expandedForm != null)
						{
							throw new RuntimeException("Expected name to be null!");
						}
						expandedForm = expl;
					}
					else if (type.equals("tty_class"))
					{
						classes.add(expl);
					}
					else
					{
						throw new RuntimeException("Unexpected type in DOC for '" + tty + "'");
					}
				}
				descInfo.close();
				ps.clearParameters();
				
				Property p = null;
				AbbreviationExpansion ae = abbreviationExpansions.get(tty);
				if (ae == null)
				{
					ConsoleUtil.printErrorln("No Abbreviation Expansion found for " + tty);
					p = makeDescriptionType(tty, expandedForm, null, classes);
				}
				else
				{
					p = makeDescriptionType(ae.getExpansion(), ae.getAbbreviation(), ae.getDescription(), classes);
				}
				
				ptDescriptions_.addProperty(p);
				
				for (String tty_class : classes)
				{
					importUtil_.addStringAnnotation(ComponentReference.fromConcept(p.getUUID()), tty_class, 
						ptUMLSAttributes_.getProperty("tty_class").getUUID(), State.ACTIVE);
				}
				
			}
			usedDescTypes.close();
			s.close();
			ps.close();
			
			if (ptDescriptions_.getProperties().size() > 0)
			{
				importUtil_.loadMetaDataItems(ptDescriptions_, metaDataRoot_.getPrimordialUuid());
			}
		}
		
		loadRelationshipMetadata();
		
		//STYPE values
		sTypes_= new HashMap<String, UUID>();
		{
			ConsoleUtil.println("Creating STYPE types");
			Statement s = db_.getConnection().createStatement();
			ResultSet rs = s.executeQuery("SELECT DISTINCT VALUE, TYPE, EXPL FROM " + tablePrefix_ + "DOC where DOCKEY like 'STYPE%'");
			while (rs.next())
			{
				String sType = rs.getString("VALUE");
				String type = rs.getString("TYPE");
				String name = rs.getString("EXPL");

				if (!type.equals("expanded_form"))
				{
					throw new RuntimeException("Unexpected type in the attribute data within DOC: '" + type + "'");
				}				
				
				ComponentReference c = ComponentReference.fromConcept(importUtil_.createConcept(
						ConverterUUID.createNamespaceUUIDFromString(ptUMLSAttributes_.getProperty("STYPE").getUUID() + ":" + name),
						name, null, null, sType, ptUMLSAttributes_.getProperty("STYPE").getUUID(), null));
				sTypes_.put(name, c.getPrimordialUuid());
				sTypes_.put(sType, c.getPrimordialUuid());
			}
			rs.close();
			s.close();
		}
		
		suppress_=  xDocLoaderHelper("SUPPRESS", "Suppress", false, ptUMLSAttributes_.getProperty("SUPPRESS").getUUID());
		
		//Not yet loading co-occurrence data yet, so don't need these yet.
		//xDocLoaderHelper("COA", "Attributes of co-occurrence", false);
		//xDocLoaderHelper("COT", "Type of co-occurrence", true);  
		
		final HashMap<String, UUID> contextTypes = xDocLoaderHelper("CXTY", "Context Type", false, sourceMetadata.getProperty("CXTY").getUUID());
		
		//not yet loading mappings - so don't need this yet
		//xDocLoaderHelper("FROMTYPE", "Mapping From Type", false);  
		//xDocLoaderHelper("TOTYPE", "Mapping To Type", false);  
		//MAPATN - not yet used in UMLS
		
		// Handle the languages
		//Not actually doing anythign with these at the moment, we just map to metadata languages.
		{
			Statement s = db_.getConnection().createStatement();
			ResultSet rs = s.executeQuery("SELECT * from " + tablePrefix_ + "DOC where DOCKEY = 'LAT' and VALUE in (select distinct LAT from " 
					+ tablePrefix_ + "CONSO where " + createSabQueryPart("", false) + ")");
			while (rs.next())
			{
				String abbreviation = rs.getString("VALUE");
				String type = rs.getString("TYPE");
				//String expansion = rs.getString("EXPL");

				if (!type.equals("expanded_form"))
				{
					throw new RuntimeException("Unexpected type in the language data within DOC: '" + type + "'");
				}

				if (abbreviation.equals("ENG") || abbreviation.equals("SPA"))
				{
					// use official ISAAC languages
					if (abbreviation.equals("ENG") || abbreviation.equals("SPA"))
					{
						//We can map these onto metadata types.
					}
					else
					{
						throw new RuntimeException("unsupported language");
					}
				}
			}
			rs.close();
			s.close();
		}
		
		// And Source Restriction Levels
		{
			ConsoleUtil.println("Creating Source Restriction Level types");
			sourceRestrictionLevels_ = new HashMap<String, UUID>();
			PreparedStatement ps = db_.getConnection().prepareStatement("SELECT VALUE, TYPE, EXPL from " + tablePrefix_ + "DOC where DOCKEY=? ORDER BY VALUE");
			ps.setString(1, "SRL");
			ResultSet rs = ps.executeQuery();
			
			String value = null;
			String description = null;
			String uri = null;
			
			//Two entries per SRL, read two rows, create an entry.
			
			while (rs.next())
			{
				String type = rs.getString("TYPE");
				String expl = rs.getString("EXPL");
				
				if (type.equals("expanded_form"))
				{
					description = expl;
				}
				else if (type.equals("uri"))
				{
					uri = expl;
				}
				else
				{
					throw new RuntimeException("oops");
				}
				
				if (value == null)
				{
					value = rs.getString("VALUE");
				}
				else
				{
					if (!value.equals(rs.getString("VALUE")))
					{
						throw new RuntimeException("oops");
					}
					
					if (description == null || uri == null)
					{
						throw new RuntimeException("oops");
					}
					
					ComponentReference c = ComponentReference.fromConcept(importUtil_.createConcept(
							ConverterUUID.createNamespaceUUIDFromString(sourceMetadata.getProperty("SRL").getUUID() + ":" + value),
							value, null, null, description, sourceMetadata.getProperty("SRL").getUUID(), null));
					sourceRestrictionLevels_.put(value, c.getPrimordialUuid());
					importUtil_.addStringAnnotation(c, uri, ptUMLSAttributes_.getProperty("URI").getUUID(), State.ACTIVE);
					type = null;
					expl = null;
					value = null;
				}
			}
			rs.close();
			ps.close();
		}

		// And Source vocabularies
		final PreparedStatement getSABMetadata = db_.getConnection().prepareStatement("Select * from " + tablePrefix_ + "SAB where (VSAB = ? or (RSAB = ? and CURVER='Y' ))");
		{
			ConsoleUtil.println("Creating Source Vocabulary types");
			ptSABs_ = new PropertyType("Source Vocabularies", true, DynamicSememeDataType.STRING){};
			ptSABs_.indexByAltNames();
			
			HashSet<String> sabList = new HashSet<>();
			sabList.addAll(sabsToInclude);
			
			Statement s = db_.getConnection().createStatement();
			ResultSet rs = s.executeQuery("select distinct SAB from RXNSAT where ATN='NDC'");
			while (rs.next())
			{
				sabList.add(rs.getString("SAB"));
			}
			rs.close();
			s.close();
			
			for (String currentSab : sabList)
			{
				s = db_.getConnection().createStatement();
				rs = s.executeQuery("SELECT SON from " + tablePrefix_ + "SAB WHERE (VSAB='" + currentSab + "' or (RSAB='" + currentSab + "' and CURVER='Y'))");
				if (rs.next())
				{
					String son = rs.getString("SON");

					Property p = ptSABs_.addProperty(son, currentSab, null);
					
					ComponentReference cr = ComponentReference.fromConcept(p.getUUID());
				
					try
					{
						//lookup the other columns for the row with this newly added RSAB terminology
						getSABMetadata.setString(1, p.getSourcePropertyAltName() == null ? p.getSourcePropertyNameFSN() : p.getSourcePropertyAltName());
						getSABMetadata.setString(2, p.getSourcePropertyAltName() == null ? p.getSourcePropertyNameFSN() : p.getSourcePropertyAltName());
						ResultSet rs2 = getSABMetadata.executeQuery();
						if (rs2.next())  //should be only one result
						{
							for (Property metadataProperty : sourceMetadata.getProperties())
							{
								String columnName = metadataProperty.getSourcePropertyAltName() == null ? metadataProperty.getSourcePropertyNameFSN() 
										: metadataProperty.getSourcePropertyAltName();
								String columnValue = rs2.getString(columnName);
								if (columnValue == null)
								{
									continue;
								}
								if (columnName.equals("SRL"))
								{
									importUtil_.addUUIDAnnotation(cr, sourceRestrictionLevels_.get(columnValue), metadataProperty.getUUID());
								}
								else if (columnName.equals("CXTY"))
								{
									importUtil_.addUUIDAnnotation(cr, contextTypes.get(columnValue), sourceMetadata.getProperty("CXTY").getUUID());
								}
								else
								{
									importUtil_.addStringAnnotation(cr, columnValue, metadataProperty.getUUID(), State.ACTIVE);
								}
							}
						}
						if (rs2.next())
						{
							throw new RuntimeException("Too many sabs.  Perhaps you should be using versioned sabs!");
						}
						rs2.close();
					}
					catch (SQLException e)
					{
						throw new RuntimeException("Error loading *SAB", e);
					}
				}
				else
				{
					throw new RuntimeException("Too few? SABs - perhaps you need to use versioned SABs.");
				}
				if (rs.next())
				{
					throw new RuntimeException("Too many SABs for '" + currentSab  + "' - perhaps you need to use versioned SABs.");
				}
				rs.close();
				s.close();
			}
			importUtil_.loadMetaDataItems(ptSABs_, metaDataRoot_.getPrimordialUuid());
			getSABMetadata.close();
		}

		// And semantic types
		{
			ConsoleUtil.println("Creating semantic types");
			Statement s = db_.getConnection().createStatement();
			ResultSet rs = s.executeQuery("SELECT distinct TUI, STN, STY from " + tablePrefix_+ "STY");
			while (rs.next())
			{
				final String tui = rs.getString("TUI");
				final String stn = rs.getString("STN");
				String sty = rs.getString("STY");

				ComponentReference c = ComponentReference.fromConcept(importUtil_.createConcept(
						ConverterUUID.createNamespaceUUIDFromString(ptUMLSAttributes_.getProperty("STY").getUUID() + ":" + sty),
						sty, null, null, null, ptUMLSAttributes_.getProperty("STY").getUUID(), null));
				semanticTypes_.put(tui, c.getPrimordialUuid());
				importUtil_.addStringAnnotation(c, tui, ptUMLSAttributes_.getProperty("TUI").getUUID(), State.ACTIVE);
				importUtil_.addStringAnnotation(c, stn, ptUMLSAttributes_.getProperty("STN").getUUID(), State.ACTIVE);
			}
			rs.close();
			s.close();
		}
	}
	
	/*
	 * Note - may return null, if there were no instances of the requested data
	 */
	private HashMap<String, UUID> xDocLoaderHelper(String dockey, String niceName, boolean loadAsDefinition, UUID parent) throws Exception
	{
		HashMap<String, UUID> result = new HashMap<>();
		ConsoleUtil.println("Creating '" + niceName + "' types");
		{
			Statement s = db_.getConnection().createStatement();
			ResultSet rs = s.executeQuery("SELECT VALUE, TYPE, EXPL FROM " + tablePrefix_ + "DOC where DOCKEY='" + dockey + "'");
			while (rs.next())
			{
				String value = rs.getString("VALUE");
				String type = rs.getString("TYPE");
				String name = rs.getString("EXPL");
				
				if (value == null)
				{
					//there is a null entry, don't care about it.
					continue;
				}

				if (!type.equals("expanded_form"))
				{
					throw new RuntimeException("Unexpected type in the attribute data within DOC: '" + type + "'");
				}
				
				UUID created = importUtil_.createConcept(ConverterUUID.createNamespaceUUIDFromString(parent + ":" + (loadAsDefinition ? value : name)),
						(loadAsDefinition ? value : name), null, (loadAsDefinition ? null : value), (loadAsDefinition ? name : null),
						parent, null).getPrimordialUuid();
				result.put((loadAsDefinition ? value : name), created);
				if (!loadAsDefinition)
				{
					result.put(value, created);
				}
			}
			rs.close();
			s.close();
		}
		if (result.size() == 0)
		{
			//This can happen, depending on what is included during the metamorphosys run
			ConsoleUtil.println("No entries found for '" + niceName + "' - skipping");
			return null;
		}
		return result;
	}
	
	private void loadRelationshipMetadata() throws Exception
	{
		ConsoleUtil.println("Creating relationship types");
		//Both of these get added as extra attributes on the relationship definition
		HashMap<String, ArrayList<String>> snomedCTRelaMappings = new HashMap<>(); //Maps something like 'has_specimen_source_morphology' to '118168003' (may be more than one target SCT code)
		HashMap<String, String> snomedCTRelMappings = new HashMap<>();  //Maps something like '118168003' to 'RO'
		
		nameToRel_ = new HashMap<>();
		
		Statement s = db_.getConnection().createStatement();
		//get the inverses of first, before the expanded forms
		ResultSet rs = s.executeQuery("SELECT DOCKEY, VALUE, TYPE, EXPL FROM " + tablePrefix_ + "DOC where DOCKEY ='REL' or DOCKEY = 'RELA' order by TYPE DESC ");
		while (rs.next())
		{
			String dockey = rs.getString("DOCKEY");
			String value = rs.getString("VALUE");
			String type = rs.getString("TYPE");
			String expl = rs.getString("EXPL");
			if (value == null)
			{
				continue;  //don't need this one
			}
			
			if (type.equals("snomedct_rela_mapping"))
			{
				ArrayList<String> targetSCTIDs = snomedCTRelaMappings.get(expl);
				if (targetSCTIDs == null)
				{
					targetSCTIDs = new ArrayList<String>();
					snomedCTRelaMappings.put(expl, targetSCTIDs);
				}
				targetSCTIDs.add(value);
			}
			else if (type.equals("snomedct_rel_mapping"))
			{
				snomedCTRelMappings.put(value, expl);
			}
			else
			{
				Relationship rel = nameToRel_.get(value);
				if (rel == null)
				{
					if (type.endsWith("_inverse"))
					{
						rel = nameToRel_.get(expl);
						if (rel == null)
						{
							rel = new Relationship(dockey.equals("RELA"));
							nameToRel_.put(value, rel);
							nameToRel_.put(expl, rel);
						}
						else
						{
							throw new RuntimeException("shouldn't happen due to query order");
						}
					}
					else
					{
						//only cases where there is no inverse
						rel = new Relationship(dockey.equals("RELA"));
						nameToRel_.put(value, rel);
					}
				}
				
				if (type.equals("expanded_form"))
				{
					rel.addDescription(value, expl);
				}
				else if (type.equals("rela_inverse") || type.equals("rel_inverse"))
				{
					rel.addRelInverse(value, expl);
				}
				else
				{
					throw new RuntimeException("Oops");
				}
			}
		}
		
		rs.close();
		s.close();
		
		HashSet<String> actuallyUsedRelsOrRelas = new HashSet<>();
		
		for (Entry<String, ArrayList<String>> x : snomedCTRelaMappings.entrySet())
		{
			if (!nameToRel_.containsKey(x.getKey()))
			{
				//metamorphosys doesn't seem to remove these when the sct rel types aren't included - just silently remove them 
				//unless it seems that they should map.
				//may_be_a appears to be a bug in RxNorm 2013-12-02.  silently ignore...
				//TODO see if they fix it in the future, make this check version specific?
				//seems to be getting worse... now it fails to remove 'has_life_circumstance' too in 2014AA, and a few others.
				//Changing to a warning.
				ConsoleUtil.printErrorln("Warning - The 'snomedct_rela_mapping' '" + x.getKey() + "' does not have a corresponding REL entry!  Skipping");
//				if (!x.getKey().equals("may_be_a") && !x.getKey().equals("has_life_circumstance"))
//				{
//					throw new RuntimeException("ERROR - No rel for " + x.getKey() + ".");
//				}
				for (String sctId : x.getValue())
				{
					snomedCTRelMappings.remove(sctId);
				}
			}
			else
			{
				for (String sctid : x.getValue())
				{
					nameToRel_.get(x.getKey()).addSnomedCode(x.getKey(), sctid);
					String relType = snomedCTRelMappings.remove(sctid);
					if (relType != null)
					{
						nameToRel_.get(x.getKey()).addRelType(x.getKey(), relType);
						//Shouldn't need this, but there are some cases where the metadata is inconsistent - with how it is actually used.
						actuallyUsedRelsOrRelas.add(relType);
					}
				}
			}
		}
		
		if (snomedCTRelMappings.size() > 0)
		{
			for (Entry<String, String> x : snomedCTRelMappings.entrySet())
			{
				ConsoleUtil.printErrorln(x.getKey() + ":" + x.getValue());
			}
			throw new RuntimeException("oops - still have (things listed above)");
			
		}
		
		ptRelationships_ = new BPT_Relations(rxNormName) {};  
		ptRelationships_.indexByAltNames();
		ptAssociations_ = new BPT_Associations() {};
		ptAssociations_.indexByAltNames();
		
		s = db_.getConnection().createStatement();
		rs = s.executeQuery("select distinct REL, RELA from " + tablePrefix_ + "REL where " + createSabQueryPart("", linkSnomedCT_));
		while (rs.next())
		{
			actuallyUsedRelsOrRelas.add(rs.getString("REL"));
			if (rs.getString("RELA") != null)
			{
				actuallyUsedRelsOrRelas.add(rs.getString("RELA"));
			}
		}
		rs.close();
		s.close();
		
		HashSet<Relationship> uniqueRels = new HashSet<>(nameToRel_.values());
		
		//Sort the generic relationships first, these are needed when processing primary
		ArrayList<Relationship> sortedRels = new ArrayList<>(uniqueRels);
		Collections.sort(sortedRels, new Comparator<Relationship>()
		{
			@Override
			public int compare(Relationship o1, Relationship o2)
			{
				if (o1.getIsRela() && !o2.getIsRela())
				{
					return 1;
				}
				if (o2.getIsRela() && !o1.getIsRela())
				{
					return -1;
				}
				return 0;
			}
		});
		
		for (final Relationship r : sortedRels)
		{
			r.setSwap(db_.getConnection(), tablePrefix_);
			if (!actuallyUsedRelsOrRelas.contains(r.getFSNName()) && !actuallyUsedRelsOrRelas.contains(r.getInverseFSNName()))
			{
				continue;
			}
			
			Property p = null;
			Boolean relTypeMap = mapToIsa.get(r.getFSNName());
			if (relTypeMap != null)  //true or false, make it a rel
			{
				p = new Property((r.getAltName() == null ? r.getFSNName() : r.getAltName()), (r.getAltName() == null ? null : r.getFSNName()),
						r.getDescription(),MetaData.IS_A.getPrimordialUuid());  //map to isA
				ptRelationships_.addProperty(p);  //conveniently, the only thing we will treat as relationships are things mapped to isa.
			}
			if (relTypeMap == null || relTypeMap == false)  //don't make it an association if set to true
			{
				p = new PropertyAssociation(null, (r.getAltName() == null ? r.getFSNName() : r.getAltName()), 
						(r.getAltName() == null ? null : r.getFSNName()), (r.getInverseAltName() == null ? r.getInverseFSNName() : r.getInverseAltName()),
						r.getDescription(), false);
				ptAssociations_.addProperty(p);
			}
			
			ComponentReference cr = ComponentReference.fromConcept(p.getUUID());
			
			//associations already handle inverse names 
			if (!(p instanceof PropertyAssociation) && r.getInverseFSNName() != null)
			{
				importUtil_.addDescription(cr, (r.getInverseAltName() == null ? r.getInverseFSNName() : r.getInverseAltName()), DescriptionType.FSN, 
						false, ptDescriptions_.getProperty("Inverse FSN").getUUID(), State.ACTIVE);
			}
			
			if (r.getAltName() != null)
			{
				//Need to create this UUID to be different than forward name, in case forward and reverse are identical (like 'RO')
				UUID descUUID = ConverterUUID.createNamespaceUUIDFromStrings(cr.getPrimordialUuid().toString(), r.getInverseFSNName(), 
						DescriptionType.SYNONYM.name(), "false", "inverse");
				//Yes, this looks funny, no its not a copy/paste error.  We swap the FSN and alt names for... it a long story.  42.
				importUtil_.addDescription(cr, descUUID, r.getInverseFSNName(), DescriptionType.SYNONYM, false, null, null, null, null,
						ptDescriptions_.getProperty("Inverse Synonym").getUUID(), State.ACTIVE, null);
			}
			
			if (r.getInverseDescription() != null)
			{
				importUtil_.addDescription(cr, r.getInverseDescription(), DescriptionType.DEFINITION, true, 
						ptDescriptions_.getProperty("Inverse Description").getUUID(), State.ACTIVE);
			}
			
			if (r.getRelType() != null)
			{
				Relationship generalRel = nameToRel_.get(r.getRelType());
				
				importUtil_.addUUIDAnnotation(cr, (mapToIsa.containsKey(generalRel.getFSNName()) ? ptRelationships_.getProperty(generalRel.getFSNName()) : 
					ptAssociations_.getProperty(generalRel.getFSNName())).getUUID(), ptRelationshipMetadata_.getProperty("General Rel Type").getUUID());
			}
			
			if (r.getInverseRelType() != null)
			{
				Relationship generalRel = nameToRel_.get(r.getInverseRelType());
				
				importUtil_.addUUIDAnnotation(cr, (mapToIsa.containsKey(generalRel.getFSNName()) ? ptRelationships_.getProperty(generalRel.getFSNName()) : 
					ptAssociations_.getProperty(generalRel.getFSNName())).getUUID(), 
						ptRelationshipMetadata_.getProperty("Inverse General Rel Type").getUUID());
			}
			
			for (String sctCode : r.getRelSnomedCode())
			{
				importUtil_.addUUIDAnnotation(cr, UuidT3Generator.fromSNOMED(sctCode), 
						ptRelationshipMetadata_.getProperty("Snomed Code").getUUID());
			}
			
			for (String sctCode : r.getInverseRelSnomedCode())
			{
				importUtil_.addUUIDAnnotation(cr, UuidT3Generator.fromSNOMED(sctCode), 
						ptRelationshipMetadata_.getProperty("Inverse Snomed Code").getUUID());
			}
		}
		
		if (ptRelationships_.getProperties().size() > 0)
		{
			importUtil_.loadMetaDataItems(ptRelationships_, metaDataRoot_.getPrimordialUuid());
		}
		if (ptAssociations_.getProperties().size() > 0)
		{
			importUtil_.loadMetaDataItems(ptAssociations_, metaDataRoot_.getPrimordialUuid());
		}
	}
	
	private void processSemanticTypes(ComponentReference concept, ResultSet rs) throws SQLException
	{
		while (rs.next())
		{
//			try
//			{
				ComponentReference annotation = ComponentReference.fromChronology(importUtil_.addUUIDAnnotation(concept, 
					semanticTypes_.get(rs.getString("TUI")), ptUMLSAttributes_.getProperty("STY").getUUID()), () -> "Sememe Member");
				if (rs.getString("ATUI") != null)
				{
					importUtil_.addStringAnnotation(annotation, rs.getString("ATUI"), ptUMLSAttributes_.getProperty("ATUI").getUUID(), State.ACTIVE);
				}
	
				if (rs.getObject("CVF") != null)  //might be an int or a string
				{
					importUtil_.addStringAnnotation(annotation, rs.getString("CVF"), ptUMLSAttributes_.getProperty("CVF").getUUID(), State.ACTIVE);
				}
//			}
//			catch (RuntimeException e)
//			{
//				//ok if dupe - this can happen due to multiple merges onto an existing SCT concept
//				if (!e.toString().contains("duplicate UUID"))
//				{
//					throw e;
//				}
//			}
		}
		rs.close();
	}
	
	private UUID createCUIConceptUUID(String cui)
	{
		return ConverterUUID.createNamespaceUUIDFromString("CUI:" + cui, true);
	}
	
	/**
	 * @throws SQLException
	 * @throws PropertyVetoException 
	 */
	private ArrayList<UUID> addRelationships(ComponentReference concept, List<REL> relationships) throws SQLException, PropertyVetoException
	{
		ArrayList<UUID> parents = new ArrayList<>();
		for (REL relationship : relationships)
		{
			relationship.setSourceUUID(concept.getPrimordialUuid());
			
			if (relationship.getSourceAUI() == null)
			{
				
				if (cuiToSCTID_.get(relationship.getTargetCUI()) != null)
				{
					if (cuiToSCTID_.get(relationship.getSourceCUI()) != null)
					{
						//Both source and target are concepts we are linking from SCT.  Don't load the rell.
						continue;
					}
					
					//map to existing target SCT concept
					relationship.setTargetUUID(sctIdToUUID_.get(cuiToSCTID_.get(relationship.getTargetCUI())));
				}
				else
				{
					//must be a concept we are creating
					relationship.setTargetUUID(createCUIConceptUUID(relationship.getTargetCUI()));	
				}
			}
			else
			{
				throw new RuntimeException("don't yet handle AUI associations");
//				relationship.setTargetUUID(createCuiSabCodeConceptUUID(relationship.getRxNormTargetCUI(), 
//						relationship.getTargetSAB(), relationship.getTargetCode()));
			}

			//We currently don't check the properties on the (duplicate) inverse rels to make sure they are all present - we assume that they 
			//created the inverse relationships as an exact copy of the primary rel direction.  So, just checking the first rel from our dupe list is good enough
			if (isRelPrimary(relationship.getRel(), relationship.getRela()))
			{
				//This can happen when the reverse of the rel equals the rel... sib/sib
				if (relCheckIsRelLoaded(relationship))
				{
					continue;
				}
				
				Property relTypeAsRel = ptRelationships_.getProperty(
						(relationship.getRela() == null ? relationship.getRel() : relationship.getRela()));
				
				PropertyAssociation relTypeAsAssn = (PropertyAssociation)ptAssociations_.getProperty(
						(relationship.getRela() == null ? relationship.getRel() : relationship.getRela()));
				
				ComponentReference r;
				if (relTypeAsRel != null)
				{
					parents.add(relationship.getTargetUUID());
					continue;
				}
				else if (relTypeAsAssn != null)
				{
					r = ComponentReference.fromChronology(importUtil_.addAssociation(concept, (relationship.getRui() != null ? 
							ConverterUUID.createNamespaceUUIDFromString("RUI:" + relationship.getRui()) : null),
							relationship.getTargetUUID(), relTypeAsAssn.getUUID(), State.ACTIVE, null, null), () -> "Association");
				}
				else
				{
					throw new RuntimeException("Unexpected rel handling");
				}

				//Add the annotations
				HashSet<String> addedRUIs = new HashSet<>();
				if (StringUtils.isNotBlank(relationship.getRela()))  //we already used rela - annotate with rel.
				{
					Property genericType = ptAssociations_.getProperty(relationship.getRel()) == null ? 
							ptRelationships_.getProperty(relationship.getRel()) :
								ptAssociations_.getProperty(relationship.getRel());
					boolean reversed = false;
					if (genericType == null && relationship.getRela().equals("mapped_from"))
					{
						//This is to handle non-sensical data in UMLS... they have no consistency in the generic rel they assign - sometimes RB, sometimes RN.
						//reverse it - currently, only an issue on 'mapped_from' rels - as the code in Relationship.java has some exceptions for this type.
						genericType = ptAssociations_.getProperty(reverseRel(relationship.getRel())) == null ? 
								ptRelationships_.getProperty(reverseRel(relationship.getRel())) :
									ptAssociations_.getProperty(reverseRel(relationship.getRel()));
						reversed = true;
					}
					importUtil_.addUUIDAnnotation(r, genericType.getUUID(), 
							ptUMLSAttributes_.getProperty(reversed ? "Generic rel type (inverse)" : "Generic rel type").getUUID());
				}
				if (StringUtils.isNotBlank(relationship.getRui()))
				{
					if (!addedRUIs.contains(relationship.getRui()))
					{
						importUtil_.addStringAnnotation(r, relationship.getRui(), ptUMLSAttributes_.getProperty("RUI").getUUID(), State.ACTIVE);
						addedRUIs.add(relationship.getRui());
						satRelStatement_.clearParameters();
						satRelStatement_.setString(1, relationship.getRui());
						ResultSet nestedRels = satRelStatement_.executeQuery();
						ArrayList<RXNSAT> satData = new ArrayList<>();
						while (nestedRels.next())
						{
							satData.add(new RXNSAT(nestedRels));
						}
						nestedRels.close();
						processSAT(r, satData, null, relationship.getSab(), null);
					}
				}
				if (StringUtils.isNotBlank(relationship.getRg()))
				{
					importUtil_.addStringAnnotation(r, relationship.getRg(), ptUMLSAttributes_.getProperty("RG").getUUID(), State.ACTIVE);
				}
				if (StringUtils.isNotBlank(relationship.getDir()))
				{
					importUtil_.addStringAnnotation(r, relationship.getDir(), ptUMLSAttributes_.getProperty("DIR").getUUID(), State.ACTIVE);
				}
				if (StringUtils.isNotBlank(relationship.getSuppress()))
				{
					importUtil_.addUUIDAnnotation(r, suppress_.get(relationship.getSuppress()), 
							ptUMLSAttributes_.getProperty("SUPPRESS").getUUID());
				}

				if (StringUtils.isNotBlank(relationship.getCvf()))
				{
					if (relationship.getCvf().equals("4096"))
					{
						importUtil_.addRefsetMembership(r, cpcRefsetConcept_.getPrimordialUuid(), State.ACTIVE, null);
					}
					else
					{
						throw new RuntimeException("Unexpected value in RXNSAT cvf column '" + relationship.getCvf() + "'");
					}
				}
				
				relCheckLoadedRel(relationship);
			}
			else
			{
				if (cuiToSCTID_.containsKey(relationship.getSourceCUI()))
				{
					//this is telling us there was a relationship from an SCT concept, to a RXNorm concept, but because we are 
					//not processing sct concept CUIs, we will never process this one in the forward direction.
					//For now, don't put it in the skip list.  
					//Perhaps, in the future, we create a stub SCT concept, and create this association to the RxNorm concept
					//but not now.
				}
				else
				{
					relCheckSkippedRel(relationship);
				}
				
			}
		}
		return parents;
	}
	
	private boolean isRelPrimary(String relName, String relaName)
	{
		if (relaName != null)
		{
			return nameToRel_.get(relaName).getFSNName().equals(relaName);
		}
		else
		{
			return nameToRel_.get(relName).getFSNName().equals(relName);
		}
	}
	
	private String reverseRel(String eitherRelType)
	{
		if (eitherRelType == null)
		{
			return null;
		}
		Relationship r = nameToRel_.get(eitherRelType);
		if (r.getFSNName().equals(eitherRelType))
		{
			return r.getInverseFSNName();
		}
		else if (r.getInverseFSNName().equals(eitherRelType))
		{
			return r.getFSNName();
		}
		else
		{
			throw new RuntimeException("gak");
		}
	}
	
	private void relCheckLoadedRel(REL rel)
	{
		loadedRels_.add(rel.getRelHash());
		skippedRels_.remove(rel.getRelHash());
	}
	
	private boolean relCheckIsRelLoaded(REL rel)
	{
		return loadedRels_.contains(rel.getRelHash());
	}
	
	/**
	 * Call this when a rel wasn't added because the rel was listed with the inverse name, rather than the primary name. 
	 */
	private void relCheckSkippedRel(REL rel)
	{
		skippedRels_.add(rel.getInverseRelHash(string -> nameToRel_.get(string)));
	}
	
	private void checkRelationships()
	{
		//if the inverse relationships all worked properly, skipped should be empty when loaded is subtracted from it.
		for (UUID uuid : loadedRels_)
		{
			skippedRels_.remove(uuid);
		}
		
		if (skippedRels_.size() > 0)
		{
			ConsoleUtil.printErrorln("Relationship design error - " +  skippedRels_.size() + " were skipped that should have been loaded");
		}
		else
		{
			ConsoleUtil.println("Yea! - no missing relationships!");
		}
	}
	
	public static void main(String[] args) throws MojoExecutionException
	{
		RxNormMojo mojo = new RxNormMojo();
		mojo.outputDirectory = new File("../rxnorm-ibdf/rxnorm/target");
		mojo.inputFileLocation = new File("../rxnorm-ibdf/rxnorm/target/generated-resources/src");
		mojo.converterVersion = "foo";
		mojo.converterOutputArtifactVersion = "bar";
		mojo.converterSourceArtifactVersion = "foooo";
		mojo.converterOutputArtifactId = "rxnorm-ibdf";
		mojo.execute();
	}
}
