package gov.va.med.term.nucc.mojo;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;

import gov.va.med.term.nucc.data.EnumValidatedTableData;
import gov.va.med.term.nucc.data.EnumValidatedTableDataReader;
import gov.va.med.term.nucc.propertyTypes.PT_Annotations;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.constants.DynamicSememeConstants;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeNidImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeStringImpl;

/**
 * 
 * {@link NUCCImportMojo}
 *
 * Goal which converts NUCC data into the workbench jbin format
 * 
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
@Mojo (name = "convert-NUCC-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class NUCCImportMojo extends ConverterBaseMojo
{
	//private static Logger log = LogManager.getLogger(NUCCImportMojo.class);
	
	private IBDFCreationUtility importUtil_;

	private HashMap<UUID, String> loadedConcepts = new HashMap<>();

	private PropertyType attributes_;
	private BPT_Refsets refsets_;

	private UUID allNuccConceptsRefset;

	private int conceptCount = 0;

	@Override
	public void execute() throws MojoExecutionException
	{
		try
		{
			super.execute();

			String temp = "Bogus date"; // TODO Find date from source
			Date date = null;
			try {
				date = new SimpleDateFormat("yyyy.MM.dd").parse(temp);
			} catch (Exception e) {
				date = new Date(); // TODO remove this when getting valid data from source
			}
			
			importUtil_ = new IBDFCreationUtility(
					Optional.of("NUCC" + " " + converterSourceArtifactVersion),
					Optional.of(MetaData.NUCC_MODULES),
					outputDirectory,
					converterOutputArtifactId,
					converterOutputArtifactVersion,
					converterOutputArtifactClassifier, false, date.getTime());
			
			attributes_ = new PT_Annotations();
			
			refsets_ = new BPT_Refsets("NUCC");
			refsets_.addProperty("All NUCC Concepts");

			// Every time concept created add membership to "All NUCC Concepts"
			allNuccConceptsRefset = refsets_.getProperty("All NUCC Concepts").getUUID();
			
			// Switch on version to select proper Columns enum to use in constructing reader
			final EnumValidatedTableDataReader<ColumnsV1> importer = new EnumValidatedTableDataReader<>(inputFileLocation, ColumnsV1.class);
			final EnumValidatedTableData<ColumnsV1> terminology = importer.process();
			
			ConsoleUtil.println("Loaded Terminology containing " + terminology.rows().size() + " entries");

			/*
			 * COLUMNS from ColumnsV1:
			 * Code, // Required FSN
			 * Grouping, // Create attribute and concepts representing each unique required value
			 * Classification, // Create attribute and concepts representing each optional unique value
			 * Specialization, // Create attribute and concepts representing each optional unique value
			 * Definition, // Optional DEFINITION
			 * Notes // Optional comment
			 */
			// Parent nuccMetadata ComponentReference
			final ComponentReference nuccMetadata = ComponentReference.fromConcept(
					createType(MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid(), "NUCC Metadata" + IBDFCreationUtility.metadataSemanticTag_));
			
			// loadTerminologyMetadataAttributes onto nuccMetadata
			importUtil_.loadTerminologyMetadataAttributes(nuccMetadata, converterSourceArtifactVersion, 
					Optional.empty(), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);

			// load metadata
			importUtil_.loadMetaDataItems(Arrays.asList(attributes_, refsets_), nuccMetadata.getPrimordialUuid());

			// Create NUCC root concept under ISAAC_ROOT
			final ConceptChronology<? extends ConceptVersion<?>> nuccRootConcept = importUtil_.createConcept("NUCC", true, MetaData.ISAAC_ROOT.getPrimordialUuid());
			ConsoleUtil.println("Created NUCC root concept " + nuccRootConcept.getPrimordialUuid() + " under ISAAC_ROOT");

			// Create concepts for each unique value in each of three Grouping, Classification and Specialization columns
			// Each concept is created as a child of its respective column type concept
			// A map of String value to ConceptChronology is maintained for later use
			final Map<String,  ConceptChronology<? extends ConceptVersion<?>>> groupingValueConceptByValueMap = new HashMap<>();
			final UUID groupingPropertyUuid = attributes_.getProperty(ColumnsV1.Grouping.toString()).getUUID();
			for (String value : terminology.getDistinctValues(ColumnsV1.Grouping)) {
				if (StringUtils.isBlank(value)) {
					throw new RuntimeException("Cannot load NUCC data with blank Grouping");
				}

				// Create the Grouping value concept as a child of both NUCC root and the Grouping property metadata concept
				// and store in map for later retrieval
				final UUID valueConceptUuid = ConverterUUID.createNamespaceUUIDFromString(groupingPropertyUuid.toString() + "|" + value, true);
				final ConceptChronology<? extends ConceptVersion<?>> valueConcept = importUtil_.createConcept(
						valueConceptUuid,
						value, null, null, null,
						groupingPropertyUuid, nuccRootConcept.getPrimordialUuid());
				ConsoleUtil.println("Created NUCC Grouping value concept " + valueConcept.getPrimordialUuid() + " for \"" + value + "\" under parents Grouping property " + groupingPropertyUuid + " and NUCC root concept " + nuccRootConcept);

				// Store Grouping value concept in map by value
				groupingValueConceptByValueMap.put(
						value,
						valueConcept);
			}

			// Create a concept for each distinct non blank Classification value and store in map for later retrieval
			final Map<String, ConceptChronology<? extends ConceptVersion<?>>> classificationValueConceptByValueMap = new HashMap<>();
			final UUID classificationPropertyUuid = attributes_.getProperty(ColumnsV1.Classification.toString()).getUUID();
			for (String value : terminology.getDistinctValues(ColumnsV1.Classification)) {
				if (StringUtils.isNotBlank(value)) {
					final UUID valueConceptUuid = ConverterUUID.createNamespaceUUIDFromString(classificationPropertyUuid.toString() + "|" + value, true);
					final ConceptChronology<? extends ConceptVersion<?>> valueConcept = importUtil_.createConcept(valueConceptUuid, value, true, classificationPropertyUuid);
					ConsoleUtil.println("Created NUCC Classification value concept " + valueConcept.getPrimordialUuid() + " for \"" + value + "\" under parent Classification property concept " + classificationPropertyUuid);
					// Store Classification value concept in map by value
					classificationValueConceptByValueMap.put(value, valueConcept);
				}
			}

			// Create a concept for each distinct Specialization value and store in map for later retrieval
			final Map<String, ConceptChronology<? extends ConceptVersion<?>>> specializationValueConceptByValueMap = new HashMap<>();
			final UUID specializationPropertyUuid = attributes_.getProperty(ColumnsV1.Specialization.toString()).getUUID();
			for (String value : terminology.getDistinctValues(ColumnsV1.Specialization)) {
				if (StringUtils.isNotBlank(value)) {
					final UUID valueConceptUuid = ConverterUUID.createNamespaceUUIDFromString(specializationPropertyUuid.toString() + "|" + value, true);
					final ConceptChronology<? extends ConceptVersion<?>> valueConcept = importUtil_.createConcept(valueConceptUuid, value, true, specializationPropertyUuid);
					ConsoleUtil.println("Created NUCC Specialization value concept " + valueConcept.getPrimordialUuid() + " for \"" + value + "\" under parent Specialization property concept " + classificationPropertyUuid);
					// Store Specialization value concept in map by value
					specializationValueConceptByValueMap.put(value, valueConcept);
				}
			}
			
			// Populate hierarchy, one row at a time, creating concepts as children of their respective Grouping concepts
			for (Map<ColumnsV1, String> row : terminology.rows()) {
				final ConceptChronology<? extends ConceptVersion<?>> groupingValueConcept = row.get(ColumnsV1.Grouping) != null ? groupingValueConceptByValueMap.get(row.get(ColumnsV1.Grouping)) : null;
				final ConceptChronology<? extends ConceptVersion<?>> classificationValueConcept = row.get(ColumnsV1.Classification) != null ? classificationValueConceptByValueMap.get(row.get(ColumnsV1.Classification)) : null;
				final ConceptChronology<? extends ConceptVersion<?>> specializationValueConcept = row.get(ColumnsV1.Specialization) != null ? specializationValueConceptByValueMap.get(row.get(ColumnsV1.Specialization)) : null;
				
				if (groupingValueConcept == null) {
					throw new RuntimeException("Cannot create NUCC concept without Grouping: " + row.toString());
				}

				try {
					// Create row concept
					UUID rowConceptUuid = ConverterUUID.createNamespaceUUIDFromString(groupingValueConcept.getPrimordialUuid().toString() + "|" + row.get(ColumnsV1.Code), true);
					final ConceptChronology<? extends ConceptVersion<?>> rowConcept = importUtil_.createConcept(
							rowConceptUuid,
							row.get(ColumnsV1.Code),
							true, groupingValueConcept.getPrimordialUuid());
					final ComponentReference rowComponentReference = ComponentReference.fromConcept(rowConcept);

					// Add required Grouping NID annotation
					importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeNidImpl(groupingValueConcept.getNid()), 
							groupingPropertyUuid, State.ACTIVE, (Long)null);

					// Add optional Classification NID annotation
					if (classificationValueConcept != null) {
						importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeNidImpl(classificationValueConcept.getNid()), 
								classificationPropertyUuid, State.ACTIVE, (Long)null);
					}
					// Add optional Specialization NID annotation
					if (specializationValueConcept != null) {
						importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeNidImpl(specializationValueConcept.getNid()), 
								specializationPropertyUuid, State.ACTIVE, (Long)null);
					}

					// Add optional Notes comment annotation
					if (StringUtils.isNotBlank(row.get(ColumnsV1.Notes))) {
						importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeStringImpl(row.get(ColumnsV1.Notes)), 
								DynamicSememeConstants.get().DYNAMIC_SEMEME_COMMENT_ATTRIBUTE.getPrimordialUuid(), State.ACTIVE, (Long)null);
					}

					// Add optional Definition description
					if (StringUtils.isNotBlank(row.get(ColumnsV1.Definition))) {
						importUtil_.addDescription(rowComponentReference, row.get(ColumnsV1.Definition), DescriptionType.DEFINITION, false, (UUID)null, State.ACTIVE);
					}

					// Add to refset allNuccConceptsRefset
					importUtil_.addRefsetMembership(rowComponentReference, allNuccConceptsRefset, State.ACTIVE, (Long)null);

					++conceptCount;
				} catch (Exception e) {
					final String msg = "Failed processing row with " + e.getClass().getSimpleName() + " " + e.getLocalizedMessage() + ": " + row;
					ConsoleUtil.println(msg);
					throw new RuntimeException(msg, e);
				}
			}
	
			ConsoleUtil.println("Metadata load stats");
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(line);
			}
			importUtil_.clearLoadStats();

			ConsoleUtil.println("Processed " + conceptCount + " concepts");
			
			ConsoleUtil.println("Load Statistics");

			// this could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "nuccUuid");

			importUtil_.shutdown();
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		}
		catch (Exception ex)
		{
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}
	}

	private ConceptChronology<? extends ConceptVersion<?>> createType(UUID parentUuid, String typeName) throws Exception
	{
		ConceptChronology<? extends ConceptVersion<?>> concept = importUtil_.createConcept(typeName, true);
		loadedConcepts.put(concept.getPrimordialUuid(), typeName);
		importUtil_.addParent(ComponentReference.fromConcept(concept), parentUuid);
		return concept;
	}

	public static void main(String[] args) throws MojoExecutionException
	{
		NUCCImportMojo i = new NUCCImportMojo();
		i.outputDirectory = new File("../nucc-ibdf/target");
		i.inputFileLocation = new File("../nucc-ibdf/target/generated-resources/src/");
		i.converterOutputArtifactVersion = "2016.01.07.foo";
		i.converterVersion = "SNAPSHOT";
		i.converterSourceArtifactVersion = "17.0";
		i.execute();
	}
}