package gov.va.med.term.cvx.mojo;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Optional;
import java.util.UUID;
import gov.va.med.term.cvx.propertyTypes.PT_Descriptions;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;

import gov.va.med.term.cvx.data.CVXCodes;
import gov.va.med.term.cvx.data.CVXCodesHelper;
import gov.va.med.term.cvx.data.CVXCodes.CVXInfo;
import gov.va.med.term.cvx.propertyTypes.PT_Annotations;
import gov.va.med.term.cvx.reader.CVXReader;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.chronicle.ObjectChronologyType;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeColumnInfo;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeData;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeDataType;
import gov.vha.isaac.ochre.api.constants.DynamicSememeConstants;
import gov.vha.isaac.ochre.model.configuration.LanguageCoordinates;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeLongImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeStringImpl;

/**
 * 
 * {@link CVXImportMojo}
 *
 * Goal which converts CVX data into the workbench jbin format
 * 
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
@Mojo (name = "convert-CVX-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class CVXImportMojo extends ConverterBaseMojo
{
	//private static Logger log = LogManager.getLogger(CVXImportMojo.class);
	
	private IBDFCreationUtility importUtil_;

	private HashMap<UUID, String> loadedConcepts = new HashMap<>();

	private PropertyType attributes_;
	private BPT_Refsets refsets_;
	private PT_Descriptions descriptions_;

	private UUID allCvxConceptsRefset;

	private int conceptCount = 0;

	@Override
	public void execute() throws MojoExecutionException
	{
		try
		{
			super.execute();

			String temp = "Bogus date"; // TODO Find date from source
			Date date = null;
			try {
				date = new SimpleDateFormat("yyyy.MM.dd").parse(temp);
			} catch (Exception e) {
				date = new Date(); // TODO remove this when getting valid data from source
			}
			
			importUtil_ = new IBDFCreationUtility(
					Optional.of("CVX" + " " + converterSourceArtifactVersion),
					Optional.of(MetaData.CVX_MODULES),
					outputDirectory,
					converterOutputArtifactId,
					converterOutputArtifactVersion,
					converterOutputArtifactClassifier, false, date.getTime());
			
			attributes_ = new PT_Annotations();
			descriptions_ = new PT_Descriptions();
			
			refsets_ = new BPT_Refsets("CVX");
			refsets_.addProperty("All CVX Concepts");

			// Every time concept created add membership to "All CVX Concepts"
			allCvxConceptsRefset = refsets_.getProperty("All CVX Concepts").getUUID();
			
			// Switch on version to select proper Columns enum to use in constructing reader
			final CVXReader importer = new CVXReader(inputFileLocation);
			final CVXCodes terminology = importer.process();
			
			ConsoleUtil.println("Loaded Terminology containing " + terminology.getCVXInfo().size() + " entries");

			/*
			 * Methods from CVXCodes.CVXInfo:
			 * getCVXCode() // float numeric id (CODE?)
			 * getShortDescription() // Required String FSN description?
			 * getFullVaccinename() // Required String preferred term description?
			 * getNotes() // Optional String comment
			 * getOchreState() // Required State (ACTIVE or INACTIVE) 
			 * getLastUpdatedDate(), // Required date ?
			 */
			// Parent cvxMetadata ComponentReference
			final ComponentReference cvxMetadata = ComponentReference.fromConcept(
					createType(MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid(), "CVX Metadata" + IBDFCreationUtility.metadataSemanticTag_));
			
			// loadTerminologyMetadataAttributes onto cvxMetadata
			importUtil_.loadTerminologyMetadataAttributes(cvxMetadata, converterSourceArtifactVersion, 
					Optional.empty(), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);

			// load metadata
			importUtil_.loadMetaDataItems(Arrays.asList(attributes_, refsets_, descriptions_), cvxMetadata.getPrimordialUuid());

			// Create CVX root concept under ISAAC_ROOT
			final ConceptChronology<? extends ConceptVersion<?>> cvxRootConcept = importUtil_.createConcept("CVX", true, MetaData.ISAAC_ROOT.getPrimordialUuid());
			ConsoleUtil.println("Created CVX root concept " + cvxRootConcept.getPrimordialUuid() + " under ISAAC_ROOT");
			
			final UUID fsnSourceDescriptionTypeUUID = descriptions_.getProperty(PT_Descriptions.Descriptions.ShortDescription.name()).getUUID();
			final UUID preferredSynonymSourceDescriptionTypeUUID = descriptions_.getProperty(PT_Descriptions.Descriptions.FullVaccinename.name()).getUUID();

			final UUID lastUpdatedPropertyUuid = attributes_.getProperty(PT_Annotations.Attribute.LastUpdated.getKey()).getUUID();
			final UUID cvxCodePropertyUuid = attributes_.getProperty(PT_Annotations.Attribute.CVXCode.getKey()).getUUID();
			final UUID cvxStatusPropertyUuid = attributes_.getProperty(PT_Annotations.Attribute.Status.getKey()).getUUID();

			final UUID cvxCodeAssemblageConceptUuid = ConverterUUID.createNamespaceUUIDFromString(cvxCodePropertyUuid.toString() + " assemblage", true);
			final ConceptChronology<? extends ConceptVersion<?>> cvxCodeAssemblageConcept = importUtil_.createConcept(cvxCodeAssemblageConceptUuid, "CVXCode Assemblage", true, cvxCodePropertyUuid);
			importUtil_.configureConceptAsDynamicRefex(
					ComponentReference.fromChronology(cvxCodeAssemblageConcept),
					"CVXCode Assemblage",
					new DynamicSememeColumnInfo[] {
							new DynamicSememeColumnInfo(
									0,
									cvxCodePropertyUuid,
									DynamicSememeDataType.STRING,
									null, 
									true,
									true) },
					ObjectChronologyType.CONCEPT,
					SememeType.STRING);
			
			final UUID cvxStatusAssemblageConceptUuid = ConverterUUID.createNamespaceUUIDFromString(cvxStatusPropertyUuid.toString() + " assemblage", true);
			final ConceptChronology<? extends ConceptVersion<?>> cvxStatusAssemblageConcept = importUtil_.createConcept(cvxStatusAssemblageConceptUuid, "CVX Status Assemblage", true, cvxStatusPropertyUuid);
			importUtil_.configureConceptAsDynamicRefex(
					ComponentReference.fromChronology(cvxStatusAssemblageConcept),
					"CVX Status Assemblage",
					new DynamicSememeColumnInfo[] {
							new DynamicSememeColumnInfo(
									0,
									cvxCodePropertyUuid,
									DynamicSememeDataType.STRING,
									null, 
									true,
									true) },
					ObjectChronologyType.CONCEPT,
					SememeType.STRING);

			UUID languageCode = MetaData.ENGLISH_LANGUAGE.getPrimordialUuid();
			UUID dialect = MetaData.US_ENGLISH_DIALECT.getPrimordialUuid();
			UUID caseSignificance = MetaData.DESCRIPTION_CASE_SENSITIVE.getPrimordialUuid();
			// Populate hierarchy, one row at a time, creating concepts as children of their respective Grouping concepts
			for (CVXInfo row : terminology.getCVXInfo()) {
				try {
					String code = CVXCodesHelper.getCVXCode(row) + "";
					String fsn = CVXCodesHelper.getShortDescription(row);
					String preferred = CVXCodesHelper.getFullVaccinename(row);
					State state = CVXCodesHelper.getOchreState(row);
					String cvxStatus = CVXCodesHelper.getStatus(row);

					// Create row concept
					UUID rowConceptUuid = ConverterUUID.createNamespaceUUIDFromString(cvxRootConcept.getPrimordialUuid().toString() + "|" + code, true);
					final ConceptChronology<? extends ConceptVersion<?>> rowConcept = importUtil_.createConcept(rowConceptUuid);
					final ComponentReference rowComponentReference = ComponentReference.fromConcept(rowConcept);
					importUtil_.addParent(rowComponentReference, cvxRootConcept.getPrimordialUuid());
					
					UUID fsnDescriptionUuid = ConverterUUID.createNamespaceUUIDFromString(rowComponentReference.getPrimordialUuid().toString() + "|" + fsn + "|" + DescriptionType.FSN.name(), true);
					importUtil_.addDescription(
							rowComponentReference,
							fsnDescriptionUuid,
							preferred,
							DescriptionType.FSN,
							true,
							dialect,
							caseSignificance,
							languageCode,
							importUtil_.getModule().getPrimordialUuid(),
							fsnSourceDescriptionTypeUUID,
							state,
							CVXCodesHelper.getLastUpdatedDate(row).getTime());
					
					UUID preferredDescriptionUuid = ConverterUUID.createNamespaceUUIDFromString(rowComponentReference.getPrimordialUuid().toString() + "|" + preferred + "|" + DescriptionType.SYNONYM.name(), true);
					importUtil_.addDescription(
							rowComponentReference,
							preferredDescriptionUuid,
							preferred,
							DescriptionType.SYNONYM,
							true,
							dialect,
							caseSignificance,
							languageCode,
							importUtil_.getModule().getPrimordialUuid(),
							preferredSynonymSourceDescriptionTypeUUID,
							state,
							CVXCodesHelper.getLastUpdatedDate(row).getTime());

					// Add required LastUpdated Long (date) annotation
					importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeLongImpl(CVXCodesHelper.getLastUpdatedDate(row).getTime()), 
							lastUpdatedPropertyUuid, state, (Long)null);

					// Add required CVXCode annotation
					importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeStringImpl(code), 
							cvxCodeAssemblageConcept.getPrimordialUuid(), state, (Long)null);

					// Add required CVX Status annotation
					importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeStringImpl(cvxStatus), 
							cvxStatusAssemblageConcept.getPrimordialUuid(), state, (Long)null);
					
					// Add optional Notes comment annotation
					if (StringUtils.isNotBlank(CVXCodesHelper.getNotes(row))) {
						importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeStringImpl(CVXCodesHelper.getNotes(row)), 
								DynamicSememeConstants.get().DYNAMIC_SEMEME_COMMENT_ATTRIBUTE.getPrimordialUuid(), state, (Long)null);
					}

					// Add to refset allCvxConceptsRefset
					importUtil_.addRefsetMembership(rowComponentReference, allCvxConceptsRefset, state, (Long)null);

					++conceptCount;
				} catch (Exception e) {
					final String msg = "Failed processing row with " + e.getClass().getSimpleName() + " " + e.getLocalizedMessage() + ": " + row;
					ConsoleUtil.println(msg);
					throw new RuntimeException(msg, e);
				}
			}
	
			ConsoleUtil.println("Metadata load stats");
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(line);
			}
			importUtil_.clearLoadStats();

			ConsoleUtil.println("Processed " + conceptCount + " concepts");
			
			ConsoleUtil.println("Load Statistics");

			// this could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "cvxUuid");

			importUtil_.shutdown();
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		}
		catch (Exception ex)
		{
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}
	}

	private ConceptChronology<? extends ConceptVersion<?>> createType(UUID parentUuid, String typeName) throws Exception
	{
		ConceptChronology<? extends ConceptVersion<?>> concept = importUtil_.createConcept(typeName, true);
		loadedConcepts.put(concept.getPrimordialUuid(), typeName);
		importUtil_.addParent(ComponentReference.fromConcept(concept), parentUuid);
		return concept;
	}

	public static void main(String[] args) throws MojoExecutionException
	{
		CVXImportMojo i = new CVXImportMojo();
		i.outputDirectory = new File("../cvx-ibdf/target");
		i.inputFileLocation = new File("../cvx-ibdf/target/generated-resources/src/");
		i.converterOutputArtifactVersion = "2016.01.07.foo";
		i.converterVersion = "SNAPSHOT";
		i.converterSourceArtifactVersion = "17.0";
		i.execute();
	}
}