package gov.vha.isaac.term.cpt.mojo;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.zip.ZipFile;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Descriptions;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeStringImpl;
import gov.vha.isaac.term.cpt.mojo.TextReader.CPTFileType;
import gov.vha.isaac.term.cpt.mojo.propertyTypes.PT_Annotations;

/**
 * 
 * {@link CPTImportMojo}
 * 
 * Goal which converts CPT data into the workbench jbin format
 *
 * @author <a href="mailto:daniel.armbrust.list@gmail.com">Dan Armbrust</a>
 */
@Mojo (name = "convert-CPT-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class CPTImportMojo extends ConverterBaseMojo
{
	private IBDFCreationUtility importUtil_;

	@Override
	public void execute() throws MojoExecutionException
	{
		try
		{
			super.execute();
			Date date = null;
			try
			{
				date = new SimpleDateFormat("yyyy").parse(converterSourceArtifactVersion);
			}
			catch (Exception e)
			{
				throw new MojoExecutionException("Failed to parse year from " + converterSourceArtifactVersion);
			}
			
			
			File zipFile = null;
			for (File f: inputFileLocation.listFiles())
			{
				if (f.getName().toLowerCase().endsWith(".zip")) 
				{
					if (zipFile != null)
					{
						throw new MojoExecutionException("Only expected to find one zip file in the folder " + inputFileLocation.getCanonicalPath());
					}
					zipFile = f;
				}
			}
			
			if (zipFile == null)
			{
				throw new MojoExecutionException("Did not find a zip file in " + inputFileLocation.getCanonicalPath());
			}
			
			ZipFile zf = new ZipFile(zipFile);
			
			HashMap<String, CPTData> data = new HashMap<>();
			
			ConsoleUtil.println("Reading LONGULT.txt");
			int read1 = TextReader.read(zf.getInputStream(zf.getEntry("LONGULT.txt")), data, CPTFileType.LONGULT);
			ConsoleUtil.println("Reading MEDU.txt");
			int read2 = TextReader.read(zf.getInputStream(zf.getEntry("MEDU.txt")), data, CPTFileType.MEDU);
			ConsoleUtil.println("Reading SHORTU.txt");
			int read3 = TextReader.read(zf.getInputStream(zf.getEntry("SHORTU.txt")), data, CPTFileType.SHORTU);
			
			zf.close();
			
			if (read1 != read2 || read1 != read3)
			{
				throw new RuntimeException("Didn't find the same number of codes in all 3 files!");
			}
			importUtil_ = new IBDFCreationUtility(Optional.of("CPT" + " " + converterSourceArtifactVersion), Optional.of(MetaData.CPT_MODULES), outputDirectory,
					converterOutputArtifactId, converterOutputArtifactVersion, converterOutputArtifactClassifier, false, date.getTime());
			
			PropertyType attributes = new PT_Annotations();
			PropertyType descriptions = new BPT_Descriptions("CPT");
			descriptions.addProperty("LONGULT", "Long Description Upper/Lower Case", null);
			descriptions.addProperty("MEDU", "Medium Description Upper Case", null);
			descriptions.addProperty("SHORTU", "Short Description Upper Case", null);
			
			BPT_Refsets refsets_ = new BPT_Refsets("CPT");
			refsets_.addProperty("All CPT Concepts");

			// Every time concept created add membership to "All CPT Concepts"
			UUID allCPTConceptsRefset = refsets_.getProperty("All CPT Concepts").getUUID();
			
			// Parent nuccMetadata ComponentReference
			final ComponentReference cptMetadata = ComponentReference.fromConcept(
					createType(MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid(), "CPT Metadata" + IBDFCreationUtility.metadataSemanticTag_));
			
			// loadTerminologyMetadataAttributes onto nuccMetadata
			importUtil_.loadTerminologyMetadataAttributes(cptMetadata, converterSourceArtifactVersion, 
					Optional.empty(), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);

			// load metadata
			importUtil_.loadMetaDataItems(Arrays.asList(attributes, refsets_, descriptions), cptMetadata.getPrimordialUuid());

			// Create CPT root concept under ISAAC_ROOT
			final ConceptChronology<? extends ConceptVersion<?>> cptRootConcept = importUtil_.createConcept("CPT", true, MetaData.ISAAC_ROOT.getPrimordialUuid());

			ConsoleUtil.println("Metadata load stats");
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(line);
			}
			importUtil_.clearLoadStats();

			String firstThree = "";
			ComponentReference parent = null;
			int cptConCount = 0;
			int groupingConCount = 0;
			
			List<CPTData> sorted = new ArrayList<>(data.values());
			Collections.sort(sorted, new Comparator<CPTData>()
			{
				@Override
				public int compare(CPTData o1, CPTData o2)
				{
					int left = Integer.parseInt(o1.code.substring(0, 3));
					int right = Integer.parseInt(o2.code.substring(0, 3));
					return Integer.compare(left,  right);
				}
			});
			
			for (CPTData d : sorted)
			{
				String temp = d.code.substring(0, 3);
				if (!temp.equals(firstThree))
				{
					//Make a new grouping concept
					firstThree = d.code.substring(0, 3);
					parent = ComponentReference.fromConcept(importUtil_.createConcept(firstThree, true, cptRootConcept.getPrimordialUuid()));
					groupingConCount++;
				}
				cptConCount++;
				ComponentReference concept = ComponentReference.fromConcept(importUtil_.createConcept(ConverterUUID.createNamespaceUUIDFromString(d.code)));
				
				importUtil_.addParent(concept, parent.getPrimordialUuid());
				importUtil_.addDescription(concept, d.code, DescriptionType.FSN, true, null, State.ACTIVE);
				importUtil_.addStaticStringAnnotation(concept, d.code, attributes.getProperty(PT_Annotations.Attribute.Code.name()).getUUID(), State.ACTIVE);
				
				importUtil_.addRefsetMembership(concept, allCPTConceptsRefset, State.ACTIVE, null);
				
				if (StringUtils.isNotBlank(d.shortu))
				{
					addDescription(concept, d.shortu, DescriptionType.SYNONYM, descriptions.getProperty("SHORTU").getUUID(), true);
				}
				if (StringUtils.isNotBlank(d.longult))
				{
					addDescription(concept, d.longult, DescriptionType.SYNONYM, descriptions.getProperty("LONGULT").getUUID(), false);
				}
				if (StringUtils.isNotBlank(d.medu))
				{
					addDescription(concept, d.medu, DescriptionType.SYNONYM, descriptions.getProperty("MEDU").getUUID(), false);
				}
			}
			
			ConsoleUtil.println("");
			ConsoleUtil.println("Load Statistics");
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(line);
			}
			
			ConsoleUtil.println("Loaded " + cptConCount + " CPT Concepts");
			ConsoleUtil.println("Created " + groupingConCount + " Grouping Concepts");

			// this could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "cptUuid");

			importUtil_.shutdown();
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		}
		catch (Exception ex)
		{
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}
	}
	
	private void addDescription(ComponentReference concept,  String text, DescriptionType descriptionType, UUID extendedType, boolean preferred)
	{
		UUID descriptionPrimordialUUID = ConverterUUID.createNamespaceUUIDFromStrings(concept.getPrimordialUuid().toString(), text, extendedType.toString(), 
				descriptionType.name(), new Boolean(preferred).toString());
		importUtil_.addDescription(concept, descriptionPrimordialUUID, text, descriptionType, preferred, extendedType, State.ACTIVE);
	}

	private ConceptChronology<? extends ConceptVersion<?>> createType(UUID parentUuid, String typeName) throws Exception
	{
		ConceptChronology<? extends ConceptVersion<?>> concept = importUtil_.createConcept(typeName, true);
		importUtil_.addParent(ComponentReference.fromConcept(concept), parentUuid);
		return concept;
	}

	public static void main(String[] args) throws MojoExecutionException
	{
		CPTImportMojo i = new CPTImportMojo();
		i.outputDirectory = new File("../cpt-ibdf/target");
		i.inputFileLocation = new File("../cpt-ibdf/target/generated-resources/src/");
		i.converterOutputArtifactVersion = "2016.01.07.foo";
		i.converterVersion = "SNAPSHOT";
		i.converterSourceArtifactVersion = "2017";
		i.execute();
	}
}