/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.isaac.utils.file_transfer;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.bind.annotation.XmlTransient;

import com.fasterxml.jackson.annotation.JsonIgnore;

/**
 * 
 * {@link Config}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
public class Config {
	public static class Artifact {
		private String group;
		private String artifact;
		private String version;
		private String classifier;
		private Set<String> types;

		Artifact() {
			// For Jackson
		}
		
		/**
		 * @param group
		 * @param artifact
		 * @param version
		 */
		public Artifact(String group, String artifact, String version, String classifier, Set<String> types) {
			super();
			this.group = group;
			this.artifact = artifact;
			this.version = version;
			this.classifier = classifier;
			this.types = types != null ? Collections.unmodifiableSet(types) : null;
		}

		/**
		 * @return the group
		 */
		public String getGroup() {
			return group;
		}

		/**
		 * @param group the group to set
		 */
		public void setGroup(String group) {
			this.group = group;
		}

		/**
		 * @return the artifact
		 */
		public String getArtifact() {
			return artifact;
		}

		/**
		 * @param artifact the artifact to set
		 */
		public void setArtifact(String artifact) {
			this.artifact = artifact;
		}

		/**
		 * @return the version
		 */
		public String getVersion() {
			return version;
		}

		/**
		 * @param version the version to set
		 */
		public void setVersion(String version) {
			this.version = version;
		}

		/**
		 * @return the classifier
		 */
		public String getClassifier() {
			return classifier;
		}

		/**
		 * @param classifier the classifier to set
		 */
		public void setClassifier(String classifier) {
			this.classifier = classifier;
		}

		/**
		 * @return the types
		 */
		public Set<String> getTypes() {
			return types != null ? Collections.unmodifiableSet(this.types) : null;
		}

		/**
		 * @param types the types to set
		 */
		public void setTypes(Set<String> types) {
			this.types = types;
		}
		/**
		 * @param types the types to set
		 */
		public void addTypes(String...types) {
			if (this.types == null) {
				this.types = new HashSet<>();
			}
			for (String type : types) {
				this.types.add(type);
			}
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#hashCode()
		 */
		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((artifact == null) ? 0 : artifact.hashCode());
			result = prime * result + ((classifier == null) ? 0 : classifier.hashCode());
			result = prime * result + ((group == null) ? 0 : group.hashCode());
			result = prime * result + ((types == null) ? 0 : types.hashCode());
			result = prime * result + ((version == null) ? 0 : version.hashCode());
			return result;
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			Artifact other = (Artifact) obj;
			if (artifact == null) {
				if (other.artifact != null)
					return false;
			} else if (!artifact.equals(other.artifact))
				return false;
			if (classifier == null) {
				if (other.classifier != null)
					return false;
			} else if (!classifier.equals(other.classifier))
				return false;
			if (group == null) {
				if (other.group != null)
					return false;
			} else if (!group.equals(other.group))
				return false;
			if (types == null) {
				if (other.types != null)
					return false;
			} else if (!types.equals(other.types))
				return false;
			if (version == null) {
				if (other.version != null)
					return false;
			} else if (!version.equals(other.version))
				return false;
			return true;
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#toString()
		 */
		@Override
		public String toString() {
			return "Artifact [group=" + group + ", artifact=" + artifact + ", version=" + version + ", classifier="
					+ classifier + ", types=" + types + "]";
		}
	}
	
	public static Config getEffectiveConfig(Config cfg) {
		Config effectiveCfg = new Config();
		effectiveCfg.setProperties(Collections.unmodifiableMap(cfg.getProperties()));
		
		Set<Artifact> effectiveArtifacts = new HashSet<>();
		for (Artifact artifact : cfg.getArtifacts()) {
			Artifact effectiveArtifact = new Artifact();
			effectiveArtifact.setGroup(artifact.getGroup());
			effectiveArtifact.setArtifact(artifact.getArtifact());
			
			String effectiveVersion = null;
			if (isVariable(artifact.getVersion())) {
				String paramNameFromVariable = getParameterNameFromVariable(artifact.getVersion());
				if (paramNameFromVariable != null) {
					effectiveVersion = cfg.getProperties().get(paramNameFromVariable);
				} else {
					effectiveVersion = artifact.getVersion();
				}
			} else {
				effectiveVersion = artifact.getVersion();
			}
			effectiveArtifact.setVersion(effectiveVersion);

			effectiveArtifact.setClassifier(artifact.getClassifier());
			effectiveArtifact.setTypes(artifact.getTypes());
			
			effectiveArtifacts.add(effectiveArtifact);
		}
		if (effectiveArtifacts.size() > 0) {
			effectiveCfg.setArtifacts(Collections.unmodifiableSet(effectiveArtifacts));
		}
		
		return effectiveCfg;
	}
	private static boolean isVariable(String str) {
		return str != null && str.trim().startsWith("${") && str.trim().endsWith("}");
	}
	private static String getParameterNameFromVariable(String str) {
		if (! isVariable(str)) {
			return null;
		}
		
		return str.trim().substring(2, str.length() - 1).trim();
	}

	private Map<String, String> properties;
	private Set<Artifact> artifacts;

	/**
	 */
	public Config() {
		super();
		this.properties = null;
		this.artifacts = null;
	}
	/**
	 * @param properties
	 * @param artifacts
	 */
	public Config(Map<String, String> properties, Set<Artifact> artifacts) {
		super();
		this.properties = properties;
		this.artifacts = artifacts;
	}

	/**
	 * @return Config with variables resolved
	 */
	@JsonIgnore // Avoid infinite recursion on serialization
	@XmlTransient // Avoid infinite recursion on serialization
	public Config getEffectiveConfig() {
		return getEffectiveConfig(this);
	}

	/**
	 * @return the properties
	 */
	public Map<String, String> getProperties() {
		return properties;
	}
	/**
	 * @param properties the properties to set
	 */
	public void setProperties(Map<String, String> properties) {
		this.properties = properties;
	}
	/**
	 * @param key
	 * @param value
	 */
	public void putProperty(String key, String value) {
		if (properties == null) {
			properties = new HashMap<>();
		}
		properties.put(key,  value);
	}

	/**
	 * @return the artifacts
	 */
	public Set<Artifact> getArtifacts() {
		return artifacts;
	}
	/**
	 * @param artifacts the artifacts to set
	 */
	public void setArtifacts(Set<Artifact> artifacts) {
		this.artifacts = artifacts;
	}
	public void addArtifact(Artifact artifact) {
		if (artifacts == null) {
			artifacts = new HashSet<>();
		}
		artifacts.add(artifact);
	}
	public void addArtifacts(Artifact...artifacts) {
		if (this.artifacts == null) {
			this.artifacts = new HashSet<>();
		}
		for (Artifact artifact : artifacts) {
			this.artifacts.add(artifact);
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((artifacts == null) ? 0 : artifacts.hashCode());
		result = prime * result + ((properties == null) ? 0 : properties.hashCode());
		return result;
	}
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Config other = (Config) obj;
		if (artifacts == null) {
			if (other.artifacts != null)
				return false;
		} else if (!artifacts.equals(other.artifacts))
			return false;
		if (properties == null) {
			if (other.properties != null)
				return false;
		} else if (!properties.equals(other.properties))
			return false;
		return true;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "Config [properties=" + properties + ", artifacts=" + artifacts + "]";
	}

	public static void main(String[] argv) throws Exception {
	}
}