/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.rf2.convert.mojo;

import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.And;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.ConceptAssertion;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.NecessarySet;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.SomeRole;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.SufficientSet;
import java.io.File;
import java.io.FileFilter;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Optional;
import java.util.TreeMap;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.EConceptUtility;
import gov.va.oia.terminology.converters.sharedUtils.EConceptUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.sql.ColumnDefinition;
import gov.va.oia.terminology.converters.sharedUtils.sql.DataType;
import gov.va.oia.terminology.converters.sharedUtils.sql.H2DatabaseHandle;
import gov.va.oia.terminology.converters.sharedUtils.sql.TableDefinition;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.LanguageCode;
import gov.vha.isaac.ochre.api.MavenConceptProxy;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.bootstrap.TermAux;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptSpecification;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.DynamicSememe;
import gov.vha.isaac.ochre.api.logic.LogicalExpression;
import gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder;
import gov.vha.isaac.ochre.api.logic.assertions.Assertion;
import gov.vha.isaac.ochre.api.util.UuidT3Generator;
import gov.vha.isaac.ochre.impl.utility.Frills;
import gov.vha.isaac.ochre.impl.utility.LanguageMap;

/**
 * Loader code to convert RxNorm into the workbench.
 */
@Mojo(name = "convert-RF2-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class RF2Mojo extends ConverterBaseMojo
{
	private EConceptUtility importUtil_;
	private H2DatabaseHandle db_;
	private boolean outputJson = false;  //Set to true to produce a json dump file
	
	String contentNameVersion_;
	String timeString_;
	
	private String CONCEPT, IDENTIFIER, RELATIONSHIP, STATED_RELATIONSHIP;
	private ArrayList<String> DESCRIPTIONS = new ArrayList<>();
	private ArrayList<String> LANGUAGES = new ArrayList<>();
	
	protected static SimpleDateFormat dateParse = new SimpleDateFormat("yyyyMMdd");
	
	private HashMap<String, TableDefinition> tables_ = new HashMap<>();
	
	//map concept UUID to a treemap that has the time and the status value.
	//Could shrink the size of this if necessary by using con sequence ids....
	private HashMap<UUID, TreeMap<Long, UUID>> conceptDefinitionStatusCache = new HashMap<>();
	
	//This cache is to work around a data problem where stated rels are missing from SCT.
	private HashSet<UUID> consWithNoStatedRel = new HashSet<>();
	
	private HashSet<UUID> neverRoleGroupSet = new HashSet<>();
	{
		neverRoleGroupSet.addAll(TermAux.PART_OF.getUuidList());
		neverRoleGroupSet.addAll(TermAux.LATERALITY.getUuidList());
		neverRoleGroupSet.addAll(TermAux.HAS_ACTIVE_INGREDIENT.getUuidList());
		neverRoleGroupSet.addAll(TermAux.HAS_DOSE_FORM.getUuidList());
	}

	private InputType inputType = null;
	
	/**
	 * Default value from SNOMED_CT_CORE_MODULE
	 */
	@Parameter(required = false) 
	private ConceptSpecification moduleUUID = MetaData.SNOMED_CT_CORE_MODULE;
	public void setModuleUUID(MavenConceptProxy conceptProxy)
	{
		moduleUUID = conceptProxy;
	}

	@Override
	public void execute() throws MojoExecutionException
	{
		try
		{
			super.execute();
			
			inputType = InputType.parse(converterOutputArtifactClassifier);

			File zipFile = init();
			loadDatabase(zipFile);

			ComponentReference rf2Metadata = ComponentReference.fromConcept(importUtil_.createConcept("RF2 Metadata " + contentNameVersion_, true));
			importUtil_.addParent(rf2Metadata, MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid());
			
			
			importUtil_.loadTerminologyMetadataAttributes(rf2Metadata, converterSourceArtifactVersion, Optional.ofNullable(timeString_), 
					converterOutputArtifactVersion, Optional.of(converterOutputArtifactClassifier), converterVersion);
			
			//process content
			
			transformConcepts();
			transformDescriptions();
			//stated first, so we can know what doesn't get stated graphs
			transformRelationships(true);
			ConsoleUtil.println("Noticed " + consWithNoStatedRel.size() + " concepts with no stated relationships");
			transformRelationships(false);
			//can clear this cache now
			ConsoleUtil.println("After copying inferred rels, still " + consWithNoStatedRel.size() + " concepts with no stated relationships");
			consWithNoStatedRel.clear();
			
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, converterOutputArtifactClassifier + "-RF2UUID");

			ConsoleUtil.println("Load Statistics");
			for (String s : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(s);
			}
			
			//shutdown
			importUtil_.shutdown();
			db_.shutdown();
			
			ConsoleUtil.println("Finished converting " + contentNameVersion_ + "-" + converterOutputArtifactClassifier);
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, converterOutputArtifactClassifier + "-ConsoleOutput.txt").toPath());
		}
		catch (Exception e)
		{
			throw new MojoExecutionException("Failure during conversion", e);
		}
	}
	
	/**
	 * This will return batches of relationships, each item the iterator returns will be all of the relationships 
	 * for a particular source concepts, while each RelBatch within the list will be all versions of a particular relationship.
	 * @param td 
	 */
	private Iterator<ArrayList<RelBatch>> getRelationships(String table, TableDefinition td) throws SQLException
	{
		PreparedStatement ps = db_.getConnection().prepareStatement("Select * from " + table + " order by sourceid, id");
		ResultSet rs = ps.executeQuery();
		
		Iterator<ArrayList<RelBatch>> iter = new Iterator<ArrayList<RelBatch>>()
		{
			RelBatch relBatchWorking = null;
			ArrayList<RelBatch> conceptRelsWorking = new ArrayList<>();
			ArrayList<RelBatch> conceptRelsNextReady = null;
			
			@Override
			public boolean hasNext()
			{
				if (conceptRelsNextReady == null)
				{
					read();
				}
				if (conceptRelsNextReady == null)
				{
					try
					{
						rs.close();
					}
					catch (SQLException e)
					{
						// noop
					}
				}
				return conceptRelsNextReady != null;
			}

			@Override
			public ArrayList<RelBatch> next()
			{
				if (conceptRelsNextReady == null)
				{
					read();
				}
				ArrayList<RelBatch> temp = conceptRelsNextReady;
				conceptRelsNextReady = null;
				return temp;
			}
			
			private void read()
			{
				
				try
				{
					while ((conceptRelsNextReady == null) && rs.next())
					{
						Rel r = new Rel(rs, td);
						if (relBatchWorking == null)
						{
							relBatchWorking = new RelBatch(r);
						}
						else if (relBatchWorking.getBatchId().equals(r.id))
						{
							relBatchWorking.addRel(r);
						}
						else  //different batchId than previous - need a new RelBatch.  Move last relBatch into conceptRels.
						{
							if (conceptRelsWorking.size() > 0 && !conceptRelsWorking.get(0).getSourceId().equals(relBatchWorking.getSourceId()))
							{
								conceptRelsNextReady = conceptRelsWorking;
								conceptRelsWorking = new ArrayList<>();
							}
							conceptRelsWorking.add(relBatchWorking);
							
							//Put this rel into a new batch.
							relBatchWorking = new RelBatch(r);
						}
					}
				}
				catch (SQLException | ParseException e)
				{
					throw new RuntimeException(e);
				}
				
				if (conceptRelsNextReady != null)
				{
					return;
				}
				
				if (conceptRelsWorking.size() > 0 && !conceptRelsWorking.get(0).getSourceId().equals(relBatchWorking.getSourceId()))
				{
					conceptRelsNextReady = conceptRelsWorking;
					conceptRelsWorking = new ArrayList<>();
					return;
				}
				
				//If we get here, the only thing left is the last relBatch.
				if (relBatchWorking != null)
				{
					conceptRelsWorking.add(relBatchWorking);
					conceptRelsNextReady = conceptRelsWorking;
					relBatchWorking = null;
					conceptRelsWorking = new ArrayList<>();
				}
			}
		};
		
		return iter;
	}

	private void transformRelationships(boolean stated) throws SQLException
	{
		ConsoleUtil.println("Converting " + (stated ? "stated" : "inferred") + " relationships into graphs");
		String table = (stated ? STATED_RELATIONSHIP : RELATIONSHIP);
		
		TableDefinition td = tables_.get(table);
		
		int graphCount = 0;
		
		Iterator<ArrayList<RelBatch>> rels = getRelationships(table, td);
		UUID lastId = null;
		
		while (rels.hasNext())
		{
			//each Rel here will be for the same sourceId.
			graphCount++;
			
			ArrayList<RelBatch> conRels = rels.next();
			long newestRelTime = 0;
			
			LogicalExpressionBuilder leb = Get.logicalExpressionBuilderService().getLogicalExpressionBuilder();
			ArrayList<Assertion> assertions = new ArrayList<>();
			HashMap<String, ArrayList<Assertion>> groupedAssertions = new HashMap<>();
			
			//Each member of a RelBatch contains the same rel ID (so different versions of the same rel)
			for (RelBatch rb : conRels)
			{
				if (!rb.isActiveNow())
				{
					//TODO handle historical relationships
				}
				else
				{
					//TODO handle history - only loading latest for now.
					Rel r = rb.getRels().last();
					
					if ((stated && r.characteristicTypeId.equals(MetaData.INFERRED.getPrimordialUuid())) 
							|| (!stated && r.characteristicTypeId.equals(MetaData.STATED.getPrimordialUuid())))
					{
						throw new RuntimeException("Unexpected - table type and characteristic type do not match!");
					}
					
					if (r.characteristicTypeId.equals(MetaData.INFERRED.getPrimordialUuid()) 
							|| r.characteristicTypeId.equals(MetaData.STATED.getPrimordialUuid()))
					{
						if (r.effectiveTime > newestRelTime)
						{
							newestRelTime = r.effectiveTime;
						}
						if (r.relGroup.trim().equals("0"))
						{
							//Don't just check primordial, IS_A has multiple UUIDs
							if (Arrays.stream(MetaData.IS_A.getUuids()).anyMatch(uuid -> uuid.equals(r.typeId)))
							{
								assertions.add(ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(r.destinationId), leb));
							}
							else
							{
								//TODO [graph] ask Keith about the never group stuff.
								//TODO [graph] handle modifier?
								//TODO [graph] handle sctid
								//TODO [graph] handle id
								//TODO [graph] maintain actual group numbers?
								if (neverRoleGroupSet.contains(r.typeId))
								{
									assertions.add(SomeRole(Get.identifierService().getConceptSequenceForUuids(r.typeId),
											ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(r.destinationId), leb)));
								}
								else
								{
									assertions.add(SomeRole(MetaData.ROLE_GROUP.getConceptSequence(),
											And(SomeRole(Get.identifierService().getConceptSequenceForUuids(r.typeId),
													ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(r.destinationId), leb)))));
								}
							}
						}
						else
						{
							ArrayList<Assertion> groupAssertions = groupedAssertions.get(r.relGroup.trim());
							if (groupAssertions == null)
							{
								groupAssertions = new ArrayList<>();
								groupedAssertions.put(r.relGroup.trim(), groupAssertions);
							}
							groupAssertions.add(SomeRole(Get.identifierService().getConceptSequenceForUuids(r.typeId),
									ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(r.destinationId), leb)));
						}
					}
					else
					{
						//kick it over into an association bucket
						//TODO should I toss these when processing inferred?
						SememeChronology<DynamicSememe<?>> assn = importUtil_.addAssociation(ComponentReference.fromConcept(r.sourceId), 
								r.id, r.destinationId, r.typeId, r.isActive ? State.ACTIVE : State.INACTIVE, r.effectiveTime, r.moduleId);
						//TODO put on modifier, group
						
						if (r.sctID != null && !r.id.equals(lastId))
						{
							importUtil_.addStaticStringAnnotation(ComponentReference.fromChronology(assn, () -> "Association"), 
									r.sctID + "", MetaData.SNOMED_INTEGER_ID.getPrimordialUuid(), State.ACTIVE);
						}
						
					}
					lastId = r.id;
				}
			}
			
			//handle relationship groups
			for (ArrayList<Assertion> groupAssertions : groupedAssertions.values())
			{
				assertions.add(SomeRole(MetaData.ROLE_GROUP.getConceptSequence(),
						And(groupAssertions.toArray(new Assertion[groupAssertions.size()]))));
			}
			
			if (assertions.size() > 0)
			{
				Boolean defined = null;
				TreeMap<Long, UUID> conDefStatus = conceptDefinitionStatusCache.get(conRels.get(0).getSourceId());
				if (conDefStatus == null)
				{
					//Try Frills - in the case of US Extension, we should have SCT loaded, pull from that.
					//Can definition status vary between stated and inferred?  Just read stated for now.
					Optional<Boolean> sctDefined = Frills.isConceptFullyDefined(ComponentReference.fromConcept(conRels.get(0).getSourceId()).getNid(), true);
					if (sctDefined.isPresent())
					{
						defined = sctDefined.get();
					}
					else
					{
						ConsoleUtil.printErrorln("No definition status found!");
					}
				}
				else
				{
					if (conDefStatus.lastEntry().getValue().equals(TermAux.SUFFICIENTLY_DEFINED.getPrimordialUuid()))
					{
						defined = true;
					}
					else if (conDefStatus.lastEntry().getValue().equals(TermAux.NECESSARY_BUT_NOT_SUFFICIENT.getPrimordialUuid()))
					{
						defined = false;
					}
					else
					{
						throw new RuntimeException("Unexpected concept definition status: " + conDefStatus.lastEntry());
					}
				}
				if (defined != null)
				{
					if (defined.booleanValue())
					{
						SufficientSet(And(assertions.toArray(new Assertion[assertions.size()])));
					}
					else
					{
						NecessarySet(And(assertions.toArray(new Assertion[assertions.size()])));
					}
					
					LogicalExpression le = leb.build();
					
					if (le.isMeaningful())
					{
						if (newestRelTime == 0)
						{
							throw new RuntimeException("Time sort failure!");
						}
						//TODO [graph] what if the modules are different across the graph rels?
						importUtil_.addRelationshipGraph(ComponentReference.fromConcept(conRels.get(0).getSourceId()), null, le, stated, newestRelTime,
								conRels.get(0).getRels().first().moduleId);
						
						if (!stated && consWithNoStatedRel.contains(conRels.get(0).getSourceId()))
						{
							// substitute inferred expression, as early SNOMED stated expressions where lost.
							importUtil_.addRelationshipGraph(ComponentReference.fromConcept(conRels.get(0).getSourceId()), null, le, true, newestRelTime,
									conRels.get(0).getRels().first().moduleId);
						}
						consWithNoStatedRel.remove(conRels.get(0).getSourceId());
					}
					else
					{
						ConsoleUtil.printErrorln("expression not meaningful?");
					}
				}
			}
			else
			{
				//TODO [graph] ask Keith about these cases where no associations get generated.
			}
			
			if (graphCount % 1000 == 0)
			{
				ConsoleUtil.showProgress();
			}
			if (graphCount % 25000 == 0)
			{
				ConsoleUtil.println("Processed " + graphCount + " relationship graphs...");
			}
			
		}
		ConsoleUtil.println("Created " + graphCount  + " graphs");
		
	}

	private void transformDescriptions() throws SQLException, ParseException, MojoExecutionException
	{
		ConsoleUtil.println("Converting descriptions");
		for (String DESCRIPTION : DESCRIPTIONS)
		{
			TableDefinition descriptionTable = tables_.get(DESCRIPTION);
			
			String lang = DESCRIPTION.split("_")[3];
			String LANGUAGE = null;
			for (String s : LANGUAGES)
			{
				if (s.split("_")[3].equals(lang))
				{
					LANGUAGE = s;
					break;
				}
			}
			if (LANGUAGE == null)
			{
				throw new MojoExecutionException("Failed to find the language table for the language: " + lang);
			}
			TableDefinition acceptabilityTable = tables_.get(LANGUAGE);
			
			ConsoleUtil.println("Processing " + descriptionTable.getTableName() + ", " + acceptabilityTable.getTableName());
			
			int descCount = 0;
			int accCount = 0;
			PreparedStatement ps = db_.getConnection().prepareStatement("Select * from " + DESCRIPTION + " order by conceptId, id");
			PreparedStatement ps2 = db_.getConnection().prepareStatement("Select * from " + LANGUAGE + " where referencedComponentId = ? ");
			UUID lastId = null;
			ResultSet descRS = ps.executeQuery();
			while (descRS.next())
			{
				descCount++;
				Long sctID = null;
				UUID id;
				if (descriptionTable.getColDataType("ID").isLong())
				{
					sctID = descRS.getLong("ID");
					id = UuidT3Generator.fromSNOMED(sctID);
				}
				else
				{
					id = UUID.fromString(descRS.getString("ID"));
				}
				long time  = dateParse.parse(descRS.getString("EFFECTIVETIME")).getTime();
				boolean active = descRS.getBoolean("ACTIVE");
				UUID moduleId = (descriptionTable.getColDataType("MODULEID").isLong() ? UuidT3Generator.fromSNOMED(descRS.getLong("MODULEID")) : 
					UUID.fromString(descRS.getString("MODULEID")));
				UUID conceptId = (descriptionTable.getColDataType("CONCEPTID").isLong() ? UuidT3Generator.fromSNOMED(descRS.getLong("CONCEPTID")) : 
					UUID.fromString(descRS.getString("CONCEPTID")));
				String languageCode = descRS.getString("LANGUAGECODE");
				UUID typeId = (descriptionTable.getColDataType("TYPEID").isLong() ? UuidT3Generator.fromSNOMED(descRS.getLong("TYPEID")) : 
					UUID.fromString(descRS.getString("TYPEID")));
				String term = descRS.getString("TERM");
				UUID caseSigId = (descriptionTable.getColDataType("CASESIGNIFICANCEID").isLong() ? UuidT3Generator.fromSNOMED(descRS.getLong("CASESIGNIFICANCEID")) : 
					UUID.fromString(descRS.getString("CASESIGNIFICANCEID")));
				
				SememeChronology<DescriptionSememe<?>> desc = importUtil_.addDescription(ComponentReference.fromConcept(conceptId), id, term, 
						DescriptionType.parse(typeId), null, null, caseSigId, 
						LanguageMap.getConceptForLanguageCode(LanguageCode.getLangCode(languageCode)).getPrimordialUuid(), 
						moduleId, null, active ? State.ACTIVE : State.INACTIVE, time);

				//add SCTID if this is the first sighting
				if (sctID != null && !id.equals(lastId))
				{
					lastId = id;
					importUtil_.addStaticStringAnnotation(ComponentReference.fromChronology(desc), sctID + "", MetaData.SNOMED_INTEGER_ID.getPrimordialUuid(), 
							State.ACTIVE);
				}
				
				ps2.clearParameters();
				if (acceptabilityTable.getColDataType("referencedComponentId").isLong())
				{
					if (sctID == null)
					{
						throw new RuntimeException("type mismatch!");
					}
					ps2.setLong(1, sctID);
				}
				else
				{
					ps2.setString(1, id.toString());
				}
				ResultSet langRS = ps2.executeQuery();
				boolean foundAcceptability = false;
				while (langRS.next())
				{
					accCount++;
					foundAcceptability = true;
					
					UUID acceptID = UUID.fromString(langRS.getString("id"));
					long acceptTime  = dateParse.parse(langRS.getString("EFFECTIVETIME")).getTime();
					boolean acceptActive = langRS.getBoolean("ACTIVE");
					UUID acceptModuleId = (acceptabilityTable.getColDataType("MODULEID").isLong() ? UuidT3Generator.fromSNOMED(langRS.getLong("MODULEID")) : 
						UUID.fromString(langRS.getString("MODULEID")));
					UUID refsetId = (acceptabilityTable.getColDataType("refsetID").isLong() ? UuidT3Generator.fromSNOMED(langRS.getLong("refsetID")) : 
						UUID.fromString(langRS.getString("refsetID")));
					UUID acceptabilityId = (acceptabilityTable.getColDataType("acceptabilityId").isLong() ? UuidT3Generator.fromSNOMED(langRS.getLong("acceptabilityId")) : 
						UUID.fromString(langRS.getString("acceptabilityId")));
					
					boolean preferred;
					if (MetaData.ACCEPTABLE.getPrimordialUuid().equals(acceptabilityId))
					{
						preferred = false;
					}
					else if (MetaData.PREFERRED.getPrimordialUuid().equals(acceptabilityId))
					{
						preferred = true;
					}
					else
					{
						throw new RuntimeException("Unexpected acceptibility: " + acceptabilityId);
					}
					
					importUtil_.addDescriptionAcceptibility(ComponentReference.fromChronology(desc), acceptID, refsetId, preferred, 
							acceptActive ? State.ACTIVE : State.INACTIVE, acceptTime, acceptModuleId);
					
				}
				if (!foundAcceptability)
				{
					ConsoleUtil.printErrorln("No acceptibility found for: " + id + " " + sctID);
				}
				

				if (descCount % 1000 == 0)
				{
					ConsoleUtil.showProgress();
				}
				if (descCount % 25000 == 0)
				{
					ConsoleUtil.println("Processed " + descCount + " descriptions with " + accCount + " acceptabilities...");
				}
				
			}
			ConsoleUtil.println("Converted " + descCount  + " descriptions");
		}
		
	}

	private void transformConcepts() throws SQLException, ParseException
	{
		ConsoleUtil.println("Converting concepts");
		TableDefinition td = tables_.get(CONCEPT);
		
		int conCount = 0;
		PreparedStatement ps = db_.getConnection().prepareStatement("Select * from " + CONCEPT + " order by id");
		UUID lastId = null;
		ResultSet rs = ps.executeQuery();
		while (rs.next())
		{
			conCount++;
			Long sctID = null;
			UUID id, moduleId, definitionStatusId;
			if (td.getColDataType("ID").isLong())
			{
				sctID = rs.getLong("ID");
				id = UuidT3Generator.fromSNOMED(sctID);
			}
			else
			{
				id = UUID.fromString(rs.getString("ID"));
			}
			
			consWithNoStatedRel.add(id);
			
			long time  = dateParse.parse(rs.getString("EFFECTIVETIME")).getTime();
			boolean active = rs.getBoolean("ACTIVE");
			moduleId = (td.getColDataType("MODULEID").isLong() ? UuidT3Generator.fromSNOMED(rs.getLong("MODULEID")) : 
				UUID.fromString(rs.getString("MODULEID")));
			definitionStatusId = (td.getColDataType("DEFINITIONSTATUSID").isLong() ? UuidT3Generator.fromSNOMED(rs.getLong("DEFINITIONSTATUSID")) : 
				UUID.fromString(rs.getString("DEFINITIONSTATUSID")));

			TreeMap<Long, UUID> conDefStatus = conceptDefinitionStatusCache.get(id);
			if (conDefStatus == null)
			{
				conDefStatus = new TreeMap<>();
				conceptDefinitionStatusCache.put(id, conDefStatus);
			}
			UUID oldValue = conDefStatus.put(time, definitionStatusId);
			if (oldValue != null && !oldValue.equals(definitionStatusId))
			{
				throw new RuntimeException("Unexpeted - multiple definition status values at the same time: " + sctID + " " + id + " " + definitionStatusId);
			}
			
			ConceptChronology<? extends ConceptVersion<?>> con = importUtil_.createConcept(id, time, active ? State.ACTIVE : State.INACTIVE, moduleId);
			if (sctID != null && !id.equals(lastId))
			{
				lastId = id;
				importUtil_.addStaticStringAnnotation(ComponentReference.fromConcept(con), sctID + "", MetaData.SNOMED_INTEGER_ID.getPrimordialUuid(), State.ACTIVE);
			}
			if (conCount % 1000 == 0)
			{
				ConsoleUtil.showProgress();
			}
			if (conCount % 25000 == 0)
			{
				ConsoleUtil.println("Processed " + conCount + " concepts...");
			}
			
		}
		ConsoleUtil.println("Converted " + conCount  + " concepts");
	}

	private void loadDatabase(File zipFile) throws Exception
	{
		long time = System.currentTimeMillis();
		db_ = new H2DatabaseHandle();
		
		File dbFile = new File(outputDirectory, contentNameVersion_ + "-" + converterOutputArtifactClassifier + ".h2.db");
		boolean createdNew = db_.createOrOpenDatabase(new File(outputDirectory, contentNameVersion_ + "-" + converterOutputArtifactClassifier));

		if (!createdNew)
		{
			ConsoleUtil.println("Using existing database.  To load from scratch, delete the file '" + dbFile.getCanonicalPath() + ".*'");
		}
		ZipFile zf = new ZipFile(zipFile);
		
		Enumeration<? extends ZipEntry> zipEntries = zf.entries();
		int tableCount = 0;
		while(zipEntries.hasMoreElements())
		{
			ZipEntry ze = zipEntries.nextElement();
			String[] structure = ze.getName().split("\\/");
			if ((structure[0].toUpperCase().equals(inputType.name()) || (structure.length > 1 && structure[1].toUpperCase().equals(inputType.name()))
					|| (structure.length > 2 && structure[2].toUpperCase().equals(inputType.name())))
					&& ze.getName().toLowerCase().endsWith(".txt"))
			{
				//One of the data files we want to load
				ConsoleUtil.println("Loading " + ze.getName());
				
				RF2FileReader fileReader = new RF2FileReader(zf.getInputStream(ze));
				
				String tableName = structure[structure.length - 1];
				tableName = tableName.substring(0,  tableName.length() - 4);
				tableName = tableName.replaceAll("-", "_");  //hyphens cause sql issues
				
				if (tableName.toLowerCase().startsWith("sct2_concept_"))
				{
					CONCEPT = tableName;
				}
				else if (tableName.toLowerCase().startsWith("sct2_description_") || tableName.toLowerCase().startsWith("sct2_textdefinition_"))
				{
					DESCRIPTIONS.add(tableName);
				}
				else if (tableName.toLowerCase().startsWith("der2_crefset_") && tableName.toLowerCase().contains("language"))
				{
					LANGUAGES.add(tableName);
				}
				else if (tableName.toLowerCase().startsWith("sct2_identifier_"))
				{
					IDENTIFIER = tableName;
				}
				else if (tableName.toLowerCase().startsWith("sct2_relationship_"))
				{
					RELATIONSHIP = tableName;
				}
				else if (tableName.toLowerCase().startsWith("sct2_statedrelationship_"))
				{
					STATED_RELATIONSHIP = tableName;
				}
				
				TableDefinition td = createTableDefinition(tableName, fileReader.getHeader(), fileReader.peekNextRow());
				tables_.put(tableName, td);
				
				if (!createdNew)
				{
					//Only need to process this far to read the metadata about the DB
					continue;
				}
				
				db_.createTable(td);
				tableCount++;
				
				int rowCount = db_.loadDataIntoTable(td, fileReader);
				fileReader.close();
				
				//don't bother indexing small tables
				if (rowCount > 10000)
				{
					HashSet<String> colsToIndex = new HashSet<String>();
					colsToIndex.add("conceptId");
					colsToIndex.add("referencedComponentId");
					colsToIndex.add("sourceId");
					
					for (String s : fileReader.getHeader())
					{
						if (colsToIndex.contains(s))
						{
							Statement statement = db_.getConnection().createStatement();
							ConsoleUtil.println("Indexing " + tableName + " on " + s);
							if (s.equals("referencedComponentId"))
							{
								statement.execute("CREATE INDEX " + tableName + "_" + s + "_index ON " + tableName  + " (" + s + ", refsetId)");
							}
							else
							{
								if (td.getColDataType("id") != null)
								{
									statement.execute("CREATE INDEX " + tableName + "_" + s + "_index ON " + tableName  + " (" + s + ", id)");
								}
								else
								{
									statement.execute("CREATE INDEX " + tableName + "_" + s + "_index ON " + tableName  + " (" + s + ")");
								}
							}
							statement.close();
						}
					}
				}
				
			}
		}
		zf.close();
		
		ConsoleUtil.println("Processing DB loaded " + tableCount + " tables in " + ((System.currentTimeMillis() - time) / 1000) + " seconds");
		if (tableCount == 0)
		{
			throw new RuntimeException("Failed to find tables in zip file!");
		}
	}

	private TableDefinition createTableDefinition(String tableName, String[] header, String[] sampleDataRow)
	{
		TableDefinition td = new TableDefinition(tableName);
		
		for(int i = 0; i < header.length; i++)
		{
			DataType dataType;
			if (header[i].equals("id") || header[i].endsWith("Id"))
			{
				//See if this looks like a UUID or a long
				try
				{
					Long.parseLong(sampleDataRow[i]);
					dataType = new DataType(DataType.SUPPORTED_DATA_TYPE.LONG, null, false);
				}
				catch (Exception e)  //Might be a null pointer if there is no data, just treat it as a string (doesn't matter)
				{
					//UUID
					dataType = new DataType(DataType.SUPPORTED_DATA_TYPE.STRING, 36, false);
				}
			}
			else if (header[i].equals("active"))
			{
				dataType = new DataType(DataType.SUPPORTED_DATA_TYPE.BOOLEAN, null, false);
			}
			else if (header[i].equals("effectiveTime") || header[i].equals("sourceEffectiveTime") || header[i].equals("targetEffectiveTime"))
			{
				dataType = new DataType(DataType.SUPPORTED_DATA_TYPE.STRING, 8, false);
			}
			else
			{
				dataType = new DataType(DataType.SUPPORTED_DATA_TYPE.STRING, null, true);
				ConsoleUtil.println("Treating " + header[i] + " as a string");
			}
			
			ColumnDefinition cd = new ColumnDefinition(header[i], dataType);
			td.addColumn(cd);
		}
		
		return td;
	}

	private File init() throws Exception
	{
		File zipFile = null;
		for (File f: inputFileLocation.listFiles())
		{
			if (f.getName().toLowerCase().endsWith(".zip")) 
			{
				if (zipFile != null)
				{
					throw new MojoExecutionException("Only expected to find one zip file in the folder " + inputFileLocation.getCanonicalPath());
				}
				zipFile = f;
			}
		}
		
		if (zipFile == null)
		{
			throw new MojoExecutionException("Did not find a zip file in " + inputFileLocation.getCanonicalPath());
		}
		
		contentNameVersion_ = zipFile.getName().substring(0, zipFile.getName().length() - 4);
		ConsoleUtil.println("Converting " + contentNameVersion_ + "-" + converterOutputArtifactClassifier);
		
		String[] temp = contentNameVersion_.split("_");
		
		SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
		long defaultTime = 0;
		for (int i = temp.length - 1; i > 0; i--)
		{
			if (temp[i].length() == 8 && NumberUtils.isDigits(temp[i]))
			{
				defaultTime = sdf.parse(temp[i]).getTime();
				timeString_ = temp[i];
				break;
			}
		}
		
		if (defaultTime == 0)
		{
			throw new MojoExecutionException("Couldn't parse date out of " + contentNameVersion_);
		}
		
		clearTargetFiles(contentNameVersion_);
		
		File[] ibdfFiles = new File[0];
		File ibdfFolder = new File(inputFileLocation, "ibdf");
		if (ibdfFolder.isDirectory())
		{
			ibdfFiles = ibdfFolder.listFiles(new FileFilter()
			{
				@Override
				public boolean accept(File pathname)
				{
					if (pathname.isFile() && pathname.getName().toLowerCase().endsWith(".ibdf"))
					{
						return true;
					}
					return false;
				}
			});
		}
		
		importUtil_ = new EConceptUtility(Optional.empty(), Optional.of(moduleUUID), outputDirectory, converterOutputArtifactId,
				converterOutputArtifactClassifier, outputJson, defaultTime, 
				Arrays.asList(new SememeType[] {SememeType.DESCRIPTION, SememeType.COMPONENT_NID, SememeType.DYNAMIC, SememeType.LONG}), true, ibdfFiles);
		
		return zipFile;
	}

	private void clearTargetFiles(String contentNameVersion)
	{
		new File(outputDirectory, converterOutputArtifactClassifier + "-RF2UUIDDebugMap.txt").delete();
		new File(outputDirectory, converterOutputArtifactClassifier + "-ConsoleOutput.txt").delete();
		new File(outputDirectory, "RF2-" + contentNameVersion + "-" + converterOutputArtifactClassifier + ".ibdf").delete();
		//For debug only, normally commented out
		//new File(outputDirectory, contentNameVersion + ".h2.db").delete();
	}
	public static void main(String[] args) throws MojoExecutionException
	{
		RF2Mojo mojo = new RF2Mojo();
		mojo.outputDirectory = new File("../rf2-ibdf/sct/target");
		mojo.inputFileLocation = new File("../rf2-ibdf/sct/target/generated-resources/src/");
		mojo.converterVersion = "foo";
		mojo.converterOutputArtifactVersion = "bar";
		mojo.converterOutputArtifactClassifier = "Full";
		mojo.converterSourceArtifactVersion = "bar";
		mojo.execute();
	}
}
