package gov.va.med.term.vhat.data;

import java.io.File;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.maven.plugin.MojoExecutionException;

import gov.va.med.term.vhat.data.dto.CodeSystem;
import gov.va.med.term.vhat.data.dto.ConceptImportDTO;
import gov.va.med.term.vhat.data.dto.DesignationExtendedImportDTO;
import gov.va.med.term.vhat.data.dto.MapEntryImportDTO;
import gov.va.med.term.vhat.data.dto.MapSetImportDTO;
import gov.va.med.term.vhat.data.dto.PropertyImportDTO;
import gov.va.med.term.vhat.data.dto.RelationshipImportDTO;
import gov.va.med.term.vhat.data.dto.SubsetImportDTO;
import gov.va.med.term.vhat.data.dto.TypeImportDTO;
import gov.va.med.term.vhat.data.dto.Version;
import gov.va.oia.terminology.converters.sharedUtils.sql.H2DatabaseHandle;

public class SqlDataReader {
	
	private H2DatabaseHandle _h2;
	
	private List<PropertyImportDTO> 	_properties;
	private Map<Long, ArrayList<PropertyImportDTO>>	_propertiesMap;
	private List<RelationshipImportDTO> _relationships;
	private Map<Long, ArrayList<RelationshipImportDTO>> _relationshipsMap;
	private List<ConceptImportDTO> 		_concepts;
	private Map<Long, ArrayList<ConceptImportDTO>>	_conceptsMap;
	private List<TypeImportDTO> 		_types;
	private List<CodeSystem> 			_codesystems;
	private List<MapSetImportDTO>		_mapsets = new ArrayList<>();
	private Map<Long, ArrayList<MapSetImportDTO>>	_mapsetsMap;
	private List<SubsetImportDTO>		_subsets = new ArrayList<>();
	private Map<Long, ArrayList<SubsetImportDTO>>	_subsetsMap;
	private List<DesignationExtendedImportDTO>	_designations;
	private Map<Long, ArrayList<DesignationExtendedImportDTO>>	_designationsMap;
	private List<MapEntryImportDTO>		_mapentries;
	private Map<Long, ArrayList<MapEntryImportDTO>>	_mapEntriesMap;
	private List<Version>				_versions;
	
	// This is a hack to keep from creating duplicate UUIDs
	HashSet<Long> _currentVuids = new HashSet<>();
	
	public static void main(String[] args) throws Exception { 
	
		/**
		 * See hack for possible duplicate UUIDs (_currentVuids), which was just for temporary use
		 * 
		 * Need to clean up all the temporary/old statements (i.e. most List<>s where replaced with Map<>s)
		 * 
		 * Look at fetchDesignations: Only getting VHAT versions or we get null VUIDs
		 * Look at fetchCodeSystems: Versions are ignored
		 * Look at fetchMapsets: Only looking at 'active' to get around 'stated graphs' issue with duplicate names
		 * fetchProperties: only querying VHAT
		 */
		new SqlDataReader();
		
	}
	
	public SqlDataReader() throws MojoExecutionException {
		new SqlDataReader(new File("c:\\va\\tmp\\vts-import"));
		javafx.application.Platform.exit();
	}
	
	public SqlDataReader(File h2DB) throws MojoExecutionException {
		
		try {
			
			_h2 = new H2DatabaseHandle(); 
			_h2.createOrOpenDatabase(h2DB);
			
		} catch (Exception e) {
			throw new MojoExecutionException("Exception", e);
		}
	
	}
	
	public void shutdown() throws MojoExecutionException
	{
		try {
			_h2.shutdown();
		} catch (Exception e) {
			throw new MojoExecutionException("Exception", e);
		}
	}
	
	public void process() throws MojoExecutionException
	{
		try {
			this.fetchSubsets();
			this.fetchMapsets();
			this.fetchMapEntries();
			this.fetchConcepts();
			this.fetchRelationships();
			this.fetchTypes();
			this.fetchCodeSystems();
			this.fetchProperties();
			this.fetchVersions();
			// This takes a long time until fixing the query
			this.fetchDesignations();
			this.shutdown();
		} 
		catch (Exception e)
		{
			throw new MojoExecutionException("Exception", e);
		}
	}
	
	private void fetchVersions() throws SQLException
	{
		String action = "add";
		String name = null;
		String description = null;
		String effectiveDateString = null;
		String releaseDateString = null;
		String source = null; //CodeSystem?
		String append = "true"; // ??
		
		SimpleDateFormat sdf = new SimpleDateFormat("dd-MMM-yy HH.mm.ss.000000000 a");

		_versions = new ArrayList<>();
		
		Statement s = _h2.getConnection().createStatement();
		String query = "SELECT DISTINCT v.description, v.name, v.effectivedate, v.releasedate, cs.name, cs.vuid"
					+ " FROM version v"
					+ " INNER JOIN codesystem cs ON cs.id=v.codesystem_id";
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			description = rs.getString(1);
			name = rs.getString(2);
			effectiveDateString = rs.getString(3);
			releaseDateString = rs.getString(4);
			source = rs.getString(5);
			append = rs.getString(6);
			
			Date effectiveDate = null;
			if (effectiveDateString != null)
			{
				try {
					effectiveDate = sdf.parse(effectiveDateString);
				} catch (Exception e) {
					//
				}
			}
			
			Date releaseDate = null;
			if (releaseDateString != null)
			{
				try {
					releaseDate = sdf.parse(releaseDateString);
				} catch (Exception e) {
					//
				}
			}
			
			_versions.add(new Version(name, source, description, effectiveDate, releaseDate, parseBoolean(append)));
		}
		
		//System.out.println("Fetched versions: " + _versions.size());

	}
	
	
	public List<Version> getVersions()
	{
		return _versions;
	}
	
	private void fetchConcepts() throws SQLException
	{
		String action = "add";
		String entityId = null;
		String name = null;
		String code = null;
		String vuidString = null;
		String active = null;

		_concepts = new ArrayList<>();
		_conceptsMap = new HashMap<>();
		ArrayList<ConceptImportDTO> _al = new ArrayList<>();

		Statement s = _h2.getConnection().createStatement();
		String query = "SELECT DISTINCT c.entity_id, c.name, c.code, c.vuid, c.active"
						+ " FROM concept c"
						+ " INNER JOIN relationship r ON r.source_entity_id=c.entity_id"
						+ " WHERE c.kind='C'"
						+ " AND c.vuid IS NOT NULL"
						+ " AND vuid REGEXP('^\\d+$')"; // I think a data import issue  
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			entityId = rs.getString(1);
			name = rs.getString(2);
			code = rs.getString(3);
			vuidString = rs.getString(4);
			active = rs.getString(5);
			
			if (active == null)
			{
				active = Boolean.TRUE.toString();
			}
			
			Long vuid = null;
			if (vuidString != null)
			{
				vuid = Long.valueOf(vuidString);
			}
			
			// Need to handle duplicate VUIDs, at least for testing
			if (!_currentVuids.contains(vuid))
			{
				_concepts.add(new ConceptImportDTO(action, name, code, vuid, parseBoolean(active)));
				
				Long entity = Long.valueOf(entityId);
				if (!_conceptsMap.containsKey(entity))
				{
					_al = new ArrayList<ConceptImportDTO>();
				}
				else
				{
					_al = _conceptsMap.get(entity);
				}
				_al.add(new ConceptImportDTO(action, name, code, vuid, parseBoolean(active)));

				_conceptsMap.put(entity, _al);
				
				_currentVuids.add(vuid);
			}
		}
		
		//System.out.println("Fetched concepts: " + _currentVuids.size());

		_currentVuids = null;
	}
	
	public List<ConceptImportDTO> getConcepts()
	{
		return _concepts;
	}
	
	public Map<Long, ArrayList<ConceptImportDTO>> getConceptsMap()
	{
		return _conceptsMap;
	} 
	
	private void fetchTypes() throws SQLException
	{
		
		String name = null;
		String kind = null;
		_types = new ArrayList<>();
				
		Statement s = _h2.getConnection().createStatement();
		String query = "SELECT DISTINCT name, kind FROM TYPE";
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			name = rs.getString(1);
			kind = rs.getString(2);
			
			_types.add(new TypeImportDTO(kind, name));
		}
	}
	
	public List<TypeImportDTO> getTypes()
	{
		return _types;
	}
	
	private void fetchProperties() throws SQLException
	{
		String action = "add";
		String typeName = null;
		String valueOld = null;
		String valueNew = null;
		String active = null;
		String conceptEntityId = null;
		
		_properties = new ArrayList<>();
		_propertiesMap = new HashMap<>();
		ArrayList<PropertyImportDTO> _al = new ArrayList<>();
		
		Statement s = _h2.getConnection().createStatement();
		String query = "SELECT DISTINCT p.property_value, p.active, t.name, p.conceptentity_id"
						+ " FROM property p"
						+ " INNER JOIN type t ON t.id=p.propertytype_id"
						+ " INNER JOIN version v ON v.id=p.version_id"
						+ " AND V.codesystem_id='1'";
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			valueNew = rs.getString(1);
			active = rs.getString(2);
			typeName = rs.getString(3);
			// valueOld? 
			conceptEntityId = rs.getString(4);
			
			if (active == null)
			{
				// default of 'Active' element is true
				active = "true";
			}
			
			Long entity = Long.valueOf(conceptEntityId);
			if (!_propertiesMap.containsKey(entity))
			{
				_al = new ArrayList<PropertyImportDTO>();
			}
			else
			{
				_al = _propertiesMap.get(entity);
			}
			_al.add(new PropertyImportDTO(action, typeName, valueOld, valueNew, parseBoolean(active)));
			
			_propertiesMap.put(entity, _al);
			
			_properties.add(new PropertyImportDTO(action, typeName, valueOld, valueNew, parseBoolean(active)));
		}
		
		//System.out.println("Fetched properties: " + _properties.size());
	}
	
	public List<PropertyImportDTO> getProperties()
	{
		return _properties;
	}
	
	public Map<Long, ArrayList<PropertyImportDTO>> getPropertiesMap()
	{
		return _propertiesMap;
	}
	
	private void fetchRelationships() throws SQLException
	{
		String action = "add";
		String typeName = null;
		String oldTargetCode = null;
		String newTargetCode = null;
		String active = null;
		String conceptEntityId = null;
		
		_relationships = new ArrayList<>();
		_relationshipsMap = new HashMap<>();
		ArrayList<RelationshipImportDTO> _al = new ArrayList<>();
		
		Statement s = _h2.getConnection().createStatement();
		String query = "SELECT t.name, c1.code, c2.code, r.active, c1.entity_id FROM relationship r"
				+ " LEFT JOIN concept c1 ON c1.entity_id=r.source_entity_id"
				+ " LEFT JOIN concept c2 ON c2.entity_id=r.target_entity_id"
				+ " LEFT JOIN type t ON r.type_id=t.id"
				+ " WHERE r.kind='C'";
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			typeName = rs.getString(1);
			oldTargetCode = rs.getString(2);
			newTargetCode = rs.getString(3);
			active = rs.getString(4);
			conceptEntityId = rs.getString(5);
			
			Long entity = Long.valueOf(conceptEntityId);
			if (!_relationshipsMap.containsKey(entity))
			{
				_al = new ArrayList<RelationshipImportDTO>();
			}
			else
			{
				_al = _relationshipsMap.get(entity);
			}
			_al.add(new RelationshipImportDTO(action, typeName, oldTargetCode, newTargetCode, parseBoolean(active)));
			
			_relationshipsMap.put(entity, _al);
			
			_relationships.add(new RelationshipImportDTO(action, typeName, oldTargetCode, newTargetCode, parseBoolean(active)));
		}
		
		//System.out.println("Fetched relationships: " + _relationships.size());
	}
	
	public List<RelationshipImportDTO> getRelationships()
	{
		return _relationships;
	}
	
	public Map<Long, ArrayList<RelationshipImportDTO>> getRelationshipsMap()
	{
		return _relationshipsMap;
	}
	
	private void fetchCodeSystems() throws SQLException
	{
		
		String codeSystemName = null;
		String vuidString = null;
		String description = null;
		String copyright = null;
		String copyrightURL = null;
		String preferredDesignationType = null;
		String action = "add";
		
		_codesystems = new ArrayList<>();
			
		Statement s = _h2.getConnection().createStatement();
		// This doesn't account for versions, just a straight fetch from the table 
		String query = "SELECT cs.name, cs.vuid, cs.description, cs.copyright, cs.copyrighturl, t.name as preferred_designation_type"
					+ " FROM CODESYSTEM cs"
					+ " INNER JOIN type t ON t.id=preferred_designation_type_id";
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			codeSystemName = rs.getString(1);
			vuidString = rs.getString(2);
			description = rs.getString(3);
			copyright = rs.getString(4);
			copyrightURL = rs.getString(5);
			preferredDesignationType = rs.getString(6);
			
			Long vuid = null;
			if (vuidString != null)
			{
				vuid = Long.valueOf(vuidString);
			}
			
			_codesystems.add(new CodeSystem(codeSystemName, vuid, description, copyright, copyrightURL, preferredDesignationType, action));
			// We don't have version in the SQL, or need a way to relate
			//cs.setVersion((Version)elementDataStack.peek().createdChildObjects.remove(0));
		}
	}
	
	public List<CodeSystem> getCodeSystems()
	{
		return _codesystems;
	}
	
	private void fetchMapsets() throws SQLException
	{
		String action = "add";
		String vuidString = null;
		String active = null;
		String name = null;
		String code = null;
		String sourceVersionName = null;
		String targetVersionName = null;
		String sourceCodeSystemName = null;
		String targetCodeSystemName = null;
		String conceptEntityId = null;

		_mapsets = new ArrayList<>();
		_mapsetsMap = new HashMap<>();
		ArrayList<MapSetImportDTO> _al = new ArrayList<>();

		Statement s = _h2.getConnection().createStatement();
		String query = "SELECT DISTINCT c.name, c.code, c.vuid, c.active, cs1.name, v1.description, cs2.name, v2.description, c.entity_id"
					+ " FROM concept c"
					+ " INNER JOIN mapsetextension ms on ms.mapsetid=c.entity_id"
					+ " INNER JOIN VERSION v1 ON v1.id=ms.sourceversionid"
					+ " INNER JOIN VERSION v2 ON v2.id=ms.targetversionid"
					+ " INNER JOIN codesystem cs1 ON cs1.id=v1.codesystem_id"
					+ " INNER JOIN codesystem cs2 ON cs2.id=v2.codesystem_id"
					+ " WHERE c.kind='M'"
					+ " AND c.active != '0'"; // To get around 'stated graph' issue for now
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			name = rs.getString(1);
			code = rs.getString(2);
			vuidString = rs.getString(3);
			active = rs.getString(4);
			sourceCodeSystemName = rs.getString(5);
			sourceVersionName = rs.getString(6);
			targetCodeSystemName = rs.getString(7);
			targetVersionName = rs.getString(8);
			conceptEntityId = rs.getString(9);
			
			if (active == null)
			{
				active = Boolean.TRUE.toString();
			}
			
			Long vuid = null;
			if (vuidString != null)
			{
				vuid = Long.valueOf(vuidString);
				
				Long entity = Long.valueOf(conceptEntityId);
				
				if (!_mapsetsMap.containsKey(entity))
				{
					_al = new ArrayList<MapSetImportDTO>();
				}
				else
				{
					_al = _mapsetsMap.get(entity);
				}
				_al.add(new MapSetImportDTO(action, name, code, vuid, parseBoolean(active), sourceCodeSystemName, sourceVersionName, targetCodeSystemName, targetVersionName));
				
				_mapsetsMap.put(entity, _al);
			}
			
			if (!_currentVuids.contains(vuid))
			{
				_mapsets.add(new MapSetImportDTO(action, name, code, vuid, parseBoolean(active), sourceCodeSystemName, sourceVersionName, targetCodeSystemName, targetVersionName));
				_currentVuids.add(vuid);
			}
		}
		
		//System.out.println("Fetched mapsets: " + _mapsets.size());
		
		
	}
	
	public List<MapSetImportDTO> getMapsets()
	{
		return _mapsets;
	}
	
	public Map<Long, ArrayList<MapSetImportDTO>> getMapsetsMap()
	{
		return _mapsetsMap;
	}
	
	private void fetchSubsets() throws SQLException
	{
		String action = "add";
		String subsetName = null;
		String vuidString = null;
		String active = null;
		String conceptEntityId = null;

		_subsets = new ArrayList<>();
		_subsetsMap = new HashMap<>();
		
		HashSet<Long> _currentVuids = new HashSet<>();
		ArrayList<SubsetImportDTO> _al = new ArrayList<>();

		Statement s = _h2.getConnection().createStatement();
		String query = "select distinct c.vuid, c.name, r.active, c.entity_id"
					+ " from relationship r"
					+ " inner join concept c on c.entity_id=r.entity_id"
					+ " where r.kind='S'";
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			vuidString = rs.getString(1);
			subsetName = rs.getString(2);
			active = rs.getString(3);
			conceptEntityId = rs.getString(4);
			
			if (active == null)
			{
				active = Boolean.TRUE.toString();
			}
			
			Long vuid = null;
			if (vuidString != null)
			{
				vuid = Long.valueOf(vuidString);
				
				Long entity = Long.valueOf(conceptEntityId);
				
				if (!_subsetsMap.containsKey(entity))
				{
					_al = new ArrayList<SubsetImportDTO>();
				}
				else
				{
					_al = _subsetsMap.get(entity);
				}
				_al.add(new SubsetImportDTO(action, vuid, subsetName, parseBoolean(active)));
				
				_subsetsMap.put(entity, _al);
			}
			
			if (vuid != null && !_currentVuids.contains(vuid))
			{
				_subsets.add(new SubsetImportDTO(action, vuid, subsetName, parseBoolean(active)));
				_currentVuids.add(vuid);
			}
		}
		
		//System.out.println("Fetched subsets: " + _subsets.size());
	}
	
	public List<SubsetImportDTO> getSubsets()
	{
		return _subsets;
	}
	
	public Map<Long, ArrayList<SubsetImportDTO>> getSubsetsMap()
	{
		return _subsetsMap;
	}
	
	private void fetchMapEntries() throws SQLException
	{
		
		String action = "add";
		String vuidString = null;
		String active = null;
		String sourceCode = null;
		String targetCode = null;
		String sequenceString = null;
		String groupingString = null;
		String effectiveDateString = null;
		String conceptEntityId = null;
		
		_mapentries = new ArrayList<>();
		_mapEntriesMap = new HashMap<>();
		ArrayList<MapEntryImportDTO> _al = new ArrayList<>();
		
		String query = "SELECT DISTINCT c.vuid, c.active, me.sourcecode, me.targetcode, r.sequence, r.grouping, me.effectivedate, r.source_entity_id"
					+ " FROM relationship r"
					+ " INNER JOIN concept c ON c.entity_id=r.target_entity_id"
					+ " INNER JOIN mapentryextension me ON me.mapentryid=c.entity_id"
					+ " WHERE r.kind='M'";
		
		Statement s = _h2.getConnection().createStatement();
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			vuidString = rs.getString(1);
			active = rs.getString(2);
			sourceCode = rs.getString(3); 
			targetCode = rs.getString(4);
			sequenceString = rs.getString(5);
			groupingString = rs.getString(6);
			effectiveDateString = rs.getString(7);
			conceptEntityId = rs.getString(8);
			
			if (active == null)
			{
				active = Boolean.TRUE.toString();
			}
			
			Long grouping = null;
			if (groupingString != null)
			{
				grouping = Long.valueOf(groupingString);
			}
			
			int sequence = 0;
			if (sequenceString != null)
			{
				sequence = Integer.valueOf(sequenceString);
			}
			
			Date effectiveDate = null;
			if (effectiveDateString != null)
			{
				try {
					effectiveDate = new SimpleDateFormat("dd-MMM-yy HH.mm.ss.000000000 a").parse(effectiveDateString);
				} catch (Exception e) {
					//
				}
			}
			
			Long vuid = null;
			if (vuidString != null)
			{
				vuid = Long.valueOf(vuidString);
				
				Long entity = Long.valueOf(conceptEntityId);
				
				if (!_mapEntriesMap.containsKey(entity))
				{
					_al = new ArrayList<MapEntryImportDTO>();
				}
				else
				{
					_al = _mapEntriesMap.get(entity);
				}
				_al.add(new MapEntryImportDTO(action, vuid, parseBoolean(active), sourceCode, targetCode, sequence, grouping, effectiveDate));
				
				_mapEntriesMap.put(entity, _al);

				_mapentries.add(new MapEntryImportDTO(action, vuid, parseBoolean(active), sourceCode, targetCode, sequence, grouping, effectiveDate));

			}
			
		}
		
		//System.out.println("Fetched mapentries: " + _mapentries.size());
	}
	
	public List<MapEntryImportDTO> getMapEntries()
	{
		return _mapentries;
	}
	
	public Map<Long, ArrayList<MapEntryImportDTO>> getMapEntriesMap()
	{
		return _mapEntriesMap;
	}
	
	
	private void fetchDesignations() throws SQLException
	{
		String action = "add";
		String typeName = null;
		String code = null;
		String valueOld = null; // Not used?
		String valueNew = null; // ??
		String moveFromConceptCode = null; // ??
		String vuidString = null;
		String active = null;
		String version = null;
		String conceptEntityId = null;

		_designations = new ArrayList<>();
		_designationsMap = new HashMap<>();
		
		ArrayList<DesignationExtendedImportDTO> _al = null;
		
		Statement s = _h2.getConnection().createStatement();
		String query = "SELECT DISTINCT t.name, c.code, c.vuid, c.name, r.active, v.id, r.source_entity_id"
						+ " FROM relationship r"
						+ " INNER JOIN concept c ON c.entity_id=r.target_entity_id"
						+ " INNER JOIN Type T ON t.id=c.type_id"
						+ " INNER JOIN version v ON V.id=r.version_id"
						+ " AND v.codesystem_id='1'" // Only 'VHAT Version' or we get null VUIDs
						+ " WHERE r.kind='D'";
		ResultSet rs = s.executeQuery(query);
		while (rs.next()) {
			typeName = rs.getString(1); //t.name
			code = rs.getString(2); //c.code
			vuidString = rs.getString(3); //c.vuid
			valueNew = rs.getString(4); //c.name
			active = rs.getString(5); //r.active
			version = rs.getString(6); //v.id
			conceptEntityId = rs.getString(7); //r.source_entity_id
			
			if (active == null)
			{
				active = Boolean.TRUE.toString();
			}
			
			Long vuid = null;
			if (vuidString != null)
			{
				vuid = Long.valueOf(vuidString);
				Long entity = Long.valueOf(conceptEntityId);
				
				if (!_designationsMap.containsKey(entity))
				{
					_al = new ArrayList<DesignationExtendedImportDTO>();
				}
				else
				{
					_al = _designationsMap.get(entity);
				}
				_al.add(new DesignationExtendedImportDTO(action, typeName, code, valueOld, valueNew, vuid, moveFromConceptCode, parseBoolean(active)));

				
				_designationsMap.put(entity, _al);
			}

		}
		
		//System.out.println("Fetched designations: " + _designationsMap.size());

	}
	
	public Map<Long, ArrayList<DesignationExtendedImportDTO>> getDesignationsMap()
	{
		return _designationsMap;
	}
	
	private boolean parseBoolean(String value)
	{
		boolean result = false;
		if (value == null || value.equalsIgnoreCase("true") || value.equals("1"))
		{
			result = true;
		}
		
		return result;
	}
	
}
