/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.rest.session;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import javax.inject.Singleton;
import org.glassfish.hk2.api.Rank;
import org.jvnet.hk2.annotations.Service;
import com.fasterxml.jackson.databind.ObjectMapper;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.UserRole;
import gov.vha.isaac.ochre.api.UserRoleService;
import gov.vha.isaac.ochre.api.util.UuidT5Generator;

/**
 * The Class MockUserService.
 *
 * {@link PrismeIntegratedUserService}
 * 
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 */
@Service(name="rest-prismeUserService")
@Rank(value = 10)
@Singleton
public class PrismeIntegratedUserService implements UserRoleService {
//	/*
//	 * Example URL for get_roles_by_token
//	 * URL url = new URL("https://DNS.aac.DNS   /rails_prisme/roles/get_roles_by_token.json?token=" + token);
//	 */
//	/*
//	 * Example SSO Token
//	 * %5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC2%5CxEE%5CxFA%5CxE1%5Cx94%5CxBF3%5CxA9%5Cx16K%22%2C+%22%7EK%5CxC4%5CxEFXk%5Cx80%5CxB1%5CxA3%5CxF3%5Cx8D%5CxB1%5Cx7F%5CxBC%5Cx02K%22%2C+%22k%5Cf%5CxDC%5CxF7%2CP%5CxB2%5Cx97%5Cx99%5Cx99%5CxE0%5CxE1%7C%5CxBF%5Cx1DK%22%2C+%22J%5Cf%5Cx9B%5CxD8w%5Cx15%5CxFE%5CxD3%5CxC7%5CxDC%5CxAC%5Cx9E%5Cx1C%5CxD0bG%22%5D
//	 */
//	//String json = "{\"roles\":[{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}],\"token_parsed?\":true,\"user\":\"DNS   ArmbrD\",\"type\":\"ssoi\",\"id\":10005}";

//	// TODO Joel implement prisme_all_roles_url=https://DNS  cttdbs80.aac.DNS   :8080/rails_prisme/roles/get_all_roles.json
//	/*
//	 * TODO implement https://DNS.aac.DNS   /rails_prisme/roles/get_all_roles
//	 * returning ["super_user","administrator","read_only","editor","reviewer","approver","manager"]
//	 */

	public static final String TEST_JSON1 = "{\"roles\":["
			+ "{\"id\":19990,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
			+ ","
			+ "{\"id\":19991,\"name\":\"editor\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
			+ ","
			+ "{\"id\":19992,\"name\":\"reviewer\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
			+ ","
			+ "{\"id\":19993,\"name\":\"administrator\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
			+ ","
			+ "{\"id\":19994,\"name\":\"manager\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
			+ ","
			+ "{\"id\":19995,\"name\":\"approver\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
			+ "],\"token_parsed?\":true,\"user\":\"DNS   ArmbrD\",\"type\":\"ssoi\",\"id\":10005}";
	public static final String TEST_JSON2 = "{\"roles\":["
			+ "{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
			+ "],\"token_parsed?\":true,\"user\":\"DNS   KniazJ\",\"type\":\"ssoi\",\"id\":10005}";
	public static final String TEST_JSON3 = "{\"roles\":["
			+ "{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
			+ ","
			+ "{\"id\":19991,\"name\":\"editor\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
			+ "],\"token_parsed?\":true,\"user\":\"DNS   EfronJ\",\"type\":\"ssoi\",\"id\":10005}";

	/**
	 * 
	 * Attempt to construct a user from a string of the following format:
	 * 
	 * {name}:{role1}[{,role2}[{,role3}[...]]]
	 * 
	 * @param arg
	 * @return
	 */
	private static Optional<String> constructTestUser(String arg) {
		try {
			String[] components = arg.split(":");

			String name = null;
			Set<UserRole> roles = new HashSet<>();
			if (components.length == 2) {
				if (components[0].matches("[A-Za-z][A-Za-z0-9_]*")) {
					name = components[0].trim();

					String[] roleStrings = components[1].split(",");

					for (int i = 0; i < roleStrings.length; ++i) {
						roles.add(UserRole.safeValueOf(roleStrings[i].trim()).get());
					}
				}
			}
			
			if (name != null && name.length() > 0 && roles.size() > 0) {
				StringBuilder builder = new StringBuilder();
				builder.append("{\"roles\":[");
				boolean addedRole = false;
				for (UserRole role : roles) {
					if (addedRole) {
						builder.append(",");
					}
					builder.append("{\"id\":19990,\"name\":\"" + role.toString() + "\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}");

					addedRole = true;
				}
				
				builder.append("],\"token_parsed?\":true,\"user\":\"" + name + "\",\"type\":\"ssoi\",\"id\":10005}");
				
				return Optional.of(builder.toString());
			}
		} catch (Exception e) {
			// ignore
		}

		return Optional.empty();
	}

	private PrismeIntegratedUserService() {
		//for HK2
	}

	/**
	 * Return a user and roles available for that user
	 *
	 * @param ssoToken
	 *            the user's SSO token string
	 * @return the user and roles available to the user
	 * 
	 * For test purposes, attempt to construct a user from a string of the following format:
	 * {name}:{role1}[{,role2}[{,role3}[...]]]
	 *
	 */
	public Optional<User> getUser(String ssoToken) {
		String jsonToUse = null;
		
		Optional<String> createdJson = constructTestUser(ssoToken);
		if (createdJson.isPresent()) {
			jsonToUse = createdJson.get();
		} else {
			if (ssoToken.equals(TEST_JSON1)) {
				jsonToUse = TEST_JSON1;
			} else if (ssoToken.equals(TEST_JSON2)) {
				jsonToUse = TEST_JSON2;
			} else if (ssoToken.equals(TEST_JSON3)) {
				jsonToUse = TEST_JSON3;
			} else if (ssoToken.equals("TEST_JSON1")) {
				jsonToUse = TEST_JSON1;
			} else if (ssoToken.equals("TEST_JSON2")) {
				jsonToUse = TEST_JSON2;
			} else if (ssoToken.equals("TEST_JSON3")) {
				jsonToUse = TEST_JSON3;
			} else {
				// Either a real SSO token or custom JSON
				jsonToUse = ssoToken;
			}
		}

		/*
		 * Example URL for get_roles_by_token
		 * URL url = new URL("https://DNS.aac.DNS   /rails_prisme/roles/get_roles_by_token.json?token=" + token);
		 */
		
		ObjectMapper mapper = new ObjectMapper();
		Map<?, ?> map = null;
		try {
			map = mapper.readValue(jsonToUse, Map.class);
		} catch (Exception e) {
			// Passed text may be random or a real SSO token, which we can't handle, so use a default
			jsonToUse = TEST_JSON1;
		}

		// TODO Joel implement access to PRISME API

		//Map map = mapper.readValue(url, Map.class);
		
		if (map == null) {
			try {
				map = mapper.readValue(jsonToUse, Map.class);
			} catch (Exception e) {
				throw new RuntimeException("Failed reading test JSON \"" + jsonToUse + "\".  Caught " + e.getClass().getName() + " " + e.getLocalizedMessage());
			}
		}

		String userName = (String)map.get("user");
		Set<UserRole> roleSet = new HashSet<>();
		Collection<?> roles = (Collection<?>)map.get("roles");
		for (Object roleMapObject : roles) {
			Map<?,?> roleMap = (Map<?,?>)roleMapObject;
			String roleName = (String)roleMap.get("name");
			
			roleSet.add(UserRole.safeValueOf(roleName).get());
		}
		
		final UUID uuidFromUserFsn = UuidT5Generator.get(MetaData.USER.getPrimordialUuid(), userName);

		User newUser = new User(userName, uuidFromUserFsn, roleSet);
		
		UserCache.put(newUser);

		return Optional.of(newUser);
	}

	/* (non-Javadoc)
	 * @see gov.vha.isaac.ochre.api.UserRoleService#getUserRoles(java.util.UUID)
	 * 
	 * This method should throw exception if the user has not already been cached
	 */
	@Override
	public Set<UserRole> getUserRoles(UUID userId)
	{
		return UserCache.get(userId).get().getRoles();
	}

	/* (non-Javadoc)
	 * @see gov.vha.isaac.ochre.api.UserRoleService#getAllUserRoles()
	 */
	@Override
	public Set<UserRole> getAllUserRoles()
	{
		// TODO Joel call PRISME for available roles and return only the intersection with the enum
		Set<UserRole> availableRoles = new HashSet<>();
		for (UserRole role : UserRole.values()) {
			if (role != UserRole.AUTOMATED) { // AUTOMATED will not be a PRISME role
				availableRoles.add(role);
			}
		}
		
		return Collections.unmodifiableSet(availableRoles);
	}
}
