/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.isaac.utils.file_transfer;

import java.io.File;
import java.net.URL;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import gov.vha.isaac.utils.file_transfer.Config.Artifact;

/**
 * 
 * {@link MavenFileUtils}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
class MavenFileUtils {
	final static String SNAPSHOT_VERSION_REGEX = "[0-9][0-9]*(\\.[0-9][0-9]*)-2[0-9][0-9][0-9][0-1][0-9][0-3][0-9]\\.[0-2][0-9][0-5][0-9][0-5][0-9]-[0-9][0-9]*";
	
	private static Logger log = LoggerFactory.getLogger(MavenFileUtils.class);

	private MavenFileUtils() {
	}

	/**
	 * 
	 * @param baseMavenURL - optional - but required if you are downloading a SNAPSHOT dependency, as this method will need to download the metadata file
	 * from the repository server in order to determine the proper version component for the SNAPSHOT.
	 * @param mavenUsername - optional - only used for a SNAPSHOT dependency
	 * @param mavenPassword - optional - only used for a SNAPSHOT dependency
	 * @param groupId
	 * @param artifactId
	 * @param version
	 * @param classifier - optional
	 * @param type
	 * @return Paths
	 * @throws Exception
	 */
	static FileTransferUtils.Paths makeMavenRelativePathForDownload(
			String baseMavenURL, String mavenUsername, String mavenPassword,
			String groupId,
			String artifactId, 
			String version,
			String classifier, String type, boolean stripSnapshotVersionDetails) throws Exception
	{
		String directoryPath = MavenFileUtils.makeMavenDirectoryRelativePath(groupId, artifactId, version);
		String snapshotVersion = "";
		String versionWithoutSnapshot = version;
	
		String localPath = null;
		String remotePath = null;
		
		if (version.endsWith("-SNAPSHOT"))
		{
			versionWithoutSnapshot = version.substring(0, version.lastIndexOf("-SNAPSHOT"));
			String metadataRelativePath = directoryPath;
			String metadataUrlPath = baseMavenURL + (baseMavenURL.endsWith("/") ? "" : "/") + metadataRelativePath;
			URL metadataUrl = new URL(metadataUrlPath + "/maven-metadata.xml");
			//URL metadataMd5Url = new URL(metadataUrlPath + "/maven-metadata.xml.md5");
			//URL metadataSha1Url = new URL(metadataUrlPath + "/maven-metadata.xml.sha1");
			
//			File metadataDir = Files.createTempDir();
//			metadataDir.deleteOnExit();
//			
//			// Download the maven-metadata.xml file
//			// Don't see any point in backgrounding this
//			//File metadataFile = new Download(mavenUsername, mavenPassword, metadataUrl, null, false, false, true, metadataDir).call();
//			File metadataFile = FileTransferUtils.download(metadataUrl, mavenUsername, mavenPassword, metadataDir);
//			metadataFile.deleteOnExit();
	
			File metadataFile = FileTransferUtils.download(metadataUrl, mavenUsername, mavenPassword, new File(directoryPath));
			
			// Download maven-metadata.xml.md5 file
			//new Download(mavenUsername, mavenPassword, metadataMd5Url, false, false, metadataDir).call();
			// Download maven-metadata.xml.sha1 file
			//new Download(mavenUsername, mavenPassword, metadataSha1Url, false, false, metadataDir).call();
			// TODO check metadata file checksum
			
			DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder;
			Document dDoc = null;
			XPath xPath = XPathFactory.newInstance().newXPath();
	
			builder = domFactory.newDocumentBuilder();
	
			dDoc = builder.parse(metadataFile);
			String timestamp = ((Node)xPath.evaluate("/metadata/versioning/snapshot/timestamp", dDoc, XPathConstants.NODE)).getTextContent();
			String buildNumber = ((Node)xPath.evaluate("/metadata/versioning/snapshot/buildNumber", dDoc, XPathConstants.NODE)).getTextContent();
			
			snapshotVersion = "-" + timestamp + "-" + buildNumber;
			
			// Delete 
			//metadataFile.delete();
			
			//The download task makes a temp folder in for this. Delete that too
			//metadataDir.delete();
		}
		
		remotePath = directoryPath + "/" + artifactId + "-" + versionWithoutSnapshot + snapshotVersion +
				(StringUtils.isNotBlank(classifier) ? "-" + classifier : "") + "." + type;
		
		if (stripSnapshotVersionDetails) {
			// Strip any snapshotVersion info from downloaded file name
			localPath = directoryPath + "/" + artifactId + "-" + version +
				(StringUtils.isNotBlank(classifier) ? "-" + classifier : "") + "." + type;
		} else {
			localPath = remotePath;
		}
		
		return new FileTransferUtils.Paths(remotePath, localPath);
	}

	static boolean artifactNameMatchesType(String filename, String type) {
		if (! StringUtils.isBlank(type) && ! type.equals("*")) {
			String[] parts = filename.split("\\.");
			
			if (parts.length > 1 && parts[parts.length - 1].matches(type)) {
				return true;
			} else if (parts.length > 2 && (parts[parts.length - 2] + "." + parts[parts.length - 1]).matches(type)) {
				return true;
			} else {
				return false;
			}
		} else {
			return true;
		}
	}

	static Set<String> getFileNamesFilteredByTypeAndClassifier(String baseUrl, Artifact artifact, String username, String password) throws Exception {
		String artifactMavenDirectoryRelativePath = MavenFileUtils.makeMavenDirectoryRelativePath(artifact.getGroup(), artifact.getArtifact(), artifact.getVersion());
		Set<String> allFilesForArtifact = MavenFileUtils.listMostRecentMavenMetadataVersionOfEachFileInDirectory(baseUrl + "/" + artifactMavenDirectoryRelativePath, username, password, artifactMavenDirectoryRelativePath);
		Set<String> filesFilteredByType = new HashSet<>();
		if (artifact.getTypes() != null && artifact.getTypes().size() > 0) {
			for (String file : allFilesForArtifact) {
				boolean matched = false;
				if (file.equals("maven-metadata.xml")) {
					matched = true;
				} else {
					for (String type : artifact.getTypes()) {
						if (artifactNameMatchesType(file, type)) {
							matched = true;
							break;
						}
					}
				}
	
				if (matched) {
					filesFilteredByType.add(file);
				}
			}
		} else {
			filesFilteredByType.addAll(allFilesForArtifact);
		}
		
		Set<String> filesFilteredByTypeAndClassifier = new HashSet<>();
		if (artifact.getClassifier() != null) {
			for (String file : filesFilteredByType) {
				if (file.equals("maven-metadata.xml") || MavenFileUtils.artifactNameMatchesClassifier(file, artifact.getArtifact(), artifact.getVersion(), artifact.getClassifier())) {
					filesFilteredByTypeAndClassifier.add(file);
				}
			}
		} else {
			filesFilteredByTypeAndClassifier.addAll(filesFilteredByType);
		}
		
		return Collections.unmodifiableSet(filesFilteredByTypeAndClassifier);
	}

	static boolean artifactNameMatchesClassifier(String filename, String artifactId, String version, String classifier) {
		if (classifier != null && ! classifier.equals("*")) {
			String classifierFromFileName = MavenFileUtils.getClassifierFromFileName(filename, artifactId, version);
			if (classifierFromFileName.matches(classifier)) {
				return true;
			} else {
				return false;
			}
		} else {
			return true;
		}
	}

	static String makeMavenDirectoryRelativePath(String groupId, String artifactId, String version) {
		String pathFromGroupId = groupId.replaceAll("\\.", "/");
	
		String directoryPath = pathFromGroupId + "/" + artifactId + "/" + version;
		
		return directoryPath;
	}

	static String getTypeFromFileName(String filename, String artifactId, String version) {
		String idRemoved = filename.replace(artifactId + "-", "");
		String versionRemoved = null;
		if (version.contains("SNAPSHOT")) {
			versionRemoved = idRemoved.replaceAll(MavenFileUtils.SNAPSHOT_VERSION_REGEX, "");
		} else {
			versionRemoved = idRemoved.replace(version, "");
		}
	
		return versionRemoved.substring(versionRemoved.indexOf('.') + 1);
	}

	static String getClassifierFromFileName(String filename, String artifactId, String version) {
		String idRemoved = filename.replace(artifactId + "-", "");
		String versionRemoved = null;
		if (version.contains("SNAPSHOT")) {
			versionRemoved = idRemoved.replaceAll(MavenFileUtils.SNAPSHOT_VERSION_REGEX, "");
		} else {
			versionRemoved = idRemoved.replace(version, "");
		}
	
		if (versionRemoved.startsWith("-")) {
			return versionRemoved.substring(1, versionRemoved.indexOf('.'));
		} else {
			return null;
		}
	}

	static Set<String> filterFileNamesWithMavenMetadata(File mavenMetadataFile, Collection<String> fileNames) throws Exception
	{
		/*
		 *	<metadata modelVersion="1.1.0">
				<groupId>gov.vha.isaac.rest</groupId>
				<artifactId>isaac-rest</artifactId>
				<version>1.9-SNAPSHOT</version>
				<versioning>
					<snapshot>
						<timestamp>20160627.203551</timestamp>
						<buildNumber>51</buildNumber>
					</snapshot>
					<lastUpdated>20160627203551</lastUpdated>
					<snapshotVersions>
						<snapshotVersion>
							<extension>war</extension>
							<value>1.9-20160627.203551-51</value>
							<updated>20160627203551</updated>
						</snapshotVersion>
						<snapshotVersion>
							<extension>pom</extension>
							<value>1.9-20160627.203551-51</value>
							<updated>20160627203551</updated>
						</snapshotVersion>
						<snapshotVersion>
							<classifier>sources</classifier>
							<extension>jar</extension>
							<value>1.9-20160627.203551-51</value>
							<updated>20160627203551</updated>
						</snapshotVersion>
					</snapshotVersions>
				</versioning>
			</metadata>
		 */
		DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder;
		Document dDoc = null;
		XPath xPath = XPathFactory.newInstance().newXPath();
		builder = domFactory.newDocumentBuilder();
		dDoc = builder.parse(mavenMetadataFile);
	
		String artifactId = XMLUtils.getStringFromXml(dDoc, "/metadata/artifactId");
	
		String timestamp = ((Node)xPath.evaluate("/metadata/versioning/snapshot/timestamp", dDoc, XPathConstants.NODE)).getTextContent();
		String buildNumber = ((Node)xPath.evaluate("/metadata/versioning/snapshot/buildNumber", dDoc, XPathConstants.NODE)).getTextContent();
	
		NodeList snapshotVersions = XMLUtils.getNodeList(dDoc, "/metadata/versioning/snapshotVersions/snapshotVersion");
	
		Set<String> filteredFileNames = new HashSet<>();
		for (int i = 0; i < snapshotVersions.getLength(); ++i) {
			Node snapshotVersionNode = snapshotVersions.item(i);
			String classifier = null;
			String extension = null;
			String value = null;
	
			NodeList snapshotVersionChildNodes = snapshotVersionNode.getChildNodes();
			for (int j = 0; j < snapshotVersionChildNodes.getLength(); ++ j) {
				Node snapshotVersionChildNode = snapshotVersionChildNodes.item(j);
				if (snapshotVersionChildNode.getNodeName().equals("classifier")) {
					classifier = snapshotVersionChildNode.getTextContent();
				} else if (snapshotVersionChildNode.getNodeName().equals("extension")) {
					extension = snapshotVersionChildNode.getTextContent();
				} else if (snapshotVersionChildNode.getNodeName().equals("value")) {
					value = snapshotVersionChildNode.getTextContent();
				}
			}
	
			filteredFileNames.add(artifactId + "-" + value + (classifier != null ? "-" + classifier : "") + "." + extension);
		}
	
		if (filteredFileNames.size() == 0) {
			String snapshotVersionSubstring = "-" + timestamp + "-" + buildNumber;
			for (String fileName : fileNames) {
				if (fileName.contains(snapshotVersionSubstring)) {
					filteredFileNames.add(fileName);
				}
			}
		}
		return Collections.unmodifiableSet(filteredFileNames);
	}

	static Set<String> listMostRecentMavenMetadataVersionOfEachFileInDirectory(String directoryUri, String username, String password) throws Exception {
		return listMostRecentMavenMetadataVersionOfEachFileInDirectory(directoryUri, username, password, (String)null);
	}
	static Set<String> listMostRecentMavenMetadataVersionOfEachFileInDirectory(String directoryUri, String username, String password, String metadataDirStr) throws Exception {
		Set<String> allFilesInDirectory = FileTransferUtils.listFilesInDirectory(directoryUri, username, password);
		Set<String> filteredSet = new HashSet<>();
		if (allFilesInDirectory.contains("maven-metadata.xml")) {
			File metadataFile = null;
			File metadataDir = null;
			
			if (directoryUri.contains(":")) {
				
				URL metadataUrl = new URL(directoryUri + "/maven-metadata.xml");

				if (metadataDirStr != null) {
					metadataFile = new File((metadataDir = new File(metadataDirStr)), "maven-metadata.xml");
					
					if (! FileTransfer.hasBeenSuccessfullyHandled(metadataFile)) {
						// TODO this should work.  Find out why it doesn't
//						FileTransferFutureTask<File> task = new FileTransferFutureTask<File>(new Download(username, password, metadataUrl, "maven-metadata.xml", false, false, true, metadataDir), "download maven-metadata.xml");
//						task.setRetryTasks(
//								new FileTransferFutureTask<File>(new Download(username, password, metadataUrl, "maven-metadata.xml", false, false, true, metadataDir), "download maven-metadata.xml (retry 1)"),
//								new FileTransferFutureTask<File>(new Download(username, password, metadataUrl, "maven-metadata.xml", false, false, true, metadataDir), "download maven-metadata.xml (retry 2)")
//								);
//						task.run();
						metadataFile = FileTransferUtils.download(metadataUrl, username, password, metadataDir, false);
						try {
							URL sha1FileUrl = new URL(metadataUrl.toString() + ".sha1");
							URL md5FileUrl = new URL(metadataUrl.toString() + ".md5");

							if (FileTransferUtils.remoteFileExists(sha1FileUrl, username, password)) {
								FileTransferUtils.download(sha1FileUrl, username, password, metadataDir, false);
							}
							if (FileTransferUtils.remoteFileExists(md5FileUrl, username, password)) {
								FileTransferUtils.download(md5FileUrl, username, password, metadataDir, false);
							}

							FileTransfer.setHasBeenHandled(metadataFile, true);
						} catch (Exception e) {
							log.warn("Failed downloading checksum file(s) for " + metadataUrl, e);
						}
					}
				} else {
					metadataDir = java.nio.file.Files.createTempDirectory("ISAAC").toFile();
					metadataDir.deleteOnExit();
					metadataFile = FileTransferUtils.download(metadataUrl, username, password, metadataDir, false);
					metadataFile.deleteOnExit();
				}
			} else {
				metadataFile = new File(directoryUri + "/maven-metadata.xml");
			}
	
			//Set<String> fileNamesFromMetadata = getFileNamesFromMavenMetadata(metadataFile);
			filteredSet.addAll(filterFileNamesWithMavenMetadata(metadataFile, allFilesInDirectory));
			
			if (metadataDirStr == null) {
				metadataFile.delete();
				metadataDir.delete();
			}
			
			filteredSet.add("maven-metadata.xml");
			
			return Collections.unmodifiableSet(filteredSet);
		} else {
			return allFilesInDirectory;
		}
	}
}
