/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.isaac.ochre.impl.utility;

import java.io.PrintStream;
import java.util.Arrays;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.chronicle.LatestVersion;
import gov.vha.isaac.ochre.api.chronicle.ObjectChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.SememeVersion;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeColumnInfo;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeData;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeUsageDescription;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.dataTypes.DynamicSememeNid;
import gov.vha.isaac.ochre.api.coordinate.LanguageCoordinate;
import gov.vha.isaac.ochre.api.coordinate.StampCoordinate;
import gov.vha.isaac.ochre.api.relationship.RelationshipVersionAdaptor;
import gov.vha.isaac.ochre.api.util.StringUtils;
import gov.vha.isaac.ochre.model.configuration.LanguageCoordinates;
import gov.vha.isaac.ochre.model.configuration.StampCoordinates;
import gov.vha.isaac.ochre.model.sememe.DynamicSememeUsageDescriptionImpl;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeUUIDImpl;
import gov.vha.isaac.ochre.model.sememe.version.ComponentNidSememeImpl;
import gov.vha.isaac.ochre.model.sememe.version.DescriptionSememeImpl;
import gov.vha.isaac.ochre.model.sememe.version.DynamicSememeImpl;
import gov.vha.isaac.ochre.model.sememe.version.LogicGraphSememeImpl;
import gov.vha.isaac.ochre.model.sememe.version.LongSememeImpl;
import gov.vha.isaac.ochre.model.sememe.version.StringSememeImpl;

/**
 * 
 * {@link ComponentDumperDiagnosticUtil}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
public class ComponentDumperDiagnosticUtil {
	private static Logger log = LogManager.getLogger(ComponentDumperDiagnosticUtil.class);
	
	private ComponentDumperDiagnosticUtil() {}

	/**
	 * Entry method for recursively dumping sememe chronology
	 * @param chronology
	 */
	public static void dumpSememeChronology(PrintStream printStream, SememeChronology<? extends SememeVersion<?>> chronology, StampCoordinate stampCoordinate, boolean requireVersions, boolean recurse, boolean uuidsOnly, boolean dynamicSememeDetails, boolean nonRecursiveStampCoord) {
		printStream.println("Found " + Frills.describeExistingSememe(chronology, uuidsOnly));
	
		dumpSememe(printStream, 0, "SEMEME", chronology, stampCoordinate, requireVersions, recurse, uuidsOnly, dynamicSememeDetails, nonRecursiveStampCoord);
	}

	/**
	 * Recursively dump sememe characteristics based on type
	 * 
	 * @param indentLevel
	 * @param label
	 * @param chronology
	 */
	public static void dumpSememe(PrintStream printStream, int indentLevel, String label, SememeChronology<? extends SememeVersion<?>> chronology, StampCoordinate stampCoordinate, boolean requireVersions, boolean recurse, boolean uuidsOnly, boolean dynamicSememeCols, boolean nonRecursiveStampCoord) {	
		int indentLevelToUse = indentLevel;

		if (chronology.getSememeType() == SememeType.DESCRIPTION) {
			@SuppressWarnings("unchecked")
			SememeChronology<? extends DescriptionSememe<?>> descriptionSememeChronology = (SememeChronology<? extends DescriptionSememe<?>>)chronology;
			dumpDescriptionChronology(printStream, indentLevelToUse, "DESCRIPTION " + label, descriptionSememeChronology, stampCoordinate, requireVersions, recurse, uuidsOnly, dynamicSememeCols, nonRecursiveStampCoord);
			return;
		}
		List<SememeVersion<?>> sememeVersions = new LinkedList<>();
		if (stampCoordinate == null) {
			for (SememeVersion<?> sememeVersion : chronology.getVersionList()) {
				sememeVersions.add(sememeVersion);
			}
			if (sememeVersions.size() > 0) {
				printStream.println(getIndent(indentLevelToUse) + label + ": ALL " + sememeVersions.size() + " VERSION(S) OF " + Frills.describeExistingSememe(chronology, uuidsOnly));
			} else {
				log.error("PASSED " + Frills.describeExistingSememe(chronology, uuidsOnly) + " AND WHICH HAS NO VERSIONS");
				if (requireVersions) {
					printStream.println(getIndent(indentLevelToUse) + label + ": NOT FOUND " + Frills.describeExistingSememe(chronology, uuidsOnly)+ " AND WHICH HAS NO VERSIONS");
					return;
				} else {
					printStream.println(getIndent(indentLevelToUse) + label + ": FOUND " + Frills.describeExistingSememe(chronology, uuidsOnly) + " AND WHICH HAS NO VERSIONS");
				}
			}
		} else {
			@SuppressWarnings({ "rawtypes", "unchecked" })
			Optional<LatestVersion<SememeVersion<?>>> optionalLatestSememeVersion = ((SememeChronology)chronology).getLatestVersion(SememeVersion.class, stampCoordinate);
			if (optionalLatestSememeVersion.isPresent()) {
				if (optionalLatestSememeVersion.get().contradictions().isPresent()) {
					// TODO Handle contradictions
				}
				
				sememeVersions.add(optionalLatestSememeVersion.get().value());
				printStream.println(getIndent(indentLevelToUse) + label + ": FOUND LATEST VERSION OF " + Frills.describeExistingSememe(chronology, uuidsOnly) + " ACCORDING TO PASSED COORDINATE: " + stampCoordinate);
			} else {
				if (requireVersions) {
					printStream.println(getIndent(indentLevelToUse) + label + ": NOT DUMPING " + Frills.describeExistingSememe(chronology, uuidsOnly) + " AND WITH NO VERSIONS CONFORMING TO PASSED STAMP COORDINATE: " + stampCoordinate);
					return;
				} else {
					printStream.println(getIndent(indentLevelToUse) + label + ": FOUND " + Frills.describeExistingSememe(chronology, uuidsOnly) + " AND WITH NO VERSIONS CONFORMING TO PASSED STAMP COORDINATE: " + stampCoordinate);
				}
			}
		}

		indentLevelToUse++;

		printStream.println(getIndent(indentLevelToUse) + label + ":      UUID(S)=" + Arrays.toString(chronology.getUuids()));
		if (! uuidsOnly) {
			printStream.println(getIndent(indentLevelToUse) + label + ":   SEMEME SEQ=" + chronology.getSememeSequence());
		}
		printStream.println(getIndent(indentLevelToUse) + label + ":   ASSEMBLAGE=" + Frills.describeExistingConcept(chronology.getAssemblageSequence(), uuidsOnly));
		printStream.println(getIndent(indentLevelToUse) + label + ": COMMIT STATE=" + chronology.getCommitState());
		printStream.println(getIndent(indentLevelToUse) + label + ": NUM VERSIONS=" + chronology.getVersionList().size());
		printStream.println(getIndent(indentLevelToUse) + label + ":  NUM SEMEMES=" + chronology.getSememeList().size());
	
		Set<Integer> modules = Frills.getAllModuleSequences(chronology);
		printStream.println(getIndent(indentLevel + 1) + label + ":  NUM MODULES=" + modules.size());
		int moduleIndex = 1;
		for (int moduleSequence : modules) {
			printStream.println(getIndent(indentLevel + 2) + "MODULE #" + moduleIndex + ": " + Frills.describeExistingConcept(moduleSequence, uuidsOnly));
		}
		DynamicSememeUsageDescription dynamicSememeUsageDescription = null;
		DynamicSememeColumnInfo[] dynamicSememeColumnInfo = null;
		if (DynamicSememeUsageDescriptionImpl.isDynamicSememe(chronology.getAssemblageSequence())) {
			dynamicSememeUsageDescription = DynamicSememeUsageDescriptionImpl.read(chronology.getAssemblageSequence());
			printStream.println(getIndent(indentLevelToUse) + label + ":   DYN SEMEME:   USAGE DESCRIPTION: " + Frills.describeExistingConcept(dynamicSememeUsageDescription.getDynamicSememeUsageDescriptorSequence(), uuidsOnly));
			printStream.println(getIndent(indentLevelToUse) + label + ":   DYN SEMEME:    USAGE DESCRIPTOR: " + dynamicSememeUsageDescription.getDynamicSememeUsageDescription());
			printStream.println(getIndent(indentLevelToUse) + label + ":   DYN SEMEME:         SEMEME NAME: " + dynamicSememeUsageDescription.getDynamicSememeName());
			if (dynamicSememeUsageDescription.getReferencedComponentTypeRestriction() != null) {
				printStream.println(getIndent(indentLevelToUse) + label + ":   DYN SEMEME:    TYPE RESTRICTION: " + dynamicSememeUsageDescription.getReferencedComponentTypeRestriction());
			}
			if (dynamicSememeUsageDescription.getReferencedComponentTypeSubRestriction() != null) {
				printStream.println(getIndent(indentLevelToUse) + label + ":   DYN SEMEME: SUBTYPE RESTRICTION: " + dynamicSememeUsageDescription.getReferencedComponentTypeSubRestriction());
			}
			dynamicSememeColumnInfo = dynamicSememeUsageDescription.getColumnInfo();
			printStream.println(getIndent(indentLevelToUse) + label + ":   DYN SEMEME:       NUM COLUMNS: " + dynamicSememeColumnInfo.length);

			if (dynamicSememeCols) {
				if (dynamicSememeColumnInfo.length > 0) {
					Arrays.sort(dynamicSememeColumnInfo);
					for (int colIndex = 0; colIndex < dynamicSememeColumnInfo.length; ++colIndex) {
						DynamicSememeColumnInfo col = dynamicSememeColumnInfo[colIndex];

						printStream.println(getIndent(indentLevelToUse) + label + ":                          COL #" + col.getColumnOrder() + ":");
						printStream.println(getIndent(indentLevelToUse) + label + ":                              COL #" + col.getColumnOrder() + "         NAME: " + col.getColumnName());
						printStream.println(getIndent(indentLevelToUse) + label + ":                              COL #" + col.getColumnOrder() + "  DESCRIPTION: " + col.getColumnDescription());
						printStream.println(getIndent(indentLevelToUse) + label + ":                              COL #" + col.getColumnOrder() + " DESC CONCEPT: " + Frills.describeExistingConcept(col.getColumnDescriptionConcept(), uuidsOnly));
						printStream.println(getIndent(indentLevelToUse) + label + ":                              COL #" + col.getColumnOrder() + "    DATA TYPE: " + col.getColumnDataType());
						if (col.getDefaultColumnValue() != null) {
							printStream.println(getIndent(indentLevelToUse) + label + ":                              COL #" + col.getColumnOrder() + "      DEFAULT: " + col.getDefaultColumnValue().dataToString());
						}
						printStream.println(getIndent(indentLevelToUse) + label + ":                              COL #" + col.getColumnOrder() + "  IS REQUIRED: " + col.isColumnRequired());
					}
				}
			}
		}
		String sememeNameFromUsageDescription = dynamicSememeUsageDescription != null ? dynamicSememeUsageDescription.getDynamicSememeName() : null;
		String sememeDescriptionFromUsageDescription = dynamicSememeUsageDescription != null ? dynamicSememeUsageDescription.getDynamicSememeUsageDescription() : null;

		int versionIndex = 1;
		for (SememeVersion<?> sememeVersion : sememeVersions) {
			int additionalIndent = 1;
			printStream.println(getIndent(indentLevelToUse) + label + ": sememe version #" + versionIndex + " of " + sememeVersions.size() + ":");

			// Switch on SememeType
			switch(chronology.getSememeType()) {
			case COMPONENT_NID:
				// COMPONENT_NID sememe contains a component NID, often referred-to as "target" NID
				printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":        TGT: " + Frills.describeExistingConcept(((ComponentNidSememeImpl)sememeVersion).getComponentNid(), uuidsOnly));
				break;
//			case DESCRIPTION:
//				// DESCRIPTION sememe contains text, author, description type, etc. Handled by specialized method.
//				@SuppressWarnings("unchecked")
//				SememeChronology<? extends DescriptionSememe<?>> descriptionChronology = (SememeChronology<? extends DescriptionSememe<?>>)chronology;
//				dumpDescriptionChronology(printStream, indentLevelToUse + additionalIndent, "DESCRIPTION " + label + ": VERSION #" + versionIndex, descriptionChronology, stampCoordinate, requireVersions, recurse);
//				//printStream.println(getIndent(indentLevel) + label + "       TEXT: " + ((DescriptionSememeImpl)sememeVersion).getText());
//				break;
			case DYNAMIC: {
				DynamicSememeImpl dynamicSememeVersion = (DynamicSememeImpl)sememeVersion;
	
				// Dynamic sememes contain an array of data
				DynamicSememeData[] data = dynamicSememeVersion.getData();
				// Only dynamic sememes specifying membership in the assemblage group should have no other data
				if (data.length == 0) {
					// MEMBERSHIP IN ASSEMBLAGE INDICATED BY DYNAMIC SEMEME ONLY
					printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":       DYNAMIC SEMEME ONLY INDICATES MEMBERSHIP IN ASSEMBLAGE " + Frills.describeExistingConcept(chronology.getAssemblageSequence(), uuidsOnly));
				} else {
					// Dump data array, element by element
					printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":       " + data.length + " DATA ELEMENTS FOR DYNAMIC SEMEME" + (sememeNameFromUsageDescription != null ? " " + sememeNameFromUsageDescription : "") + (sememeDescriptionFromUsageDescription != null ? " (" + sememeDescriptionFromUsageDescription + ")" : "") + ":");
					for (int index = 0; index < data.length; ++index) {
						// Get values from usage description, if available
						String columnNameFromUsageDescription = (dynamicSememeColumnInfo != null && dynamicSememeColumnInfo[index] != null) ? dynamicSememeColumnInfo[index].getColumnName() : null;
						String columnDescriptionFromUsageDescription = (dynamicSememeColumnInfo != null && dynamicSememeColumnInfo[index] != null) ? dynamicSememeColumnInfo[index].getColumnDescription() : null;
						printStream.print(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":       " + getIndent(1) + "ELEMENT " + (index + 1) + " (TYPE=" + (data[index] != null ? data[index].getDynamicSememeDataType() : null) + (columnNameFromUsageDescription != null ? ", NAME=\""+ columnNameFromUsageDescription + "\"" : "") + (columnDescriptionFromUsageDescription != null ? ", DESC=\""+ columnDescriptionFromUsageDescription + "\"" : "") + "): ");
						if (data[index] == null) {
							printStream.println("null");
							continue;
						}
						// Display based on data element type
						switch (data[index].getDynamicSememeDataType()) {
						case NID:
							// Display the description for the concept corresponding to the contained NID
							printStream.println(Frills.describeExistingConcept(((DynamicSememeNid)data[index]).getDataNid(), uuidsOnly));
							break;
						case UUID:
							// Display the description for the concept corresponding to the contained UUID
							printStream.println(Frills.describeExistingConcept(((DynamicSememeUUIDImpl)data[index]).getDataUUID(), uuidsOnly));
							break;
						case UNKNOWN:
						case DOUBLE:
						case BOOLEAN:
						case FLOAT:
						case INTEGER:
						case LONG:
						case STRING:
						case ARRAY:
						case BYTEARRAY:
						case POLYMORPHIC:
							// Let built-in method display data for these types
							printStream.println(data[index].dataToString());
							break;
						default:
							break;
						}
					}
				}
				break;
			}
			case LOGIC_GRAPH: {
				LogicGraphSememeImpl logicGraphSememeVersion = (LogicGraphSememeImpl)sememeVersion;
				Set<Integer> parentSequences = Frills.getParentConceptSequencesFromLogicGraph(logicGraphSememeVersion);
				printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ": " + parentSequences.size() + " PARENT(S) INDICATED BY LOGIC GRAPH:");
				int i = 0;
				for (int parentSequence : parentSequences) {
					i++;
					printStream.println(getIndent(indentLevelToUse + additionalIndent + 1) + "       PARENT " + i + ": " + Frills.describeExistingConcept(parentSequence, uuidsOnly));
				}
				break;
			}
			case MEMBER:
				printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ": ASSEMBLAGE: " + Frills.describeExistingConcept(sememeVersion.getAssemblageSequence(), uuidsOnly));
				break;
			case LONG:
				printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":       LONG: " + ((LongSememeImpl)sememeVersion).getLongValue());
				break;
			case RELATIONSHIP_ADAPTOR:
				printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":     ORIGIN: " + Frills.describeExistingConcept(((RelationshipVersionAdaptor<?>)sememeVersion).getOriginSequence(), uuidsOnly));
				printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":DESTINATION: " + Frills.describeExistingConcept(((RelationshipVersionAdaptor<?>)sememeVersion).getDestinationSequence(), uuidsOnly));
				printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":      GROUP: " + ((RelationshipVersionAdaptor<?>)sememeVersion).getGroup());
				printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":       TYPE: " + Frills.describeExistingConcept(((RelationshipVersionAdaptor<?>)sememeVersion).getTypeSequence(), uuidsOnly));
				break;
			case STRING:
				printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":       TEXT: " + ((StringSememeImpl)sememeVersion).getString());
				break;
			case UNKNOWN:
				break;
			default:
				break;
			}
			printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":     MODULE: " + Frills.describeExistingConcept(sememeVersion.getModuleSequence(), uuidsOnly));
			printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":      STATE: " + sememeVersion.getState());
			printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":       DATE: " + new Date(sememeVersion.getTime()));
			printStream.println(getIndent(indentLevelToUse + additionalIndent) + "VERSION #" + versionIndex + ":     AUTHOR: " + Frills.describeExistingConcept(sememeVersion.getAuthorSequence(), uuidsOnly));
		}
	
		if (recurse && chronology.getSememeList().size() > 0) {
			printStream.println(getIndent(indentLevelToUse) + label + ": " + chronology.getSememeList().size() + " NESTED SEMEMES: ");
			dumpNonDescriptionSememes(printStream, indentLevelToUse + 1, chronology, (nonRecursiveStampCoord ? null : stampCoordinate), requireVersions, recurse, uuidsOnly, dynamicSememeCols, false);
		}
	}

	/**
	 * Recursively dump non-description sememes attached to the passed object chronology
	 * 
	 * @param indentLevel
	 * @param componentChronology
	 */
	public static void dumpNonDescriptionSememes(PrintStream printStream, int indentLevel, ObjectChronology<?> componentChronology, StampCoordinate stampCoordinate, boolean requireVersions, boolean recurse, boolean uuidsOnly, boolean dynamicSememeDetails, boolean nonRecursiveStampCoord) {
		int sememeNum = 0;
		// Iterate all sememes attached to component
		for (SememeChronology<? extends SememeVersion<?>> sememeChronology : componentChronology.getSememeList()) {
			// Dump attached non-description sememe characteristics,
			// ignoring description sememes

			boolean skipDueToStampCoordinate = false;
			if (stampCoordinate != null) {
				skipDueToStampCoordinate = ! Frills.getLatestVersion((SememeChronology)sememeChronology, SememeVersion.class, stampCoordinate).isPresent();
			}
			if (sememeChronology.getSememeType() != SememeType.DESCRIPTION && ! skipDueToStampCoordinate) {
				sememeNum++;

				dumpSememe(printStream, indentLevel, "SEMEME #" + sememeNum, sememeChronology, (nonRecursiveStampCoord ? null : stampCoordinate), requireVersions, recurse, uuidsOnly, dynamicSememeDetails, false);
			}
		}
	}

	private static List<SememeChronology<? extends SememeVersion<?>>> getNonDescriptionSememes(ObjectChronology<?> chronology) {
		List<SememeChronology<? extends SememeVersion<?>>> nonDescriptionSememes = new LinkedList<>();
		for (SememeChronology<? extends SememeVersion<?>> sememeChronology : chronology.getSememeList()) {
			if (sememeChronology.getSememeType() != SememeType.DESCRIPTION) {
				nonDescriptionSememes.add(sememeChronology);
			}
		}
		return nonDescriptionSememes;
	}

	/**
	 * Recursively dump characteristics of passed description and all of its attached sememes.
	 * 
	 * @param indentLevel
	 * @param label
	 * @param descriptionChronology
	 */
	public static void dumpDescriptionChronology(PrintStream printStream, int indentLevel, String label, SememeChronology<? extends DescriptionSememe<?>> descriptionChronology, StampCoordinate stampCoordinate, boolean requireVersions, boolean recurse, boolean uuidsOnly, boolean dynamicSememeDetails, boolean nonRecursiveStampCoord) {		
		List<DescriptionSememe<?>> descriptionVersions = new LinkedList<>();
		if (stampCoordinate == null) {
			for (DescriptionSememe<?> sememeVersion : descriptionChronology.getVersionList()) {
				descriptionVersions.add(sememeVersion);
			}
			if (descriptionVersions.size() > 0) {
				printStream.println(getIndent(indentLevel) + label  + ": ALL " + descriptionVersions.size() + " VERSION(S) OF " + Frills.describeExistingSememe(descriptionChronology, uuidsOnly));
			} else {
				log.error("PASSED " + Frills.describeExistingSememe(descriptionChronology, uuidsOnly) + " AND WHICH HAS NO VERSIONS");
				if (requireVersions) {
					printStream.println(getIndent(indentLevel) + label  + ": NOT DUMPING " + Frills.describeExistingSememe(descriptionChronology, uuidsOnly) + " AND WHICH HAS NO VERSIONS");
					return;
				} else {
					printStream.println(getIndent(indentLevel) + label  + ": FOUND " + Frills.describeExistingSememe(descriptionChronology, uuidsOnly) + " AND WHICH HAS NO VERSIONS");
				}
			}
		} else {
			@SuppressWarnings({ "rawtypes", "unchecked" })
			Optional<LatestVersion<DescriptionSememe<?>>> optionalLatestSememeVersion = ((SememeChronology)descriptionChronology).getLatestVersion(DescriptionSememe.class, stampCoordinate);
			if (optionalLatestSememeVersion.isPresent()) {
				if (optionalLatestSememeVersion.get().contradictions().isPresent()) {
					// TODO Handle contradictions
				}
				
				descriptionVersions.add(optionalLatestSememeVersion.get().value());
				printStream.println(getIndent(indentLevel) + label  + ": FOUND LATEST VERSION OF " + Frills.describeExistingSememe(descriptionChronology, uuidsOnly) + " ACCORDING TO PASSED COORDINATE: " + stampCoordinate);
			} else {
				if (requireVersions) {
					printStream.println(getIndent(indentLevel) + label  + ": NOT DUMPING " + Frills.describeExistingSememe(descriptionChronology, uuidsOnly) + " AND WITH NO VERSIONS CONFORMING TO PASSED STAMP COORDINATE: " + stampCoordinate);
					return;
				} else {
					printStream.println(getIndent(indentLevel) + label  + ": FOUND " + Frills.describeExistingSememe(descriptionChronology, uuidsOnly) + " AND WITH NO VERSIONS CONFORMING TO PASSED STAMP COORDINATE: " + stampCoordinate);
				}
			}
		}

		printStream.println(getIndent(indentLevel + 1) + label + ":      UUID(S)=" + Arrays.toString(descriptionChronology.getUuids()));
		if (! uuidsOnly) {
			printStream.println(getIndent(indentLevel + 1) + label + ":   SEMEME SEQ=" + descriptionChronology.getSememeSequence());
		}
		printStream.println(getIndent(indentLevel + 1) + label + ":   ASSEMBLAGE=" + Frills.describeExistingConcept(descriptionChronology.getAssemblageSequence(), uuidsOnly));
		printStream.println(getIndent(indentLevel + 1) + label + ": COMMIT STATE=" + descriptionChronology.getCommitState());
		printStream.println(getIndent(indentLevel + 1) + label + ": NUM VERSIONS=" + descriptionChronology.getVersionList().size());
		printStream.println(getIndent(indentLevel + 1) + label + ":  NUM SEMEMES=" + descriptionChronology.getSememeList().size());

		Set<Integer> modules = Frills.getAllModuleSequences(descriptionChronology);
		printStream.println(getIndent(indentLevel + 1) + label + ": NUM MODULES: " + modules.size());
		int moduleIndex = 1;
		for (int moduleSequence : modules) {
			printStream.println(getIndent(indentLevel + 2) + "MODULE #" + moduleIndex++ + ": " + Frills.describeExistingConcept(moduleSequence, uuidsOnly));
		}

		// Get all versions of passed description
		int versionNum = 0;
		if (stampCoordinate == null) {
			printStream.println(getIndent(indentLevel + 1) + label + ": ALL " + descriptionVersions.size() + " DESCRIPTION VERSION(S):");
		} else {
			printStream.println(getIndent(indentLevel + 1) + label + ": LATEST DESCRIPTION VERSION ACCORDING TO PASSED COORDINATE " + stampCoordinate + ":");
		}
		
		for (DescriptionSememe<?> version : descriptionVersions) {
			versionNum++;
			// Dump characteristics of current description version
			printStream.println(getIndent(indentLevel + 2) + "DESCRIPTION VERSION #" + versionNum + ":");
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ":  DESC TYPE: " + Frills.describeExistingConcept(version.getDescriptionTypeConceptSequence(), uuidsOnly));
			//printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ": ASSEMBLAGE: " + Frills.describeExistingConcept(version.getAssemblageSequence()));
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ":     MODULE: " + Frills.describeExistingConcept(version.getModuleSequence(), uuidsOnly));
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ":      STATE: " + version.getState());
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ":       TEXT: " + version.getText());
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ":       DATE: " + new Date(version.getTime()));
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ":     AUTHOR: " + Frills.describeExistingConcept(version.getAuthorSequence(), uuidsOnly));
		}
	
		if (recurse) {
			// Get and dump all sememes nested on description
			printStream.println(getIndent(indentLevel + 1) + label + ": ALL " + getNonDescriptionSememes(descriptionChronology).size() + " NON-DESCRIPTION SEMEMES:");
			dumpNonDescriptionSememes(printStream, indentLevel + 2, descriptionChronology, (nonRecursiveStampCoord ? null : stampCoordinate), requireVersions, recurse, uuidsOnly, dynamicSememeDetails, false);
		}
	}

	/**
	 * Recursively dump characteristics of all description sememes attached to the concept,
	 * including all versions of each description.
	 * 
	 * @param indentLevel - for indented formatting
	 * @param conceptChronology - passed concept chronology
	 */
	public static void dumpDescriptions(PrintStream printStream, int indentLevel, ConceptChronology<? extends ConceptVersion<?>> conceptChronology, StampCoordinate stampCoordinate, boolean requireVersions, boolean recurse, boolean uuidsOnly, boolean dynamicSememeDetails, boolean nonRecursiveStampCoord) {
		int descriptionIndex = 0;
		// For each description attached to the passed concept chronology
		for (SememeChronology<? extends DescriptionSememe<?>> descriptionChronology : conceptChronology.getConceptDescriptionList()) {
			boolean skipDueToStampCoord = false;
			if (stampCoordinate != null) {
				@SuppressWarnings("unchecked")
				SememeChronology<DescriptionSememeImpl> descriptionSememeImplChronology = (SememeChronology<DescriptionSememeImpl>)descriptionChronology;
				skipDueToStampCoord = ! Frills.getLatestVersion(descriptionSememeImplChronology, stampCoordinate).isPresent();
			}
			if (! skipDueToStampCoord) {
				descriptionIndex++;
				// Recursively dump the characteristics of the current description,
				// including all versions and attached sememes
				dumpDescriptionChronology(printStream, indentLevel + 1, "DESCRIPTION #" + descriptionIndex, descriptionChronology, (nonRecursiveStampCoord ? null : stampCoordinate), requireVersions, recurse, uuidsOnly, dynamicSememeDetails, false);
			}
		}
	}
	
	/**
	 * Recursively dump to a PrintStream the characteristics of the passed concept chronology.
	 * First, recursively dump attached description sememes, then non-description sememes.
	 * 
	 * @param indentLevel - for indented formatting
	 * @param conceptChronology - passed concept chronology
	 */
	public static void dumpConceptChronology(
			PrintStream printStream,
			int indentLevel,
			ConceptChronology<? extends ConceptVersion<?>> conceptChronology,
			StampCoordinate stampCoordinate,
			LanguageCoordinate languageCoordinate,
			boolean requireVersions,
			boolean recurse,
			boolean listChildren,
			boolean uuidsOnly,
			boolean dynamicSememeDetails,
			boolean nonRecursiveStampCoord) {
		
		List<ConceptVersion<?>> conceptVersions = new LinkedList<>();
		if (stampCoordinate == null) {
			for (ConceptVersion<?> conceptVersion : conceptChronology.getVersionList()) {
				conceptVersions.add(conceptVersion);
			}
			if (conceptVersions.size() > 0) {
				printStream.println(getIndent(indentLevel) + "ALL " + conceptVersions.size() + " VERSION(S) OF " + Frills.describeExistingConcept(conceptChronology, uuidsOnly));
			} else {
				log.error("PASSED " + Frills.describeExistingConcept(conceptChronology, uuidsOnly) + " HAS NO VERSIONS");
				if (requireVersions) {
					printStream.println(getIndent(indentLevel) + "NOT DUMPING " + Frills.describeExistingConcept(conceptChronology, uuidsOnly) + " WITH NO VERSIONS");
					return;
				} else {
					printStream.println(getIndent(indentLevel) + "FOUND " + Frills.describeExistingConcept(conceptChronology, uuidsOnly) + " WITH NO VERSIONS");
				}
			}
		} else {
			@SuppressWarnings({ "rawtypes", "unchecked" })
			Optional<LatestVersion<ConceptVersion<?>>> optionalLatestConceptVersion = ((ConceptChronology)conceptChronology).getLatestVersion(ConceptVersion.class, stampCoordinate);
			if (optionalLatestConceptVersion.isPresent()) {
				if (optionalLatestConceptVersion.get().contradictions().isPresent()) {
					// TODO Handle contradictions
				}
				
				conceptVersions.add(optionalLatestConceptVersion.get().value());
				
				// Display concept default description text
				printStream.println(getIndent(indentLevel) + "FOUND LATEST VERSION OF " + Frills.describeExistingConcept(conceptChronology, uuidsOnly) + " ACCORDING TO PASSED COORDINATE :" + stampCoordinate);
			} else {
				if (requireVersions) {
					printStream.println(getIndent(indentLevel) + "NOT DUMPING " + Frills.describeExistingConcept(conceptChronology, uuidsOnly) + " WITH NO VERSIONS CONFORMING TO PASSED STAMP COORDINATE: " + stampCoordinate);
					return;
				} else {
					printStream.println(getIndent(indentLevel) + "FOUND " + Frills.describeExistingConcept(conceptChronology, uuidsOnly) + " WITH NO VERSIONS CONFORMING TO PASSED STAMP COORDINATE: " + stampCoordinate);
				}
			}
		}

		printStream.println(getIndent(indentLevel + 1) + "CONCEPT          UUID(S)=" + Arrays.toString(conceptChronology.getUuids()));
		if (! uuidsOnly) {
			printStream.println(getIndent(indentLevel + 1) + "CONCEPT      CONCEPT SEQ=" + conceptChronology.getConceptSequence());
		}
		printStream.println(getIndent(indentLevel + 1) + "CONCEPT     COMMIT STATE=" + conceptChronology.getCommitState());
		printStream.println(getIndent(indentLevel + 1) + "CONCEPT     NUM VERSIONS=" + conceptChronology.getVersionList().size());
		printStream.println(getIndent(indentLevel + 1) + "CONCEPT NUM DESCRIPTIONS=" + conceptChronology.getConceptDescriptionList().size());
		printStream.println(getIndent(indentLevel + 1) + "CONCEPT      NUM SEMEMES=" + conceptChronology.getSememeList().size());

		Optional<LatestVersion<DescriptionSememe<?>>> fullySpecifiedDescription = conceptChronology.getFullySpecifiedDescription(languageCoordinate != null ? languageCoordinate : LanguageCoordinates.getUsEnglishLanguageFullySpecifiedNameCoordinate(), stampCoordinate != null ? stampCoordinate : StampCoordinates.getDevelopmentLatest());
		if (fullySpecifiedDescription.isPresent()) {
			printStream.println(getIndent(indentLevel + 1) + "CONCEPT  FSN DESCRIPTION=" + (fullySpecifiedDescription.isPresent() ? (fullySpecifiedDescription.get().value().getText() + " (UUID=" + fullySpecifiedDescription.get().value().getPrimordialUuid() + ")") : "(NOT FOUND)"));
		} else {
			printStream.println(getIndent(indentLevel + 1) + "CONCEPT  FSN DESCRIPTION=(NOT FOUND)");

			log.error("NO FSN FOUND FOR " + Frills.describeExistingConcept(conceptChronology, uuidsOnly));
		}
		Optional<LatestVersion<DescriptionSememe<?>>> preferredDescription = conceptChronology.getPreferredDescription(languageCoordinate != null ? languageCoordinate : LanguageCoordinates.getUsEnglishLanguagePreferredTermCoordinate(), stampCoordinate != null ? stampCoordinate : StampCoordinates.getDevelopmentLatest());
		printStream.println(getIndent(indentLevel + 1) + "CONCEPT PREF DESCRIPTION=" + (preferredDescription.isPresent() ? preferredDescription.get().value().getText() + " (UUID=" + preferredDescription.get().value().getPrimordialUuid() + ")" : "(NOT FOUND)"));


		if (DynamicSememeUsageDescriptionImpl.isDynamicSememe(conceptChronology.getConceptSequence())) {
			DynamicSememeUsageDescription dynamicSememeUsageDescriptionObject = DynamicSememeUsageDescriptionImpl.read(conceptChronology.getConceptSequence());
			String dynamicSememeUsageDescription = dynamicSememeUsageDescriptionObject.getDynamicSememeUsageDescription();
			String dynamicSememeName = dynamicSememeUsageDescriptionObject.getDynamicSememeName();
			printStream.println(getIndent(indentLevel + 1) + "     DYNAMIC SEMEME NAME=" + dynamicSememeName);
			printStream.println(getIndent(indentLevel + 1) + "    DYNAMIC SEMEME USAGE=" + dynamicSememeUsageDescription);
			printStream.println(getIndent(indentLevel + 1) + "              NUM USAGES=" + Get.sememeService().getSememeSequencesFromAssemblage(conceptChronology.getConceptSequence()).size());
		}

		Set<Integer> modules = Frills.getAllModuleSequences(conceptChronology);
		printStream.println(getIndent(indentLevel + 1) + "CONCEPT      NUM MODULES=" + modules.size());
		int moduleIndex = 1;
		for (int moduleSequence : modules) {
			printStream.println(getIndent(indentLevel + 2) + "MODULE #" + moduleIndex + ": " + Frills.describeExistingConcept(moduleSequence, uuidsOnly));
		}
		
		Set<Integer> children = Frills.getAllChildrenOfConcept(conceptChronology.getConceptSequence(), false, false);
		printStream.print(getIndent(indentLevel + 1) + "CONCEPT HAS " + children.size() + " CHILDREN");
		if (listChildren && children.size() > 0) {
			printStream.println(":");
			int childIndex = 1;
			for (int childConceptSequence : children) {
				printStream.println(getIndent(indentLevel + 2) + "CHILD CONCEPT #" + childIndex++ + ": " + Frills.describeExistingConcept(childConceptSequence, uuidsOnly));
			}
		} else {
			printStream.println();
		}

		printStream.println(getIndent(indentLevel + 1) + conceptVersions.size() + " CONCEPT VERSION(S):");
		int versionNum = 0;
		for (ConceptVersion<?> conceptVersion : conceptVersions) {
			versionNum++;
			printStream.println(getIndent(indentLevel + 2) + "CONCEPT VERSION #" + versionNum + ":");
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ": MODULE: " + Frills.describeExistingConcept(conceptVersion.getModuleSequence(), uuidsOnly));
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ":  STATE: " + conceptVersion.getState());
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ":   DATE: " + new Date(conceptVersion.getTime()));
			printStream.println(getIndent(indentLevel + 3) + "VERSION #" + versionNum + ": AUTHOR: " + Frills.describeExistingConcept(conceptVersion.getAuthorSequence(), uuidsOnly));
		}

		if (recurse) {
			// Get all description sememes attached to the passed concept chronology
			List<SememeChronology<? extends DescriptionSememe<?>>> descriptionList = conceptChronology.getConceptDescriptionList();
			printStream.println(getIndent(indentLevel + 1) + "ALL " + descriptionList.size() + " DESCRIPTIONS FOR " + Frills.describeExistingConcept(conceptChronology, uuidsOnly));

			// Recursively dump all description sememes attached to the passed concept chronology
			dumpDescriptions(printStream, indentLevel + 1, conceptChronology, (nonRecursiveStampCoord ? null : stampCoordinate), requireVersions, recurse, uuidsOnly, dynamicSememeDetails, false);

			printStream.println(getIndent(indentLevel + 1) + "NON-DESCRIPTION SEMEMES ON " + Frills.describeExistingConcept(conceptChronology, uuidsOnly));
			// Recursively dump all non-description sememes attached to passed concept chronology
			dumpNonDescriptionSememes(printStream, indentLevel + 2, conceptChronology, (nonRecursiveStampCoord ? null : stampCoordinate), requireVersions, recurse, uuidsOnly, dynamicSememeDetails, false);
		}
	}

	/**
	 * Recursively dump to a PrintStream the characteristics of the passed concept chronology.
	 * First, recursively dump attached description sememes, then non-description sememes.
	 * 
	 * @param conceptChronology - passed concept chronology
	 */
	public static void dumpConceptChronology(PrintStream printStream, ConceptChronology<? extends ConceptVersion<?>> conceptChronology, StampCoordinate stampCoordinate, LanguageCoordinate languageCooridinate, boolean requireVersions, boolean recurse, boolean listChildren, boolean uuidsOnly, boolean dynamicSememeDetails, boolean nonRecursiveStampCoord) {
		dumpConceptChronology(printStream, 0, conceptChronology, stampCoordinate, languageCooridinate, requireVersions, recurse, listChildren, uuidsOnly, dynamicSememeDetails, nonRecursiveStampCoord);
	}

	/**
	 * Helper method for indented formatting
	 * @param indentLevel
	 * @return
	 */
	private static String getIndent(int indentLevel) {
		StringBuffer buffer = new StringBuffer();
		for (int indent = 0; indent < indentLevel; ++indent) {
			buffer.append("\t");
		}
		return buffer.toString();
	}

//	private String spacesForStringLength(String str) {
//		return str.replaceAll(".", " ");
//	}

	private static String indent(String str, int level) {
		String indent = getIndent(level);

		StringBuilder outputBuilder = new StringBuilder();
		
		String[] rows = str.split("\n");
		for (int i = 0; i < rows.length; ++i) {
			if (!StringUtils.isBlank(rows[i])) {
				outputBuilder.append(indent + rows[i]);
			}
			if (i < rows.length - 1) {
				outputBuilder.append("\n");
			}
		}
		
		return outputBuilder.toString();
	}

//	private static String prefix(String str, String prefix) {
//		StringBuilder outputBuilder = new StringBuilder();
//		
//		String[] rows = str.split("\n");
//		for (int i = 0; i < rows.length; ++i) {
//			if (!StringUtils.isBlank(rows[i])) {
//				outputBuilder.append(prefix + rows[i]);
//			}
//			if (i < rows.length - 1) {
//				outputBuilder.append("\n");
//			}
//		}
//		
//		return outputBuilder.toString();
//	}

	public static void main(String...args) {
		String testString1 = "row 1\n\t\trow 2\n\t\t\trow 3";
		String testString2 = "row 1\nrow 2\n";
		String testString3 = "\nrow 2\nrow 3";

		System.out.println();
		System.out.println(indent(testString1, 3));
		System.out.println();
		System.out.println(indent(testString2, 3));
		System.out.println();
		System.out.println(indent(testString3, 3));
	}
}
