/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.ochre.api.util;

import java.util.Optional;
import java.util.UUID;

import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.chronicle.ObjectChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.version.SememeVersion;
import gov.vha.isaac.ochre.api.externalizable.OchreExternalizable;
import gov.vha.isaac.ochre.api.externalizable.OchreExternalizableObjectType;
import gov.vha.isaac.ochre.api.identity.StampedVersion;

/**
 * 
 * {@link LoggingUtils}
 *
 * @author <a href="mailto:joelk@joelk.com">joelk</a>
 *
 */
public class LoggingUtils {
	private LoggingUtils() {}

	public static String describeConcept(int id) {
		if (id >= 0) {
			Optional<? extends ConceptChronology<? extends ConceptVersion<?>>> optionalConcept = Get.conceptService().getOptionalConcept(id);
			if (optionalConcept.isPresent()) {
				return describe(optionalConcept.get());
			} else {
				return "NONEXISTENT CONCEPT INT ID=" + id;
			}
		} else {
			Optional<? extends ObjectChronology<? extends StampedVersion>> object = Get.identifiedObjectService().getIdentifiedObjectChronology(id);
			if (object.isPresent()) {
				if (object.get().getOchreObjectType() == OchreExternalizableObjectType.CONCEPT) {
					return describe(object.get());
				} else {
					return "NON CONCEPT " + describe(object.get());
				}
			} else {
				return "NONEXISTENT CONCEPT NID=" + id;
			}
		}
	}

	public static String describeSememe(int id) {
		if (id >= 0) {
			Optional<? extends SememeChronology<? extends SememeVersion<?>>> optionalSememe = Get.sememeService().getOptionalSememe(id);
			if (optionalSememe.isPresent()) {
				return describe(optionalSememe.get());
			} else {
				return "NONEXISTENT SEMEME INT ID=" + id;
			}
		} else {
			Optional<? extends ObjectChronology<? extends StampedVersion>> object = Get.identifiedObjectService().getIdentifiedObjectChronology(id);
			if (object.isPresent()) {
				if (object.get().getOchreObjectType() == OchreExternalizableObjectType.SEMEME) {
					return describe(object.get());
				} else {
					return "NON SEMEME " + describe(object.get());
				}
			} else {
				return "NONEXISTENT SEMEME NID=" + id;
			}
		}
	}

	public static String describe(UUID uuid) {
		if (Get.identifierService().hasUuid(uuid)) {
			Optional<? extends ObjectChronology<? extends StampedVersion>> object = Get.identifiedObjectService().getIdentifiedObjectChronology(Get.identifierService().getNidForUuids(uuid));
			if (object.isPresent()) {
				return describe(object.get());
			}
		}

		return "NONEXISTENT UUID=" + uuid;
	}

	public static String describe(OchreExternalizable ochreObject) {
		if (ochreObject == null) {
			return "NULL OBJECT";
		}

		String objectDescription = ochreObject.getOchreObjectType().toString();
		if (ochreObject instanceof ConceptChronology) {
			ConceptChronology<?> conceptChronology = (ConceptChronology<?>)ochreObject;
			try {
				objectDescription += " " + Get.conceptDescriptionOfAnyStateText(conceptChronology.getNid()) + " (" + conceptChronology.getPrimordialUuid() + ")";
			} catch (Exception e) {
				objectDescription += " " + conceptChronology.getPrimordialUuid();
			}
		} else if (ochreObject instanceof SememeChronology) {
			SememeChronology<?> sememeChronology = (SememeChronology<?>)ochreObject;
			try {
				objectDescription = sememeChronology.getSememeType() + " " + objectDescription + " " + sememeChronology.getPrimordialUuid() + " of assemblage " + describe(Get.conceptService().getConcept(sememeChronology.getAssemblageSequence())) + " attached to " + describe(Get.identifiedObjectService().getIdentifiedObjectChronology(sememeChronology.getReferencedComponentNid()).orElse(null));
			} catch (Exception e) {
				objectDescription = sememeChronology.getSememeType() + " " + objectDescription + " " + sememeChronology.getPrimordialUuid() + " of assemblage SEQ=" + sememeChronology.getAssemblageSequence() + " attached to component NID=" + sememeChronology.getReferencedComponentNid();
			}
		}

		return objectDescription;
	}
}
