/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.ochre.deployment.hapi.extension.hl7.message;

import static gov.vha.isaac.ochre.api.constants.Constants.DATA_STORE_ROOT_LOCATION_PROPERTY;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

import gov.vha.isaac.ochre.access.maint.deployment.dto.PublishChecksumMessage;
import gov.vha.isaac.ochre.access.maint.deployment.dto.PublishChecksumMessageDTO;
import gov.vha.isaac.ochre.access.maint.deployment.dto.PublishSiteDiscoveryMessage;
import gov.vha.isaac.ochre.access.maint.deployment.dto.PublishSiteDiscoveryMessageDTO;
import gov.vha.isaac.ochre.access.maint.deployment.dto.PublishUpdateMessage;
import gov.vha.isaac.ochre.access.maint.deployment.dto.PublishUpdateMessageDTO;
import gov.vha.isaac.ochre.access.maint.deployment.dto.Site;
import gov.vha.isaac.ochre.access.maint.deployment.dto.SiteDTO;
import gov.vha.isaac.ochre.api.ConfigurationService;
import gov.vha.isaac.ochre.api.LookupService;
import gov.vha.isaac.ochre.api.util.DBLocator;
import gov.vha.isaac.ochre.services.dto.publish.ApplicationProperties;
import gov.vha.isaac.ochre.services.dto.publish.HL7ApplicationProperties;
import gov.vha.isaac.ochre.services.dto.publish.HL7MessageProperties;
import gov.vha.isaac.ochre.services.dto.publish.MessageProperties;
import javafx.concurrent.Task;

/**
 * Various units tests for HL7 messaging module.
 *
 * {@link TestHL7Emulator}
 *
 * @author <a href="mailto:nmarques@westcoastinformatics.com">Nuno Marques</a>
 */
public class TestHL7Emulator
{
	private static final Logger LOG = LogManager.getLogger(TestHL7Emulator.class);
	private static ApplicationProperties applicationProperties_ = getDefaultServerProperties();

	@BeforeClass
	public static void initialize() throws Exception {
		HL7Messaging.enableListener(applicationProperties_);
	}

	@AfterClass
	public static void finish() throws Exception {
		LookupService.shutdownSystem();
	}


	//@Test
	public void testSendChecksumMessageGood() throws Throwable {
		
		String hl7Message = "Radiology Procedures";

		Site site = new SiteDTO();
		site.setId(1);
		site.setVaSiteId("950");
		site.setGroupName("");
		site.setName("STLVETSDEV");
		site.setType("");
		site.setMessageType("T");

		PublishChecksumMessage publishMessage = new PublishChecksumMessageDTO(4, site, hl7Message);

		List<PublishChecksumMessage> publishMessages = new ArrayList<PublishChecksumMessage>();
		publishMessages.add(publishMessage);

		Task<Void> t = HL7Messaging.checksum(publishMessages, getDefaultMessageProperties()).get(0);
		t.get();
	}



	//@Test
	public void testSendSiteDataMessageGood() throws Throwable {
		
		String hl7Message = "Vital Qualifiers";

		Site site = new SiteDTO();
		site.setId(1);
		site.setVaSiteId("950");
		site.setGroupName("");
		site.setName("STLVETSDEV");
		site.setType("");
		site.setMessageType("T");

		PublishSiteDiscoveryMessage publishMessage = new PublishSiteDiscoveryMessageDTO(9, site, hl7Message);

		List<PublishSiteDiscoveryMessage> publishMessages = new ArrayList<PublishSiteDiscoveryMessage>();
		publishMessages.add(publishMessage);

		Task<Void> t = HL7Messaging.discovery(publishMessages, getDefaultMessageProperties()).get(0);
		t.get();
	}
	
	//@Test
	public void testSendSiteUpdateMessageGoodFromFile() throws Throwable {
		
		String hl7Message = "Reactions";

		Site site = new SiteDTO();
		site.setId(1);
		site.setVaSiteId("950");
		site.setGroupName("");
		site.setName("STLVETSDEV");
		site.setType("");
		site.setMessageType("T");

		PublishUpdateMessage publishMessage = new PublishUpdateMessageDTO(4, site, hl7Message);
		publishMessage.setSubset("4712076");
		ClassLoader classLoader = getClass().getClassLoader();
		File file = new File(classLoader.getResource("hl7.update.msg.txt").getFile());
		System.out.println(file.getAbsolutePath());
		StringBuilder rawMessage = new StringBuilder();
				
		BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file)));
		String line;
		while ((line = reader.readLine()) != null) {
			rawMessage.append(line).append(Character.toString((char) 13));
		}
		reader.close();
		
		publishMessage.setRawHL7Message(rawMessage.toString());

		List<PublishUpdateMessage> publishMessages = new ArrayList<PublishUpdateMessage>();
		publishMessages.add(publishMessage);

		Task<Void> t = HL7Messaging.update(publishMessages, getDefaultMessageProperties()).get(0);

		t.get();
		
	}
	
	//@Test
	public void testSendSiteUpdateMessageGood() throws Throwable {
		
		isaacInit();
		
		String hl7Message = "Reactions";

		Site site = new SiteDTO();
		site.setId(1);
		site.setVaSiteId("950");
		site.setGroupName("");
		site.setName("STLVETSDEV");
		site.setType("");
		site.setMessageType("T");

		PublishUpdateMessage publishMessage = new PublishUpdateMessageDTO(4, site, hl7Message);
		publishMessage.setSubset("4712076");

		List<PublishUpdateMessage> publishMessages = new ArrayList<PublishUpdateMessage>();
		publishMessages.add(publishMessage);

		Task<Void> t = HL7Messaging.update(publishMessages, getDefaultMessageProperties()).get(0);
		t.get();

		isaacStop();
	}


	private static ApplicationProperties getDefaultServerProperties() {

		ApplicationProperties appProp = new HL7ApplicationProperties();

		// Application Server Message String
		appProp.setApplicationServerName("ISAAC");
		appProp.setApplicationVersion("8.4.0.0");

		// Listener Port
		appProp.setListenerPort(49990);

		// Sending Facility Site ID
		appProp.setSendingFacilityNamespaceId("200ET1");

		// Target Vitria Interface Engine
		appProp.setInterfaceEngineURL(
				"http://localhost:8081/isaac-hl7-emulator/EmulatorServlet");

		// Encoding type
		appProp.setHl7EncodingType("VB");

		appProp.setUseInterfaceEngine(false);

		return appProp;

	}

	private static MessageProperties getDefaultMessageProperties() {

		MessageProperties messageProperties = new HL7MessageProperties();

		// Settings used by the converter to configure the sending application.
		messageProperties.setSendingApplicationNamespaceIdUpdate("VETS UPDATE");
		messageProperties.setSendingApplicationNamespaceIdMD5("VETS MD5");
		messageProperties.setSendingApplicationNamespaceIdSiteData("VETS DATA");

		// Target Application at VistA sites
		messageProperties.setReceivingApplicationNamespaceIdUpdate("XUMF UPDATE");
		messageProperties.setReceivingApplicationNamespaceIdMD5("XUMFMD5");
		messageProperties.setReceivingApplicationNamespaceIdSiteData("XUMF DATA");

		// Message Version ID
		messageProperties.setVersionId("2.4");

		// acceptAcknowledgementType
		messageProperties.setAcceptAcknowledgementType("AL");
		messageProperties.setApplicationAcknowledgementType("AL");

		messageProperties.setCountryCode("USA");

		// MFI field values
		messageProperties.setMasterFileIdentifier("Standard Terminology");
		messageProperties.setNameOfCodingSystem("ERT");
		messageProperties.setFileLevelEventCode("MUP");
		messageProperties.setResponseLevelCode("NE");

		// MFE field values
		messageProperties.setRecordLevelEventCode("NE");

		// QRD field values
		messageProperties.setQueryFormatCode("R");
		messageProperties.setQueryPriority("I");
		messageProperties.setQueryId("Standard Terminology Query");
		messageProperties.setQueryLimitedRequestQuantity(99999);
		// appProp.setQueryLimitedRequestUnits();

		messageProperties.setQueryWhoSubjectFilterIdNumber("ALL");
		messageProperties.setQueryWhatDepartmentDataCodeIdentifier("VA");

		// CE static field values
		messageProperties.setSubFieldSeparator("@");

		return messageProperties;

	}
	

	
	/**
	 * Isaac init.
	 */
	private void isaacInit() {
		LOG.info("Isaac Init called");

		try {
			LOG.info("ISAAC Init thread begins");

			if (StringUtils.isBlank(System.getProperty(DATA_STORE_ROOT_LOCATION_PROPERTY))) {
				// if there isn't an official system property set, check this
				// one.
				String sysProp = System.getProperty("isaacDatabaseLocation");

				File dataStoreLocation = DBLocator.findDBFolder(new File(sysProp));

				if (!dataStoreLocation.exists()) {
					throw new RuntimeException("Couldn't find a data store from the input of '"
							+ dataStoreLocation.getAbsoluteFile().getAbsolutePath() + "'");
				}
				if (!dataStoreLocation.isDirectory()) {
					throw new RuntimeException(
							"The specified data store: '" + dataStoreLocation.getAbsolutePath() + "' is not a folder");
				}

				// use the passed in JVM parameter location
				LookupService.getService(ConfigurationService.class).setDataStoreFolderPath(dataStoreLocation.toPath());
				System.out.println("  Setup AppContext, data store location = " + dataStoreLocation.getAbsolutePath());
			}

			// status_.set("Starting ISAAC");
			LookupService.startupIsaac();

			// status_.set("Ready");
			System.out.println("Done setting up ISAAC");

		} catch (Exception e) {
			LOG.error("Failure starting ISAAC", e);
		}
	}

	/**
	 * Isaac stop.
	 */
	private void isaacStop() {
		LOG.info("Stopping ISAAC");
		LookupService.shutdownSystem();
		LOG.info("ISAAC stopped");
	}
}
