/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright 
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.ochre.deployment.hapi.extension.hl7.message;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.TaxonomyService;
import gov.vha.isaac.ochre.api.chronicle.LatestVersion;
import gov.vha.isaac.ochre.api.collections.ConceptSequenceSet;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptService;
import gov.vha.isaac.ochre.api.component.concept.ConceptSnapshot;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.DynamicSememe;
import gov.vha.isaac.ochre.api.component.sememe.version.SememeVersion;
import gov.vha.isaac.ochre.api.component.sememe.version.StringSememe;
import gov.vha.isaac.ochre.api.coordinate.StampCoordinate;
import gov.vha.isaac.ochre.api.coordinate.StampPrecedence;
import gov.vha.isaac.ochre.associations.AssociationInstance;
import gov.vha.isaac.ochre.associations.AssociationUtilities;
import gov.vha.isaac.ochre.impl.utility.Frills;
import gov.vha.isaac.ochre.model.configuration.LanguageCoordinates;
import gov.vha.isaac.ochre.model.coordinate.StampCoordinateImpl;
import gov.vha.isaac.ochre.model.coordinate.StampPositionImpl;
import gov.vha.isaac.ochre.services.business.DataChange.DataChangeType;
import gov.vha.isaac.ochre.services.dto.publish.MessageProperties;
import gov.vha.isaac.ochre.services.dto.publish.NameValueDTO;
import gov.vha.isaac.ochre.services.dto.publish.PublishConceptDTO;
import gov.vha.isaac.ochre.services.dto.publish.PublishRegionDTO;

/**
 * {@link HL7Update}
 *
 * @author <a href="mailto:nmarques@westcoastinformatics.com">Nuno Marques</a>
 */
public class HL7Update {

	private Logger log = LogManager.getLogger(HL7Update.class);

	private StampCoordinate STAMP_COORDINATES = null;

	private TaxonomyService ts = Get.taxonomyService();

	// TODO: Source all the following hardcoded UUID values from MetaData, once
	// available
	// ConceptChronology: VHAT Attribute Types <261>
	// uuid:8287530a-b6b0-594d-bf46-252e09434f7e
	// VHAT Metadata -> "Attribute Types"
	private final UUID vhatPropertyTypesUUID = UUID.fromString("8287530a-b6b0-594d-bf46-252e09434f7e");
	private final int vhatPropertyTypesNid = Get.identifierService().getNidForUuids(vhatPropertyTypesUUID);

	// conceptChronology: CODE (ISAAC) <77>
	// uuid:803af596-aea8-5184-b8e1-45f801585d17
	private final UUID codeAssemblageUUID = MetaData.CODE.getPrimordialUuid();
	private final int codeAssemblageConceptSeq = Get.identifierService().getConceptSequenceForUuids(codeAssemblageUUID);

	// ConceptChronology: Preferred Name (ISAAC) <257>
	// uuid:a20e5175-6257-516a-a97d-d7f9655916b8
	// VHAT Description Types -> Preferred Name
	private final UUID preferredNameExtendedType = UUID.fromString("a20e5175-6257-516a-a97d-d7f9655916b8");

	public List<PublishRegionDTO> exportForHL7Update(Long vuid, MessageProperties messageProperties) {

		log.info("HERE !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");

		STAMP_COORDINATES = new StampCoordinateImpl(StampPrecedence.PATH,
				new StampPositionImpl(System.currentTimeMillis(), MetaData.DEVELOPMENT_PATH.getConceptSequence()),
				ConceptSequenceSet.EMPTY, State.ANY_STATE_SET);

		List<PublishRegionDTO> pubRegionList = new ArrayList<>();

		pubRegionList.add(new PublishRegionDTO(messageProperties.getMasterFileIdentifier(), null));

		Optional<Integer> nid = Frills.getNidForVUID(vuid);
		Optional<ConceptSnapshot> conceptSnapshot = Frills.getConceptSnapshot(nid.get(), null, null);

		Set<Integer> childIds = new HashSet<Integer>();

		if (conceptSnapshot.isPresent()) {
			childIds = Frills.getAllChildrenOfConcept(conceptSnapshot.get().getConceptSequence(), true, false);
		}

		ConceptService conceptService = Get.conceptService();
		List<PublishConceptDTO> publishConceptDTOList = new ArrayList<PublishConceptDTO>();

		for (Integer id : childIds) {
			ConceptChronology<? extends ConceptVersion<?>> concept = conceptService.getConcept(id);
			PublishConceptDTO pubConcept = getDesignations(concept);
			pubConcept.setPropertyList(getProperties(concept.getNid()));
			pubConcept.setRelationshipList(getRelationships(concept));
			publishConceptDTOList.add(pubConcept);
		}

		PublishRegionDTO publishRegionDTO = new PublishRegionDTO(conceptSnapshot.get().getConceptDescriptionText(),
				publishConceptDTOList);

		pubRegionList.add(publishRegionDTO);

		return pubRegionList;
	}

	private PublishConceptDTO getDesignations(ConceptChronology<?> concept) {

		List<PublishConceptDTO> publishConcept = new ArrayList<>(1);
		List<NameValueDTO> designationsList = new ArrayList<>();

		Get.sememeService().getSememesForComponent(concept.getNid()).forEach(sememe -> {

			if (sememe.getSememeType() == SememeType.DESCRIPTION) {

				@SuppressWarnings({ "rawtypes", "unchecked" })
				Optional<LatestVersion<DescriptionSememe>> descriptionVersion = ((SememeChronology) sememe)
						.getLatestVersion(DescriptionSememe.class, STAMP_COORDINATES);

				if (descriptionVersion.isPresent()) {

					@SuppressWarnings({ "rawtypes", "unchecked" })
					List<DescriptionSememe<?>> descSememeList = ((SememeChronology) sememe)
							.getVisibleOrderedVersionList(STAMP_COORDINATES);
					Collections.reverse(descSememeList);

					for (DescriptionSememe<?> ds : descSememeList) {

						// TODO: is DataChangeType used?
						// determine type of datachangetype

						publishConcept.add(new PublishConceptDTO(ds.getText(), getCodeFromNid(ds.getNid()),
								(ds.getState() == State.ACTIVE), DataChangeType.CHANGED));

						// if active get properties
						if (ds.getState() == State.ACTIVE
								&& ds.getDescriptionTypeConceptSequence() != MetaData.SYNONYM.getConceptSequence()) {

							Get.sememeService().getSememesForComponent(sememe.getNid()).forEach((nestedSememe) -> {

								// skip code and vuid properties - they are
								// handled already
								if (nestedSememe.getAssemblageSequence() != MetaData.VUID.getConceptSequence()
										&& nestedSememe.getAssemblageSequence() != codeAssemblageConceptSeq
										&& ts.wasEverKindOf(nestedSememe.getAssemblageSequence(),
												vhatPropertyTypesNid)) {
									designationsList.add(getProperty(nestedSememe));
								}
							});
						}
					}
				}
			}
		});

		publishConcept.get(0).setDesignationList(designationsList);
		return publishConcept.get(0);
	}

	private List<NameValueDTO> getProperties(int componentNid) {

		ArrayList<NameValueDTO> properties = new ArrayList<>();
		Get.sememeService().getSememesForComponent(componentNid).forEach((sememe) -> {
			// skip code and vuid properties - they have special handling
			if (sememe.getAssemblageSequence() != MetaData.VUID.getConceptSequence()
					&& sememe.getAssemblageSequence() != codeAssemblageConceptSeq
					&& ts.wasEverKindOf(sememe.getAssemblageSequence(), vhatPropertyTypesNid)) {

				properties.add(getProperty(sememe));
			}
		});

		return properties;
	}

	// was buildProperty in VetsExporter
	private NameValueDTO getProperty(SememeChronology<?> sememe) {

		String name = getPreferredNameDescription(
				Get.identifierService().getConceptNid(sememe.getAssemblageSequence()));
		String value = new String();

		if (sememe.getSememeType() == SememeType.DYNAMIC) {

			@SuppressWarnings({ "unchecked", "rawtypes" })
			Optional<LatestVersion<? extends DynamicSememe>> dynamicSememeVersion = ((SememeChronology) sememe)
					.getLatestVersion(DynamicSememe.class, STAMP_COORDINATES);

			if (dynamicSememeVersion.isPresent() && dynamicSememeVersion.get().value().getData() != null
					&& dynamicSememeVersion.get().value().getData().length > 0) {

				value = dynamicSememeVersion.get().value().getData()[0] == null ? null
						: dynamicSememeVersion.get().value().getData()[0].dataToString();
			}
		} else if (sememe.getSememeType() == SememeType.STRING) {

			@SuppressWarnings({ "unchecked", "rawtypes" })
			Optional<LatestVersion<? extends StringSememe>> stringSememeVersion = ((SememeChronology) sememe)
					.getLatestVersion(StringSememe.class, STAMP_COORDINATES);

			if (stringSememeVersion.isPresent()) {
				@SuppressWarnings({ "rawtypes", "unchecked" })
				List<StringSememe<?>> stringSememeList = ((SememeChronology) sememe)
						.getVisibleOrderedVersionList(STAMP_COORDINATES);
				Collections.reverse(stringSememeList);
				value = stringSememeList.toString();
			}
		}

		log.debug("NAME: {} | VALUE: {}", name, value);
		return new NameValueDTO(name, value);
	}

	/**
	 *
	 * @param concept
	 * @return a List of NameValueDTO objects for the concept
	 */
	private List<NameValueDTO> getRelationships(ConceptChronology<?> concept) {

		List<NameValueDTO> relationships = new ArrayList<>();

		for (AssociationInstance ai : AssociationUtilities.getSourceAssociations(concept.getNid(), STAMP_COORDINATES)) {
			try {
				Long code = null;

				if (ai.getTargetComponent().isPresent()
						&& !"has_parent".equalsIgnoreCase(ai.getAssociationType().getDescription())) {

					code = getCodeFromNid(ai.getTargetComponent().get().getNid());

					log.debug("\t\tR NAME: {} | VALUE: {} ACTIVE: {}", ai.getAssociationType().getAssociationName(),
							code, ai.getData().getState());

					if (ai.getData().getState() == State.ACTIVE) {
						relationships.add(
								new NameValueDTO(ai.getAssociationType().getAssociationName(), Long.toString(code)));
					}

				}
			} catch (Exception e) {
				log.error("Association build failure", e);
			}
		}

		return relationships;
	}

	/**
	 *
	 * @param concept
	 * @return a String containing the parent name of the concept
	 */
	private String getParentName(ConceptChronology<?> concept) {
		String name = new String();
		for (AssociationInstance ai : AssociationUtilities.getSourceAssociations(concept.getNid(), STAMP_COORDINATES)) {
			try {
				if (ai.getTargetComponent().isPresent()
						&& "has_parent".equalsIgnoreCase(ai.getAssociationType().getDescription())) {

					name = getPreferredNameDescription(
							Get.identifierService().getNidForUuids(ai.getTargetComponent().get().getPrimordialUuid()));
				}
			} catch (Exception e) {
				log.error("Get Parent assoication failure");
			}
		}
		log.debug("D: {}", name);
		return name;
	}

	private String getPreferredNameDescription(int conceptNid) {

		ArrayList<String> descriptions = new ArrayList<>(1);
		ArrayList<String> inactiveDescriptions = new ArrayList<>(1);

		Get.sememeService().getDescriptionsForComponent(conceptNid).forEach(sememeChronology -> {

			@SuppressWarnings({ "unchecked", "rawtypes" })
			Optional<LatestVersion<DescriptionSememe<?>>> descriptionSememe = ((SememeChronology) sememeChronology)
					.getLatestVersion(DescriptionSememe.class, STAMP_COORDINATES);

			if (descriptionSememe.isPresent() && preferredNameExtendedType
					.equals(Frills.getDescriptionExtendedTypeConcept(STAMP_COORDINATES, sememeChronology.getNid(), true)
							.orElse(null))) {
				if (descriptionSememe.get().value().getState() == State.ACTIVE) {
					descriptions.add(descriptionSememe.get().value().getText());
				} else {
					inactiveDescriptions.add(descriptionSememe.get().value().getText());
				}
			}
		});

		if (descriptions.size() == 0) {
			descriptions.addAll(inactiveDescriptions);
		}

		// TODO: check if this is needed
		if (descriptions.size() == 0) {
			// This doesn't happen for concept that represent subsets, for
			// example.
			// log.debug("Failed to find a description flagged as preferred on concept "
			// + Get.identifierService().getUuidPrimordialForNid(conceptNid));
			String description = Frills.getDescription(conceptNid, STAMP_COORDINATES,
					LanguageCoordinates.getUsEnglishLanguagePreferredTermCoordinate()).orElse("ERROR!");
			if (description.equals("ERROR!")) {
				log.error("Failed to find any description on concept "
						+ Get.identifierService().getUuidPrimordialForNid(conceptNid));
			}
			return description;
		}
		if (descriptions.size() > 1) {
			log.warn("Found " + descriptions.size() + " descriptions flagged as the 'Preferred' vhat type on concept "
					+ Get.identifierService().getUuidPrimordialForNid(conceptNid));
		}

		return descriptions.get(0);
	}

	/**
	 * 
	 * @param componentNid
	 * @return the Code value found based on the Nid
	 */
	private Long getCodeFromNid(int componentNid) {

		Optional<SememeChronology<? extends SememeVersion<?>>> sc = Get.sememeService()
				.getSememesForComponentFromAssemblage(componentNid, codeAssemblageConceptSeq).findFirst();
		if (sc.isPresent()) {
			// There was a bug in the older terminology loaders which loaded
			// 'Code' as a static sememe, but marked it as a dynamic sememe.
			// So during edits, new entries would get saves as dynamic sememes,
			// while old entries were static. Handle either....

			if (sc.get().getSememeType() == SememeType.STRING) {
				@SuppressWarnings({ "unchecked", "rawtypes" })
				Optional<LatestVersion<StringSememe<?>>> sv = ((SememeChronology) sc.get())
						.getLatestVersion(StringSememe.class, STAMP_COORDINATES);
				if (sv.isPresent()) {
					return Long.parseLong(sv.get().value().getString());
				}
			} else if (sc.get().getSememeType() == SememeType.DYNAMIC) {
				// this path will become dead code, after the data is fixed.
				@SuppressWarnings({ "unchecked", "rawtypes" })
				Optional<LatestVersion<? extends DynamicSememe>> sv = ((SememeChronology) sc.get())
						.getLatestVersion(DynamicSememe.class, STAMP_COORDINATES);
				if (sv.isPresent()) {
					if (sv.get().value().getData() != null && sv.get().value().getData().length == 1) {
						return Long.parseLong(sv.get().value().getData()[0].dataToString());
					}
				}
			} else {
				log.error("Unexpected sememe type for 'Code' sememe on nid " + componentNid);
			}
		}
		return null;
	}

}
