/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.loinc;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.TreeMap;
import org.apache.commons.io.input.BOMInputStream;
import com.opencsv.CSVReader;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.vha.isaac.ochre.api.util.AlphanumComparator;

/**
 * 
 * {@link LoincCsvFileReader}
 *
 * Reads the CSV formatted release files of LOINC, and the custom release notes file
 * to extract the date and time information.
 * 
 * @author <a href="mailto:daniel.armbrust.list@gmail.com">Dan Armbrust</a>
 */
public class LoincCsvFileReader extends LOINCReader
{
	String[] header;
	CSVReader reader;
	String version = null;
	String release = null;
	private TreeMap<String, Long> versionTimeMap = new TreeMap<String, Long>(new AlphanumComparator(true));
	
	public LoincCsvFileReader(File f, boolean populateVersionTimeMap) throws IOException
	{
		ConsoleUtil.println("Using the data file " + f.getAbsolutePath());
		//Their new format includes the (optional) UTF-8 BOM, which chokes java for stupid legacy reasons.
		reader = new CSVReader(new BufferedReader(new InputStreamReader(new BOMInputStream(new FileInputStream(f)))));
		header = readLine();
		
		readReleaseNotes(f.getParentFile(), populateVersionTimeMap);
	}
	
	public LoincCsvFileReader(InputStream is) throws IOException
	{
		//Their new format includes the (optional) UTF-8 BOM, which chokes java for stupid legacy reasons.
		reader = new CSVReader(new BufferedReader(new InputStreamReader(new BOMInputStream(is))));
		header = readLine();
	}
	
	@SuppressWarnings("deprecation")
	public void readReleaseNotes(File dataFolder, boolean populateVersionTimeMap) throws IOException
	{
		File relNotes  = null;
		for (File f : dataFolder.listFiles())
		{
			if (f.getName().toLowerCase().contains("releasenotes.txt"))
			{
				relNotes = f;
				break;
			}
		}
		if (relNotes.exists())
		{
			SimpleDateFormat[] sdf = new SimpleDateFormat[] {new SimpleDateFormat("MMM dd, yyyy"), new SimpleDateFormat("MMM yyyy"), 
					new SimpleDateFormat("MMMyyyy"), new SimpleDateFormat("MM/dd/yy")};
			BufferedReader br = new BufferedReader(new FileReader(relNotes));
			String line = br.readLine();
			boolean first = true;
			String versionCache = null;
			while (line != null)
			{
				if (line.matches("\\s*\\|\\s*Version [\\w\\.\\-\\(\\)]*\\s*\\|\\s*"))
				{
					String temp = line.substring(line.indexOf("Version") + "Version ".length());
					versionCache = temp.replace('|', ' ').trim();
					if (first)
					{
						version = versionCache;
					}
				}
				if (line.matches("\\s*\\|\\s*Released [\\w\\s/,]*\\|"))
				{
					String temp = line.substring(line.indexOf("Released") + "Released ".length());
					temp = temp.replace('|', ' ').trim();
					if (first)
					{
						release = temp;
						first = false;
						if (!populateVersionTimeMap)
						{
							break;
						}
					}
					Long time = -1l;
					for (SimpleDateFormat f : sdf)
					{
						try
						{
							time = f.parse(temp).getTime();
							break;
						}
						catch (ParseException e)
						{
							//noop
						}
					}
					if (time < 0)
					{
						throw new IOException("Failed to parse " + temp);
					}
					if (versionCache == null)
					{
						ConsoleUtil.printErrorln("No version for line " + line);
					}
					else
					{
						versionTimeMap.put(versionCache, time);
					}
					versionCache = null;
				}
				line = br.readLine();
			}
			br.close();
			if (populateVersionTimeMap)
			{
				//release notes is missing this one...set it to a time before 2.03.
				if (!versionTimeMap.containsKey("2.02"))
				{
					Date temp = new Date(versionTimeMap.get("2.03"));
					
						temp.setMonth(temp.getMonth() -1);
						versionTimeMap.put("2.02", temp.getTime());
					
				}
				//Debug codel
//				ConsoleUtil.println("Release / Time map read from readme file:");
//				for (Entry<String, Long> x : versionTimeMap.entrySet())
//				{
//					ConsoleUtil.println(x.getKey() + " " + new Date(x.getValue()).toString());
//				}
			}
		}
		else
		{
			ConsoleUtil.printErrorln("Couldn't find release notes file - can't read version or release date!");
		}
	}
	
	@Override
	public String getVersion()
	{
		return version;
	}

	@Override
	public String getReleaseDate()
	{
		return release;
	}

	@Override
	public String[] getHeader()
	{
		return header;
	}

	@Override
	public String[] readLine() throws IOException
	{
		String[] temp = reader.readNext();
		if (temp != null)
		{
			if (fieldCount_ == 0)
			{
				fieldCount_ = temp.length;
				int i = 0;
				for (String s : temp)
				{
					fieldMapInverse_.put(i, s);
					fieldMap_.put(s, i++);
				}
			}
			else if (temp.length < fieldCount_)
			{
				temp = Arrays.copyOf(temp, fieldCount_);
			}
			else if (temp.length > fieldCount_)
			{
				throw new RuntimeException("Data error - to many fields found on line: " + Arrays.toString(temp));
			}
		}
		return temp;
	}

	@Override
	public void close() throws IOException
	{
		reader.close();
	}

	public TreeMap<String, Long> getTimeVersionMap()
	{
		return versionTimeMap;
	}
}
