/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.ochre.mojo.profileSync;

import java.io.BufferedReader;
import java.io.Console;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Parameter;
import gov.vha.isaac.ochre.api.LookupService;
import gov.vha.isaac.ochre.api.sync.SyncFiles;
import gov.vha.isaac.ochre.mojo.external.QuasiMojo;

/**
 * {@link ProfilesMojoBase}
 * 
 * This allows authentication to be passed in via system property, parameter, or, will 
 * prompt for the username/password (if allowed by the system property 'profileSyncNoPrompt')
 * IN THAT ORDER.  System properties have the highest priority.
 * 
 * To prevent prompting during automated runs - set the system property 'profileSyncNoPrompt=true'
 * To set the username via system property - set 'profileSyncUsername=username'
 * To set the password via system property - set 'profileSyncPassword=password'
 * 
 * To enable authentication without prompts, using public keys - set both of the following 
 *   'profileSyncUsername=username'
 *   'profileSyncNoPrompt=true' 
 *   
 * This will cause a public key authentication to be attempted using the ssh credentials found 
 * in the current users .ssh folder (in their home directory)
 *
 * @author <a href="mailto:daniel.armbrust.list@gmail.com">Dan Armbrust</a> 
 */
public abstract class ProfilesMojoBase extends QuasiMojo
{
	// For disabling Profile Sync entirely
	public static final String PROFILE_SYNC_DISABLE = "profileSyncDisable";
	
	// For preventing command line prompts for credentials during automated runs - set this system property to true.
	public static final String PROFILE_SYNC_NO_PROMPTS = "profileSyncNoPrompt";
	
	// Allow setting the username via a system property
	public static final String PROFILE_SYNC_USERNAME_PROPERTY = "profileSyncUsername";
	
	// Allow setting the password via a system property
	public static final String PROFILE_SYNC_PASSWORD_PROPERTY = "profileSyncPassword";
	
	private boolean disableHintGiven = false;
	
	/**
	 * The location of the (already existing) profiles folder which should be shared via SCM.
	 */
	@Parameter (required = true)
	File userProfileFolderLocation = null;
	
	/**
	 * The location URL to use when connecting to the sync service
	 */
	@Parameter (required = true)
	String changeSetURL = null;
	
	/**
	 * The Type of the specified changeSetURL - should be GIT or SVN
	 */
	@Parameter (required = true)
	String changeSetURLType = null;
	
	/**
	 * The username to use for remote operations
	 */
	@Parameter (required = false)
	private String profileSyncUsername = null;
	
	/**
	 * The password to use for remote operations
	 */
	@Parameter (required = false)
	private String profileSyncPassword = null;
	
	private static String username = null;
	private static String password = null;

	public ProfilesMojoBase() throws MojoExecutionException
	{
		super();
	}
	
	/**
	 * @see org.apache.maven.plugin.Mojo#execute()
	 */
	@Override
	public void execute() throws MojoExecutionException
	{
		if (StringUtils.isNotBlank(changeSetURL) && !changeSetURLType.equalsIgnoreCase("GIT") && !changeSetURLType.equalsIgnoreCase("SVN")) 
		{
			throw new MojoExecutionException("Change set URL type must be GIT or SVN");
		}
	}

	protected boolean skipRun()
	{
		if (Boolean.getBoolean(PROFILE_SYNC_DISABLE))
		{
			return true;
		}
		if (StringUtils.isBlank(changeSetURL))
		{
			getLog().info("No SCM configuration will be done - no 'changeSetUrl' parameter was provided");
			return true;
		}
		else
		{
			return false;
		}
	}
	
	protected SyncFiles getProfileSyncImpl() throws MojoExecutionException
	{
		if (changeSetURLType.equalsIgnoreCase("GIT"))
		{
			SyncFiles svc = LookupService.getService(SyncFiles.class, "GIT");
			if (svc == null)
			{
				throw new MojoExecutionException("Unable to load the GIT implementation of the ProfileSyncI interface." + 
						"  Is gov.vha.isaac.gui.modules.sync-git listed as a dependency for the mojo execution?");
			}
			svc.setRootLocation(userProfileFolderLocation);
			return svc;
		}
		else if (changeSetURLType.equalsIgnoreCase("SVN"))
		{
			SyncFiles svc = LookupService.getService(SyncFiles.class, "SVN");
			if (svc == null)
			{
				throw new MojoExecutionException("Unable to load the SVN implementation of the ProfileSyncI interface." + 
						"  Is gov.vha.isaac.gui.modules.sync-svn listed as a dependency for the mojo execution?");
			}
			svc.setRootLocation(userProfileFolderLocation);
			return svc;
		}
		else
		{
			throw new MojoExecutionException("Unsupported change set URL Type");
		}
	}
	
	/**
	 * Does the necessary substitution to put the contents of getUserName() into the URL, if a known pattern needing substitution is found.
	 *  ssh://someuser@csfe.aceworkspace.net:29418/... for example needs to become:
	 *  ssh://<getUsername()>@csfe.aceworkspace.net:29418/...
	 * @throws MojoExecutionException 
	 */
	protected String getURL() throws MojoExecutionException
	{
		return getProfileSyncImpl().substituteURL(changeSetURL, getUsername());
	}
	
	protected String getUsername() throws MojoExecutionException
	{
		if (username == null)
		{
			username = System.getProperty(PROFILE_SYNC_USERNAME_PROPERTY);
			
			//still blank, try property
			if (StringUtils.isBlank(username))
			{
				username = profileSyncUsername;
			}
			
			//still no username, prompt if allowed
			if (StringUtils.isBlank(username) && !Boolean.getBoolean(PROFILE_SYNC_NO_PROMPTS))
			{
				Callable<Void> callable = new Callable<Void>()
				{
					@Override
					public Void call() throws Exception
					{
						if (!disableHintGiven)
						{
							System.out.println("To disable remote sync during build, add '-D" + PROFILE_SYNC_DISABLE + "=true' to your maven command");
							disableHintGiven = true;
						}
						
						try
						{
							System.out.println("Enter the " + changeSetURLType + " username for the Profiles/Changset remote store (" +
									changeSetURL + "):");
							BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
							username = br.readLine();
						}
						catch (IOException e)
						{
							throw new MojoExecutionException("Error reading username from console");
						}
						return null;
					}
				};
				
				try
				{
					Executors.newSingleThreadExecutor(new ThreadFactory()
					{
						@Override
						public Thread newThread(Runnable r)
						{
							Thread t = new Thread(r, "User Prompt Thread");
							t.setDaemon(true);
							return t;
						}
					}).submit(callable).get(2, TimeUnit.MINUTES);
				}
				catch (TimeoutException | InterruptedException e)
				{
					throw new MojoExecutionException("Username not provided within timeout");
				}
				catch (ExecutionException ee)
				{
					throw (ee.getCause() instanceof MojoExecutionException ? (MojoExecutionException)ee.getCause() : 
						new MojoExecutionException("Unexpected", ee.getCause()));
				}
			}
		}
		return username;
	}
	
	protected String getPassword() throws MojoExecutionException
	{
		if (password == null)
		{
			password = System.getProperty(PROFILE_SYNC_PASSWORD_PROPERTY);
			
			//still blank, try the passed in param
			if (StringUtils.isBlank(password))
			{
				password = profileSyncPassword;
			}
			
			//still no password, prompt if allowed
			if (StringUtils.isBlank(password) && !Boolean.getBoolean(PROFILE_SYNC_NO_PROMPTS))
			{
				Callable<Void> callable = new Callable<Void>()
				{
					@Override
					public Void call() throws Exception
					{
						try
						{
							if (!disableHintGiven)
							{
								System.out.println("To disable remote sync during build, add '-D" + PROFILE_SYNC_DISABLE + "=true' to your maven command");
								disableHintGiven = true;
							}
							System.out.println("Enter the " + changeSetURLType + " password for the Profiles/Changset remote store: (" +
									changeSetURL + "):");
							
							//Use console if available, for password masking
							Console console = System.console();
							if (console != null)
							{
								password = new String(console.readPassword());
							}
							else
							{
								BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
								password = br.readLine();
							}
						}
						catch (IOException e)
						{
							throw new MojoExecutionException("Error reading password from console");
						}
						return null;
					}
				};
				
				try
				{
					Executors.newSingleThreadExecutor(new ThreadFactory()
					{
						@Override
						public Thread newThread(Runnable r)
						{
							Thread t = new Thread(r, "User Password Prompt Thread");
							t.setDaemon(true);
							return t;
						}
					}).submit(callable).get(2, TimeUnit.MINUTES);
				}
				catch (TimeoutException | InterruptedException e)
				{
					throw new MojoExecutionException("Password not provided within timeout");
				}
				catch (ExecutionException ee)
				{
					throw (ee.getCause() instanceof MojoExecutionException ? (MojoExecutionException)ee.getCause() : 
						new MojoExecutionException("Unexpected", ee.getCause()));
				}
			}
		}
		return password;
	}
}