package gov.va.med.term.vhat.mojo;

import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.And;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.ConceptAssertion;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.NecessarySet;
import java.io.File;
import java.io.FileWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import gov.va.med.term.vhat.data.TerminologyDataReader;
import gov.va.med.term.vhat.data.dto.ConceptImportDTO;
import gov.va.med.term.vhat.data.dto.DesignationImportDTO;
import gov.va.med.term.vhat.data.dto.MapEntryImportDTO;
import gov.va.med.term.vhat.data.dto.MapSetImportDTO;
import gov.va.med.term.vhat.data.dto.PropertyImportDTO;
import gov.va.med.term.vhat.data.dto.RelationshipImportDTO;
import gov.va.med.term.vhat.data.dto.SubsetImportDTO;
import gov.va.med.term.vhat.data.dto.SubsetMembershipImportDTO;
import gov.va.med.term.vhat.data.dto.TypeImportDTO;
import gov.va.med.term.vhat.propertyTypes.PT_Annotations;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.EConceptUtility;
import gov.va.oia.terminology.converters.sharedUtils.EConceptUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Associations;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Descriptions;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Relations;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.Property;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyAssociation;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.ValuePropertyPair;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.component.sememe.SememeChronology;
import gov.vha.isaac.ochre.api.component.sememe.version.DescriptionSememe;
import gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder;
import gov.vha.isaac.ochre.api.logic.assertions.Assertion;
import gov.vha.isaac.ochre.api.logic.assertions.ConceptAssertion;

/**
 * Goal which converts VHAT data into the workbench jbin format
 */
@Mojo (name = "convert-VHAT-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class VHATImportMojo extends ConverterBaseMojo
{
	private Map<String, List<RelationshipImportDTO>> relationshipMap = null;
	private TerminologyDataReader importer_;
	private EConceptUtility importUtil_;

	private HashMap<String, String> referencedConcepts = new HashMap<String, String>();
	private HashMap<String, String> loadedConcepts = new HashMap<String, String>();
	private UUID rootConceptUUID;

	private PropertyType attributes_, descriptions_, associations_, relationships_;
	private BPT_Refsets refsets_;

	private UUID allVhatConceptsRefset;

	private HashSet<Long> conceptsWithNoDesignations = new HashSet<Long>();
	private int mapEntryCount = 0;
	private int mapSetCount = 0;

	@Override
	public void execute() throws MojoExecutionException
	{

		try
		{
			super.execute();

			String temp = converterOutputArtifactVersion.substring(0, 11);

			importUtil_ = new EConceptUtility(Optional.empty(), Optional.of(MetaData.VHA_MODULE), outputDirectory, converterOutputArtifactId,
				converterOutputArtifactClassifier, false, new SimpleDateFormat("yyyy.MM.dd").parse(temp).getTime());
			
			attributes_ = new PT_Annotations();
			descriptions_ = new BPT_Descriptions("VHAT");
			associations_ = new BPT_Associations("VHAT");
			relationships_ = new BPT_Relations("VHAT");
			refsets_ = new BPT_Refsets("VHAT");
			refsets_.addProperty("All VHAT Concepts");

			importer_ = new TerminologyDataReader(inputFileLocation);
			List<ConceptImportDTO> items = importer_.process();

			relationshipMap = importer_.getRelationshipsMap();
			List<TypeImportDTO> dto = importer_.getTypes();

			ComponentReference vhatMetadata = ComponentReference.fromConcept(createType(MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid(), "VHAT Metadata"));
			
			importUtil_.loadTerminologyMetadataAttributes(vhatMetadata, converterSourceArtifactVersion, 
					Optional.empty(), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);

			// read in the dynamic types
			for (TypeImportDTO typeImportDTO : dto)
			{
				if (typeImportDTO.getKind().equals("DesignationType"))
				{
					Property p = descriptions_.addProperty(typeImportDTO.getName());
					//Add some rankings for FSN / synonym handling
					if (p.getSourcePropertyNameFSN().equals("Fully Specified Name"))
					{
						p.setPropertySubType(BPT_Descriptions.FSN);
					}
					else if (p.getSourcePropertyNameFSN().equals("Preferred Name"))
					{
						p.setPropertySubType(BPT_Descriptions.SYNONYM);
					}
					else if (p.getSourcePropertyNameFSN().equals("Synonym"))
					{
						p.setPropertySubType(BPT_Descriptions.SYNONYM + 1);
					}
				}
				else if (typeImportDTO.getKind().equals("RelationshipType"))
				{
					//currently loading isA as a graph rel, and an assoiation
					if (typeImportDTO.getName().equals("has_parent"))
					{
						Property p = relationships_.addProperty(typeImportDTO.getName());
						p.setWBPropertyType(MetaData.IS_A.getPrimordialUuid());
					}
					associations_.addProperty(new PropertyAssociation(associations_, typeImportDTO.getName(), typeImportDTO.getName(), null, 
							typeImportDTO.getName(), false));
				}
				else if (typeImportDTO.getKind().equals("PropertyType"))
				{
					attributes_.addProperty(typeImportDTO.getName());
				}
			}
			
			//get the refset names
			for (SubsetImportDTO subset : importer_.getSubsets())
			{
				refsets_.addProperty(subset.getSubsetName());
			}

			importUtil_.loadMetaDataItems(Arrays.asList(descriptions_, attributes_, associations_, relationships_, refsets_), 
					vhatMetadata.getPrimordialUuid());
//			loadedConcepts.put(refsets_.getRefsetIdentityParent().toString(), eConceptUtil_.PROJECT_REFSETS_NAME);

			ConsoleUtil.println("Metadata load stats");
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(line);
			}
			
			importUtil_.clearLoadStats();

			allVhatConceptsRefset = refsets_.getProperty("All VHAT Concepts").getUUID();
			loadedConcepts.put(allVhatConceptsRefset.toString(), "All VHAT Concepts");

			Map<Long, Set<Long>> subsetMembershipMap = new HashMap<Long, Set<Long>>();
			// get the subset memberships to build the refset for each subset
			for (ConceptImportDTO item : items)
			{
				List<DesignationImportDTO> designations = item.getDesignations();
				for (DesignationImportDTO designation : designations)
				{
					List<SubsetMembershipImportDTO> subsets = designation.getSubsets();
					for (SubsetMembershipImportDTO subsetMembership : subsets)
					{
						Set<Long> vuids = subsetMembershipMap.get(subsetMembership.getVuid());
						if (vuids == null)
						{
							vuids = new HashSet<Long>();
							subsetMembershipMap.put(subsetMembership.getVuid(), vuids);
						}
						vuids.add(designation.getVuid());
					}
				}
			}
			
			// create all the subsets - option 1 - this could be swapped out with option 2, below (which is currently commented out)
			List<SubsetImportDTO> subsets = importer_.getSubsets();
			for (SubsetImportDTO subset : subsets)
			{
				loadRefset(subset.getSubsetName(), subsetMembershipMap.get(subset.getVuid()));
			}

			int conCount = 0;
			for (ConceptImportDTO item : items)
			{
				conCount++;
				writeEConcept(item);
				if (conCount % 500 == 0)
				{
					ConsoleUtil.showProgress();
				}
				if (conCount % 10000 == 0)
				{
					ConsoleUtil.println("Processed " + conCount + " concepts");
				}
			}
			
			ConsoleUtil.println("Processed " + conCount + " concepts");

			ArrayList<String> missingConcepts = new ArrayList<String>();

			for (String refUUID : referencedConcepts.keySet())
			{
				if (loadedConcepts.get(refUUID) == null)
				{
					missingConcepts.add(refUUID);
					ConsoleUtil.printErrorln("Data error - The concept " + refUUID + " - " + referencedConcepts.get(refUUID)
							+ " was referenced, but not loaded - will be created as '-MISSING-'");
				}
			}

			if (missingConcepts.size() > 0)
			{
				ConceptChronology<? extends ConceptVersion<?>> missingParent = importUtil_.createConcept("Missing Concepts", true);
				importUtil_.addParent(ComponentReference.fromChronology(missingParent), rootConceptUUID);
				for (String refUUID : missingConcepts)
				{
					ComponentReference c = ComponentReference.fromChronology(importUtil_.createConcept(UUID.fromString(refUUID), "-MISSING-", true));
					importUtil_.addParent(c, missingParent.getPrimordialUuid());
				}
			}

			if (mapEntryCount > 0)
			{
				ConsoleUtil.println("Skipped " + mapEntryCount + " MapEntry objects");
			}
			if (mapSetCount > 0)
			{
				ConsoleUtil.println("Skipped " + mapSetCount + " MapSet objects");
			}

			// Put in names instead of IDs so the load stats print nicer:
			Hashtable<String, String> stringsToSwap = new Hashtable<String, String>();
			for (SubsetImportDTO subset : subsets)
			{
				stringsToSwap.put(subset.getVuid() + "", subset.getSubsetName());
			}
			

			ConsoleUtil.println("Load Statistics");
			// swap out vuids with names to make it more readable...
			for (String line : importUtil_.getLoadStats().getSummary())
			{
				Enumeration<String> e = stringsToSwap.keys();
				while (e.hasMoreElements())
				{
					String current = e.nextElement();
					line = line.replaceAll(current, stringsToSwap.get(current));
				}
				ConsoleUtil.println(line);
			}

			// this could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "vhatUuid");

			if (conceptsWithNoDesignations.size() > 0)
			{
				ConsoleUtil.printErrorln(conceptsWithNoDesignations.size() + " concepts were found with no descriptions at all.  These were assigned '-MISSING-'");
				FileWriter fw = new FileWriter(new File(outputDirectory, "NoDesignations.txt"));
				for (Long l : conceptsWithNoDesignations)
				{
					fw.write(l.toString());
					fw.write(System.getProperty("line.separator"));
				}
				fw.close();
			}
			importUtil_.shutdown();
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		}
		catch (Exception ex)
		{
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}

	}

	private void writeEConcept(ConceptImportDTO conceptDto) throws Exception
	{
		if (conceptDto instanceof MapEntryImportDTO)
		{
			mapEntryCount++;
		}
		else if (conceptDto instanceof MapSetImportDTO)
		{
			mapSetCount++;
		}
		else
		{
			//TODO create time on concepts?
			ComponentReference concept = ComponentReference.fromConcept(importUtil_.createConcept(getConceptUuid(conceptDto.getVuid().toString())));
			loadedConcepts.put(concept.getPrimordialUuid().toString(), conceptDto.getVuid().toString());
			importUtil_.addStaticStringAnnotation(concept, conceptDto.getVuid().toString(), attributes_.getProperty("VUID").getUUID(), State.ACTIVE);

			List<DesignationImportDTO> designationDto = conceptDto.getDesignations();
			ArrayList<ValuePropertyPairExtended> descriptionHolder = new ArrayList<>(designationDto.size());
			for (DesignationImportDTO didto : designationDto)
			{
				descriptionHolder.add(new ValuePropertyPairExtended(didto.getValueNew(), getDescriptionUuid(didto.getVuid().toString()),
						descriptions_.getProperty(didto.getTypeName()), didto, !didto.isActive()));
			}
			
			for (PropertyImportDTO property : conceptDto.getProperties())
			{
				importUtil_.addStringAnnotation(concept, property.getValueNew(), attributes_.getProperty(property.getTypeName()).getUUID(), State.ACTIVE);
			}

			List<SememeChronology<DescriptionSememe<?>>> wbDescriptions = importUtil_.addDescriptions(concept, descriptionHolder);
			
			//Descriptions have now all been added to the concepts - now we need to process the rest of the ugly bits of vhat
			//and place them on the descriptions.
			for (int i = 0; i < descriptionHolder.size(); i++)
			{
				ValuePropertyPairExtended vpp = descriptionHolder.get(i);
				SememeChronology<DescriptionSememe<?>> desc = wbDescriptions.get(i);
				
				if (vpp.getValue().equals("VHAT"))
				{
					// On the root node, we need to add some extra attributes
					importUtil_.addDescription(concept, "VHAT", DescriptionType.SYNONYM, true, null, null, State.ACTIVE);
					importUtil_.addDescription(concept, "VHA Terminology", DescriptionType.SYNONYM, false, null, null, State.ACTIVE);
					ConsoleUtil.println("Root concept FSN is 'VHAT' and the UUID is " + concept.getPrimordialUuid());
					importUtil_.addParent(concept, MetaData.ISAAC_ROOT.getPrimordialUuid());
					rootConceptUUID = concept.getPrimordialUuid();
				}
				importUtil_.addStaticStringAnnotation(ComponentReference.fromChronology(desc, () -> "Description"), vpp.getDesignationImportDTO().getVuid() + "", 
					attributes_.getProperty("VUID").getUUID(), State.ACTIVE);

				// VHAT is kind of odd, in that the attributes are attached to the description, rather than the concept.
				for (PropertyImportDTO property : vpp.getDesignationImportDTO().getProperties())
				{
					importUtil_.addStringAnnotation(ComponentReference.fromChronology(desc, () -> "Description"), property.getValueNew(), 
							attributes_.getProperty(property.getTypeName()).getUUID(), State.ACTIVE);
				}
			}
			
			if (descriptionHolder.size() == 0)
			{
				// Seems like a data error - but it is happening... no descriptions at all.....
				conceptsWithNoDesignations.add(conceptDto.getVuid());
				// The workbench implodes if you don't have a fully specified name....
				importUtil_.addDescription(concept, "-MISSING-", DescriptionType.FSN, true, null, null, State.ACTIVE);
			}

			List<RelationshipImportDTO> relationshipImports = relationshipMap.get(conceptDto.getCode());
			LogicalExpressionBuilder leb = Get.logicalExpressionBuilderService().getLogicalExpressionBuilder();
			ArrayList<ConceptAssertion> assertions = new ArrayList<>();
			if (relationshipImports != null)
			{
				for (RelationshipImportDTO relationshipImportDTO : relationshipImports)
				{
					UUID sourceUuid = getConceptUuid(relationshipImportDTO.getSourceCode());
					UUID targetUuid = getConceptUuid(relationshipImportDTO.getNewTargetCode());

					referencedConcepts.put(targetUuid.toString(), relationshipImportDTO.getNewTargetCode());

					if (!sourceUuid.equals(concept.getPrimordialUuid()))
					{
						throw new MojoExecutionException("Design failure!");
					}

					importUtil_.addAssociation(concept, null, targetUuid, associations_.getProperty(relationshipImportDTO.getTypeName()).getUUID(),
							State.ACTIVE, null, null);
					
					//If it is an isA rel, also create it as a rel.
					
					if (relationships_.getProperty(relationshipImportDTO.getTypeName()) != null)
					{
						if (relationships_.getProperty(relationshipImportDTO.getTypeName()).getWBTypeUUID().equals(MetaData.IS_A.getPrimordialUuid()))
						{
							assertions.add(ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(targetUuid), leb));
						}
						else
						{
							throw new RuntimeException("Only is_a is handled so far as a relationship");
						}
					}
				}
				if (assertions.size() > 0)
				{
					NecessarySet(And(assertions.toArray(new Assertion[assertions.size()])));
					importUtil_.addRelationshipGraph(concept, null, leb.build(), true, null, null);
				}
			}

			importUtil_.addRefsetMembership(concept, allVhatConceptsRefset, State.ACTIVE, null);
		}
	}

	private ConceptChronology<? extends ConceptVersion<?>> createType(UUID parentUuid, String typeName) throws Exception
	{
		ConceptChronology<? extends ConceptVersion<?>> concept = importUtil_.createConcept(typeName, true);
		loadedConcepts.put(concept.getPrimordialUuid().toString(), typeName);
		importUtil_.addParent(ComponentReference.fromConcept(concept), parentUuid);
		return concept;
	}

	private void loadRefset(String typeName, Set<Long> refsetMembership) throws Exception
	{
		UUID concept = refsets_.getProperty(typeName).getUUID();
		loadedConcepts.put(concept.toString(), typeName);

		if (refsetMembership != null)
		{
			for (Long memberVuid : refsetMembership)
			{
				importUtil_.addRefsetMembership(ComponentReference.fromSememe(getDescriptionUuid(memberVuid.toString())), concept, State.ACTIVE, null);
			}
		}
	}

	private UUID getConceptUuid(String codeId)
	{
		return ConverterUUID.createNamespaceUUIDFromString("code:" + codeId, true);
	}

	private UUID getDescriptionUuid(String descriptionId)
	{
		return ConverterUUID.createNamespaceUUIDFromString("description:" + descriptionId, true);
	}

	public static void main(String[] args) throws MojoExecutionException
	{
		VHATImportMojo i = new VHATImportMojo();
		i.outputDirectory = new File("../vhat-ibdf/target");
		i.inputFileLocation = new File("../vhat-ibdf/target/generated-resources/src/");
		i.converterOutputArtifactVersion = "2016.01.07.foo";
		i.converterVersion = "SNAPSHOT";
		i.execute();
	}
	
	private class ValuePropertyPairExtended extends ValuePropertyPair
	{
		private DesignationImportDTO didto_;
		public ValuePropertyPairExtended(String value, UUID descriptionUUID, Property property, DesignationImportDTO didto, boolean disabled)
		{
			super(value, descriptionUUID, property);
			didto_ = didto;
			setDisabled(disabled);
		}
		
		public DesignationImportDTO getDesignationImportDTO()
		{
			return didto_;
		}
	}
}