/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.loinc.techPreview;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Optional;
import java.util.UUID;
import org.antlr.v4.runtime.tree.ParseTree;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.EConceptUtility;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.ValuePropertyPair;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.loinc.LOINCReader;
import gov.vha.isaac.loinc.LoincCsvFileReader;
import gov.vha.isaac.loinc.techPreview.propertyTypes.PT_Annotations;
import gov.vha.isaac.loinc.techPreview.propertyTypes.PT_Descriptions;
import gov.vha.isaac.loinc.techPreview.propertyTypes.PT_Refsets;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.logic.LogicalExpression;
import gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder;
import gov.vha.isaac.ochre.logic.provider.ISAACVisitor;
import se.liu.imt.mi.snomedct.expression.tools.SNOMEDCTParserUtil;

/**
 * 
 * Loader code to convert Loinc into the ISAAC datastore.
 * 
 * Paths are typically controlled by maven, however, the main() method has paths configured so that they
 * match what maven does for test purposes.
 */
@Mojo( name = "convert-loinc-tech-preview-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES )
public class LoincTPLoaderMojo extends ConverterBaseMojo
{
	private static final String necessarySctid = "900000000000074008";
	private static final String sufficientSctid = "900000000000073002";
	private static final String eol = System.getProperty("line.separator");
	
	/**
	 * we also read a native loinc input file - get that version too
	 */
	@Parameter (required = true, defaultValue = "${loinc-src-data.version}")
	protected String converterSourceLoincArtifactVersion;
	
	@Override
	public void execute() throws MojoExecutionException
	{
		ConsoleUtil.println("LOINC Tech Preview Processing Begins " + new Date().toString());
		super.execute();
	
		ConsoleUtil.println("Processing LOINC");
		LOINCReader loincData = null;
		File tpZipFile = null;
		int expLineNumber = 1;
		
		BufferedWriter loincExpressionDebug = null;
		
		try
		{
			if (!inputFileLocation.isDirectory())
			{
				throw new MojoExecutionException("LoincDataFiles must point to a directory containing the 3 required loinc data files");
			}
			
			for (File f : inputFileLocation.listFiles())
			{
				if (f.getName().toLowerCase().equals("loinc.csv"))
				{
					loincData = new LoincCsvFileReader(f, false);
				}
				if (f.isFile() && f.getName().toLowerCase().endsWith(".zip"))
				{
					if (tpZipFile != null)
					{
						throw new RuntimeException("Found multiple zip files in " + inputFileLocation.getAbsolutePath());
					}
					tpZipFile = f;
				}
			}
			if (loincData == null)
			{
				throw new MojoExecutionException("Could not find the loinc data file in " + inputFileLocation.getAbsolutePath());
			}
			if (tpZipFile == null)
			{
				throw new RuntimeException("Couldn't find the tech preview zip file in " + inputFileLocation.getAbsolutePath());
			}
			loincExpressionDebug = new BufferedWriter(new FileWriter(new File(outputDirectory, "ExpressionDebug.log")));
			
			SimpleDateFormat dateReader = new SimpleDateFormat("MMMMMMMMMMMMM yyyy"); //Parse things like "June 2014"
			Date releaseDate = dateReader.parse(loincData.getReleaseDate());
			
			File[] ibdfFiles = new File(inputFileLocation, "ibdf").listFiles(new FileFilter()
			{
				@Override
				public boolean accept(File pathname)
				{
					if (pathname.isFile() && pathname.getName().toLowerCase().endsWith(".ibdf"))
					{
						return true;
					}
					return false;
				}
			});
			
			importUtil_ = new EConceptUtility(Optional.empty(), Optional.of(MetaData.LOINC_SOLOR_MODULE), outputDirectory, converterOutputArtifactId,
				converterOutputArtifactClassifier, false, releaseDate.getTime(),  
				Arrays.asList(new SememeType[] {SememeType.DESCRIPTION, SememeType.COMPONENT_NID, SememeType.LOGIC_GRAPH}), false, ibdfFiles);

			ConsoleUtil.println("Loading Metadata");
			
			// Set up a meta-data root concept
			ComponentReference metadata = ComponentReference.fromConcept(importUtil_.createConcept("LOINC Tech Preview Metadata", true,
					MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid()));
			
			importUtil_.loadTerminologyMetadataAttributes(metadata, converterSourceArtifactVersion, 
					Optional.of(loincData.getReleaseDate()), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);
			importUtil_.addStaticStringAnnotation(metadata, converterSourceLoincArtifactVersion,  MetaData.SOURCE_ARTIFACT_VERSION.getPrimordialUuid(), State.ACTIVE);
			
			PT_Refsets refsets = new PT_Refsets();
			PT_Annotations annotations = new PT_Annotations(new ArrayList<>());
			PT_Descriptions descTypes = new PT_Descriptions();
			
			importUtil_.loadMetaDataItems(Arrays.asList((new PropertyType[] {refsets, annotations, descTypes})), metadata.getPrimordialUuid());
			
			
			//TODO do I need any other attrs right now?

			ConsoleUtil.println("Reading data file into memory.");
			int conCounter = 0;

			HashMap<String, String[]> loincNumToData = new HashMap<>();
			{
				String[] line = loincData.readLine();
				while (line != null)
				{
					if (line.length > 0)
					{
						loincNumToData.put(line[loincData.getFieldMap().get("LOINC_NUM")], line);
					}
					line = loincData.readLine();
					if (loincNumToData.size() % 1000 == 0)
					{
						ConsoleUtil.showProgress();
					}
					if (loincNumToData.size() % 10000 == 0)
					{
						ConsoleUtil.println("Read " + loincNumToData.size() + " lines");
					}
				}
			}
			loincData.close();

			ConsoleUtil.println("Read " + loincNumToData.size()  + " data lines from file");
			
			/*
			 * Columns in this data file are:
			 * id - A UUID for this row
			 * effectiveTime
			 * active - 1 for active
			 * moduleId
			 * refsetId
			 * referencedComponentId
			 * mapTarget - LOINC_NUM
			 * Expression - the goods
			 * definitionStatusId
			 * correlationId
			 * contentOriginId
			 */
			
			loincExpressionDebug.write("line number,expression id,converted expression" + eol);

			
			ConsoleUtil.println("Processing Expressions / Creating Concepts");
			
			LoincExpressionReader ler = new LoincExpressionReader(tpZipFile);
			String[] expressionLine = ler.readLine();
			while (expressionLine != null)
			{
				try
				{
					if (expressionLine.length > 0)
					{
						String[] loincConceptData = loincNumToData.get(expressionLine[ler.getPositionForColumn("mapTarget")]);
						
						if (loincConceptData == null)
						{
							ConsoleUtil.printErrorln("Skipping line " + expLineNumber + " because I can't find loincNum " + expressionLine[ler.getPositionForColumn("mapTarget")]);
						}
						
						boolean active = expressionLine[ler.getPositionForColumn("active")].equals("1");
						if (!active)
						{
							ConsoleUtil.printErrorln("Skipping line " + expLineNumber + " because it is inactive");
						}
						
						if (active && loincConceptData != null)
						{
							ParseTree parseTree;
							String definitionSctid = expressionLine[ler.getPositionForColumn("definitionStatusId")];
							if (definitionSctid.equals(sufficientSctid))
							{
								parseTree = SNOMEDCTParserUtil.parseExpression(expressionLine[ler.getPositionForColumn("Expression")]);
							}
							else if (definitionSctid.equals(necessarySctid))
							{
								//See <<< black magic from http://ihtsdo.org/fileadmin/user_upload/doc/download/doc_CompositionalGrammarSpecificationAndGuide_Current-en-US_INT_20150708.pdf?ok
								parseTree = SNOMEDCTParserUtil.parseExpression("<<< " + expressionLine[ler.getPositionForColumn("Expression")]);
							}
							else
							{
								throw new RuntimeException("Unexpected definition status: " + definitionSctid + " on line " + expLineNumber);
							}

							LogicalExpressionBuilder defBuilder = Get.logicalExpressionBuilderService().getLogicalExpressionBuilder();
							ISAACVisitor visitor = new ISAACVisitor(defBuilder);
							visitor.visit(parseTree);
							LogicalExpression expression = defBuilder.build();
							
							UUID expressionId = UUID.fromString(expressionLine[ler.getPositionForColumn("id")]);
							
							loincExpressionDebug.write(expLineNumber + "," + expressionId + "," + expression.toString() + eol);
							
							
							//Build up a concept with the attributes we want, and the expression from the tech preview
							
							String loincNum = loincConceptData[loincData.getPositionForColumn("LOINC_NUM")];
							ComponentReference concept = ComponentReference.fromConcept(importUtil_.createConcept(buildUUID(loincNum)));
							conCounter++;
							importUtil_.addRelationshipGraph(concept, expressionId, expression, true, null, null);
							importUtil_.addRefsetMembership(concept, PT_Refsets.Refsets.ALL.getProperty().getUUID(), State.ACTIVE, null);
							
							//add descriptions
							ArrayList<ValuePropertyPair> descriptions = new ArrayList<>();
							
							for (String property : descTypes.getPropertyNames())
							{
								String data = loincConceptData[loincData.getPositionForColumn(property)];
								if (!StringUtils.isBlank(data))
								{
									descriptions.add(new ValuePropertyPair(data, descTypes.getProperty(property)));
								}
							}
							
							importUtil_.addDescriptions(concept, descriptions);
							
							//add attributes
							for (String property : annotations.getPropertyNames())
							{
								String data = loincConceptData[loincData.getPositionForColumn(property)];
								if (!StringUtils.isBlank(data))
								{
									importUtil_.addStringAnnotation(concept, data, annotations.getProperty(property).getUUID(), State.ACTIVE);
								}
							}
						}
					}
				}
				catch (Exception e)
				{
					getLog().error("Failed with expression line number at " + expLineNumber + " " + e + " skipping line");
				}
				
				expressionLine = ler.readLine();
				expLineNumber++;
			}
			
			loincExpressionDebug.close();

			ConsoleUtil.println("Created " + conCounter + " concepts total");
			

			ConsoleUtil.println("Data Load Summary:");
			for (String s : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println("  " + s);
			}
			
			ConsoleUtil.println("Finished");
		}

		catch (Exception ex)
		{
			throw new MojoExecutionException("Failed with expression line number at " + expLineNumber, ex);
		}
		finally
		{
			try
			{
				if (importUtil_ != null)
				{
					importUtil_.shutdown();
				}
				if (loincData != null)
				{
					loincData.close();
				}
				if (loincExpressionDebug != null)
				{
					loincExpressionDebug.close();
				}
			}
			catch (IOException e)
			{
				throw new RuntimeException("Failure", e);
			}
		}
	}
	
	private UUID buildUUID(String uniqueIdentifier)
	{
		return ConverterUUID.createNamespaceUUIDFromString(uniqueIdentifier, true);
	}
}
