/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.isaac.loinc.standard;

import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.And;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.ConceptAssertion;
import static gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder.NecessarySet;
import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.TreeMap;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.EConceptUtility;
import gov.va.oia.terminology.converters.sharedUtils.EConceptUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.Property;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.ValuePropertyPair;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.loinc.LOINCReader;
import gov.vha.isaac.loinc.LoincCsvFileReader;
import gov.vha.isaac.loinc.NameMap;
import gov.vha.isaac.loinc.TxtFileReader;
import gov.vha.isaac.loinc.standard.propertyTypes.PT_Annotations;
import gov.vha.isaac.loinc.standard.propertyTypes.PT_Associations;
import gov.vha.isaac.loinc.standard.propertyTypes.PT_Descriptions;
import gov.vha.isaac.loinc.standard.propertyTypes.PT_Refsets;
import gov.vha.isaac.loinc.standard.propertyTypes.PT_Relations;
import gov.vha.isaac.loinc.standard.propertyTypes.PT_SkipAxis;
import gov.vha.isaac.loinc.standard.propertyTypes.PT_SkipClass;
import gov.vha.isaac.loinc.standard.propertyTypes.PT_SkipOther;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.logic.LogicalExpressionBuilder;
import gov.vha.isaac.ochre.api.logic.assertions.ConceptAssertion;

/**
 * 
 * Loader code to convert Loinc into isaac.
 * 
 * Paths are typically controlled by maven, however, the main() method has paths configured so that they
 * match what maven does for test purposes.
 */
@Mojo( name = "convert-loinc-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES )
public class LoincImportMojo extends ConverterBaseMojo
{
	// Need a handle to these
	private PropertyType pt_SkipAxis_;
	private PropertyType pt_SkipClass_;

	private final ArrayList<PropertyType> propertyTypes_ = new ArrayList<PropertyType>();
	
	protected Hashtable<String, Integer> fieldMap_;
	protected Hashtable<Integer, String> fieldMapInverse_;
	
	private HashMap<String, HashMap<String, String>> mapToData = new HashMap<>();

	// Various caches for performance reasons
	private Hashtable<String, PropertyType> propertyToPropertyType_ = new Hashtable<String, PropertyType>();

	private final SimpleDateFormat sdf_ = new SimpleDateFormat("yyyyMMdd");

	Hashtable<UUID, ComponentReference> concepts_ = new Hashtable<UUID, ComponentReference>();

	private NameMap classMapping_;
	
	private TreeMap<String, Long> versionTimeMap_;
	
	private int skippedDeletedItems = 0;
	
	/**
	 * Used for debug. Sets up the same paths that maven would use.... allow the code to be run standalone.
	 */
	public static void main(String[] args) throws Exception
	{
		LoincImportMojo loincConverter = new LoincImportMojo();
		loincConverter.outputDirectory = new File("../loinc-ibdf/target/");
		loincConverter.inputFileLocation = new File("../loinc-ibdf/target/generated-resources/src");
		loincConverter.converterVersion = "foo";
		loincConverter.converterOutputArtifactVersion = "foo";
		loincConverter.converterOutputArtifactClassifier = "foo";
		loincConverter.converterSourceArtifactVersion = "foo";
		loincConverter.execute();
	}
	
	@Override
	protected boolean supportsAnnotationSkipList()
	{
		return true;
	}

	private void initProperties()
	{
		propertyTypes_.add(new PT_Annotations(annotationSkipList));
		propertyTypes_.add(new PT_Descriptions());
		propertyTypes_.add(new PT_Associations());
		propertyTypes_.add(pt_SkipAxis_);
		propertyTypes_.add(pt_SkipClass_);
		propertyTypes_.add(new PT_SkipOther(annotationSkipList));
		propertyTypes_.add(new PT_Refsets());
		propertyTypes_.add(new PT_Relations());
	}

	@Override
	public void execute() throws MojoExecutionException
	{
		ConsoleUtil.println("LOINC Processing Begins " + new Date().toString());
		
		LOINCReader loincData = null;
		LOINCReader mapTo = null;
		LOINCReader sourceOrg = null;
		LOINCReader loincMultiData = null;

		try
		{
			super.execute();

			if (!inputFileLocation.isDirectory())
			{
				throw new MojoExecutionException("LoincDataFiles must point to a directory containing the required loinc data files");
			}
			

			for (File f : inputFileLocation.listFiles())
			{
				if (f.getName().toLowerCase().equals("loincdb.txt"))
				{
					loincData = new TxtFileReader(f);
				}
				else if (f.getName().toLowerCase().equals("loinc.csv"))
				{
					loincData = new LoincCsvFileReader(f, true);
					versionTimeMap_ = ((LoincCsvFileReader)loincData).getTimeVersionMap();
				}
				else if (f.getName().toLowerCase().equals("map_to.csv"))
				{
					mapTo = new LoincCsvFileReader(f, false);
				}
				else if (f.getName().toLowerCase().equals("source_organization.csv"))
				{
					sourceOrg = new LoincCsvFileReader(f, false);
				}
				else if (f.getName().toLowerCase().endsWith("multi-axial_hierarchy.csv"))
				{
					loincMultiData = new LoincCsvFileReader(f, false);
				}
				else if (f.getName().toLowerCase().endsWith(".zip"))
				{
					//New zip file set
					
					@SuppressWarnings("resource")
					ZipFile zf = new ZipFile(f);
					Enumeration<? extends ZipEntry> zipEntries = zf.entries();
					while(zipEntries.hasMoreElements())
					{
						ZipEntry ze = zipEntries.nextElement();
						
						//see {@link SupportedConverterTypes}
						if (f.getName().toLowerCase().contains("text"))
						{
							if (ze.getName().toLowerCase().endsWith("loinc.csv"))
							{
								ConsoleUtil.println("Using the data file " + f.getAbsolutePath() + " - " + ze.getName());
								loincData = new LoincCsvFileReader(zf.getInputStream(ze));
								((LoincCsvFileReader)loincData).readReleaseNotes(f.getParentFile(), true);
								versionTimeMap_ = ((LoincCsvFileReader)loincData).getTimeVersionMap();
							}
							else if (ze.getName().toLowerCase().endsWith("map_to.csv"))
							{
								ConsoleUtil.println("Using the data file " + f.getAbsolutePath() + " - " + ze.getName());
								mapTo = new LoincCsvFileReader(zf.getInputStream(ze));
							}
							else if (ze.getName().toLowerCase().endsWith("source_organization.csv"))
							{
								ConsoleUtil.println("Using the data file " + f.getAbsolutePath() + " - " + ze.getName());
								sourceOrg = new LoincCsvFileReader(zf.getInputStream(ze));
							}
						}
						else if (f.getName().toLowerCase().contains("multi-axial_hierarchy"))
						{
							if (ze.getName().toLowerCase().contains("multi-axial"))
							{
								ConsoleUtil.println("Using the data file " + f.getAbsolutePath() + " - " + ze.getName());
								loincMultiData = new LoincCsvFileReader(zf.getInputStream(ze));
							}
						}
					}
				}
			}

			if (loincData == null)
			{
				throw new MojoExecutionException("Could not find the loinc data file in " + inputFileLocation.getAbsolutePath());
			}
			if (loincMultiData == null)
			{
				throw new MojoExecutionException("Could not find the multi-axial file in " + inputFileLocation.getAbsolutePath());
			}
			
			SimpleDateFormat dateReader = new SimpleDateFormat("MMMMMMMMMMMMM yyyy"); //Parse things like "June 2014"
			Date releaseDate = dateReader.parse(loincData.getReleaseDate());
			
			importUtil_ = new EConceptUtility(Optional.empty(), Optional.of(MetaData.LOINC_MODULE), outputDirectory, converterOutputArtifactId,
				converterOutputArtifactClassifier, false, releaseDate.getTime());
			
			pt_SkipAxis_ = new PT_SkipAxis();
			pt_SkipClass_ = new PT_SkipClass();
			
			String version = loincData.getVersion() ;
			fieldMap_ = loincData.getFieldMap();
			fieldMapInverse_ = loincData.getFieldMapInverse();

			String mapFileName = null;

			if (version.contains("2.36"))
			{
				PropertyType.setSourceVersion(1);
				mapFileName = "classMappings-2.36.txt";
			}
			else if (version.contains("2.38"))
			{
				PropertyType.setSourceVersion(2);
				mapFileName = "classMappings-2.36.txt";  // Yes, wrong one, never made the file for 2.38
			}
			else if (version.contains("2.40"))
			{
				PropertyType.setSourceVersion(3);
				mapFileName = "classMappings-2.40.txt";
			}
			else if (version.contains("2.44"))
			{
				PropertyType.setSourceVersion(4);
				mapFileName = "classMappings-2.44.txt";
			}
			else if (version.contains("2.46"))
			{
				PropertyType.setSourceVersion(4);
				mapFileName = "classMappings-2.46.txt";
			}
			else if (version.contains("2.48"))
			{
				PropertyType.setSourceVersion(4);
				mapFileName = "classMappings-2.48.txt";
			}
			else if (version.contains("2.50"))
			{
				PropertyType.setSourceVersion(5);
				mapFileName = "classMappings-2.52.txt";  //never did a 2.50, skipped to 2.52
			}
			else if (version.contains("2.52"))
			{
				PropertyType.setSourceVersion(6);
				mapFileName = "classMappings-2.52.txt";
			}
			else if (version.contains("2.54"))
			{
				PropertyType.setSourceVersion(7);
				mapFileName = "classMappings-2.54.txt";
			}
			else
			{
				ConsoleUtil.printErrorln("ERROR: UNTESTED VERSION - NO TESTED PROPERTY MAPPING EXISTS!");
				PropertyType.setSourceVersion(7);
				mapFileName = "classMappings-2.54.txt";
			}

			classMapping_ = new NameMap(mapFileName);
			
			if (mapTo != null)
			{
				String[] line = mapTo.readLine();
				while (line != null)
				{
					if (line.length > 0)
					{
						HashMap<String, String> nestedData = mapToData.get(line[0]);
						if (nestedData == null)
						{
							nestedData = new HashMap<>();
							mapToData.put(line[0], nestedData);
						}
						if (nestedData.put(line[1], line[2]) != null)
						{
							throw new Exception("Oops - " + line[0] + " " + line[1] + " " + line[2]);
						}
					}
					line = mapTo.readLine();
				}
			}

			initProperties();

			ConsoleUtil.println("Loading Metadata");

			// Set up a meta-data root concept
			ComponentReference metadata = ComponentReference.fromConcept(importUtil_.createConcept("LOINC Metadata", true, MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid()));
			
			importUtil_.loadTerminologyMetadataAttributes(metadata, converterSourceArtifactVersion, 
					Optional.of(loincData.getReleaseDate()), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);

			importUtil_.loadMetaDataItems(propertyTypes_, metadata.getPrimordialUuid());

			// Load up the propertyType map for speed, perform basic sanity check
			for (PropertyType pt : propertyTypes_)
			{
				for (String propertyName : pt.getPropertyNames())
				{
					if (propertyToPropertyType_.containsKey(propertyName))
					{
						ConsoleUtil.printErrorln("ERROR: Two different property types each contain " + propertyName);
					}
					propertyToPropertyType_.put(propertyName, pt);
				}
			}
			
			if (sourceOrg != null)
			{
				ComponentReference sourceOrgConcept = ComponentReference.fromConcept(importUtil_.createConcept("Source Organization", true, metadata.getPrimordialUuid()));
				String[] line = sourceOrg.readLine();
				while (line != null)
				{
					//﻿"COPYRIGHT_ID","NAME","COPYRIGHT","TERMS_OF_USE","URL"
					if (line.length > 0)
					{
						ComponentReference c = ComponentReference.fromConcept(importUtil_.createConcept(line[0], false, sourceOrgConcept.getPrimordialUuid()));
						importUtil_.addDescription(c, line[1], DescriptionType.SYNONYM, true, propertyToPropertyType_.get("NAME").getProperty("NAME").getUUID(), null, State.ACTIVE);
						importUtil_.addStringAnnotation(c, line[2], propertyToPropertyType_.get("COPYRIGHT").getProperty("COPYRIGHT").getUUID(), State.ACTIVE);
						importUtil_.addStringAnnotation(c, line[3], propertyToPropertyType_.get("TERMS_OF_USE").getProperty("TERMS_OF_USE").getUUID(), State.ACTIVE);
						importUtil_.addStringAnnotation(c, line[4], propertyToPropertyType_.get("URL").getProperty("URL").getUUID(), State.ACTIVE);
					}
					line = sourceOrg.readLine();
				}
			}

			UUID loincAllConceptsRefset = PT_Refsets.Refsets.ALL.getProperty().getUUID();

			// The next line of the file is the header.
			String[] headerFields = loincData.getHeader();

			// validate that we are configured to map all properties properly
			checkForLeftoverPropertyTypes(headerFields);
			
			ConsoleUtil.println("Metadata summary:");
			for (String s : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println("  " + s);
			}
			importUtil_.clearLoadStats();

			// Root
			ComponentReference rootConcept = ComponentReference.fromConcept(importUtil_.createConcept("LOINC", true, MetaData.ISAAC_ROOT.getPrimordialUuid()));
			importUtil_.addDescription(rootConcept, "Logical Observation Identifiers Names and Codes", DescriptionType.SYNONYM, false, null, null, State.ACTIVE);
			ConsoleUtil.println("Root concept FSN is 'LOINC' and the UUID is " + rootConcept.getPrimordialUuid());

			concepts_.put(rootConcept.getPrimordialUuid(), rootConcept);

			// Build up the Class metadata
			ComponentReference classConcept = ComponentReference.fromConcept(importUtil_.createConcept(pt_SkipClass_.getPropertyTypeUUID(), pt_SkipClass_.getPropertyTypeDescription(),
					true, rootConcept.getPrimordialUuid()));
			concepts_.put(classConcept.getPrimordialUuid(), classConcept);

			for (String property : pt_SkipClass_.getPropertyNames())
			{
				ComponentReference temp = ComponentReference.fromConcept(importUtil_.createConcept(pt_SkipClass_.getProperty(property).getUUID(), property, 
						true, classConcept.getPrimordialUuid()));
				concepts_.put(temp.getPrimordialUuid(), temp);
				importUtil_.configureConceptAsAssociation(temp.getPrimordialUuid(), null);
			}

			// And the axis metadata
			ComponentReference axisConcept = ComponentReference.fromConcept(importUtil_.createConcept(pt_SkipAxis_.getPropertyTypeUUID(), 
					pt_SkipAxis_.getPropertyTypeDescription(), true, rootConcept.getPrimordialUuid()));
			concepts_.put(axisConcept.getPrimordialUuid(), axisConcept);

			for (String property : pt_SkipAxis_.getPropertyNames())
			{
				ComponentReference temp = ComponentReference.fromConcept(importUtil_.createConcept(pt_SkipAxis_.getProperty(property).getUUID(), property, 
						true, axisConcept.getPrimordialUuid()));
				concepts_.put(temp.getPrimordialUuid(), temp);
				importUtil_.configureConceptAsAssociation(temp.getPrimordialUuid(), null);
			}

			// load the data
			ConsoleUtil.println("Processing file....");

			int dataRows = 0;
			{
				String[] line = loincData.readLine();
				dataRows++;
				while (line != null)
				{
					if (line.length > 0)
					{
						processDataLine(line);
					}
					line = loincData.readLine();
					dataRows++;
					if (dataRows % 1000 == 0)
					{
						ConsoleUtil.showProgress();
					}
					if (dataRows % 10000 == 0)
					{
						ConsoleUtil.println("Processed " + dataRows + " lines");
					}
				}
			}
			loincData.close();

			ConsoleUtil.println("Read " + dataRows + " data lines from file");

			ConsoleUtil.println("Processing multi-axial file");

			{
				// header - PATH_TO_ROOT,SEQUENCE,IMMEDIATE_PARENT,CODE,CODE_TEXT
				int lineCount = 0;
				String[] line = loincMultiData.readLine();
				while (line != null)
				{
					lineCount++;
					if (line.length > 0)
					{
						processMultiAxialData(rootConcept.getPrimordialUuid(), line);
					}
					line = loincMultiData.readLine();
					if (lineCount % 1000 == 0)
					{
						ConsoleUtil.showProgress();
					}
				}
				loincMultiData.close();
				ConsoleUtil.println("Read " + lineCount + " data lines from file.  Creating graphs and hierarcy concepts");
				for (Entry<UUID, HashSet<UUID>> items : multiaxialPathsToRoot.entrySet())
				{
					UUID source = items.getKey();
					HashSet<UUID> parents = items.getValue();
					LogicalExpressionBuilder leb = Get.logicalExpressionBuilderService().getLogicalExpressionBuilder();
					ConceptAssertion[] assertions = new ConceptAssertion[parents.size()];
					int i = 0;
					for (UUID parent : parents)
					{
						assertions[i++] = ConceptAssertion(Get.identifierService().getConceptSequenceForUuids(parent), leb);
					}
					NecessarySet(And(assertions));
					importUtil_.addRelationshipGraph(ComponentReference.fromConcept(source), null, leb.build(), true, null,  null);
				}
			}

			ConsoleUtil.println("Creating all concepts refset");

			//Add all of the concepts to a refset
			for (ComponentReference concept : concepts_.values())
			{
				importUtil_.addRefsetMembership(concept, loincAllConceptsRefset, State.ACTIVE, null);
			}
			
			ConsoleUtil.println("Processed " + concepts_.size() + " concepts total");

			ConsoleUtil.println("Data Load Summary:");
			for (String s : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println("  " + s);
			}

			ConsoleUtil.println("Skipped " + skippedDeletedItems + " Loinc codes because they were flagged as DELETED and they had no desriptions.");
			
			// this could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "loincUuid");
			ConsoleUtil.println("LOINC Processing Completes " + new Date().toString());
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		}
		catch (Exception ex)
		{
			try
			{
				// make sure this is dumped
				ConverterUUID.dump(outputDirectory, "loincUuid");
			}
			catch (IOException e)
			{
				//noop
			}
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}
		finally
		{
			try
			{
				if (importUtil_ != null)
				{
					importUtil_.shutdown();
				}
				if (loincData != null)
				{
					loincData.close();
				}
				if (loincMultiData != null)
				{
					loincMultiData.close();
				}
				if (mapTo != null)
				{
					mapTo.close();
				}
				if (sourceOrg != null)
				{
					sourceOrg.close();
				}
			}
			catch (IOException e)
			{
				throw new MojoExecutionException(e.getLocalizedMessage(), e);
			}
		}
	}

	private void processDataLine(String[] fields) throws ParseException, IOException
	{
		Integer index = fieldMap_.get("VersionLastChanged");  // They changed this in 2.54 release
		Long time = null;
		if (index != null)
		{
			time = versionTimeMap_.get(fields[index]);
			if (time == null)
			{
				throw new IOException("Couldn't find time for version " + fields[index]);
			}
		}
		else
		{
			index = fieldMap_.get("DATE_LAST_CHANGED");  // They changed this in 2.38 release
			if (index == null)
			{
				index = fieldMap_.get("DT_LAST_CH");
			}
			String lastChanged = fields[index];
			time = (StringUtils.isBlank(lastChanged) ? null : sdf_.parse(lastChanged).getTime());
		}

		State status = mapStatus(fields[fieldMap_.get("STATUS")]);

		String code = fields[fieldMap_.get("LOINC_NUM")];

		ComponentReference concept = ComponentReference.fromConcept(importUtil_.createConcept(buildUUID(code), time, status, null));
		
		ArrayList<ValuePropertyPair> descriptions = new ArrayList<>();

		for (int fieldIndex = 0; fieldIndex < fields.length; fieldIndex++)
		{
			if (fields[fieldIndex] != null && fields[fieldIndex].length() > 0)
			{
				PropertyType pt = propertyToPropertyType_.get(fieldMapInverse_.get(fieldIndex));
				if (pt == null)
				{
					ConsoleUtil.printErrorln("ERROR: No property type mapping for the property " + fieldMapInverse_.get(fieldIndex) + ":" + fields[fieldIndex]);
					continue;
				}

				Property p = pt.getProperty(fieldMapInverse_.get(fieldIndex));

				if (pt instanceof PT_Annotations)
				{
					if ((p.getSourcePropertyNameFSN().equals("COMMON_TEST_RANK") || p.getSourcePropertyNameFSN().equals("COMMON_ORDER_RANK") 
							|| p.getSourcePropertyNameFSN().equals("COMMON_SI_TEST_RANK")) && fields[fieldIndex].equals("0"))
					{
						continue;  //Skip attributes of these types when the value is 0
					}
					importUtil_.addStringAnnotation(concept, fields[fieldIndex], p.getUUID(), (p.isDisabled() ? State.INACTIVE : State.ACTIVE));
				}
				else if (pt instanceof PT_Descriptions)
				{
					//Gather for later
					descriptions.add(new ValuePropertyPair(fields[fieldIndex], p));
				}
				else if (pt instanceof PT_SkipAxis)
				{
					// See if this class object exists yet.
					UUID potential = ConverterUUID.createNamespaceUUIDFromString(pt_SkipAxis_.getPropertyTypeDescription() + ":" +
							fieldMapInverse_.get(fieldIndex) + ":" + fields[fieldIndex], true);

					ComponentReference axisConcept = concepts_.get(potential);
					if (axisConcept == null)
					{
						axisConcept = ComponentReference.fromConcept(importUtil_.createConcept(potential, fields[fieldIndex], true));
						importUtil_.addParent(axisConcept, pt_SkipAxis_.getProperty(fieldMapInverse_.get(fieldIndex)).getUUID());
						concepts_.put(axisConcept.getPrimordialUuid(), axisConcept);
					}
					importUtil_.addAssociation(concept, null, axisConcept.getPrimordialUuid(), pt_SkipAxis_.getProperty(fieldMapInverse_.get(fieldIndex)).getUUID(),
							State.ACTIVE, concept.getTime(), null);
				}
				else if (pt instanceof PT_SkipClass)
				{
					// See if this class object exists yet.
					UUID potential = ConverterUUID.createNamespaceUUIDFromString(pt_SkipClass_.getPropertyTypeDescription() + ":" +
							fieldMapInverse_.get(fieldIndex) + ":" + fields[fieldIndex], true);

					ComponentReference classConcept = concepts_.get(potential);
					if (classConcept == null)
					{
						classConcept = ComponentReference.fromConcept(importUtil_.createConcept(potential, classMapping_.getMatchValue(fields[fieldIndex]), true));
						if (classMapping_.hasMatch(fields[fieldIndex]))
						{
							importUtil_.addStringAnnotation(classConcept, fields[fieldIndex], propertyToPropertyType_.get("ABBREVIATION").getProperty("ABBREVIATION")
									.getUUID(), State.ACTIVE);
						}
						importUtil_.addParent(classConcept, pt_SkipClass_.getProperty(fieldMapInverse_.get(fieldIndex)).getUUID());
						concepts_.put(classConcept.getPrimordialUuid(), classConcept);
					}
					importUtil_.addAssociation(concept, null, classConcept.getPrimordialUuid(), pt_SkipClass_.getProperty(fieldMapInverse_.get(fieldIndex)).getUUID(),
							State.ACTIVE, concept.getTime(), null);
				}
				else if (pt instanceof PT_Relations)
				{
					//This will only ever be is_a
					UUID parent = buildUUID(fields[fieldIndex]);
					importUtil_.addParent(concept, parent, pt.getProperty(fieldMapInverse_.get(fieldIndex)), null);
				}
				else if (pt instanceof PT_Associations)
				{
					importUtil_.addAssociation(concept, null, buildUUID(fields[fieldIndex]), pt.getProperty(fieldMapInverse_.get(fieldIndex)).getUUID(), 
							State.ACTIVE, concept.getTime(), null);
				}
				else if (pt instanceof PT_SkipOther)
				{
					importUtil_.getLoadStats().addSkippedProperty();
				}
				else
				{
					ConsoleUtil.printErrorln("oops - unexpected property type: " + pt);
				}
			}
		}
		
		//MAP_TO moved to a different file in 2.42.
		HashMap<String, String> mappings = mapToData.get(code);
		if (mappings != null)
		{
			for (Entry<String, String> mapping : mappings.entrySet())
			{
				String target = mapping.getKey();
				String comment = mapping.getValue();
				ComponentReference cr = ComponentReference.fromChronology(importUtil_.addAssociation(concept, null, buildUUID(target), 
						propertyToPropertyType_.get("MAP_TO").getProperty("MAP_TO").getUUID(), State.ACTIVE, concept.getTime(), null), () -> "Association");
				if (comment != null && comment.length() > 0)
				{
					importUtil_.addStringAnnotation(cr, comment, propertyToPropertyType_.get("COMMENT").getProperty("COMMENT").getUUID(), State.ACTIVE);
				}
			}
		}
		
		//Now add all the descriptions
		if (descriptions.size() == 0)
		{
			if ("DEL".equals(fields[fieldMap_.get("CHNG_TYPE")]))
			{
				//They put a bunch of these in 2.44... leaving out most of the important info... just makes a mess.  Don't load them.
				skippedDeletedItems++;
				return;
			}
			else
			{
				ConsoleUtil.printErrorln("ERROR: no name for " + code);
				importUtil_.addFullySpecifiedName(concept, code);
			}
		}
		else
		{
			importUtil_.addDescriptions(concept, descriptions);
		}

		ComponentReference current = concepts_.put(concept.getPrimordialUuid(), concept);
		if (current != null)
		{
			ConsoleUtil.printErrorln("Duplicate LOINC code (LOINC_NUM):" + code);
		}
	}

	private HashMap<UUID, HashSet<UUID>> multiaxialPathsToRoot = new HashMap<>();
	
	private void processMultiAxialData(UUID rootConcept, String[] line)
	{
		// PATH_TO_ROOT,SEQUENCE,IMMEDIATE_PARENT,CODE,CODE_TEXT
		// This file format used to be a disaster... but it looks like since 2.40, they encode proper CSV, so I've thrown out the custom parsing.
		// If you need the old custom parser that reads the crap they used to produce as 'CSV', look at the SVN history for this method. 

		String pathString = line[0];
		String[] pathToRoot = (pathString.length() > 0 ? pathString.split("\\.") : new String[] {});

		String sequence = line[1];
		
		String immediateParentString = line[2];

		UUID immediateParent = (immediateParentString == null || immediateParentString.length() == 0 ? rootConcept : buildUUID(immediateParentString));

		String code = line[3];

		String codeText = line[4];

		if (code.length() == 0 || codeText.length() == 0)
		{
			ConsoleUtil.printErrorln("missing code or text!");
		}

		UUID potential = buildUUID(code);

		ComponentReference concept = concepts_.get(potential);
		if (concept == null)
		{
			concept = ComponentReference.fromConcept(importUtil_.createConcept(potential));
			if (sequence != null && sequence.length() > 0)
			{
				importUtil_.addStringAnnotation(concept, sequence, propertyToPropertyType_.get("SEQUENCE").getProperty("SEQUENCE").getUUID(), State.ACTIVE);
			}

			if (immediateParentString != null && immediateParentString.length() > 0)
			{
				importUtil_.addStringAnnotation(concept, immediateParentString, propertyToPropertyType_.get("IMMEDIATE_PARENT").getProperty("IMMEDIATE_PARENT")
						.getUUID(), State.ACTIVE);
			}

			ValuePropertyPair vpp = new ValuePropertyPair(codeText, propertyToPropertyType_.get("CODE_TEXT").getProperty("CODE_TEXT"));
			importUtil_.addDescriptions(concept, Arrays.asList(vpp));  //This will get added as FSN

			HashSet<UUID> parents = multiaxialPathsToRoot.get(concept.getPrimordialUuid());
			if (parents == null)
			{
				parents = new HashSet<UUID>();
				multiaxialPathsToRoot.put(concept.getPrimordialUuid(), parents);
			}
			parents.add(immediateParent);

			if (pathString != null && pathString.length() > 0)
			{
				importUtil_.addStringAnnotation(concept, pathString, propertyToPropertyType_.get("PATH_TO_ROOT").getProperty("PATH_TO_ROOT").getUUID(), State.ACTIVE);
			}
			importUtil_.addStringAnnotation(concept, code, propertyToPropertyType_.get("CODE").getProperty("CODE").getUUID(), State.ACTIVE);

			concepts_.put(concept.getPrimordialUuid(), concept);
		}

		// Make sure everything in pathToRoot is linked.
		checkPath(concept, pathToRoot);
	}

	private void checkPath(ComponentReference concept, String[] pathToRoot)
	{
		// The passed in concept should have a relation to the item at the end of the root list.
		for (int i = (pathToRoot.length - 1); i >= 0; i--)
		{
			UUID target = buildUUID(pathToRoot[i]);
			HashSet<UUID> parents = multiaxialPathsToRoot.get(concept.getPrimordialUuid());
			if (parents == null)
			{
				parents = new HashSet<UUID>();
				multiaxialPathsToRoot.put(concept.getPrimordialUuid(), parents);
			}
			parents.add(target);
			
			concept = concepts_.get(target);
			if (concept == null)
			{
				ConsoleUtil.printErrorln("Missing concept! " + pathToRoot[i]);
				break;
			}
		}
	}

	private State mapStatus(String status) throws IOException
	{
		if (status.equals("ACTIVE") || status.equals("TRIAL") || status.equals("DISCOURAGED"))
		{
			return State.ACTIVE;
		}
		else if (status.equals("DEPRECATED"))
		{
			return State.INACTIVE;
		}
		else
		{
			ConsoleUtil.printErrorln("No mapping for status: " + status);
			return State.ACTIVE;
		}
	}

	private void checkForLeftoverPropertyTypes(String[] fileColumnNames) throws Exception
	{
		for (String name : fileColumnNames)
		{
			PropertyType pt = propertyToPropertyType_.get(name);
			if (pt == null)
			{
				ConsoleUtil.printErrorln("ERROR:  No mapping for property type: " + name);
			}
		}
	}

	/**
	 * Utility to help build UUIDs in a consistent manner.
	 */
	private UUID buildUUID(String uniqueIdentifier)
	{
		return ConverterUUID.createNamespaceUUIDFromString(uniqueIdentifier, true);
	}
}
