/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.va.isaac.gui.refexViews.refexCreation.wizardPages;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.isaac.AppContext;
import gov.va.isaac.gui.refexViews.refexCreation.PanelControllersI;
import gov.va.isaac.gui.refexViews.refexCreation.ScreensController;
import gov.va.isaac.gui.refexViews.refexCreation.SememeGUIData;
import gov.va.isaac.gui.refexViews.util.DynamicSememeDataColumnListCell;
import gov.vha.isaac.ochre.api.chronicle.ObjectChronologyType;
import gov.vha.isaac.ochre.api.component.sememe.SememeType;
import gov.vha.isaac.ochre.api.component.sememe.version.dynamicSememe.DynamicSememeColumnInfo;
import gov.vha.isaac.ochre.impl.utility.Frills;
import javafx.fxml.FXML;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.ListCell;
import javafx.scene.control.ListView;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.Region;
import javafx.stage.Stage;
import javafx.util.Callback;
/**
 * 
 * {@link SummaryController}
 *
 * @author <a href="jefron@apelon.com">Jesse Efron</a>
 * @author <a href="mailto:daniel.armbrust.list@gmail.com">Dan Armbrust</a>
 */
public class SummaryController implements PanelControllersI {
	@FXML private Label actualRefexName;
	@FXML private Label actualRefexDescription;
	@FXML private Label actualParentConcept;
	@FXML private Label actualComponentTypeRestriction;
	@FXML private Label actualComponentTypeSubRestriction;
	@FXML private BorderPane summaryPane;
	@FXML private ListView<DynamicSememeColumnInfo> detailsListView;
	@FXML private Button cancelButton;
	@FXML private Button startOverButton;
	@FXML private Button commitButton;
	@FXML private Button backButton;

	ScreensController processController_;
	Region sceneParent_;

	private final Logger logger = LoggerFactory.getLogger(SummaryController.class);

	@Override
	public void initialize() {
		assert actualComponentTypeRestriction != null : "fx:id=\"actualComponentTypeRestriction\" was not injected: check your FXML file 'summary.fxml'.";
		assert actualComponentTypeSubRestriction != null : "fx:id=\"actualComponentTypeSubRestriction\" was not injected: check your FXML file 'summary.fxml'.";
		
		assert actualRefexDescription != null : "fx:id=\"actualRefexDescription\" was not injected: check your FXML file 'summary.fxml'.";
		assert actualRefexName != null : "fx:id=\"actualRefexName\" was not injected: check your FXML file 'summary.fxml'.";
		assert cancelButton != null : "fx:id=\"cancelButton\" was not injected: check your FXML file 'summary.fxml'.";
		assert startOverButton != null : "fx:id=\"startOverButton\" was not injected: check your FXML file 'summary.fxml'.";
		assert commitButton != null : "fx:id=\"commitButton\" was not injected: check your FXML file 'summary.fxml'.";
		assert backButton != null : "fx:id=\"commitButton\" was not injected: check your FXML file 'summary.fxml'.";
		assert summaryPane != null : "fx:id=\"summaryPane\" was not injected: check your FXML file 'summary.fxml'.";
		assert actualParentConcept != null : "fx:id=\"actualParentConcept\" was not injected: check your FXML file 'summary.fxml'.";
		assert detailsListView != null : "fx:id=\"detailsListView\" was not injected: check your FXML file 'summary.fxml'.";

		cancelButton.setOnAction(e -> ((Stage)summaryPane.getScene().getWindow()).close());
	
		commitButton.setOnAction(e -> 
		{
			try
			{
				AppContext.getRuntimeGlobals().disableAllCommitListeners();
				storeValues();
			} catch (Exception e1) {
				logger.error("Coudn't Disable WF Init & Commit New Dynamic Sememe Concept", e1);
			}
			finally
			{
			    AppContext.getRuntimeGlobals().enableAllCommitListeners();
			}

			((Stage)summaryPane.getScene().getWindow()).close();
		});

		startOverButton.setOnAction(e -> processController_.showFirstScreen());
		backButton.setOnAction(e -> processController_.showPreviousScreen());
		
		detailsListView.setCellFactory(new Callback<ListView<DynamicSememeColumnInfo>, ListCell<DynamicSememeColumnInfo>>()
		{
			@Override
			public ListCell<DynamicSememeColumnInfo> call(ListView<DynamicSememeColumnInfo> param)
			{
				return new DynamicSememeDataColumnListCell();
			}
		});
	}

	private void setupRefexContent(SememeGUIData refexData) {
		actualRefexName.setText(refexData.getRefexName());
		actualRefexDescription.setText(refexData.getRefexDescription());
		actualParentConcept.setText(refexData.getParentConcept().getConceptDescriptionText());
		actualComponentTypeRestriction.setText(refexData.getComponentRestrictionType() == ObjectChronologyType.UNKNOWN_NID ? "No Restriction" 
				: refexData.getComponentRestrictionType().toString()); 
		actualComponentTypeSubRestriction.setText(refexData.getComponentSubRestrictionType() == SememeType.UNKNOWN ? "No Subtype Restriction" 
				: refexData.getComponentSubRestrictionType().toString()); 
		
		detailsListView.getItems().clear();
		detailsListView.getItems().addAll(refexData.getColumnInfo());
		detailsListView.scrollTo(0);
	}

	public void storeValues() {
		try {
			SememeGUIData refexData = processController_.getWizardData();
			Frills.createNewDynamicSememeUsageDescriptionConcept(refexData.getRefexName(),
					refexData.getRefexName(), refexData.getRefexDescription(), refexData.getColumnInfo().toArray(new DynamicSememeColumnInfo[0]), 
					refexData.getParentConcept().getNid(), 
					refexData.getComponentRestrictionType(), 
					refexData.getComponentSubRestrictionType());
		} catch (RuntimeException e) {
			logger.error("Unable to create and/or commit refset concept and metadata", e);
			AppContext.getCommonDialogs().showErrorDialog("Error Creating Sememe", "Unexpected error creating the Sememe", e.getMessage(), summaryPane.getScene().getWindow());
		}
	}
	
	public void updateValues(SememeGUIData refexData)
	{
		setupRefexContent(refexData);
	}

	/**
	 * @see gov.va.isaac.gui.refexViews.refexCreation.PanelControllersI#finishInit(gov.va.isaac.gui.refexViews.refexCreation.ScreensController, javafx.scene.Parent)
	 */
	@Override
	public void finishInit(ScreensController screenController, Region parent)
	{
		processController_ = screenController;
		sceneParent_ = parent;
	}

	/**
	 * @see gov.va.isaac.gui.refexViews.refexCreation.PanelControllersI#getParent()
	 */
	@Override
	public Region getParent()
	{
		return sceneParent_;
	}
}
