/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.va.isaac.gui.conceptCreation.wizardPages;

import gov.va.isaac.gui.ConceptNode;
import gov.va.isaac.gui.SimpleDisplayConcept;
import gov.va.isaac.gui.util.ErrorMarkerUtils;
import gov.va.isaac.util.UpdateableBooleanBinding;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.Get;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.scene.Node;
import javafx.scene.control.ChoiceBox;
import javafx.scene.control.TextField;

/**
 * {@link RelRow}
 *
 * @author <a href="mailto:daniel.armbrust.list@gmail.com">Dan Armbrust</a>
 */
public class RelRow
{
	ConceptNode relationship;
	ConceptNode target;
	ChoiceBox<RoleType> type;
	Node typeNode;
	TextField group;
	Node groupNode;
	
	//TODO (artf231869) rework this binding code - the updates to the utility code merge the SimpleStringProperty aspect into the UpdatedableBooleanBinding - 
	//so the binding now carries the details about valid, and the reason why not valid, if not - which vastly simplifies piecing this together.  Ask Dan...
	private UpdateableBooleanBinding rowValid;
	SimpleStringProperty groupFieldInvalidReason_ = new SimpleStringProperty("");
	SimpleStringProperty typeFieldInvalidReason_ = new SimpleStringProperty("A Type selection is required");
	
	// TODO (artf231869) add validation of this type (unless drools covers it)
	SimpleStringProperty typeTargetInvalidReason_ = new SimpleStringProperty("Relationship type may not be same as relationship target");

	// TODO (artf231869) Need to add soon
	SimpleStringProperty typeIsaInvalidReason_ = new SimpleStringProperty("Relationship type may not be IS_A as that is defined in first screen");

	public RelRow()
	{
		ObservableList<SimpleDisplayConcept> dropDownOptions = FXCollections.observableArrayList();
		dropDownOptions.add(new SimpleDisplayConcept(Get.conceptSnapshot().getConceptSnapshot(MetaData.IS_A.getNid())));
		relationship = new ConceptNode(null, true, dropDownOptions, null);
		target = new ConceptNode(null, true);
		
		//TODO (artf231868) add validation icons / reasons
		type = new ChoiceBox<RoleType>(FXCollections.observableArrayList(RoleType.All_Role, RoleType.Some_Role));
		type.valueProperty().addListener(new ChangeListener<RoleType>()
		{
			@Override
			public void changed(ObservableValue<? extends RoleType> observable, RoleType oldValue, RoleType newValue)
			{
				
				String type = newValue.toString().trim();
				
				if (type.length() == 0)
				{
					typeFieldInvalidReason_.set("A Type selection is required");
				}
				else
				{
					typeFieldInvalidReason_.set("");
				}
			}
		});
		
		typeNode = ErrorMarkerUtils.setupErrorMarker(type, typeFieldInvalidReason_);
		
		group = new TextField("0");
		group.setMinWidth(45.0);  //TODO (artf231869) figure out why this is needed
		group.textProperty().addListener(new ChangeListener<String>()
		{
			@Override
			public void changed(ObservableValue<? extends String> observable, String oldValue, String newValue)
			{
				
				String group = newValue.trim();
				
				if (group.length() == 0)
				{
					groupFieldInvalidReason_.set("A group is required");
				}
				else
				{
					try
					{
						Integer.parseInt(group);
						groupFieldInvalidReason_.set("");
					}
					catch (NumberFormatException e)
					{
						groupFieldInvalidReason_.set("The group field must be a number");
					}
				}
			}
		});
		
		groupNode = ErrorMarkerUtils.setupErrorMarker(group, groupFieldInvalidReason_);
		rowValid = new UpdateableBooleanBinding()
		{
			{
				setComputeOnInvalidate(true);
				bind(relationship.isValid(), target.isValid(), typeFieldInvalidReason_, groupFieldInvalidReason_);
			}
			@Override
			protected boolean computeValue()
			{
				return (relationship.isValid().get() && target.isValid().get() && 
						typeFieldInvalidReason_.get().length() == 0 && groupFieldInvalidReason_.get().length() == 0);
			}
		};
	}

	public BooleanBinding isValid()
	{
		return rowValid;
	}
	
	/**
	 * @return the relationship
	 */
	public ConceptNode getRelationshipNode()
	{
		return relationship;
	}
	
	public int getRelationshipNid()
	{
		return relationship.getConcept().getNid();
	}

	public ConceptNode getTargetNode()
	{
		return target;
	}
	
	public int getTargetNid()
	{
		return target.getConcept().getNid();
	}

	/**
	 * @return the type
	 */
	public Node getTypeNode()
	{
		return typeNode;
	}

	public RoleType getType()
	{
		if ("All Role".equalsIgnoreCase(type.getSelectionModel().getSelectedItem().toString()))
		{
			return RoleType.All_Role;
		}
		else
		{
			return RoleType.Some_Role;
		}
	}

	/**
	 * @return the group
	 */
	public Node getGroupNode()
	{
		return groupNode;
	}
	
	public int getGroup()
	{
		return Integer.parseInt(group.getText().trim());
	}
}
