/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.isaac.utils.file_transfer;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.util.concurrent.Callable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * {@link SimpleUpload}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
public class SimpleUpload implements Callable<URL>
{
	private static Logger log = LoggerFactory.getLogger(SimpleUpload.class);
	String username_, password_;
	File sourceFile_;
	URL destinationUrl_;
	boolean overwrite_;
	
	/**
	 * @param username (optional) used if provided
	 * @param password (optional) used if provided
	 * @param sourceFile The file to upload
	 * @param destinationUrl The URL to upload to
	 * @throws IOException
	 */
	public SimpleUpload(
			String username,
			String password,
			File sourceFile,
			URL destinationUrl,
			boolean overwrite) throws IOException
	{
		this.username_ = username;
		this.password_ = password;
		
		this.sourceFile_ = sourceFile;
		this.destinationUrl_ = destinationUrl;
		
		this.overwrite_ = overwrite;
	}
	
	public static URL upload(File file, URL destinationUrl, String username, String password, boolean overwrite) throws Exception
	{
		OutputStream out = null;
		FileInputStream fis = null;
		HttpURLConnection httpCon = null;
		try
		{
			if (FileTransfer.hasBeenSuccessfullyHandled(file.getAbsolutePath()) || (! overwrite && FileTransferUtils.remoteFileExists(destinationUrl, username, password))) {
				log.debug("Not uploading already uploaded file " + file.getAbsolutePath() + " to " + destinationUrl.toString());
				FileTransfer.setHasBeenHandled(file.getAbsolutePath(), true);

				return destinationUrl;
			} else {
				FileTransfer.setHasBeenHandled(file.getAbsolutePath(), false);
			}

			log.debug("Uploading " + file.getAbsolutePath() + " to " + destinationUrl.toString());

			httpCon = NetworkUtils.getConnection(destinationUrl, username, password);
			httpCon.setDoOutput(true);
			httpCon.setRequestMethod("PUT");
			httpCon.setConnectTimeout(30 * 1000);
			httpCon.setReadTimeout(60 * 60 * 1000);
			//httpCon.setChunkedStreamingMode(8192);
			long fileLength = file.length();
			httpCon.setFixedLengthStreamingMode(fileLength);
			out = httpCon.getOutputStream();

			byte[] buf = new byte[8192];
			fis = new FileInputStream(file);
			int read = 0;
			while ((read = fis.read(buf, 0, buf.length)) > 0)
			{
				out.write(buf, 0, read);
			}
			out.flush();
			out.close();
			fis.close();

			log.debug("Done uploading " + file.getAbsolutePath() + " to " + destinationUrl.toString());

			FileTransfer.setHasBeenHandled(file.getAbsolutePath(), true);

			StringBuilder sb = new StringBuilder();
			try {
				InputStream is = httpCon.getInputStream();
				read = 0;
				byte[] buffer = new byte[1024];
				CharBuffer cBuffer = ByteBuffer.wrap(buffer).asCharBuffer();
				while (read != -1)
				{
					read = is.read(buffer);
					if (read > 0)
					{
						sb.append(cBuffer, 0, read);
					}
				}
			} catch (IOException e) {
				log.debug("Unable to read server response message after uploading file " + file.getCanonicalFile().getAbsolutePath() + ") to destination (" + destinationUrl + ")");
			}
			if (sb.toString().trim().length() > 0)
			{
				throw new Exception("The server reported an error during the publish operation of file " + file.getCanonicalFile().getAbsolutePath() + ":  " + sb.toString());
			}
		}
		catch (Exception e1)
		{
			log.error("Failed to upload file (" + file.getCanonicalFile().getAbsolutePath() + ") to destination (" + destinationUrl + ")", e1);
		}
		finally {
			if (httpCon != null) {
				httpCon.disconnect();
			}
			if (out != null) {
				out.flush();
				out.close();
			}
			if (fis != null) {
				fis.close();
			}
		}

		return destinationUrl;
	}

	/* (non-Javadoc)
	 * @see java.util.concurrent.Callable#call()
	 */
	public URL call() throws Exception
	{
		return upload(sourceFile_, destinationUrl_, username_, password_, overwrite_);
	}
}