package gov.vha.isaac.utils.file_transfer;

import java.io.File;
import java.io.IOException;
import java.net.Authenticator;
import java.net.MalformedURLException;
import java.net.PasswordAuthentication;
import java.net.URL;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;

import gov.vha.isaac.utils.file_transfer.Config.Artifact;

/**
 * 
 * {@link FileTransfer}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 * FileTransfer is a command-line utility for copying maven artifacts from one repo to another.
 * It downloads from specified repo URL to local filesystem (current directory)
 * or uploads from local file system (current directory) to specified repo URL
 * 
 * FileTransfer and its helper classes contain many static methods, but the package is
 * not designed to be used as a toolkit, but rather as a command-line application.'
 * This is because many of the methods interact with a cache of handled/unhandled files,
 * which prevents redundant downloads and uploads, and which if not used as coded in the
 * command-line tool could yield unexpected results.
 * 
 */
public class FileTransfer
{
	private static Logger log = LoggerFactory.getLogger(FileTransfer.class);
	
	/**
	 * FILE_ACTION_RESULTS is a static cache of files
	 * for which upload or downloaded has been attempted.
	 * 
	 * It is used to (optionally) prevent redundant downloads
	 * and to generate a summary report before exiting
	 */
	private static Map<String, Boolean> FILE_ACTION_RESULTS = Collections.synchronizedMap(new ConcurrentHashMap<>());

	/**
	 * @param file
	 * @return
	 * @throws IOException
	 * 
	 * Returns true IFF a File's canonical name has been cached
	 * and associated with a boolean success indicator of true
	 */
	static boolean hasBeenSuccessfullyHandled(File file) throws IOException {
		return hasBeenSuccessfullyHandled(file.getAbsolutePath().toLowerCase());
	}
	/**
	 * @param filePath
	 * @return
	 * @throws IOException
	 * 
	 * Returns true IFF a File's canonical name has been cached
	 * and associated with a boolean success indicator of true
	 */
	static boolean hasBeenSuccessfullyHandled(String filePath) throws IOException {
		String canonicalFileName = new File(filePath).getCanonicalFile().toString();
		return FILE_ACTION_RESULTS.get(canonicalFileName) != null && FILE_ACTION_RESULTS.get(canonicalFileName);
	}
	/**
	 * @param file
	 * @return
	 * @throws IOException
	 * 
	 * Returns true IFF a File's canonical name has been cached
	 * and associated with a boolean success indicator of false
	 */
	static boolean hasBeenUnsuccessfullyHandled(File file) throws IOException {
		return hasBeenUnsuccessfullyHandled(file.getAbsolutePath().toLowerCase());
	}
	/**
	 * @param filePath
	 * @return
	 * @throws IOException
	 * 
	 * Returns true IFF a File's canonical name has been cached
	 * and associated with a boolean success indicator of false
	 */
	static boolean hasBeenUnsuccessfullyHandled(String filePath) throws IOException {
		String canonicalFileName = new File(filePath).getCanonicalFile().toString();
	return FILE_ACTION_RESULTS.get(canonicalFileName) != null && ! FILE_ACTION_RESULTS.get(canonicalFileName);
	}
	/**
	 * @param file
	 * @param succeeded
	 * @throws IOException
	 * 
	 * Associates a File's canonical name with a success indicator of true or false
	 */
	static void setHasBeenHandled(File file, boolean succeeded) throws IOException {
		setHasBeenHandled(file.getAbsoluteFile().toString(), succeeded);
	}
	/**
	 * @param filePath
	 * @param succeeded
	 * @throws IOException
	 * 
	 * Associates a File's canonical name with a success indicator of true or false
	 */
	static void setHasBeenHandled(String filePath, boolean succeeded) throws IOException {
		String canonicalFileName = new File(filePath).getCanonicalFile().toString();
		FILE_ACTION_RESULTS.put(canonicalFileName, succeeded);
	}

	/**
	 * ThreadPool allowing multiple concurrent upload/download actions
	 * and ensuring shutdown on exit.
	 */
	private static ExecutorService executor = null;

	static ExecutorService getExecutorService() {
		if (executor == null) {
			executor = Executors.newFixedThreadPool(4);
		}
		
		return executor;
	}

	/**
	 * @param config json config file specifying target artifacts
	 * @param baseUrl source (for download) repository
	 * @param username remote repository user name
	 * @param password remote repository password
	 * @param stripSnapshotVersionDetails boolean specifying whether tamestamp file component should be stripped
	 * @param overwrite boolean specifying whether existing files should be overwritten
	 * @param ignoreChecksum boolean specifying whether checksum verification problem should cause failure for respective file
	 * @return
	 * @throws MalformedURLException
	 * @throws Exception
	 */
	public static Map<String, Boolean> download(
			Config config,
			String baseUrl,
			String username, String password,
			boolean stripSnapshotVersionDetails,
			boolean overwrite,
			boolean ignoreChecksum) throws MalformedURLException, Exception {
		FILE_ACTION_RESULTS.clear();
		
		Set<FileTransferFutureTask<File>> tasks = new HashSet<>();
		for (Artifact artifact : config.getEffectiveConfig().getArtifacts()) {
			Set<String> filesFilteredByTypeAndClassifier = MavenFileUtils.getFileNamesFilteredByTypeAndClassifier(baseUrl, artifact, username, password);
			
			for (String filename : filesFilteredByTypeAndClassifier) {
				FileTransferFutureTask<File> task = null;

				if (filename.equals("maven-metadata.xml")) {
					String directoryPath = MavenFileUtils.makeMavenDirectoryRelativePath(artifact.getGroup(), artifact.getArtifact(), artifact.getVersion());
					URL url = new URL(baseUrl
							+ "/"
							+ directoryPath + "/" + "maven-metadata.xml");
					File destinationDir = new File(directoryPath).getAbsoluteFile();

					task = new FileTransferFutureTask<File>(new Download(username, password, url, "maven-metadata.xml", false, ignoreChecksum, overwrite, destinationDir), "download maven-metadata.xml");
					task.setRetryTasks(
							new FileTransferFutureTask<File>(new Download(username, password, url, "maven-metadata.xml", false, ignoreChecksum, overwrite, destinationDir), "download maven-metadata.xml (retry 1)"),
							new FileTransferFutureTask<File>(new Download(username, password, url, "maven-metadata.xml", false, ignoreChecksum, overwrite, destinationDir), "download maven-metadata.xml (retry 2)")
							);

					tasks.add(task);

					getExecutorService().execute(task);
				} else {
					String classifier = MavenFileUtils.getClassifierFromFileName(filename, artifact.getArtifact(), artifact.getVersion());
					String type = MavenFileUtils.getTypeFromFileName(filename, artifact.getArtifact(), artifact.getVersion());

					final FileTransferUtils.Paths mavenRelativePath = MavenFileUtils.makeMavenRelativePathForDownload(baseUrl, username, password, 
							artifact.getGroup(), artifact.getArtifact(), artifact.getVersion(), classifier, type, stripSnapshotVersionDetails);
					File destinationDir = new File(mavenRelativePath.getLocal()).getParentFile().getAbsoluteFile();
					URL url = new URL(baseUrl
							+ "/"
							+ mavenRelativePath.getRemote());
					task = new FileTransferFutureTask<File>(new Download(username, password, url, FileTransferUtils.basename(mavenRelativePath.getLocal()), false, ignoreChecksum, overwrite, destinationDir), "download " + url + " to " + mavenRelativePath.getLocal());
					task.setRetryTasks(
							new FileTransferFutureTask<File>(new Download(username, password, url, FileTransferUtils.basename(mavenRelativePath.getLocal()), false, ignoreChecksum, overwrite, destinationDir), "download " + url + " to " + mavenRelativePath.getLocal() + " (retry 1)"),
							new FileTransferFutureTask<File>(new Download(username, password, url, FileTransferUtils.basename(mavenRelativePath.getLocal()), false, ignoreChecksum, overwrite, destinationDir), "download " + url + " to " + mavenRelativePath.getLocal() + " (retry 2)")
							);
				}

				tasks.add(task);

				getExecutorService().execute(task);
			}
		}
		
		for (FileTransferFutureTask<File> executedTask : tasks) {
			try {
				executedTask.get();
			} catch (Exception e) {
				log.error("Failed waiting for task to complete: " + executedTask.getDescription());
				for (FileTransferFutureTask<File> retry : executedTask.getRetryTasks()) {
					try {
						retry.get();
					} catch (Exception e1) {
						log.error("Failed running: " + retry.getDescription());
					}
				}
			}
		}

		getExecutorService().shutdown();
		
		return Collections.unmodifiableMap(FILE_ACTION_RESULTS);
	}

	/**
	 * @param config json config file specifying target artifacts
	 * @param baseUrl destination (for upload) repository
	 * @param username remote repository user name
	 * @param password remote repository password
	 * @param stripSnapshotVersionDetails boolean specifying whether tamestamp file component should be stripped
	 * @param overwrite boolean specifying whether existing files should be overwritten
	 * @return
	 * @throws MalformedURLException
	 * @throws Exception
	 */
	public static Map<String, Boolean> upload(
			Config config,
			String baseUrl,
			String username,
			String password,
			boolean stripSnapshotVersionDetails,
			boolean overwrite) throws MalformedURLException, Exception {
		FILE_ACTION_RESULTS.clear();
		
		Set<FileTransferFutureTask<URL>> uploadTasks = new HashSet<>();
		for (Artifact artifact : config.getEffectiveConfig().getArtifacts()) {
			Set<String> filesFilteredByTypeAndClassifier = MavenFileUtils.getFileNamesFilteredByTypeAndClassifier(".", artifact, username, password);
			for (String filename : filesFilteredByTypeAndClassifier) {
				FileTransferFutureTask<URL> task = null;
				if (filename.endsWith(".sha1") || filename.endsWith(".md5")) {
					// ignore, as handled by Upload
				} else {
					Upload upload = null;
					
					if (filename.equals("maven-metadata.xml")) {
						// This constructor uses literal file basename, not name constructed from maven file name components
						upload = new Upload(
								username, password,
								baseUrl,
								artifact.getGroup(),
								artifact.getArtifact(),
								artifact.getVersion(),
								filename,
								overwrite
								);
					} else {
						String classifier = MavenFileUtils.getClassifierFromFileName(filename, artifact.getArtifact(), artifact.getVersion());
						String type = MavenFileUtils.getTypeFromFileName(filename, artifact.getArtifact(), artifact.getVersion());

						// This constructor constructs filename based on maven file name components and metadata
						upload = new Upload(
								username, password,
								baseUrl,
								artifact.getGroup(),
								artifact.getArtifact(),
								artifact.getVersion(),
								classifier,
								type,
								stripSnapshotVersionDetails,
								overwrite
								);
					}
					
					task = new FileTransferFutureTask<URL>(upload, "upload of " + filename + " to " + baseUrl);
					task.setRetryTasks(
							new FileTransferFutureTask<URL>(upload, "upload of " + filename + " to " + baseUrl + " (retry 1)"),
							new FileTransferFutureTask<URL>(upload, "upload of " + filename + " to " + baseUrl + " (retry 2)")
							);
					
					// Upload sha1 and md5 files, if exist
					for (FileTransferFutureTask<URL> simpleUploadSubTask : upload.generateSubTasks()) {
						uploadTasks.add(simpleUploadSubTask);
						getExecutorService().execute(simpleUploadSubTask);
					}
				}
				if (task != null) {
					uploadTasks.add(task);
					getExecutorService().execute(task);
				}
			}
		}

		for (FileTransferFutureTask<URL> executedTask : uploadTasks) {
			try {
				executedTask.get();
			} catch (Exception e) {
				log.error("Failed waiting for task to complete: " + executedTask.getDescription());
				for (FileTransferFutureTask<URL> retry : executedTask.getRetryTasks()) {
					try {
						retry.get();
					} catch (Exception e1) {
						log.error("Failed running: " + retry.getDescription());
					}
				}
			}
		}

		getExecutorService().shutdown();
		
		return Collections.unmodifiableMap(FILE_ACTION_RESULTS);
	}

	public static void main(String[] args) throws Exception
	{
		try
		{
			// create Options object
			Options options = new Options();

			// add cfg option
			Option cfgFileOption = new Option("cfg", true, "configuration file");
			cfgFileOption.setRequired(true);
			options.addOption(cfgFileOption);
			// add uploadOption option
			Option uploadOption = new Option("upload", false, "perform upload");
			options.addOption(uploadOption);
			// add downloadOption option
			Option downloadOption = new Option("download", false, "perform download");
			options.addOption(downloadOption);
			// add user option
			Option userOption = new Option("user", true, "repository user name");
			userOption.setRequired(true);
			options.addOption(userOption);
			// add login option
			Option passwordOption = new Option("password", true, "repository password");
			passwordOption.setRequired(true);
			options.addOption(passwordOption);
			// add base url option
			Option urlOption = new Option("url", true, "base url");
			urlOption.setRequired(true);
			options.addOption(urlOption);
			// add maintain snapshot version option
			Option stripSnapshotVersionDetailsOption = new Option("stripSnapshotVersionDetails", false, "strip snapshot version details from downloaded filenames");
			options.addOption(stripSnapshotVersionDetailsOption);
			// enable SSL verification
			Option sslVerify = new Option("sslVerify", false, "enable SSL hostname verification");
			options.addOption(sslVerify);
			// Force overwrite of existing artifacts
			Option overwrite = new Option("overwrite", false, "force overwrite of existing artifacts");
			options.addOption(overwrite);
			// Force overwrite of existing artifacts
			Option ignoreChecksum = new Option("ignoreChecksum", false, "ignore failure of checksum verification");
			options.addOption(ignoreChecksum);

			CommandLineParser parser = new DefaultParser();
			CommandLine cmd = parser.parse(options, args);
			
			// Validate arguments here
			if (cmd.hasOption(uploadOption.getOpt()) && cmd.hasOption(downloadOption.getOpt())) {
				throw new IllegalArgumentException("Incompatible arguments \"-" + uploadOption.getOpt() + "\" and \"-" + downloadOption.getOpt() + "\"");
			} else if (! cmd.hasOption(uploadOption.getOpt()) && ! cmd.hasOption(downloadOption.getOpt())) {
				throw new IllegalArgumentException("Missing required argument \"-" + uploadOption.getOpt() + "\" or \"-" + downloadOption.getOpt() + "\"");
			}

			final String username = cmd.getOptionValue(userOption.getOpt());
			if (StringUtils.isBlank(username)) {
				throw new IllegalArgumentException("Missing required argument \"-" + userOption.getOpt() + "\"");
			}
			final String password = cmd.getOptionValue(passwordOption.getOpt());
			if (StringUtils.isBlank(password)) {
				throw new IllegalArgumentException("Missing required argument \"-" + passwordOption.getOpt() + "\"");
			}
			final String cfgFileName = cmd.getOptionValue(cfgFileOption.getOpt());

			if(cfgFileName == null) {
				throw new IllegalArgumentException("Missing commandline option " + cfgFileOption);
			}

			final String baseUrl = cmd.getOptionValue(urlOption.getOpt());
			if (StringUtils.isBlank(baseUrl)) {
				throw new IllegalArgumentException("Missing required argument \"-" + urlOption.getOpt() + "\"");
			}

			if (! cmd.hasOption(sslVerify.getOpt())) {
				NetworkUtils.disableSSLVerification();
			}

			ObjectMapper mapper = new ObjectMapper();
			Config cfg = mapper.readValue(new File(cfgFileName), Config.class);

			Authenticator.setDefault (new Authenticator() {
				protected PasswordAuthentication getPasswordAuthentication() {
					return new PasswordAuthentication (username, password.toCharArray());
				}
			});

			// Uncomment to disable SNAPSHOT handling
//			for (Artifact artifact : cfg.getEffectiveConfig().getArtifacts()) {
//				if (artifact.getVersion().contains("SNAPSHOT")) {
//					throw new RuntimeException("SNAPSHOT versions not currently supported");
//				}
//			}
			if (cmd.hasOption(downloadOption.getOpt())) {
				// IF DOWNLOAD
				download(cfg, baseUrl, username, password, cmd.hasOption(stripSnapshotVersionDetailsOption.getOpt()), cmd.hasOption(overwrite.getOpt()), cmd.hasOption(ignoreChecksum.getOpt()));
			} else {
				// IF UPLOAD
				upload(cfg, baseUrl, username, password, cmd.hasOption(stripSnapshotVersionDetailsOption.getOpt()), cmd.hasOption(overwrite.getOpt()));
			}
		}
		catch (Exception e)
		{
			log.error("Caught " + e.getClass().getName() + " " + e.getLocalizedMessage());
			e.printStackTrace();
			throw e;
		}
		finally {
			getExecutorService().shutdown();

			int succeeded= 0;
			int failed = 0;
			for (Map.Entry<String, Boolean> entry : FileTransfer.FILE_ACTION_RESULTS.entrySet()) {
				if (entry.getValue()) {
					succeeded++;
					System.out.println("SUCCEEDED: " + entry.getKey());
				}
			}
			for (Map.Entry<String, Boolean> entry : FileTransfer.FILE_ACTION_RESULTS.entrySet()) {
				if (! entry.getValue()) {
					failed++;
					System.out.println("FAILED: " + entry.getKey());
				}
			}
			System.out.println("TOTAL: " + (succeeded + failed) + ", SUCCEEDED: " + succeeded + ", FAILED: " + failed);
		}
	}
}