package gov.va.caret.util.test;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import gov.va.caret.model.BgLae;
import gov.va.caret.model.StiPt;
import gov.va.caret.model.WorIm;
import gov.va.caret.model.support.BgLaeJUnit;
import gov.va.caret.model.support.StiPtJUnit;
import gov.va.caret.model.support.WorImJUnit;
import gov.va.caret.model.support.occ.AddressChange;
import gov.va.caret.model.support.occ.CalculationChange;
import gov.va.caret.model.support.occ.CalculationInfo;
import gov.va.caret.model.support.occ.DateRange;
import gov.va.caret.model.support.occ.FinalPayment;
import gov.va.caret.model.support.occ.InitialPayment;
import gov.va.caret.model.support.occ.MultiFacetedConfig;
import gov.va.caret.model.support.occ.Reinstatement;
import gov.va.caret.model.support.occ.StipendConfig;
import gov.va.caret.model.support.occ.TierChange;
import gov.va.caret.util.Toolbox;
import junit.framework.TestResult;

public class CalculationInfoTest extends org.junit.Assert implements junit.framework.Test  {

	static Logger logger = LogManager.getLogger(CalculationInfoTest.class);
	
	@Test
	public void testPaymentDate() {
		assertTrue( Toolbox.parseDate("04/01/2018").equals( applicationApproval_sameMonthMar20.getPaymentDate() ) );
		assertTrue( Toolbox.parseDate("04/01/2018").equals( applicationApproval_1stDaySameMonthMar01.getPaymentDate() ) );
		assertTrue( Toolbox.parseDate("04/01/2018").equals( applicationApproval_priorMonthFeb15.getPaymentDate() ) );
		assertTrue( Toolbox.parseDate("04/01/2018").equals( applicationApproval_2MonthsBackJan15.getPaymentDate() ) );
	}
	
	@Test
	public void testRetroactiveMonths() {
		assertTrue( applicationApproval_sameMonthMar20.getRetroactiveMonths() == 0 );
		assertTrue( applicationApproval_1stDaySameMonthMar01.getRetroactiveMonths() == 0 );
		assertTrue( applicationApproval_priorMonthFeb15.getRetroactiveMonths() == 0 );//
		assertTrue( applicationApproval_2MonthsBackJan15.getRetroactiveMonths() == 1 );
	}
	
	@Test
	public void testBackPayAmount() {
		assertTrue( applicationApproval_sameMonthMar20.getBackPayAmount().equals(BigDecimal.ZERO)  );//
		assertTrue( applicationApproval_1stDaySameMonthMar01.getBackPayAmount().equals(BigDecimal.ZERO) );
		assertTrue( applicationApproval_priorMonthFeb15.getBackPayAmount().toString().equals("1004.36")  );
		assertTrue( applicationApproval_2MonthsBackJan15.getBackPayAmount().toString().equals("850.30") );
	}
	
	@Test
	public void testTotalDays() {
		assertTrue( applicationApproval_sameMonthMar20.getTotalDays( ) == 12 );
		assertTrue( applicationApproval_1stDaySameMonthMar01.getTotalDays() == 0 );
		assertTrue( applicationApproval_priorMonthFeb15.getTotalDays() == 14 );
		assertTrue( applicationApproval_2MonthsBackJan15.getTotalDays() == 17 );
	}
	
	@Test
	public void testProratedAmount() {
		
		assertTrue( applicationApproval_sameMonthMar20.getProratedAmount().toString().equals("537.96") );
		assertTrue( applicationApproval_1stDaySameMonthMar01.getProratedAmount().equals( BigDecimal.ZERO ) );
		assertTrue( applicationApproval_priorMonthFeb15.getProratedAmount().toString().equals("1004.36") );
		assertTrue( applicationApproval_2MonthsBackJan15.getProratedAmount().toString().equals("304.81") );
		
	}

	@Test
	public void testInitialPayment() {
		assertTrue( applicationApproval_sameMonthMar20.getFinalAmount().toString().equals("537.96") );
		assertTrue( applicationApproval_1stDaySameMonthMar01.getFinalAmount().toString().equals("1090.98") );
		assertTrue( applicationApproval_priorMonthFeb15.getFinalAmount().toString().equals("3186.32") );//
		assertTrue( applicationApproval_2MonthsBackJan15.getFinalAmount().toString().equals("1395.79") );
		
		
		logger.info( "complexApplicationApproval_4MonthsBackNov15.totalStipend " +  ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getFinalAmount() );
		for (  int i = 0; i < ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getLastRecurringConfigs().size(); i++) {
			StipendConfig oldConfig =  ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getLastRecurring( i );
			logger.info( "===============");
			logger.info( "Hourly:" + oldConfig.getBlsRateHourly() + "(" + oldConfig.getTier() + ")" );
			if ( i == ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getLastRecurringConfigs().size() - 1 ) {
				logger.info( "Daily:" +  ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getLastRecurringDailyRate(i) );
				logger.info( "Days:" +  ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getTotalDays() );
			}
			logger.info( "Monthly:" +  ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getLastRecurringMonthlyRate(i) );
			logger.info( "Start:" + oldConfig.getStipendStartDateStr() );
			logger.info( "End:" + oldConfig.getStipendEndDateStr() );
		}
		logger.info( "===============");
		logger.info( "Config Hourly:" +  ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getHourlyRate() + "(" +  ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getStipendPaymentConfig().getTier() + ")" );
		logger.info( "Config Monthly:" +  ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getMonthlyRate() );
		logger.info( "Config Start:" + Toolbox.formatDate( complexApplicationApproval_4MonthsBackNov15.getStipendPaymentConfig().getStipendStartDate() ) );
		logger.info( "Config Payment:" + (2 + ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getPaymentMonth() )%12 + "/" + ((InitialPayment)complexApplicationApproval_4MonthsBackNov15).getPaymentYear() );
	}
	
	
	@Test
	public void testHoldRemovalAmountRecurring() {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(Toolbox.parseDate("03/01/18"));
		logger.info( "Last Paid:" + Toolbox.formatDateCprs( calendar.getTime() ) ) ;
		
		//Last payment applied 3/1/18 for February
		BgLae march1stLastPaid = new BgLaeJUnit();
		march1stLastPaid.setOwnerId( 2018 * 12 + calendar.get(Calendar.MONTH) );
		march1stLastPaid.setBgLaeId( 2000 );
		march1stLastPaid.setBoxGpId( 200 );
		
		StiPt recurringConfigOnHoldSinceMarch = new StiPtJUnit();
		recurringConfigOnHoldSinceMarch.setTier("2");
		recurringConfigOnHoldSinceMarch.setBlsRateHourly("12.54");
		recurringConfigOnHoldSinceMarch.setTierHours("108.75");
		recurringConfigOnHoldSinceMarch.setPayment("1363.73");
		recurringConfigOnHoldSinceMarch.setStipendStartDate(Toolbox.parseDate("01/11/2018"));
		recurringConfigOnHoldSinceMarch.setStipendStatus( StipendConfig.ON_HOLD );
		recurringConfigOnHoldSinceMarch.setStipendType( StipendConfig.RECURRING );
		recurringConfigOnHoldSinceMarch.setPayslipDate(Toolbox.parseDate( "02/01/2018") );
		recurringConfigOnHoldSinceMarch.setCycleId( march1stLastPaid.getBgLaeId() );
		
		StipendConfig configHeldSinceMarch = new StipendConfig( recurringConfigOnHoldSinceMarch );
		
		assertTrue( configHeldSinceMarch.getHoldAmount( march1stLastPaid, Toolbox.parseDate("04/01/18") ).toString().equals("1363.73") );
		assertTrue( configHeldSinceMarch.getHoldAmount( march1stLastPaid, Toolbox.parseDate("05/01/18") ).toString().equals("2727.46") );
		assertTrue( configHeldSinceMarch.getHoldAmount( march1stLastPaid, Toolbox.parseDate("06/01/18") ).toString().equals("4091.19") );
		
		
	}
	
	
	@Test
	public void testStakeHolderScenario1() {
		
		/**
		 * Scenario 1:  Veteran and Caregiver Old Address: 1145 Lollipop St, Walden CO 80480   (17.60/hr)
            Veteran and Caregiver New Address: 221 Caring Rd, Ft Collins, CO 80517 (14.21/hr)
			You received an OCC Alert for a Tier Change from the Caregiver Support Coordinator at the VAMC on October 5,
			2018. The current tier level is 2 and is being lowered to tier 1 effective September 10, 2018 due to a
			reassessment. There was an address change that was updated in CareT September 10, 2018 as well. The current
			payment file is for October but will be paid on November 1, 2018. 
		 * 
		 */
		
		StiPt medTierHighBlsNoEntitledDaysFeb15 = new StiPtJUnit();
		medTierHighBlsNoEntitledDaysFeb15.setBlsRateHourly("17.60");
		medTierHighBlsNoEntitledDaysFeb15.setEntitledDays(0);
		medTierHighBlsNoEntitledDaysFeb15.setTier("2");
		medTierHighBlsNoEntitledDaysFeb15.setPayment("1914.00");
		medTierHighBlsNoEntitledDaysFeb15.setStipendStartDate(Toolbox.parseDate("02/15/2018"));
		medTierHighBlsNoEntitledDaysFeb15.setTierHours("108.75");
		
		StiPt lowTierHighBlsNoMediumEntitledDaysSept10 = new StiPtJUnit();
		lowTierHighBlsNoMediumEntitledDaysSept10.setBlsRateHourly("17.60");
		lowTierHighBlsNoMediumEntitledDaysSept10.setEntitledDays(0);
		lowTierHighBlsNoMediumEntitledDaysSept10.setTier("1");
		lowTierHighBlsNoMediumEntitledDaysSept10.setPayment("765.60");
		lowTierHighBlsNoMediumEntitledDaysSept10.setStipendStartDate(Toolbox.parseDate("09/10/2018"));
		lowTierHighBlsNoMediumEntitledDaysSept10.setTierHours("43.50");
		
		StiPt lowTierHighBlsNoMediumEntitledDaysNov01 = new StiPtJUnit();
		lowTierHighBlsNoMediumEntitledDaysNov01.setBlsRateHourly("14.21");
		lowTierHighBlsNoMediumEntitledDaysNov01.setEntitledDays(0);
		lowTierHighBlsNoMediumEntitledDaysNov01.setTier("1");
		lowTierHighBlsNoMediumEntitledDaysNov01.setPayment("615.14");
		lowTierHighBlsNoMediumEntitledDaysNov01.setStipendStartDate(Toolbox.parseDate("10/01/2018"));
		lowTierHighBlsNoMediumEntitledDaysNov01.setTierHours("43.50");
		
		
		Calendar october1st2018paymentDate = Calendar.getInstance();
		october1st2018paymentDate.setTime( Toolbox.parseDate( "10/01/2018") );
		Calendar november1st2018paymentDate = Calendar.getInstance();
		november1st2018paymentDate.setTime( Toolbox.parseDate( "11/01/2018") );
		
		try {
			CalculationChange tierChange_DecrPayScenario1Part1 = 
					new TierChange(new WorImJUnit(), lowTierHighBlsNoMediumEntitledDaysSept10, medTierHighBlsNoEntitledDaysFeb15, october1st2018paymentDate);
			assertTrue( tierChange_DecrPayScenario1Part1.getRecoupmentAmount().toString().equals("792.96") );

			
			CalculationChange addressChange_DecrPayScenario1Part2 = 
					new AddressChange(new WorImJUnit(), lowTierHighBlsNoMediumEntitledDaysNov01, lowTierHighBlsNoMediumEntitledDaysSept10, november1st2018paymentDate);
			assertTrue( addressChange_DecrPayScenario1Part2.getFinalAmount().toString().equals("615.14") );
			
			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	@Test
	public void testStakeHolderScenario2() {
		
		/**
		 * Scenario 2: Veteran and Caregiver Address: 221 Caring Rd, Ft Collins, CO 80517 (14.21/hr)
			The CSC sends you an email and states the Veteran and Caregiver have been reinstated as of 10/5/2018. 
			The tier level was a tier 1 and has been changed to a tier 3 effective 5/15/2018. The Caregivers 90-day
			stipend benefits ended 6/14/2018. The current payment file is for October but will pay on November 1, 2018.
		 * 
		 */
		
		StiPt highTierMedBlsNoEntitledDaysOct05 = new StiPtJUnit();
		highTierMedBlsNoEntitledDaysOct05.setBlsRateHourly("14.21");
		highTierMedBlsNoEntitledDaysOct05.setEntitledDays(0);
		highTierMedBlsNoEntitledDaysOct05.setTier("3");
		highTierMedBlsNoEntitledDaysOct05.setPayment("2472.54");
		highTierMedBlsNoEntitledDaysOct05.setStipendStartDate(Toolbox.parseDate("10/05/2018"));
		highTierMedBlsNoEntitledDaysOct05.setTierHours("174.00");
		
		Calendar november1st2018paymentDate = Calendar.getInstance();
		november1st2018paymentDate.setTime( Toolbox.parseDate( "11/01/2018") );
		
		try {
			List<StipendConfig> lastRecurringConfigs = new ArrayList<StipendConfig>();
			lastRecurringConfigs.add( new StipendConfig(highTierMedBlsNoEntitledDaysOct05) );
			
			CalculationChange reinstatement_Scenario2 = new Reinstatement( new WorImJUnit(), lastRecurringConfigs, november1st2018paymentDate);

			assertTrue( reinstatement_Scenario2.getTotalDays() == 27 );
			assertTrue( reinstatement_Scenario2.getFinalAmount().toString().equals("2194.83") );
			
			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	@Test
	public void testStakeHolderScenario3() {
		
		/**
		 * Scenario 3:  Veteran and Caregiver Address: 221 Caring Rd, Ft Collins, CO 80517 (14.21/hr)
            The CSC calls you 10/5/2018 and says she forgot to update CareT with the tier decrease that was 
            effective 4/5/2018 as a tier 1 on the last reassessment. The current tier is 2 in CareT. The CSC 
            also states the Veteran called her today and no longer wants his girlfriend to be his caregiver 
            as of 10/5/18 (benefits end date November 4). The current payment file is for October but will 
            pay November 1, 2018.
		 * 
		 */
		StiPt lowTierHighBlsNoMediumEntitledDaysJan01 = new StiPtJUnit();
		lowTierHighBlsNoMediumEntitledDaysJan01.setBlsRateHourly("14.21");
		lowTierHighBlsNoMediumEntitledDaysJan01.setEntitledDays(0);
		lowTierHighBlsNoMediumEntitledDaysJan01.setTier("2");
		lowTierHighBlsNoMediumEntitledDaysJan01.setPayment("1545.34");
		lowTierHighBlsNoMediumEntitledDaysJan01.setStipendStartDate(Toolbox.parseDate("01/10/2018"));
		lowTierHighBlsNoMediumEntitledDaysJan01.setTierHours("108.75");
		
		StiPt lowTierMedBlsNoEntitledDaysApril05 = new StiPtJUnit();
		lowTierMedBlsNoEntitledDaysApril05.setBlsRateHourly("14.21");
		lowTierMedBlsNoEntitledDaysApril05.setEntitledDays(0);
		lowTierMedBlsNoEntitledDaysApril05.setTier("1");
		lowTierMedBlsNoEntitledDaysApril05.setPayment("615.14");
		lowTierMedBlsNoEntitledDaysApril05.setStipendStartDate(Toolbox.parseDate("04/05/2018"));
		lowTierMedBlsNoEntitledDaysApril05.setTierHours("43.50");
		
		StiPt lowTierMedBls30EntitledDaysOct05 = new StiPtJUnit();
		lowTierMedBls30EntitledDaysOct05.setBlsRateHourly("14.21");
		lowTierMedBls30EntitledDaysOct05.setEntitledDays(30);
		lowTierMedBls30EntitledDaysOct05.setTier("1");
		lowTierMedBls30EntitledDaysOct05.setPayment("615.14");
		lowTierMedBls30EntitledDaysOct05.setStipendStartDate(Toolbox.parseDate("10/05/2018"));
		lowTierMedBls30EntitledDaysOct05.setTierHours("43.50");
		
		Calendar november1st2018paymentDate = Calendar.getInstance();
		november1st2018paymentDate.setTime( Toolbox.parseDate( "11/01/2018") );
		
		try {
			CalculationChange tierChange_DecrPayScenario3Part1 = 
					new TierChange(new WorImJUnit(), lowTierMedBlsNoEntitledDaysApril05, lowTierHighBlsNoMediumEntitledDaysJan01, november1st2018paymentDate);
			logger.info( "tierChange_DecrPayScenario3Part1.getRecoupmentAmount().toString():" + tierChange_DecrPayScenario3Part1.getRecoupmentAmount().toString() ) ;

			assertTrue( tierChange_DecrPayScenario3Part1.getRecoupmentAmount().toString().equals("10064.78") );
			assertTrue( tierChange_DecrPayScenario3Part1.isRecoupable() );
			
			
			CalculationInfo finalPayment_Scenario3Part2 = new FinalPayment(new WorImJUnit(), november1st2018paymentDate, lowTierMedBls30EntitledDaysOct05 );
			
			assertTrue( finalPayment_Scenario3Part2.getFinalAmount().toString().equals("696.42") );
			assertTrue( finalPayment_Scenario3Part2.getTotalDays(  ) == 4 );
			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	
	@Test
	public void testHoldRemovalAmountOneTime() {
		
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(Toolbox.parseDate("03/01/18"));
		logger.info( "Last Paid:" + Toolbox.formatDateCprs( calendar.getTime() ) ) ;
		
		//Last payment applied 3/1/18 for February
		BgLae march1stLastPaid = new BgLaeJUnit();
		march1stLastPaid.setOwnerId( 2018 * 12 + calendar.get(Calendar.MONTH) );
		march1stLastPaid.setBgLaeId( 2000 );
		march1stLastPaid.setBoxGpId( 200 );
				
		StiPt onTimeconfigOnHoldSinceMarch = new StiPtJUnit();
		onTimeconfigOnHoldSinceMarch.setOneTimePayment("5000.00");
		onTimeconfigOnHoldSinceMarch.setStipendStartDate(Toolbox.parseDate("01/11/2018"));
		onTimeconfigOnHoldSinceMarch.setStipendStatus( StipendConfig.ON_HOLD );
		onTimeconfigOnHoldSinceMarch.setStipendType( StipendConfig.ONE_TIME );
		onTimeconfigOnHoldSinceMarch.setPayslipDate( null );
		onTimeconfigOnHoldSinceMarch.setCycleId( march1stLastPaid.getBgLaeId() );
		StipendConfig oneTimeconfigHeldSinceMarch = new StipendConfig( onTimeconfigOnHoldSinceMarch );

		assertTrue( oneTimeconfigHeldSinceMarch.getHoldAmount( march1stLastPaid, Toolbox.parseDate("06/01/18") ).toString().equals("5000.00") );
	}
	
	@Test
	public void testRecoupmentAmount() {
		
		assertTrue( tierChange_DecrPay2Feb15_Jan15.getRecoupmentAmount().toString().equals("2181.96")  );//753.34  //20390.98 //20552.41
//		assertTrue( tierChange_DecrPay.getRecoupmentAmount().toString().)
//		tierChange_DecrPay,
//		tierChange_DecrPay2,
//		tierChange_IncrPay,
//		tierChange_IncrPay2
		
	}
	
	@Test
	public void testCalculationChange() {
		
		
		
		assertTrue( tierChange_DecrPayApr05_Feb15.getMonthlyRate().compareTo( tierChange_DecrPayApr05_Feb15.getMonthlyRate() ) == 0 );
		MultiFacetedConfig recurringPayment = ((MultiFacetedConfig)tierChange_DecrPayApr05_Feb15);
		assertTrue( tierChange_DecrPayApr05_Feb15.getMonthlyRate().compareTo( recurringPayment.getLastRecurringMonthlyRate( recurringPayment.getLastRecurringConfigs().size()-1 ) ) < 0 );
		
//		recurringPayment = ((RecurringPayment)tierChange_DecrPay2Feb15_Jan15);
//		assertTrue( tierChange_DecrPay2Feb15_Jan15.getMonthlyRate().compareTo( recurringPayment.getLastRecurringMonthlyRate( recurringPayment.getLastRecurringConfigs().size()-1 ) ) < 0 );

		recurringPayment = ((MultiFacetedConfig)tierChange_IncrPayMar20_Jan15);
		assertTrue( tierChange_IncrPayMar20_Jan15.getMonthlyRate().compareTo( recurringPayment.getLastRecurringMonthlyRate( recurringPayment.getLastRecurringConfigs().size()-1 ) ) > 0 );
		
		recurringPayment = ((MultiFacetedConfig)tierChange_IncrPay2Apr05_Feb15);
		assertTrue( tierChange_IncrPay2Apr05_Feb15.getMonthlyRate().compareTo( recurringPayment.getLastRecurringMonthlyRate( recurringPayment.getLastRecurringConfigs().size()-1 ) ) > 0 );
		
		assertTrue( tierChange_IncrPayMar20_Jan15.getBackPayAmount().compareTo(BigDecimal.ZERO) > 0 );
		
//		assertTrue( tierChange_IncrPay2.getMonthlyRate().compareTo( tierChange_IncrPay2.getOldMonthlyRate() ) > 0 );
//		assertTrue( tierChange_IncrPay2.getMonthlyRate().compareTo( tierChange_IncrPay2.getOldMonthlyRate() ) > 0 );
		
//		assertTrue( applicationApproval_sameMonth.getRetroactiveMonths() == 0 );
//		assertTrue( applicationApproval_priorMonth.getRetroactiveMonths() == 1 );
//		assertTrue( applicationApproval_1stDaySameMonth.getRetroactiveMonths() == 0 );
	}
	
	@Test
	public void testComplexTierChange() {
		
		//assert 3 payment definitions, all with different monthly/hourly rates
		assertTrue( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().size() == 3 );
		
		//assert that the tier increased
		assertTrue ( Integer.valueOf(((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(0).getTier()) < Integer.valueOf(((TierChange)complexTierChange_IncrPay).getStipendPaymentConfig().getTier() ) );
		assertTrue ( Integer.valueOf(((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(1).getTier()) < Integer.valueOf(((TierChange)complexTierChange_IncrPay).getStipendPaymentConfig().getTier() ) );
		assertTrue ( Integer.valueOf(((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(2).getTier()) < Integer.valueOf(((TierChange)complexTierChange_IncrPay).getStipendPaymentConfig().getTier() ) );
		
		//assert that none of the payment definitions have the same bls-hourly rate
		assertTrue ( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(0).getBlsRateHourly() != ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(1).getBlsRateHourly() ) ;
		assertTrue ( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(0).getBlsRateHourly() != ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(2).getBlsRateHourly() ) ;
		assertTrue ( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(1).getBlsRateHourly() != ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(2).getBlsRateHourly() ) ;
		
		//assert that none of the payment definitions have the same stipend start date
		assertTrue ( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(0).getStipendStartDate() != ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(1).getStipendStartDate() ) ;
		assertTrue ( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(0).getStipendStartDate() != ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(2).getStipendStartDate() ) ;
		assertTrue ( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(1).getStipendStartDate() != ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(2).getStipendStartDate() ) ;
		
		//assert that none of the payment definitions have the same stipend end date
		assertTrue ( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(0).getStipendEndDate() != ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(1).getStipendEndDate() ) ;
		assertTrue ( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(0).getStipendEndDate() != ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(2).getStipendEndDate() ) ;
		assertTrue ( ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(1).getStipendEndDate() != ((TierChange)complexTierChange_IncrPay).getLastRecurringConfigs().get(2).getStipendEndDate() ) ;
		
		debugComplexTierChange((TierChange)complexTierChange_IncrPay);
		debugComplexTierChange((TierChange)complexTierChange_IncrPay2);
		
	}	
	
	public void debugComplexTierChange( TierChange complexTierChange_IncrPay ) {
		
		logger.info( "complexTierChange_IncrPay.totalStipend " +  (complexTierChange_IncrPay).getFinalAmount() );
		for (  int i = 0; i < (complexTierChange_IncrPay).getLastRecurringConfigs().size(); i++) {
			StipendConfig oldConfig = complexTierChange_IncrPay.getLastRecurringConfigs().get( i );
			logger.info( "===============");
			logger.info( "Hourly:" + oldConfig.getBlsRateHourly() + "(" + oldConfig.getTier() + ")" );
			logger.info( "Monthly:" + (complexTierChange_IncrPay).getLastRecurringMonthlyRate(i) );
			logger.info( "Start:" + oldConfig.getStipendStartDateStr() );
			logger.info( "End:" + oldConfig.getStipendEndDateStr() );
			logger.info( "Monthly Rate Diff:" + (complexTierChange_IncrPay).getMonthlyRateDiff(i) );

			logger.info( "*Hourly:" + (complexTierChange_IncrPay).getAdjustedConfig(i).getBlsRateHourly() + "(" + (complexTierChange_IncrPay).getAdjustedConfig(i).getTier() + ")" );
			logger.info( "*Monthly:" + (complexTierChange_IncrPay).getAdjustedNewMonthlyRate(i) );
			logger.info( "Start:" + (complexTierChange_IncrPay).getAdjustedConfig(i).getStipendStartDateStr() );
			logger.info( "End:" + (complexTierChange_IncrPay).getAdjustedConfig(i).getStipendEndDateStr() );
//			logger.info( "*Monthly Rate Diff:" + (complexTierChange_IncrPay).getMonthlyRateDiff0(i) );
			
		}
		logger.info( "===============");
		logger.info( "Config Hourly:" + (complexTierChange_IncrPay).getHourlyRate() + "(" + (complexTierChange_IncrPay).getStipendPaymentConfig().getTier() + ")" );
		logger.info( "Config Monthly:" + (complexTierChange_IncrPay).getMonthlyRate() );
		logger.info( "Config Start:" + Toolbox.formatDate( complexTierChange_IncrPay.getStipendPaymentConfig().getStipendStartDate() ) );
		logger.info( "Config Payment:" + (complexTierChange_IncrPay).getPaymentMonth() + "/" + (complexTierChange_IncrPay).getPaymentYear() );
	}
//	
//	@Test
//	public void testRecoupable() {
//		
//		assertTrue( applicationApproval_sameMonth.getRetroactiveMonths() == 0 );
//		assertTrue( applicationApproval_priorMonth.getRetroactiveMonths() == 1 );
//		assertTrue( applicationApproval_1stDaySameMonth.getRetroactiveMonths() == 0 );
//		
//	}	
//	
//	@Test
//	public void testDailyRate() {
//		
//		assertTrue( applicationApproval_sameMonth.getRetroactiveMonths() == 0 );
//		assertTrue( applicationApproval_priorMonth.getRetroactiveMonths() == 1 );
//		assertTrue( applicationApproval_1stDaySameMonth.getRetroactiveMonths() == 0 );
//		
//	}
//	
	@Test
	public void testPaymentSchedule() {
		
		int monthKey = 2018 * 12 + 4; //May 
		
		long vendrId1 = 101l;
		long vendrId2 = 102l;
		
		
		
		StiPt recurring1 = new StiPtJUnit();
		recurring1.setStipendType(StipendConfig.RECURRING);
		recurring1.setStipendStatus(StipendConfig.PAYMENT_FINANCE);
		recurring1.setVendrId( vendrId1 );
		recurring1.setCycleId( monthKey );
		recurring1.setPayment("500.0");
		
		
		
		
	}
	
	@Override
	public int countTestCases() {
		return 0;
	}

	@Override
	public void run(TestResult result) {
		
	}
	
	@Before
	public void setUp() {
		try{
			
			
			
		}catch(Exception ex){
			ex.printStackTrace();
			fail("Init failed...");
		}
	}
	
	static CalculationInfo complexApplicationApproval_4MonthsBackNov15,
							applicationApproval_sameMonthMar20,
							applicationApproval_priorMonthFeb15,
							applicationApproval_1stDaySameMonthMar01,
							applicationApproval_2MonthsBackJan15,
							
							revocation_MediumEntitledDays,
							revocation_NoEntitledDays,
							revocation_NoEntitledDaysBackDated,
							revocation_MediumEntitledDaysBackDated,
							
							reinstatementNoChangeNoEntitlement,
							reinstatementNoChangeBackDatedNoEntitlement,
							reinstatementBackDatedHighEntitlement,
							reinstatementBackDatedNoEntitlement,
							
							reinstatementBackDatedTierIncrease,
							reinstatementBackDatedTierDecrease;
							
							
	
	static CalculationChange addressChange_DecrPay,
							addressChange_IncrPay,
							
							tierChange_DecrPayApr05_Feb15,
							tierChange_DecrPay2Feb15_Jan15,
							tierChange_IncrPayMar20_Jan15,
							tierChange_IncrPay2Apr05_Feb15,
							complexTierChange_IncrPay,
							complexTierChange_IncrPay2,
							complexTierChange_DecrPay;
	
	
	static Date paymentDate = Toolbox.parseDate("04/01/2018");


	static StiPt lowTierHighBlsNoEntitledDaysNov30 = new StiPtJUnit();
	static StiPt medTierMedBlsNoEntitledDaysOct10 = new StiPtJUnit();
	static StiPt medTierMedBlsNoEntitledDaysNov15 = new StiPtJUnit();
	static StiPt mediumTierHighBlsNoEntitledDaysNov30 = new StiPtJUnit();
	static StiPt highTierHighBlsNoEntitledDaysNov30 = new StiPtJUnit();
	
	static StiPt lowTierMedBlsMediumEntitledDaysJan15 = new StiPtJUnit();
	static StiPt medTierMedBlsMediumEntitledDaysJan15 = new StiPtJUnit();
	static StiPt medTierlowBlsMediumEntitledDaysJan01 = new StiPtJUnit();
	static StiPt medTierlowBlsMediumEntitledDaysJan01_V2 = new StiPtJUnit();
	static StiPt medTierlowBlsMediumEntitledDaysFeb01 = new StiPtJUnit();
	static StiPt highTierMedBlsMediumEntitledDaysJan15 = new StiPtJUnit();
	
	static StiPt lowTierMedBlsLowEntitledDaysFeb15 = new StiPtJUnit();
	static StiPt mediumTierMedBlsLowEntitledDaysFeb15 = new StiPtJUnit();
	static StiPt highTierMedBlsLowEntitledDaysFeb15 = new StiPtJUnit();
	
	static StiPt lowTierlowBlsHighEntitledDaysMar01 = new StiPtJUnit();
	static StiPt highTierlowBlsHighEntitledDaysMar01 = new StiPtJUnit();
	static StiPt highTierMedBlsHighEntitledDaysNov15 = new StiPtJUnit();
	static StiPt highTierlowBlsHighEntitledDaysDec10 = new StiPtJUnit();
	static StiPt mediumTierMedBlsNoEntitledDaysMar20 = new StiPtJUnit();

	static StiPt lowTierLowBlsMediumEntitledDaysApr05 = new StiPtJUnit();
	static StiPt lowTierMedBlsLowEntitledDaysApr05 = new StiPtJUnit();
	static StiPt highTierMedBlsLowEntitledDaysApr05 = new StiPtJUnit();
	static StiPt mediumTierLowBlsNoEntitledDaysMay10 = new StiPtJUnit();
	
	static {
		
		lowTierHighBlsNoEntitledDaysNov30.setBlsRateHourly("25.08");
		lowTierHighBlsNoEntitledDaysNov30.setEntitledDays(0);
		lowTierHighBlsNoEntitledDaysNov30.setTier("1");
		lowTierHighBlsNoEntitledDaysNov30.setPayment("1087.85");
		lowTierHighBlsNoEntitledDaysNov30.setStipendStartDate( Toolbox.parseDate("11/30/2017") );
		lowTierHighBlsNoEntitledDaysNov30.setTierHours("43.50");
		
		medTierMedBlsNoEntitledDaysNov15.setBlsRateHourly("12.54");
		medTierMedBlsNoEntitledDaysNov15.setEntitledDays(0);
		medTierMedBlsNoEntitledDaysNov15.setTier("2");
		medTierMedBlsNoEntitledDaysNov15.setPayment("1363.73");
		medTierMedBlsNoEntitledDaysNov15.setStipendStartDate( Toolbox.parseDate("11/15/2017") );
		medTierMedBlsNoEntitledDaysNov15.setTierHours("108.75");
		
		medTierMedBlsNoEntitledDaysOct10.setBlsRateHourly("11.02");
		medTierMedBlsNoEntitledDaysOct10.setEntitledDays(0);
		medTierMedBlsNoEntitledDaysOct10.setTier("2");
		medTierMedBlsNoEntitledDaysOct10.setPayment("1198.42");
		medTierMedBlsNoEntitledDaysOct10.setStipendStartDate( Toolbox.parseDate("10/10/2017") );
		medTierMedBlsNoEntitledDaysOct10.setTierHours("108.75");
		
		mediumTierHighBlsNoEntitledDaysNov30.setBlsRateHourly("25.08");
		mediumTierHighBlsNoEntitledDaysNov30.setEntitledDays(0);
		mediumTierHighBlsNoEntitledDaysNov30.setTier("2");
		mediumTierHighBlsNoEntitledDaysNov30.setPayment("2727.46");
		mediumTierHighBlsNoEntitledDaysNov30.setStipendStartDate( Toolbox.parseDate("11/30/2017") );
		mediumTierHighBlsNoEntitledDaysNov30.setTierHours("108.75");
		
		highTierHighBlsNoEntitledDaysNov30.setBlsRateHourly("25.08");
		highTierHighBlsNoEntitledDaysNov30.setEntitledDays(0);
		highTierHighBlsNoEntitledDaysNov30.setTier("3");
		highTierHighBlsNoEntitledDaysNov30.setPayment("4351.39");
		highTierHighBlsNoEntitledDaysNov30.setStipendStartDate( Toolbox.parseDate("11/30/2017") );
		highTierHighBlsNoEntitledDaysNov30.setTierHours("174.00");
		
		lowTierMedBlsMediumEntitledDaysJan15.setBlsRateHourly("12.54");
		lowTierMedBlsMediumEntitledDaysJan15.setEntitledDays(60);
		lowTierMedBlsMediumEntitledDaysJan15.setTier("1");
		lowTierMedBlsMediumEntitledDaysJan15.setPayment("545.49");
		lowTierMedBlsMediumEntitledDaysJan15.setStipendStartDate(Toolbox.parseDate("01/15/2018"));
		lowTierMedBlsMediumEntitledDaysJan15.setTierHours("43.50");

		medTierMedBlsMediumEntitledDaysJan15.setBlsRateHourly("12.54");
		medTierMedBlsMediumEntitledDaysJan15.setEntitledDays(60);
		medTierMedBlsMediumEntitledDaysJan15.setTier("2");
		medTierMedBlsMediumEntitledDaysJan15.setPayment("1363.73");
		medTierMedBlsMediumEntitledDaysJan15.setStipendStartDate(Toolbox.parseDate("01/15/2018"));
		medTierMedBlsMediumEntitledDaysJan15.setTierHours("108.75");
		
		medTierlowBlsMediumEntitledDaysJan01.setBlsRateHourly("12.85");
		medTierlowBlsMediumEntitledDaysJan01.setEntitledDays(0);
		medTierlowBlsMediumEntitledDaysJan01.setTier("2");
		medTierlowBlsMediumEntitledDaysJan01.setPayment("1397.44");
		medTierlowBlsMediumEntitledDaysJan01.setStipendStartDate(Toolbox.parseDate("01/01/2018"));
		medTierlowBlsMediumEntitledDaysJan01.setTierHours("108.75");
		
		medTierlowBlsMediumEntitledDaysJan01_V2.setBlsRateHourly("11.52");
		medTierlowBlsMediumEntitledDaysJan01_V2.setEntitledDays(0);
		medTierlowBlsMediumEntitledDaysJan01_V2.setTier("2");
		medTierlowBlsMediumEntitledDaysJan01_V2.setPayment("1252.80");
		medTierlowBlsMediumEntitledDaysJan01_V2.setStipendStartDate(Toolbox.parseDate("01/01/2018"));
		medTierlowBlsMediumEntitledDaysJan01_V2.setTierHours("108.75");
		
		medTierlowBlsMediumEntitledDaysFeb01.setBlsRateHourly("6.27");
		medTierlowBlsMediumEntitledDaysFeb01.setEntitledDays(60);
		medTierlowBlsMediumEntitledDaysFeb01.setTier("2");
		medTierlowBlsMediumEntitledDaysFeb01.setPayment("681.86");
		medTierlowBlsMediumEntitledDaysFeb01.setStipendStartDate(Toolbox.parseDate("02/01/2018"));
		medTierlowBlsMediumEntitledDaysFeb01.setTierHours("108.75");
		
		highTierMedBlsMediumEntitledDaysJan15.setBlsRateHourly("12.54");
		highTierMedBlsMediumEntitledDaysJan15.setEntitledDays(60);
		highTierMedBlsMediumEntitledDaysJan15.setTier("3");
		highTierMedBlsMediumEntitledDaysJan15.setPayment("2181.96");
		highTierMedBlsMediumEntitledDaysJan15.setStipendStartDate(Toolbox.parseDate("01/15/2018"));
		highTierMedBlsMediumEntitledDaysJan15.setTierHours("174.00");
		
		lowTierMedBlsLowEntitledDaysFeb15.setBlsRateHourly("12.54");
		lowTierMedBlsLowEntitledDaysFeb15.setEntitledDays(30);
		lowTierMedBlsLowEntitledDaysFeb15.setTier("1");
		lowTierMedBlsLowEntitledDaysFeb15.setPayment("545.49");
		lowTierMedBlsLowEntitledDaysFeb15.setStipendStartDate(Toolbox.parseDate("02/15/2018"));
		lowTierMedBlsLowEntitledDaysFeb15.setTierHours("43.50");
		
		mediumTierMedBlsLowEntitledDaysFeb15.setBlsRateHourly("12.54");//old
		mediumTierMedBlsLowEntitledDaysFeb15.setEntitledDays(30);
		mediumTierMedBlsLowEntitledDaysFeb15.setTier("2");
		mediumTierMedBlsLowEntitledDaysFeb15.setPayment("1363.73");
		mediumTierMedBlsLowEntitledDaysFeb15.setStipendStartDate(Toolbox.parseDate("02/15/2018"));
		mediumTierMedBlsLowEntitledDaysFeb15.setTierHours("108.75");
		
		highTierMedBlsLowEntitledDaysFeb15.setBlsRateHourly("12.54");
		highTierMedBlsLowEntitledDaysFeb15.setEntitledDays(30);
		highTierMedBlsLowEntitledDaysFeb15.setTier("3");
		highTierMedBlsLowEntitledDaysFeb15.setPayment("2181.96");
		highTierMedBlsLowEntitledDaysFeb15.setStipendStartDate(Toolbox.parseDate("02/15/2018"));
		highTierMedBlsLowEntitledDaysFeb15.setTierHours("174.00");
		
		lowTierlowBlsHighEntitledDaysMar01.setBlsRateHourly("6.27");
		lowTierlowBlsHighEntitledDaysMar01.setEntitledDays(90);
		lowTierlowBlsHighEntitledDaysMar01.setTier("1");
		lowTierlowBlsHighEntitledDaysMar01.setPayment("272.75");
		lowTierlowBlsHighEntitledDaysMar01.setStipendStartDate(Toolbox.parseDate("03/01/2018"));
		lowTierlowBlsHighEntitledDaysMar01.setTierHours("43.50");
		
		highTierlowBlsHighEntitledDaysMar01.setBlsRateHourly("6.27");
		highTierlowBlsHighEntitledDaysMar01.setEntitledDays(90);
		highTierlowBlsHighEntitledDaysMar01.setTier("3");
		highTierlowBlsHighEntitledDaysMar01.setPayment("1090.98");
		highTierlowBlsHighEntitledDaysMar01.setStipendStartDate(Toolbox.parseDate("03/01/2018"));
		highTierlowBlsHighEntitledDaysMar01.setTierHours("174.00");
		
		highTierlowBlsHighEntitledDaysDec10.setBlsRateHourly("6.27");
		highTierlowBlsHighEntitledDaysDec10.setEntitledDays(90);
		highTierlowBlsHighEntitledDaysDec10.setTier("3");
		highTierlowBlsHighEntitledDaysDec10.setPayment("1090.98");
		highTierlowBlsHighEntitledDaysDec10.setStipendStartDate(Toolbox.parseDate("12/10/2017"));
		highTierlowBlsHighEntitledDaysDec10.setTierHours("174.00");
		
		highTierMedBlsHighEntitledDaysNov15.setBlsRateHourly("11.52");
		highTierMedBlsHighEntitledDaysNov15.setEntitledDays(90);
		highTierMedBlsHighEntitledDaysNov15.setTier("3");
		highTierMedBlsHighEntitledDaysNov15.setPayment("2004.48");
		highTierMedBlsHighEntitledDaysNov15.setStipendStartDate(Toolbox.parseDate("11/15/2017"));
		highTierMedBlsHighEntitledDaysNov15.setTierHours("174.00");
		
		mediumTierMedBlsNoEntitledDaysMar20.setBlsRateHourly("12.54");
		mediumTierMedBlsNoEntitledDaysMar20.setEntitledDays(0);
		mediumTierMedBlsNoEntitledDaysMar20.setTier("2");
		mediumTierMedBlsNoEntitledDaysMar20.setPayment("1363.73");
		mediumTierMedBlsNoEntitledDaysMar20.setStipendStartDate( Toolbox.parseDate("03/20/2018") );
		mediumTierMedBlsNoEntitledDaysMar20.setTierHours("108.75");
		
		lowTierLowBlsMediumEntitledDaysApr05.setBlsRateHourly("6.27");
		lowTierLowBlsMediumEntitledDaysApr05.setEntitledDays(60);
		lowTierLowBlsMediumEntitledDaysApr05.setTier("1");
		lowTierLowBlsMediumEntitledDaysApr05.setPayment("272.75");
		lowTierLowBlsMediumEntitledDaysApr05.setStipendStartDate(Toolbox.parseDate("04/05/2018"));
		lowTierLowBlsMediumEntitledDaysApr05.setTierHours("43.50");
		
		lowTierMedBlsLowEntitledDaysApr05.setBlsRateHourly("12.54");
		lowTierMedBlsLowEntitledDaysApr05.setEntitledDays(30);
		lowTierMedBlsLowEntitledDaysApr05.setTier("1");
		lowTierMedBlsLowEntitledDaysApr05.setPayment("545.49");
		lowTierMedBlsLowEntitledDaysApr05.setStipendStartDate(Toolbox.parseDate("04/05/2018"));
		lowTierMedBlsLowEntitledDaysApr05.setTierHours("43.50");
		
		highTierMedBlsLowEntitledDaysApr05.setBlsRateHourly("12.54");
		highTierMedBlsLowEntitledDaysApr05.setEntitledDays(30);
		highTierMedBlsLowEntitledDaysApr05.setTier("3");
		highTierMedBlsLowEntitledDaysApr05.setPayment("2181.96");
		highTierMedBlsLowEntitledDaysApr05.setStipendStartDate(Toolbox.parseDate("04/05/2018"));
		highTierMedBlsLowEntitledDaysApr05.setTierHours("174.00");
		
		mediumTierLowBlsNoEntitledDaysMay10.setBlsRateHourly("6.27");
		mediumTierLowBlsNoEntitledDaysMay10.setEntitledDays(0);
		mediumTierLowBlsNoEntitledDaysMay10.setTier("2");
		mediumTierLowBlsNoEntitledDaysMay10.setPayment("681.86");
		mediumTierLowBlsNoEntitledDaysMay10.setStipendStartDate( Toolbox.parseDate("05/10/2018") );
		mediumTierLowBlsNoEntitledDaysMay10.setTierHours("108.75");
	}
	
	
	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		logger.info(">>>CalculationInfoTest SET UP");
		
		Calendar calendar = Calendar.getInstance();
		calendar.set( Calendar.MONTH, 6 );
		calendar.set( Calendar.YEAR, 2018 );
		calendar.set( Calendar.DAY_OF_MONTH, 1);
		

		Date start = calendar.getTime();
		calendar.roll( Calendar.DAY_OF_MONTH, false);
		Date end = calendar.getTime();
		
		logger.info("start:" + Toolbox.formatDate(start) + " end:" + Toolbox.formatDate(end));
		
		logger.info("max:" + Toolbox.formatDate( DateRange.max( start, end ) ) );
			
		
		
		WorIm workItem = new WorImJUnit();
		StiPt caret0 = new StiPtJUnit();
		caret0.setBlsRateHourly("12.54");
		caret0.setEntitledDays(60);
		caret0.setTier("2");
		caret0.setPayment("1363.73");
		caret0.setStipendStartDate(Toolbox.parseDate("01/05/2018"));
		caret0.setTierHours("108.75");
		calendar.setTime( Toolbox.parseDate("02/01/2018") );
		if ( logger.isDebugEnabled() ) {
			logger.debug(new InitialPayment(workItem, Collections.singletonList(new StipendConfig(caret0)), calendar));
		}

		
		StiPt caret = new StiPtJUnit();
		caret.setBlsRateHourly("12.54");
		caret.setEntitledDays(60);
		caret.setTier("2");
		caret.setPayment("1363.73");
		caret.setStipendStartDate(Toolbox.parseDate("01/11/2018"));
		caret.setTierHours("108.75");
		caret.setPayslipDate(Toolbox.parseDate( "02/01/2018") );
//		calendar.setTime( Toolbox.parseDate("03/01/2018") );

		calendar.setTime( paymentDate );
		Calendar april1st2018paymentDate = Calendar.getInstance();
		april1st2018paymentDate.setTime( Toolbox.parseDate( "04/01/2018") );
		
		applicationApproval_1stDaySameMonthMar01 = new InitialPayment(workItem, Collections.singletonList(new StipendConfig(highTierlowBlsHighEntitledDaysMar01)), april1st2018paymentDate);
		applicationApproval_priorMonthFeb15 = new InitialPayment(workItem, Collections.singletonList(new StipendConfig(highTierMedBlsLowEntitledDaysFeb15)), april1st2018paymentDate);
		applicationApproval_sameMonthMar20 = new InitialPayment(workItem, Collections.singletonList(new StipendConfig(mediumTierMedBlsNoEntitledDaysMar20)), april1st2018paymentDate);
		applicationApproval_2MonthsBackJan15 = new InitialPayment(workItem, Collections.singletonList(new StipendConfig(lowTierMedBlsMediumEntitledDaysJan15)), april1st2018paymentDate);

		List<StipendConfig> lastRecurringConfigs0 = new ArrayList<StipendConfig>();
		lastRecurringConfigs0.add( new StipendConfig(medTierMedBlsNoEntitledDaysNov15) );//payments started Nov 15
		lastRecurringConfigs0.add( new StipendConfig(medTierlowBlsMediumEntitledDaysJan01) );
		complexApplicationApproval_4MonthsBackNov15 = new InitialPayment(workItem, lastRecurringConfigs0, april1st2018paymentDate);
		logger.info(  complexApplicationApproval_4MonthsBackNov15.getFinalAmount() );

		addressChange_DecrPay = new AddressChange(workItem, highTierlowBlsHighEntitledDaysMar01, highTierMedBlsLowEntitledDaysFeb15, april1st2018paymentDate);
		addressChange_IncrPay = new AddressChange(workItem, highTierMedBlsLowEntitledDaysFeb15, highTierlowBlsHighEntitledDaysMar01, april1st2018paymentDate);
//		
		tierChange_DecrPayApr05_Feb15 = new TierChange(workItem, lowTierMedBlsLowEntitledDaysApr05, mediumTierMedBlsLowEntitledDaysFeb15, april1st2018paymentDate);
		tierChange_DecrPay2Feb15_Jan15 = new TierChange(workItem, lowTierMedBlsLowEntitledDaysFeb15, highTierMedBlsMediumEntitledDaysJan15, april1st2018paymentDate);
//		
		tierChange_IncrPayMar20_Jan15 = new TierChange(workItem, mediumTierMedBlsNoEntitledDaysMar20, lowTierMedBlsMediumEntitledDaysJan15, april1st2018paymentDate);
		tierChange_IncrPay2Apr05_Feb15 = new TierChange(workItem, highTierMedBlsLowEntitledDaysApr05, mediumTierMedBlsLowEntitledDaysFeb15, april1st2018paymentDate);
		
		Calendar may1st2018paymentDate = Calendar.getInstance();
		may1st2018paymentDate.setTime( Toolbox.parseDate("05/01/2018") );
		
		List<StipendConfig> lastRecurringConfigs = new ArrayList<StipendConfig>();
		lastRecurringConfigs.add( new StipendConfig(medTierMedBlsNoEntitledDaysNov15) );//payments started Nov 15
		lastRecurringConfigs.add( new StipendConfig(medTierlowBlsMediumEntitledDaysJan01) );//Annual BLS rate increase for defined Address
		lastRecurringConfigs.add( new StipendConfig(medTierlowBlsMediumEntitledDaysFeb01) );//Address change, new lower bls rate starting Feb 1
		complexTierChange_IncrPay = new TierChange(workItem, highTierlowBlsHighEntitledDaysDec10, lastRecurringConfigs, may1st2018paymentDate);//tier increase backdate to Dec 10
		
		lastRecurringConfigs = new ArrayList<StipendConfig>();
		lastRecurringConfigs.add( new StipendConfig(medTierMedBlsNoEntitledDaysOct10) );//payments started Oct 10
		lastRecurringConfigs.add( new StipendConfig(medTierlowBlsMediumEntitledDaysJan01_V2) );//Annual BLS rate increase for defined Address
		
		Calendar Nov1st2018paymentDate = Calendar.getInstance();
		Nov1st2018paymentDate.setTime( Toolbox.parseDate("11/01/2018") );
		complexTierChange_IncrPay2 = new TierChange(workItem, highTierMedBlsHighEntitledDaysNov15, lastRecurringConfigs, Nov1st2018paymentDate);//tier increase backdate to Nov 15
		
		complexTierChange_DecrPay = new TierChange(workItem, highTierMedBlsLowEntitledDaysApr05, mediumTierMedBlsLowEntitledDaysFeb15, april1st2018paymentDate);
		
		revocation_MediumEntitledDays = new FinalPayment(workItem, april1st2018paymentDate, mediumTierMedBlsLowEntitledDaysFeb15);
		revocation_NoEntitledDays = new FinalPayment(workItem, april1st2018paymentDate, mediumTierMedBlsLowEntitledDaysFeb15);
		
		revocation_NoEntitledDaysBackDated = new FinalPayment(workItem, april1st2018paymentDate, mediumTierMedBlsLowEntitledDaysFeb15);
		revocation_MediumEntitledDaysBackDated = new FinalPayment(workItem, april1st2018paymentDate, mediumTierMedBlsLowEntitledDaysFeb15);
		
		lastRecurringConfigs = new ArrayList<StipendConfig>();
		lastRecurringConfigs.add( new StipendConfig(lowTierMedBlsLowEntitledDaysApr05) );
		lastRecurringConfigs.add( new StipendConfig(lowTierMedBlsLowEntitledDaysFeb15) );
		reinstatementNoChangeNoEntitlement  = new Reinstatement(workItem, lastRecurringConfigs, may1st2018paymentDate);
		
		Calendar june1st2018paymentDate = Calendar.getInstance();
		june1st2018paymentDate.setTime( Toolbox.parseDate("05/01/2018") );
		
		lastRecurringConfigs = new ArrayList<StipendConfig>();
		lastRecurringConfigs.add( new StipendConfig(lowTierMedBlsLowEntitledDaysApr05) );
		lastRecurringConfigs.add( new StipendConfig(lowTierMedBlsLowEntitledDaysFeb15) );
		reinstatementNoChangeBackDatedNoEntitlement  = new Reinstatement(workItem, lastRecurringConfigs, may1st2018paymentDate);
		
		lastRecurringConfigs = new ArrayList<StipendConfig>();
		lastRecurringConfigs.add( new StipendConfig(highTierlowBlsHighEntitledDaysMar01) );
		lastRecurringConfigs.add( new StipendConfig(lowTierLowBlsMediumEntitledDaysApr05) );
		reinstatementBackDatedHighEntitlement = new Reinstatement(workItem, lastRecurringConfigs, april1st2018paymentDate);
		
		lastRecurringConfigs = new ArrayList<StipendConfig>();
		lastRecurringConfigs.add( new StipendConfig(highTierlowBlsHighEntitledDaysMar01) );
		reinstatementBackDatedNoEntitlement = new Reinstatement(workItem, lastRecurringConfigs, april1st2018paymentDate);
		
		lastRecurringConfigs = new ArrayList<StipendConfig>();
		lastRecurringConfigs.add( new StipendConfig(lowTierlowBlsHighEntitledDaysMar01) );
		lastRecurringConfigs.add( new StipendConfig(highTierlowBlsHighEntitledDaysMar01) );
		reinstatementBackDatedTierIncrease = new Reinstatement(workItem, lastRecurringConfigs, april1st2018paymentDate);
		
		lastRecurringConfigs = new ArrayList<StipendConfig>();
		lastRecurringConfigs.add( new StipendConfig(highTierlowBlsHighEntitledDaysMar01) );
		lastRecurringConfigs.add( new StipendConfig(lowTierlowBlsHighEntitledDaysMar01) );
		reinstatementBackDatedTierDecrease = new Reinstatement(workItem, lastRecurringConfigs, april1st2018paymentDate);
		
		if ( logger.isDebugEnabled() ) {
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& application-Approval &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(applicationApproval_sameMonthMar20.toString());
			
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& application-Approval &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(applicationApproval_priorMonthFeb15.toString());
			
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& application-Approval &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(applicationApproval_1stDaySameMonthMar01.toString());
			
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& application-Approval &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(applicationApproval_2MonthsBackJan15.toString());
			
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& address-Change-Decreased-Pay &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(addressChange_DecrPay.toString());
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& address-Change-Increased-Pay &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(addressChange_IncrPay.toString());
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& tier-Change-Decreased-Pay &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(tierChange_DecrPayApr05_Feb15.toString());
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& tier-Change-Decreased-Pay2 &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(tierChange_DecrPay2Feb15_Jan15.toString());
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& tier-Change-Increased-Pay &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(tierChange_IncrPayMar20_Jan15.toString());
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& revocation-Medium-Entitled-Days &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
//			logger.debug(revocation_MediumEntitledDays.toString());
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& Reinstatement &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
			logger.debug(reinstatementNoChangeNoEntitlement.toString());
			
			
			
		}
	}
	@AfterClass
	public static void tearDownAfterClass() throws Exception {
	}
	
	
//	public static Object rollUntilActivated ( Class cl, StipendConfig targetConfig, Calendar rollingDate, Object previousActivated, StipendConfig activeConfig ) throws InstantiationException, IllegalAccessException, IllegalArgumentException, InvocationTargetException, NoSuchMethodException, SecurityException {
//		
//		Object activated;
//		for ( ; targetConfig.getEventDate().after( rollingDate.getTime() ); ) {
//			if ( rollingDate.get(Calendar.MONTH) == Calendar.DECEMBER ) {
//				rollingDate.roll(Calendar.YEAR, true);
//			}
//			rollingDate.roll(Calendar.MONTH, true);
//			
//			logger.info( "Payment Month Date:" + Toolbox.formatDate( rollingDate.getTime() ) );
//			
//			if ( targetConfig.getEventDate().before( rollingDate.getTime() ) ) {
////				initialConfig = nextConfig;
//				break;
//			}
//				
//			if ( activeConfig != null ) {
//				logger.info( "\n Pertinent to:(" + previousActivated.getClass().getSimpleName() +")" );
//				if ( activeConfig.getLastAppliedDate() == null ) {
//					//first payment...
//					logger.info( "First Payment of..." );
//					logger.info( previousActivated );
//					activeConfig.setLastAppliedDate( rollingDate.getTime() );
//				} else {
//					System.out.print( "Continue:");
//					logger.info( previousActivated );
//					activeConfig.setLastAppliedDate( rollingDate.getTime() );
//				}
//			} else {
//				logger.info( "CareT: NO Payment events found for this VCG..." );
//			}
//		}
//		if ( CalculationChange.class.isAssignableFrom(cl) ) {
//			Constructor<?> cons = cl.getConstructor( StipendConfig.class, StipendConfig.class, Integer.class, Integer.class );
//			activated = cons.newInstance(targetConfig, activeConfig, rollingDate.get(Calendar.MONTH), rollingDate.get(Calendar.YEAR) );
//		} else {
//			Constructor<?> cons = cl.getConstructor( StipendConfig.class, Integer.class, Integer.class );
//			activated = cons.newInstance(targetConfig, rollingDate.get(Calendar.MONTH), rollingDate.get(Calendar.YEAR) );
//		}
//		
//		if ( targetConfig.getLastAppliedDate() != null ) {
//			logger.info( "------------------");
//			logger.info( "Monthly Payment Applied:" + targetConfig.getPayment( ) );
//			logger.info( "------------------");
//		} else {
//			logger.info( "\n \n ------------------------");
//			logger.info( activated );
//		}
//		
//		targetConfig.setLastAppliedDate( rollingDate.getTime() );
//		
//		return activated;
//	}
	
	public static void main ( String[] args ) {
		
		Date last = null;
		
		Calendar calendar = Calendar.getInstance();
		calendar.set( Calendar.MONTH, 5 );
		calendar.set( Calendar.YEAR, 2018 );
		calendar.set( Calendar.DAY_OF_MONTH, 1);

		Date start = calendar.getTime();
		calendar.roll( Calendar.DAY_OF_MONTH, false);
		Date end = calendar.getTime();
		
		
		
	}
	
}
