package gov.va.caret.service.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.util.ArrayUtil;
import com.liferay.portal.kernel.util.LocaleUtil;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.kernel.util.StringPool;
import com.liferay.portal.kernel.workflow.WorkflowConstants;
import com.liferay.portal.model.Organization;
import com.liferay.portal.model.Role;
import com.liferay.portal.model.RoleConstants;
import com.liferay.portal.model.User;
import com.liferay.portal.security.auth.PrincipalThreadLocal;
import com.liferay.portal.security.permission.PermissionChecker;
import com.liferay.portal.security.permission.PermissionCheckerFactoryUtil;
import com.liferay.portal.security.permission.PermissionThreadLocal;
import com.liferay.portal.service.GroupLocalServiceUtil;
import com.liferay.portal.service.OrganizationLocalServiceUtil;
import com.liferay.portal.service.RoleLocalServiceUtil;
import com.liferay.portal.service.ServiceContext;
import com.liferay.portal.service.ServiceContextThreadLocal;
import com.liferay.portal.service.UserLocalServiceUtil;
import com.liferay.portal.service.UserServiceUtil;
import com.liferay.portal.util.PortalUtil;

import gov.va.caret.model.Facil;
import gov.va.caret.model.Persn;
import gov.va.caret.model.impl.PersnImpl;
import gov.va.caret.model.support.Person;
import gov.va.caret.security.CAction;
import gov.va.caret.security.CodifiedRole;
import gov.va.caret.service.CaretLocalServiceUtil;
import gov.va.caret.service.FacilLocalServiceUtil;
import gov.va.caret.service.PersnLocalServiceUtil;
import gov.va.caret.service.base.CaretServiceBaseImpl;
import gov.va.caret.util.CaretStrPool;
import gov.va.caret.util.CaretUtil;
import gov.va.caret.util.TemplateHandler;
import gov.va.caret.util.Toolbox;

/**
 * The implementation of the caret remote service.
 *
 * <p>
 * All custom service methods should be put in this class. Whenever methods are added, rerun ServiceBuilder to copy their definitions into the {@link gov.va.caret.service.CaretService} interface.
 *
 * <p>
 * This is a remote service. Methods of this service are expected to have security checks based on the propagated JAAS credentials because this service can be accessed remotely.
 * </p>
 *
 * @author caret
 * @see gov.va.caret.service.base.CaretServiceBaseImpl
 * @see gov.va.caret.service.CaretServiceUtil
 */
public class CaretServiceImpl extends CaretServiceBaseImpl {
    private static Persn NOT_FOUND_PERSON = new PersnImpl(){
		private static final long serialVersionUID = 1L;
		public String getStatus() {  return "Person Not found!"; };
    };

	/*
     * NOTE FOR DEVELOPERS:
     *
     * Never reference this interface directly. Always use {@link gov.va.caret.service.CaretServiceUtil} to access the caret remote service.
     */
	Log _log = LogFactoryUtil.getLog(CaretServiceImpl.class);

	private long getCompanyId(){
		if ( _log.isDebugEnabled() ){
			_log.info("comanyId=" + PortalUtil.getDefaultCompanyId() );
		}
		return PortalUtil.getDefaultCompanyId();
	}
	
	public boolean isPermitted ( String action ) {
		PermissionChecker permissionChecker = PermissionThreadLocal.getPermissionChecker();
		if ( permissionChecker != null ){ 
			try { 
				 return CAction.canDoCaret( permissionChecker, CAction.CaretCan.valueOf(action), getClinicId(getCompanyId()) );
			} catch (Exception e ){
				_log.error("Permissioning Error", e);
			}
		} else {
			_log.error("PermissionChecker is null ");
		}
		return false;
	}
	
	
	public Persn activationBySecId ( String secId, boolean deactivate ) throws PortalException, SystemException {
		PermissionChecker permissionChecker = PermissionThreadLocal.getPermissionChecker();
		String permissionLog = ", you cannot update [permissionChecker==null]";
		if ( permissionChecker != null ){
			long companyId = getCompanyId();
			if ( CAction.canDoCaret( permissionChecker, CAction.CaretCan.VIEW_VETERANS, getClinicId(companyId) ) ){
				int status = WorkflowConstants.STATUS_APPROVED;
				User user = findUser( companyId, secId, null );
				if ( user!= null ){
					Persn p = PersnLocalServiceUtil.getByUser( user.getUserId() );
					if ( deactivate ){
						status = WorkflowConstants.STATUS_INACTIVE;
						p.setStatus("DEACTIVATED_" + p.getStatus() );
					} else {
						p.setStatus("ACTIVATED_" + p.getStatus() );
					}
					user = UserLocalServiceUtil.updateStatus(user.getUserId(), status, getCaretServiceContext() );
					return p;
				} else {
					permissionLog = ", user not found";
				}
			} else {
				permissionLog = ", you do not have permission";
			}
		}
		_log.info("samAccountName =" + secId + permissionLog );
		return NOT_FOUND_PERSON;
	}
	
	public Persn getPersonBySecId ( String secId ) throws PortalException, SystemException {
		
		PermissionChecker permissionChecker = PermissionThreadLocal.getPermissionChecker();
		String permissionLog = ", you cannot update [permissionChecker==null]";
		if ( permissionChecker != null ){ 
			long companyId = getCompanyId();
			if ( CAction.canDoCaret( permissionChecker, CAction.CaretCan.VIEW_VETERANS, getClinicId(companyId)  ) ){
				User user = findUser( companyId, secId, null );
				if ( user!= null ){
					Persn persn = PersnLocalServiceUtil.getByUser( user.getUserId() );
					persn.setStatus( (user.isActive()? "ACTIVE_": "INACTIVE_" ) + user.getStatus() );
					return persn;
				}
			}
		}
		_log.info("secId =" + secId + permissionLog );
		
		return NOT_FOUND_PERSON;
	}
	
	
	public String getXmlBySecId ( String secId ) throws PortalException, SystemException {
		
		PermissionChecker permissionChecker = PermissionThreadLocal.getPermissionChecker();
		String xmlResponse = null;
		if ( permissionChecker != null ){ 
			long companyId = getCompanyId();
			if ( CAction.canDoCaret( permissionChecker, CAction.CaretCan.VIEW_VETERANS, getClinicId(companyId)  ) ){
				TemplateHandler templateHandler = new TemplateHandler();
				User user = findUser( companyId, secId, null );
				Map<String,Object> templateParams;
				if ( user!= null ){
					Persn persn = PersnLocalServiceUtil.getByUser( user.getUserId() );
					templateParams = persn.getModelAttributes();
					templateParams.put("externalId", secId );
					templateParams.put("officeLocation", persn.getAddress() );
					
					String facility = StringPool.BLANK;
					String facilityName = StringPool.BLANK;
					String visnName = StringPool.BLANK;
					for ( Organization org : OrganizationLocalServiceUtil.getUserOrganizations( user.getUserId() ) ){
						if ( !org.getName().startsWith("VISN") ){
							facilityName = facilityName + org.getName() + StringPool.COMMA;
							facility = facility + FacilLocalServiceUtil.getByOrgId( org.getOrganizationId() ).getFacilityNumber() + StringPool.COMMA;
						} else {
							visnName = visnName + org.getName() + StringPool.COMMA;
						}
					}
					templateParams.put("facility",  facility.isEmpty()? facility : facility.substring(0, facility.length()-1) );
					templateParams.put("facilityName", facilityName.isEmpty()? facilityName : facilityName.substring(0, facilityName.length()-1) );
					templateParams.put("visn", visnName.isEmpty()? visnName : visnName.substring(0, visnName.length() -1) );
					
					String roles = "";
					for ( Role role : RoleLocalServiceUtil.getSubtypeRoles( PropsUtil.get( "roles.regular.subtypes" ) ) ){
						if ( RoleLocalServiceUtil.hasUserRole( user.getUserId(), role.getRoleId() ) ){
							roles = roles + CodifiedRole.getCode( role.getName() ) + StringPool.COMMA;
						}
					}
					templateParams.put("roles", roles.substring( 0, Math.max(roles.length()-1,0) ) );
					templateParams.put("active", String.valueOf(user.isActive()) );
					templateParams.put("jobTitle", user.getJobTitle() );
					templateParams.put("birthDateStr", persn.getBirthDate() == null? StringPool.BLANK: Toolbox.formatDate( persn.getBirthDate() ) );
					templateParams.put("deceasedDateStr", persn.getDeceasedDate() == null? StringPool.BLANK: Toolbox.formatDate( persn.getDeceasedDate() ) );
				} else {
					templateParams = Person.DEFAULT_PERSON.getModelAttributes();
					templateParams.put("externalId", secId );
					templateParams.put("facility", StringPool.BLANK );
					templateParams.put("officeLocation", StringPool.BLANK );
					templateParams.put("facilityName", StringPool.BLANK );
					templateParams.put("roles", StringPool.BLANK );
					templateParams.put("active", "false" );
					templateParams.put("jobTitle", StringPool.BLANK);
					templateParams.put("deceasedDateStr", StringPool.BLANK);
					templateParams.put("birthDateStr", StringPool.BLANK);
					templateParams.put("visn", StringPool.BLANK);
				}
				try {
					xmlResponse = templateHandler.loadTemplate(templateParams, "/gov/va/oit/oed/vaww/templates", "activationResponse.ftl" );
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		_log.info("secId =" + xmlResponse  );
		
		return xmlResponse;
	}
	
	public String updateUserXml ( Persn person, String[] rolesArr, Facil facil ) throws PortalException, SystemException {
		person = updateUser(person, rolesArr, facil );
		
		TemplateHandler templateHandler = new TemplateHandler();
		User user = findUser( getCompanyId(), person.getExternalId(), null );
		
		Map<String,Object> templateParams = person.getModelAttributes();
		templateParams.put("externalId", user.getScreenName() );
		templateParams.put("officeLocation", person.getAddress2() );
		
		String facility = StringPool.BLANK;
		String facilityName = StringPool.BLANK;
		String visnName = StringPool.BLANK;
		for ( Organization org : OrganizationLocalServiceUtil.getUserOrganizations( user.getUserId() ) ){
			if ( !org.getName().startsWith("VISN") ){
				facilityName = facilityName + org.getName() + StringPool.COMMA;
				facility = facility + FacilLocalServiceUtil.getByOrgId( org.getOrganizationId() ).getFacilityNumber() + StringPool.COMMA;
			} else {
				visnName = visnName + org.getName() + StringPool.COMMA;
			}
		}
		templateParams.put("facility", facility.isEmpty()? facility : facility.substring(0, facility.length()-1)  );
		templateParams.put("facilityName", facilityName.isEmpty()? facilityName : facilityName.substring(0, facilityName.length()-1) );
		templateParams.put("visn", visnName.isEmpty()? visnName : visnName.substring(0, visnName.length() -1) );
		
		String roles = "";
		for ( Role role : RoleLocalServiceUtil.getSubtypeRoles( PropsUtil.get( "roles.regular.subtypes" ) ) ){
			if ( RoleLocalServiceUtil.hasUserRole( user.getUserId(), role.getRoleId() ) ){
				roles = roles + CodifiedRole.getCode( role.getName() ) + StringPool.COMMA;
			}
		}
		templateParams.put("roles", roles.substring( 0, Math.max(roles.length()-1,0) ) );
		templateParams.put("active", String.valueOf(user.isActive()) );
		templateParams.put("jobTitle", user.getJobTitle() );
		templateParams.put("birthDateStr", person.getBirthDate() == null? StringPool.BLANK: Toolbox.formatDate( person.getBirthDate() ) );
		templateParams.put("deceasedDateStr", person.getDeceasedDate() == null? StringPool.BLANK: Toolbox.formatDate( person.getDeceasedDate() ) );
		
		String xmlResponse = StringPool.BLANK;
		try {
			xmlResponse = templateHandler.loadTemplate(templateParams, "/gov/va/oit/oed/vaww/templates", "activationResponse.ftl" );
		} catch (IOException e) {
			e.printStackTrace();
		}
		_log.info("secId =" + xmlResponse  );

		return xmlResponse;
	}
	
	public Persn updateUser ( Persn person, String[] roles, Facil facility ) throws PortalException, SystemException {
		
		_log.info("secID (ExternalId)=" + person.getExternalId() + 
				"\n jobTitle(Prefix)=" + person.getPrefix() + 
				"\n  officeLocation(Address)=" + person.getAddress2() + 
				"\n  facility(FacilityNumber)=" + facility.getFacilityNumber() + 
				"\n  facilityName(Location)=" + facility.getLocation() +
				"\n  email=" + person.getEmail() + " phoneNumber=" + person.getPhone() + 
				"\n  roles:" + Arrays.toString(roles) + "\n FirstName=" + 
				person.getFirstName() + " LastName="  + person.getLastName() );

		String secId = person.getExternalId();
		PermissionChecker permissionChecker = PermissionThreadLocal.getPermissionChecker();
		String permissionLog = ", you cannot update [permissionChecker==null]";
		
		if ( permissionChecker != null ){ 
			long companyId = getCompanyId();
			if ( CAction.canDoCaret( permissionChecker, CAction.CaretCan.VIEW_VETERANS, getClinicId(companyId)   ) ) {
				try{
					User user = findUser( companyId, secId, null );
					Persn updatePerson;
					if ( user!= null ){

						getValidRoles( companyId, roles, facility, new long[]{user.getUserId()}, true);
						
						updatePerson = PersnLocalServiceUtil.getByUser( user.getUserId() );
						
						updatePerson.setFirstName( person.getFirstName() );
						updatePerson.setMiddleName( person.getMiddleName() );
						updatePerson.setLastName( person.getLastName() );
						
						CaretLocalServiceUtil.save(updatePerson);
						updatePerson.setStatus( (user.isActive()? "ACTIVE_": "INACTIVE_" ) + person.getStatus() );
						UserLocalServiceUtil.updateJobTitle( user.getUserId(), person.getPrefix() );
					} else {
						_log.info("New user...");
						String jobTitle = person.getPrefix();

						long[] orgId = new long[]{0};
						long[] roleIds = ArrayUtil.toLongArray( getValidRoles( companyId, roles, facility, orgId, false) );
						
						int[] birthday;
						if ( person.getBirthDate() != null){
							Calendar c = Calendar.getInstance();
							c.setTime( person.getBirthDate() );
							birthday = new int[]{ c.get(Calendar.MONTH), c.get(Calendar.DAY_OF_MONTH), c.get(Calendar.YEAR) };
						} else {
							_log.info("missing birthday assignment, using default...");
							birthday = new int[]{10,15,1899};
						}
						
						boolean isMale;
						if ( person.getGender() == null || person.getGender().trim().isEmpty() ){
							_log.info("missing gender assignment, using default...");
							isMale = true;
						} else {
							isMale = 'm' == person.getGender().toLowerCase().trim().charAt(0);
						}
						try{
							user = UserLocalServiceUtil.getUserByEmailAddress( companyId, person.getEmail() );
							person.setEmail(null);
						} catch ( PortalException p ){
							//ignore missing user...
						}
						
						if ( orgId[0] == 0 ){
							orgId = new long[]{} ;
						}
							
						user = setUser( companyId, CaretUtil.getClinicSite( ), secId, person.getEmail(), 
									person.getFirstName(), person.getMiddleName(), person.getLastName(),
									isMale, roleIds, orgId, birthday, jobTitle  );
						
//						UserServiceUtil.updateUser(userId, oldPassword, newPassword1, newPassword2, passwordReset, reminderQueryQuestion, reminderQueryAnswer, screenName, emailAddress, facebookId, openId, languageId, timeZoneId, greeting, comments, firstName, middleName, lastName, prefixId, suffixId, male, birthdayMonth, birthdayDay, birthdayYear, smsSn, aimSn, facebookSn, icqSn, jabberSn, msnSn, mySpaceSn, skypeSn, twitterSn, ymSn, jobTitle, groupIds, organizationIds, roleIds, userGroupRoles, userGroupIds, serviceContext);
						
						updatePerson = PersnLocalServiceUtil.getByUser( user.getUserId() );
						
						_log.info("Created user with screenName=" + user.getScreenName() );
						
						updatePerson = updateUser(person, roles, facility);
						
					}
					updatePerson.setExternalId( secId );
					return updatePerson;
				}catch(PortalException p){
					_log.warn(p);
				}
			} else {
				permissionLog = ", you do not have permission";
			}
		}
		_log.info("secId =" + secId + permissionLog );
		return NOT_FOUND_PERSON;
	}

	private Long getClinicId( long companyId ) throws PortalException, SystemException {
		return GroupLocalServiceUtil.getGroup(companyId, CaretUtil.getClinicSite() ).getGroupId();
	}

	private List<Long> getValidRoles( long companyId, String[] roles, Facil facil, long[] updaterId, boolean updateNow ) throws PortalException, SystemException {
		long orgId = facil == null? 0: facil.getGroupId();
		Set<Long> organIds = new HashSet<Long>();
		long userId = updaterId[0];
		List<Long> roleIds = new ArrayList<Long>();
		for ( String roleCode: roles ){
			Role role = RoleLocalServiceUtil.fetchRole( companyId, CodifiedRole.getRoleName( roleCode ) );
			String name = role.getName().toLowerCase();
			if (  name.contains("visn") || name.contains("csc") ){
				if ( facil != null ){
					String[] facils = facil.getFacilityNumber().split(",");
					for ( String facilName: facils ){
						Facil facilGroup = null;
						if ( !facilName.startsWith("VISN") ) {
							facilGroup = FacilLocalServiceUtil.getByFacilityNumber( facilName );
						}
						if ( facilGroup != null ){
							orgId = facilGroup.getGroupId();
						} else {
							try{
								orgId = OrganizationLocalServiceUtil.getOrganizationId( getCompanyId(), facilName );
							} catch ( Exception e ){}
						}
						if ( orgId > 0 ){
							organIds.add( orgId );
						}
					}
				}
			}
			roleIds.add( role.getRoleId() );
		}
		
		if ( updateNow && !roleIds.isEmpty() ){
			CodifiedRole.updateClinicRole( userId, ArrayUtil.toLongArray(roleIds), organIds.toArray(new Long[organIds.size()]) );
		} else {
			updaterId[0] = orgId;
		}
		return roleIds;
	}


	private User setUser( long comanyId, String site, String screenName, String email, String firstName, String middleName, String lastName, 
			boolean isMale, long[] roles, long[] orgId, int[] bday, String jobTitle  ) throws SystemException {

		String password = screenName+".";
		try {
			User adm = UserLocalServiceUtil.getRoleUsers(RoleLocalServiceUtil.getRole(comanyId, RoleConstants.ADMINISTRATOR).getRoleId()).get(0);
			PrincipalThreadLocal.setName( adm.getUserId() );
			PermissionThreadLocal.setPermissionChecker( PermissionCheckerFactoryUtil.create(adm) );
			
			return UserServiceUtil.addUser(comanyId, true, password,
					password, false, screenName, email, 
					0l, StringPool.BLANK, LocaleUtil.getDefault(), firstName, middleName, 
					lastName, 0, 0, isMale, bday[0], bday[1], bday[2], jobTitle, 
					new long[]{ GroupLocalServiceUtil.getGroup( comanyId, site ).getGroupId() }, orgId, 
					roles, new long[]{}, false, new ServiceContext());
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}
	
	private User findUser( long companyId, String screenName, String email ) throws SystemException {
		User user = null;
		try {
			user = UserLocalServiceUtil.getUserByScreenName( companyId, screenName );
		} catch (PortalException e) {} //ignoring exception thrown when the user does not exist... 
//		if ( user == null && !Toolbox.isEmpty(email) ){
//			try {
//				user = UserLocalServiceUtil.getUserByEmailAddress( companyId, email );
//			} catch (PortalException e) {} //ignoring exception thrown when the user does not exist... 
//			if ( user != null ){
//				_log.warn("WARNING: Could not find by SCREEN_NAME, found user by EMAIL: " + user.getEmailAddress() );
//			}
//		}
		return user;
	}
	
	private ServiceContext getCaretServiceContext() {
		PermissionChecker permissionChecker = PermissionThreadLocal.getPermissionChecker();
		
		ServiceContext sc = ServiceContextThreadLocal.getServiceContext();
		if ( sc == null ) {
			sc = new ServiceContext();
			ServiceContextThreadLocal.pushServiceContext(sc);
		}
		sc.setAddGuestPermissions(false);
		
		sc.setCreateDate ( new java.util.Date() );
		sc.setModifiedDate( sc.getCreateDate() );
		sc.setAttribute( CaretStrPool.USER_ID, permissionChecker.getUserId() );
		
		sc.setAttribute( CaretStrPool.ROLE_ID, 0 );
		sc.setAttribute( CaretStrPool.DASHBOARD_ROLE_NAME, "SERVICE" );
		
		return sc;
	}
}
