package gov.va.caret.model.support.occ;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;

import gov.va.caret.model.BlsRe;
import gov.va.caret.model.StiPt;
import gov.va.caret.model.WorIm;
import gov.va.caret.model.support.VcgSupport;
import gov.va.caret.service.BlsReLocalServiceUtil;
import gov.va.caret.service.VcgLocalServiceUtil;
import gov.va.caret.util.Toolbox;


public class InitialPayment extends MultiFacetedConfig implements BackPay, ComplexChange {

	private BigDecimal totalStipend;  				//[Monthly Amount] + [Prorated Amount] + [Back Pay] OCC Super must review if > 35k
	private BigDecimal backPay = null;				//[Number of Retro Months] * [Monthly Amount]
	private static Log _log = LogFactoryUtil.getLog( InitialPayment.class );
			
	public InitialPayment(WorIm workItem, StiPt config, Calendar paymentMonthDate ) {
		super(workItem, config, paymentMonthDate);
		if ( workItem != null ) {
			setUp( config.getStipendStartDate() );
		}
	}
	
	public InitialPayment(WorIm workItem, List<StipendConfig> configs, Calendar paymentMonthDate ) {
		super(workItem, configs.get(0), paymentMonthDate);
		lastRecurringConfigs = configs;
		setUp( config.getStipendStartDate() );
	}

	public BigDecimal getMonthlyBackPayAmount(){
		if ( lastRecurringConfigs != null && !lastRecurringConfigs.isEmpty() ) {
			if ( lastRecurringConfigs.size() > 1 ) {
				BigDecimal monthlyBackPayAmount = BigDecimal.ZERO;
				for ( int ndx = 0; ndx < lastRecurringConfigs.size(); ndx++ ) {
					BigDecimal monthlyBackPayAmountNdx =  new BigDecimal( getRetroactiveMonths( ndx ) ).multiply( getLastRecurringMonthlyRate( ndx ) ).setScale( 2, RoundingMode.HALF_UP );
					monthlyBackPayAmount = monthlyBackPayAmount.add( monthlyBackPayAmountNdx );
					lastRecurringConfigs.get(ndx).setMonthlyBackPayAmount( monthlyBackPayAmountNdx );
				}
				return monthlyBackPayAmount;
			} else {
				if ( lastRecurringConfigs.get(0).getMonthlyBackPayAmount() == null ) {
					lastRecurringConfigs.get(0).setMonthlyBackPayAmount( new BigDecimal( getRetroactiveMonths( 0 ) ).multiply( getLastRecurringMonthlyRate( 0 ) ).setScale( 2, RoundingMode.HALF_UP ) );
				}
				return lastRecurringConfigs.get(0).getMonthlyBackPayAmount(); 
			}
		}
		return new BigDecimal( getRetroactiveMonths() ).multiply( getMonthlyRate() ).setScale( 2, RoundingMode.HALF_UP );
	}
	
	//Monthly Stipend Payment+Backpay Prorate Amount = Total Stipend Payment
	public BigDecimal getBackPayAmount(){
		if ( isRecurring() && getStipendPaymentConfig().getPayslipDate() != null ) {
			return defaultValue;
		}
		Calendar calendar = Calendar.getInstance();
		calendar.setTime( config.getStipendStartDate() );
		if ( getPaymentDate().compareTo( StipendConfig.getPayday( calendar ).getTime() ) <= 0 ){
			return defaultValue;
		}
		if ( backPay == null ){
			backPay = getMonthlyBackPayAmount().add( getProratedAmount() );
		}
		return backPay;
	}
	
	
	public BigDecimal getFinalAmount(){
		if ( totalStipend == null ){
			if ( StipendConfig.ONE_TIME.equals( getStipendPaymentConfig().getStipendType() ) || BigDecimal.ZERO.equals( getMonthlyRate() ) ) {
				return new BigDecimal(getStipendPaymentConfig().getOneTimePayment());
			}
			if ( isRecurring() && getStipendPaymentConfig().getPayslipDate() != null ) {
				totalStipend = getMonthlyRate();
			} else {
				Calendar lastDayStipendStartMonth = Calendar.getInstance();
				lastDayStipendStartMonth.setTime( config.getStipendStartDate() );
				
				if ( lastDayStipendStartMonth.get(Calendar.DATE) > 1 && getPaymentDate().compareTo( StipendConfig.getPayday( lastDayStipendStartMonth ).getTime() ) == 0 ){
					totalStipend = getProratedAmount();
				} else {
					totalStipend = getMonthlyRate().add( getBackPayAmount() );
				}
			}
		}
		return totalStipend;
	}
	
	public BigDecimal getMonthlyRate (  ) {
		if ( monthlyRate == null && config != null ) {
			if ( !Toolbox.isEmpty( config.getPayment() ) ) {
				monthlyRate = new BigDecimal( config.getPayment() );
			}
		}
		if ( monthlyRate == null ) {
			monthlyRate = new BigDecimal( config.getTierHours() ).multiply( new BigDecimal( config.getBlsRateHourly() ) ).setScale( 2, RoundingMode.HALF_UP );
			config.setPayment( monthlyRate.toPlainString() );
		}
		return monthlyRate;
	}
	
	public static String getFinalAmount ( String stipendStartDate, String payment, String tierHours, String blsRateHourly, Calendar payday  ) {
		return InitialPayment.getFinalAmount(Toolbox.parseDateTime(stipendStartDate), payment, tierHours, blsRateHourly, payday );
	}
	
	public static String getFinalAmount ( Date stipendStartDate, String payment, String tierHours, String blsRateHourly, Calendar payday  ) {
		return new InitialPayment(null, StipendConfig.getStipendConfig( stipendStartDate, payment, tierHours, blsRateHourly ), payday).getFinalAmount().toString();
	}
	
	//Junit tested...
	public int getTotalDays( ) {
		if ( getPaymentDate().before( config.getStipendStartDate() ) ) {
			return 0;
		}
		if ( isRecurring() && getStipendPaymentConfig().getPayslipDate() != null ) {
			return 0;
		} else {
			Calendar calendar = Calendar.getInstance();
			if ( isMultiYearMonth() ) {
				calendar.setTime( lastRecurringConfigs.get(lastRecurringConfigs.size()-1).getStipendStartDate() );
			} else {
				calendar.setTime( config.getStipendStartDate() );
			}

			if ( getPaymentDate().compareTo( StipendConfig.getPayday( calendar ).getTime() ) < 0 ){
				return 0;
			}
			int applicationDay = calendar.get( Calendar.DAY_OF_MONTH );
			if ( applicationDay == 1) {
				return 0;				//full month will be accounted
			}
			calendar.set( Calendar.DAY_OF_MONTH, 1 );
			calendar.roll(Calendar.DAY_OF_MONTH, false );
			int endDateCurrentPaymentMonth = calendar.get( Calendar.DAY_OF_MONTH );
			return endDateCurrentPaymentMonth - applicationDay + 1;//to make the approval date inclusive
		}
	}
	
	//Application Date - End Date of Current Payment Month = Total Days
	public BigDecimal getProratedAmount(){  //getProratedAmount()
		if (  isRecurring() && getStipendPaymentConfig().getPayslipDate() != null ) {
			return defaultValue;
		}
		int totaldays = getTotalDays();
		if ( totaldays == 0 ) {
			return defaultValue;
		}
		return new BigDecimal( totaldays ).multiply( getDailyRate() ).setScale( 2, RoundingMode.HALF_UP );
	}
	
	public String getOneTime(int i) {
		return getLastRecurringConfigs().get(i).getOneTimePayment();
	}
	
	@Override
	public BigDecimal getOneTimePayment() {
		if ( config.getPayslipDate() == null ) {
			return getBackPayAmount();
		}
		return defaultValue;
	}
	
	public boolean isRecoupable(){
		return false;
	}
	public boolean isBackPayDetected() {
		return getBackPayAmount().compareTo( BigDecimal.ZERO ) > 0; 
	}
	
	public boolean isProrateFormula() {
		return true;
	}

	public String toString () {
		return  "Stipend Payment Date: " + formatDate( getPaymentDate() ) + 
				"\n        With Begin Date: " + formatDate( config.getStipendStartDate() ) +
				"\n          Tier Level of: " + config.getTier() + 
				"\n Total Monthly Hours of: " + config.getTierHours() + 
				"\n   Local hourly Rate of: " + config.getBlsRateHourly() +
				"\n          Daily Rate of: " + getDailyRate() +
				"\n        Retro Months of: " + getRetroactiveMonths(  ) +
				"\n          Total Days of: " + getTotalDays(  ) +
				"\n           Payment owed: " + getFinalAmount(  ) +
				"\n------------------------------------------------" +
				"\n MonthlyRecurringAmount: " + getMonthlyRecurringAmount()+
				"\n          BackPayAmount: " + getBackPayAmount()+
				"\n            FinalAmount: " + getFinalAmount()+
				//getRecoupmentAmount();
				"\n         ProratedAmount: " + getProratedAmount()+
				"\n            MonthlyRate: " + getMonthlyRate()+
				"\n              DailyRate: " + getDailyRate()+
				"\n             HourlyRate: " + getHourlyRate()+
				"\n           ProratedDate: " + getProratedDate()+
				"\n      RetroactiveMonths: " + getRetroactiveMonths();
	}

	public void setEffectiveDates ( Date effectiveDate ) {
		
		if ( lastRecurringConfigs != null && !lastRecurringConfigs.isEmpty() && lastRecurringConfigs.size() > 1 ) {
			//sort from most current to least current effective date
			Collections.sort( lastRecurringConfigs, new Comparator<StipendConfig>() {
				@Override
				public int compare(StipendConfig last, StipendConfig first) { 
					return first.getEffectiveDate().compareTo(last.getEffectiveDate());
				}
			});
		}

		List<StipendConfig> inclusive = new ArrayList<StipendConfig>();
		StipendConfig lastStipendConfig = null;
		for ( StipendConfig stipendConfig: lastRecurringConfigs ) {
			if ( lastStipendConfig != null ) {
				stipendConfig.setStipendEndDate( lastStipendConfig.getEffectiveDate() );
			} else {
				stipendConfig.setStipendEndDate( this.getPaymentDate() );
			}
			if ( stipendConfig.getStipendApprovedDate() == null ) {
				if ( config.getStipendStartDate().after( stipendConfig.getStipendStartDate() ) ) {
					stipendConfig.setStipendStartDate( config.getStipendStartDate() );
				}
			}
			lastStipendConfig = stipendConfig;
			inclusive.add( lastStipendConfig );
			_log.info( lastStipendConfig.getStipendStartDateStr() + "-" + lastStipendConfig.getStipendEndDateStr() );
		}
		
		if ( this.config.getStiPtId() > 0 && inclusive.size() > 1) {
			this.config.setStipendStartDate( inclusive.get( 0 ).getStipendStartDate() );
			this.config.setPayment( inclusive.get( 0 ).getPayment() );
			this.config.setBlsRateHourly( inclusive.get( 0).getBlsRateHourly() );
			inclusive.set( 0, this.config );
		}
		lastRecurringConfigs = inclusive;
	}
	
	public List<BlsRe> getYears() throws PortalException, SystemException {
		
		Calendar effectiveDateCalendar = Calendar.getInstance();
		
		effectiveDateCalendar.setTime( getPaymentDate() );
		int maxYear = effectiveDateCalendar.get( Calendar.YEAR ) - ( getPaymentMonth() > 0 ? 1: 2);
		
		effectiveDateCalendar.setTime( config.getStipendStartDate() );
		int minYear = effectiveDateCalendar.get( Calendar.YEAR ) -1;

		VcgSupport vcgSupport = new VcgSupport( VcgLocalServiceUtil.getVcg( this.config.getVcgId() ) );

		if ( minYear == maxYear ) {
			BlsRe blsRe = BlsReLocalServiceUtil.findByLessYear(vcgSupport.getVeteran().getZip(), minYear+1);
			if ( !Toolbox.isEmpty( config.getBlsRateHourly() ) ) {
				blsRe.setH75( Double.valueOf( config.getBlsRateHourly() ) );
			}
			_log.info( "Year:" +  blsRe.getYear() );
			return Collections.singletonList( blsRe );
		}
		
		List<BlsRe> list = BlsReLocalServiceUtil.findBlsByZip( vcgSupport.getVeteran().getZip() );
		List<BlsRe> copy = new ArrayList<BlsRe>( );
		Set<Integer> years = new HashSet<Integer>();
		for ( BlsRe bls: list ){
			if ( bls.getYear() <= maxYear && bls.getYear() >= minYear && !years.contains(bls.getYear() ) ){
				years.add( bls.getYear() );
				copy.add(bls);
			}
		}
		
		if ( years.isEmpty() ) {
			_log.warn("No current BLS Rate found for zip " + vcgSupport.getVeteran().getZip() + ", using last available BLS rate ");
			copy = (List<BlsRe>) Collections.singleton( BlsReLocalServiceUtil.findByLessYear(vcgSupport.getVeteran().getZip(), maxYear) );
		}
		
		//sort from most current to least current effective date
		Collections.sort( copy, new Comparator<BlsRe>() {
			@Override
			public int compare(BlsRe first, BlsRe last) { 
				return Integer.valueOf(first.getYear()).compareTo(last.getYear());
			}
		});
		
		return copy;
	}
	
	public void loadMultiYearAspect ( Date stipendStartDate ) throws PortalException, SystemException {
			
		List<BlsRe> copy = getYears();
		if ( copy.size() > 1 ) {
			Date startDate = stipendStartDate;
			lastRecurringConfigs = new ArrayList<StipendConfig>();
			Double rate = 0.0;
			for ( BlsRe nextRate : copy) {
				if ( rate >= Double.valueOf( nextRate.getH75() )) continue; //ignore rate drop rates
				Calendar nextCal = Calendar.getInstance();
				nextCal.set( nextRate.getYear() + 1, Calendar.JANUARY, 1, 0, 0, 1);
				if ( nextCal.getTime().after( config.getStipendStartDate() ) ) {
					startDate = nextCal.getTime();
				}
				StiPt newStipendConfig = StipendConfig.getStipendConfig( startDate, 
													null,
													this.config.getTierHours(),
													nextRate.getH75().toString() );
				newStipendConfig.setTier( config.getTier() );
				lastRecurringConfigs.add( new StipendConfig( newStipendConfig ) );
			}
		} else {
			lastRecurringConfigs = Collections.singletonList( this.config );
		}
	}
	
	public boolean isMultiYearMonth() {
		return lastRecurringConfigs != null && lastRecurringConfigs.size() > 1;
	}
	
	public StipendConfig getStipendPaymentConfig (){
		if ( !isMultiYearMonth() && this.config != null ) {
			return this.config;
		}
		return lastRecurringConfigs.get(0);
	}
	
	public void setUp( Date stipendStartDate ) {
		
		if ( (lastRecurringConfigs == null || lastRecurringConfigs.isEmpty()) ) {
			try {
				loadMultiYearAspect(stipendStartDate);
			} catch (PortalException | SystemException e) {
				e.printStackTrace();
			}
		}
		
		setEffectiveDates( stipendStartDate );
	}

	@Override
	public BigDecimal getProratedAmount(int ndx) {
		return getProratedAmount();
	}

	@Override
	public BigDecimal getMonthlyRateDiff(int ndx) {
		return BigDecimal.ZERO;
	}

	@Override
	public BigDecimal getOneTimePayment(int ndx) {
		return new BigDecimal( getOneTime( ndx ) );
	}

}

