package gov.va.caret.pdf;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.sql.Blob;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.pdfbox.exceptions.COSVisitorException;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDDocumentCatalog;
import org.apache.pdfbox.pdmodel.interactive.form.PDAcroForm;
import org.apache.pdfbox.pdmodel.interactive.form.PDField;

import com.liferay.counter.service.CounterLocalServiceUtil;
import com.liferay.portal.kernel.dao.jdbc.OutputBlob;
import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.util.PortalUtil;

import gov.va.caret.ApplicationWorkFlowException;
import gov.va.caret.model.Docum;
import gov.va.caret.model.VcgAn;
import gov.va.caret.model.support.Person;
import gov.va.caret.model.support.Person.Role;
import gov.va.caret.service.DocumLocalServiceUtil;
import gov.va.caret.service.thread.Monitor;
import gov.va.caret.util.CaretUtil;
import gov.va.iam.IamService;

public class PdfHandler {

	private static final int FIELD_DISABLE = 8388609;
	private static Map<String, Map<String, String>> fieldMap = new HashMap<String, Map<String, String>>();
	private static Log _log = LogFactoryUtil.getLog(PdfHandler.class);
	
	public PdfHandler() {
		for ( Map<String, String> i: fieldMap.values() ){
			if ( i.size() > 0 ){
				return;
			}
		}
		try {
			init();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	@SuppressWarnings("rawtypes")
	private void init() throws IOException {
		
		PDDocument pdfDocument = PDDocument.load( getSeedUrl() );
		PDDocumentCatalog docCatalog = pdfDocument.getDocumentCatalog();
		PDAcroForm acroForm = docCatalog.getAcroForm();
		List fields = acroForm.getFields();
		
		Map<String, String> pdfSeed = new HashMap<String, String>();
		Properties props = new Properties();
		props.load(getClass().getClassLoader().getResourceAsStream("pdf.properties"));
		
		for (Iterator fieldsIter = fields.iterator(); fieldsIter.hasNext(); ) {
			loadSeed( (PDField) fieldsIter.next(), pdfSeed, props );
		}
		pdfDocument.close();
		
		for ( Role role : Person.Role.values() ){
			fieldMap.put(role.getLabel(), new HashMap<String, String>() );
		}
		
		
		for ( Object prop: props.keySet() ){
			String propKey = prop.toString();
			int index = propKey.indexOf('.');
			if ( index > 0 ) {
				fieldMap.get( propKey.substring(0, index) ).put( propKey.substring(index+1 ), pdfSeed.get( props.getProperty( propKey ) ) );
			}
		}
		_log.info("Completed init() of pdf form seed");
	}

	@SuppressWarnings("rawtypes")
	private void loadSeed(PDField field, Map<String, String> pdfSeed, Properties props) throws IOException {
		List kids = field.getKids();
		if (kids != null) {
			for ( Iterator iterator = kids.iterator(); iterator.hasNext(); ) {
				Object pdfObj = iterator.next();
				if (pdfObj instanceof PDField) {
					loadSeed((PDField) pdfObj, pdfSeed, props);
				}
			}
		} else {
			if ( props.values().contains( field.getPartialName() ) ){
				pdfSeed.put( field.getPartialName(), field.getFullyQualifiedName() );
			}
		}
	}
	
	private void setField( PDDocument pdfDocument, String name, Object object ) throws IOException {
        PDDocumentCatalog docCatalog = pdfDocument.getDocumentCatalog();
        PDAcroForm acroForm = docCatalog.getAcroForm();
        acroForm.setXFA(null);
        PDField field = acroForm.getField( name );
        if( field != null ){
        	if ( field.getClass() == org.apache.pdfbox.pdmodel.interactive.form.PDTextbox.class ) {
        		pdfDocument.getDocumentCatalog().getAcroForm().getField( name ).setValue( String.valueOf(object) );
        	} else if ( field.getClass() == org.apache.pdfbox.pdmodel.interactive.form.PDCheckbox.class ) {
        		if ( Boolean.TRUE == object ){
        			((org.apache.pdfbox.pdmodel.interactive.form.PDCheckbox) pdfDocument.getDocumentCatalog().getAcroForm().getField( name ) ).check();
        		}
        	}
        	field.setFieldFlags(FIELD_DISABLE);
        }
        else {
            _log.warn( "No field found with name:" + name );
        }
    }
	
	public boolean signUpdateDocum( VaForm1010cgSection form ) throws IOException, COSVisitorException {
		
		PDDocument pdfDoc = null;			
		Docum docum = null;
		Person person = form.getIntent();

		long locked = Monitor.lock(form.getApplicationId(), person.getRootUserId());
		if ( locked != person.getRootUserId() ){
			_log.error( person.getRootUserId() + " could not lock pdf " + form.getApplicationId() );
			return false;
		}

		try {
			docum = DocumLocalServiceUtil.getAppByOwnerId( form.getApplicationId() );
			if ( docum != null ) { 
				Blob blob = docum.getDocument();
				byte[] binaryData = blob.getBytes(1, (int) blob.length());
				InputStream inputStream = new ByteArrayInputStream(binaryData);
				pdfDoc = PDDocument.load(inputStream);
			}
		} catch ( ApplicationWorkFlowException | SQLException e ){
			ApplicationWorkFlowException.handleException(e);
		}
		
		if ( pdfDoc == null ){
			_log.info("New App Document for " + form.getApplicationId() );
			pdfDoc = PDDocument.load( getSeedUrl() );
		}
		form.setPdfDoc(pdfDoc);
		
		ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		setFields( form );
		if ( form.isFinal() ){
			_log.info("==========finalizing pdf");
			for ( Person blank : form.getBlanks() ){
				setFields(pdfDoc, blank, blank.getPdfAttributes( Collections.<String, Object> emptyMap() ) );
				String name = fieldMap.get(blank.getRole().getLabel()).get("signature");
				pdfDoc.getDocumentCatalog().getAcroForm().getField( name ).setFieldFlags(FIELD_DISABLE);
			}
		}
		pdfDoc.save( outStream );
		pdfDoc.close();
      	outStream.flush();
		if ( !eSigForm( person, outStream ) && CaretUtil.isMviEnabled() ){
			_log.error( person.getRootUserId() + " could not sign pdf " + form.getApplicationId() );
			Monitor.unlock( form.getApplicationId() );
			return false;
		}
		
		ByteArrayInputStream in = new ByteArrayInputStream(outStream.toByteArray());
		OutputBlob outputBlob = new OutputBlob ( in, in.available() );
		
		try {
			if ( person.isVeteranRole() ){
				docum = DocumLocalServiceUtil.createDocum( CounterLocalServiceUtil.increment( Docum.class.getName() ) );
				docum.setDocument( outputBlob );
				docum.setDeleted( false );
				docum.setClassPk( form.getApplicationId() );
				docum.setClassId( PortalUtil.getClassNameId(VcgAn.class) );
				docum.setType( FileType.TEN_TEN_CG.name() );
				docum.setName( "1010CG_" + person.getRootUserId() + ".pdf" );
				DocumLocalServiceUtil.addDocum( docum );
				_log.info("==========created pdf:" + docum.getDocumId() );
			} else {
				docum.setDocument( outputBlob );
				DocumLocalServiceUtil.updateDocum( docum );
				_log.info("==========updated pdf" + docum.getDocumId() );
			}
		} catch ( SystemException e ) {
			ApplicationWorkFlowException.handleException(e);
		}
		Monitor.unlock( form.getApplicationId() );
		return true;
	}
	
	public void setFields( VaForm1010cgSection form ) throws IOException, COSVisitorException {
		setFields( form.getPdfDoc(), form.getIntent(), form.getIntent().getPdfAttributes( form.getExtraFields() ) );
	}
	
	public void setFields( PDDocument pdfDoc, Person person, Map<String,Object> modelAtts ) throws IOException, COSVisitorException {
		
		for ( String key : modelAtts.keySet() ){
			String fieldName = fieldMap.get( person.getRole().getLabel() ).get( key );
			if ( fieldName == null ) continue;
			
			_log.debug(fieldName +  "=" + key);
			setField( pdfDoc, fieldName, modelAtts.get(key) );
		}
	}

	public boolean eSigForm( Person person, ByteArrayOutputStream outStream ) {
		byte[] b = null;
		try {
			String field = fieldMap.get(person.getRole().getLabel()).get("signature");
			int index = field.lastIndexOf('.');
			if (index > 0) {
				field = field.substring(index + 1);
				_log.info("signing field:" + field);
				if ( _log.isDebugEnabled() ){
					_log.debug("signPdf(" + person.getEsiguserId() + ", "+ person.getFullName() +", "+ person.getEsigEmail() + ", file, offset:"+ person.getEsigGmtoffset() + ")" );
				}
				b = IamService.signPdf(person.getEsiguserId(), person.getFullName(), person.getEsigEmail(),
						outStream.toByteArray(), field, person.getEsigGmtoffset());
				_log.debug("eSig call completed..." + b.length);

				outStream.reset();
				outStream.write(b);
				outStream.flush();
			}
		} catch (Exception e) {
			ApplicationWorkFlowException.handleException(e);
			return false;
		}
		return true;
	}

	public static URL getSeedUrl() {
		return PdfHandler.class.getClassLoader().getResource("seed1010.pdf");
	}
}
