unit fODAllergyCheck;
{ ------------------------------------------------------------------------------
  Update History

  2017-02-07: NSR#20071211 (Changes to Pharmacy Allergy Package)
  2017-12-20: SDS NSR#20131101 (Prevent Ordering Meds)
  ------------------------------------------------------------------------------- }
interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, ORFn, ORNet, rOrders, rODAllergy,
  Winapi.RichEdit, ShellAPI, Vcl.ExtCtrls, fBase508Form, VA508AccessibilityManager, Vcl.ComCtrls;

type
  TfrmAllergyCheck = class(TForm)
    reInfo: TRichEdit;
    lblOverride: TLabel;
    cbAllergyReason: TComboBox;
    btnContinue: TButton;
    btnCancel: TButton;
    lblComment: TLabel;
    amgrMain : TVA508AccessibilityManager;
    cbComment: TComboBox;
    pnlBottom: TPanel;
    pnlSeverity: TPanel;
    pnlComment: TPanel;
    pnlOverride: TPanel;
    imgSeverityLeft: TImage;
    imgSeverityRight: TImage;
    procedure cbAllergyReasonChange(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure AdjustButtonSize(pButton: TButton);
    procedure AdjustFormItemPositions;
    procedure ConfigureSeverityBanner;
    procedure FormCreate(Sender: TObject);
  private
    fOrderBlockedForLocalLifeThreateningAllergy: boolean;
    procedure setButtonStatus;
  public
    MedIEN : Integer;
    parentorder : TForm;
    procedure WndProc(var Msg: TMessage); override; // Used to fire URL's in TRichEdit
    procedure setup(aMedIEN:Integer;aSL: TStringList);
  end;

function MedFieldsNeeded(aMedIEN:Integer; var aReason:String; var aComment:String):Boolean;

implementation

{$R *.dfm}

function MedFieldsNeeded(aMedIEN:Integer; var aReason:String; var aComment:String):Boolean;
var
  AllergyCheck: TfrmAllergyCheck;
  SL: TSTringList;
begin
  Result := True;
  SL := TStringList.Create;
  aReason := '';
  aCOmment := '';
  try
    OrderChecksOnMedicationSelect(SL, 'PSI', aMedIEN);
    if SL.Count > 0 then
      begin
        AllergyCheck := TfrmAllergyCheck.Create(nil);
        try
          AllergyCheck.setup(aMedIEN,SL);
          Result := AllergyCheck.ShowModal = mrOk;
          aReason := AllergyCheck.cbAllergyReason.Text;
          aComment := AllergyCheck.cbComment.Text;
        finally
          FreeAndNil(AllergyCheck);
        end;
      end;
  finally
    SL.Free;
  end;
end;

////////////////////////////////////////////////////////////////////////////////

procedure TfrmAllergyCheck.cbAllergyReasonChange(Sender: TObject);
begin
  setButtonStatus;
end;


procedure TfrmAllergyCheck.setButtonStatus;
begin
  btnContinue.Enabled := (Length(Trim(cbAllergyReason.Text)) >= 4) AND
    (Pos('^', cbAllergyReason.Text) = 0) and (not fOrderBlockedForLocalLifeThreateningAllergy);
end;

procedure TfrmAllergyCheck.FormCreate(Sender: TObject);
begin
  fOrderBlockedForLocalLifeThreateningAllergy:= false;
end;

procedure TfrmAllergyCheck.FormShow(Sender: TObject);
var
 item: TVA508AccessibilityItem;
begin
  inherited;
  Font.Size := mainFontSize + 1;
  AdjustFormItemPositions;

  item := amgrMain.AccessData.FindItem(cbComment);
  amgrMain.AccessData[item.INDEX].AccessText := Trim(cbComment.Text);

  setButtonStatus;
end;

procedure TfrmAllergyCheck.setup(aMedIEN: Integer; aSL: TStringList);
var
  ss: TSTrings;
  OCList: TStringList;
  gridText, substr, s, commentstr: string;
  i,j : Integer;
  remOC: TStringList;
  mask: NativeInt; // Fixing Defect 352249
  enablecomment : Boolean;
begin
  enablecomment := False;
  commentstr := '';
  mask := SendMessage(reInfo.Handle, EM_GETEVENTMASK, 0, 0);
  SendMessage(reInfo.Handle, EM_SETEVENTMASK, 0, mask or ENM_LINK);
  SendMessage(reInfo.Handle, EM_AUTOURLDETECT, Integer(true), 0);

  remOC := TStringList.Create;

  ss := GetAllergyReasonList(MedIEN,'A');
  cbAllergyReason.Items.Assign(ss);
  ss.Free; // releasing list created by GetAllergyReasonList
  ss := GetAllergyReasonList(MedIEN,'AR');
  cbComment.Items.Assign(ss);
  ss.Free; // releasing list created by GetAllergyReasonList

  OCList := aSL;
  for i := 0 to OCList.Count - 1 do
  begin
    s := Piece(OCList[i], U, 4);
    enablecomment := StrToBool(Piece(OCList[i], U, 5));
    if enablecomment then
      commentstr := Piece(OCList[i], U, 7);
    gridText := s;
    substr := Copy(s, 0, 2);
    if substr = '||' then
      begin
        gridText := '';
        substr := Copy(s, 3, Length(s));
        GetXtraTxt(remOC, Piece(substr, '&', 1), Piece(substr, '&', 2));
        reInfo.Lines.Add('(' + inttostr(i + 1) + ' of ' + inttostr(OCList.Count) + ')  ');
        for j := 0 to remOC.Count - 1 do
          reInfo.Lines.Add('      ' + remOC[j]);
      end
    else
      reInfo.Lines.Add(gridText + CRLF);
  end;
  remOC.Free;

  // SDS - NSR 20131101 - block order if (1) locally entered and (2) life threatening
  // ORWDXC ALLERGY does not return a severity, so we have to search text - I dislike parsing text, but it works
  // I was informed by stakeholders we will NOT receive a severity at all for remotely entered allergies - I could not independently verify this on our test account
  // if that "fact" is wrong, then we might block a remotely-entered LIFE THREATENING allergy, and this fix will fail
  // I will ensure field test instructions have a test case to ensure a REMOTELY entered LIFE THREATENING allergy to test that scenario
  fOrderBlockedForLocalLifeThreateningAllergy:= pos('LIFE THREATENING', upperCase(reInfo.Lines.Text)) > 0;


  if enablecomment = True then
  begin
    lblComment.Visible := enablecomment;
    cbComment.Visible := lblComment.Visible;
    cbComment.Text := commentstr;
  end;


  if fOrderBlockedForLocalLifeThreateningAllergy then
  begin
    reInfo.Lines.Insert(0, 'THIS ORDER HAS BEEN BLOCKED BY A DOCUMENTED LIFE-THEATENING ALLERGY OR ADVERSE REACTION!');
    reInfo.Lines.Insert(1, 'To place this order an authorized user must edit this locally entered allergy to reduce severity' + sLineBreak
    );
    pnlComment.visible:= false;
    pnlOverride.Visible:= false;
  end;

  ConfigureSeverityBanner; // SDS NSR#20131101 (Prevent Ordering Meds)
end;

procedure TfrmAllergyCheck.WndProc(var Msg: TMessage);
var
  p: TENLink;
  sURL: string;
begin
  if (Msg.Msg = WM_NOTIFY) then
    begin
      if (PNMHDR(Msg.lParam).code = EN_LINK) then
        begin
          p := TENLink(Pointer(TWMNotify(Msg).NMHdr)^);
          if (p.Msg = WM_LBUTTONDOWN) then
            begin
              try
                SendMessage(reInfo.Handle, EM_EXSETSEL, 0, Longint(@(p.chrg)));
                sURL := reInfo.SelText;
                ShellExecute(Handle, 'open', PChar(sURL), NIL, NIL, SW_SHOWNORMAL);
              except
                ShowMessage('Error opening HyperLink');
              end;
            end;
        end;
    end;
  inherited;
end;

procedure TfrmAllergyCheck.AdjustButtonSize(pButton: TButton);
// SDS NSR#20131101 (Prevent Ordering Meds)
// set height of bottom panel, set width of buttons
const
  Gap = 5;
begin
  pnlBottom.Height:= Canvas.TextHeight(pButton.Caption) + Gap +
    pnlBottom.Margins.Top * 2 + pButton.Margins.Top * 2;
  if pButton.Height < Canvas.TextHeight(pButton.Caption) then // CQ2737  GE
    pButton.Height := Canvas.TextHeight(pButton.Caption) + Gap; // CQ2737  GE
end;

procedure TfrmAllergyCheck.AdjustFormItemPositions;
// SDS NSR#20131101 (Prevent Ordering Meds)
// let alignment do most of the work
begin
  reInfo.Font.Size := mainFontSize;
  AdjustButtonSize(btnContinue);
  AdjustButtonSize(btnCancel);
  pnlComment.Height:= Canvas.TextHeight(lblComment.Caption) + pnlComment.Margins.Top * 2;
  pnlOverride.Height:= pnlComment.Height;
  self.Height:= (pnlSeverity.Height * 4) + pnlBottom.Height + pnlOverride.Height + pnlComment.Height;
end;

procedure TfrmAllergyCheck.ConfigureSeverityBanner;
// SDS, NSR 20131101
// display a very prominent severity banner with icons and severity
// load an appropriate Windows standard icon based on severity
// // 101: IDI_WARNING, 102: IDI_QUESTION, 103: IDI_ERROR, 104: IDI_INFORMATION
var
  s: string;

  procedure ConfigureSeverityPanel(AColor, ATextColor: TColor; AIcon: integer; AText: string);
  begin
    imgSeverityLeft.Picture.Icon.Handle:= LoadImage(GetModuleHandle('user32'), MAKEINTRESOURCE(AIcon), IMAGE_ICON, 64, 64, LR_DEFAULTCOLOR);
    imgSeverityRight.Picture.Icon.Handle:= imgSeverityLeft.Picture.Icon.Handle;
    pnlSeverity.Caption:= AText;
    pnlSeverity.Color:= AColor;
    pnlSeverity.Font.Color:= ATextColor;
  end;

begin
  s:= uppercase(reInfo.Text);
  // we try to find text indicating severity, starting with most severe first
  if pos(sAllergySeverityStrings[asLifeThreatening], s) > 0 then
    ConfigureSeverityPanel(clRed, clWhite, 103, sAllergySeverityStrings[asLifeThreatening])
  else if pos(sAllergySeverityStrings[asSevere], s) > 0 then
    ConfigureSeverityPanel(clYellow, clBlack, 101, sAllergySeverityStrings[asSevere])
  else if pos(sAllergySeverityStrings[asModerate], s) > 0 then
    ConfigureSeverityPanel(clBtnFace, clBlack, 104, sAllergySeverityStrings[asModerate])
  else if pos(sAllergySeverityStrings[asMild], s) > 0 then
    ConfigureSeverityPanel(clBtnFace, clBlack, 104, sAllergySeverityStrings[asMild])
  else
    ConfigureSeverityPanel(clBtnFace, clBlack, 104, sAllergySeverityStrings[asUnknown]);
end;


end.
