unit fSurrogateEdit;

{
  This unit implements class TfrmSurrogateEdit that provides data for
  Surrogate Management Functionality within CPRS Graphical User Interface (GUI)
  (Request #20071216)
}

interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants,
  System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, fBase508Form, VA508AccessibilityManager,
  Vcl.StdCtrls, ORDtTm, ORCtrls, Vcl.ExtCtrls, Vcl.ComCtrls, ORDtTmRng,
  System.Actions, Vcl.ActnList;

type
  TfrmSurrogateEdit = class(TfrmBase508Form)
    pnlBottom: TPanel;
    bvlBottom: TBevel;
    Panel1: TPanel;
    btnOK: TButton;
    btnCancel: TButton;
    pnlSurrogateTools: TPanel;
    cboSurrogate: TORComboBox;
    ordtbStart: TORDateBox;
    ordtbStop: TORDateBox;
    VA508StaticText1: TVA508StaticText;
    VA508StaticText2: TVA508StaticText;
    VA508StaticTextName: TVA508StaticText;
    btnRestore: TButton;
    stxtAllowedRange: TVA508StaticText;
    ALMain: TActionList;
    ActionReset: TAction;
    procedure ordtbStartDateDialogClosed(Sender: TObject);
    procedure ordtbStopDateDialogClosed(Sender: TObject);
    procedure cboSurrogateNeedData(Sender: TObject; const StartFrom: string;
      Direction, InsertAt: Integer);
    procedure FormShow(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure cboSurrogateChange(Sender: TObject);
    procedure pnlSurrogateToolsDblClick(Sender: TObject);
    procedure ALMainUpdate(Action: TBasicAction; var Handled: Boolean);
    procedure ActionResetExecute(Sender: TObject);
  private
    fData: String;
    fStart: String;
    fStop: String;
    fMin: String;
    fMax: String;
    fSurrogate: String;
    procedure setValues(aName, aStart, aStop, aMin, aMax, aData: String;
      bLast: Boolean = false);
    procedure setButtonStatus;
  public
  end;

var
  frmSurrogateEdit: TfrmSurrogateEdit;

const
  defaultSurrPeriod = 7; // days to plan ahead   //  V32 Defect 101 SDS 4/3/2017 - change from 1 month to 1 week

function editSurrogate(var aSurrogate, aStart, aStop, aMin, aMax, aData: String;
  bLast: Boolean = false): Integer;

implementation

uses
  Math, ORFn, rCore, System.UITypes, fOptionsSurrogate;

{$R *.dfm}

function editSurrogate(var aSurrogate, aStart, aStop, aMin, aMax, aData: String;
  bLast: Boolean = false): Integer;
begin
  if not assigned(frmSurrogateEdit) then
    Application.CreateForm(TfrmSurrogateEdit, frmSurrogateEdit);
  try
    ResizeAnchoredFormToFont(frmSurrogateEdit);
    with frmSurrogateEdit do
    begin
      setValues(aSurrogate, aStart, aStop, aMin, aMax, aData, bLast);
      Result := ShowModal;
      if Result = mrOK then
      begin
        aData := cboSurrogate.ItemID;
        aSurrogate := cboSurrogate.Text;
        aStart := FormatDateTime(fmtListDateTimeSec, ordtbStart.DateSelected);
        aStop := FormatDateTime(fmtListDateTimeSec, ordtbStop.DateSelected);
      end;
    end;
  finally
    frmSurrogateEdit.Release;
    frmSurrogateEdit := nil;
  end;
end;

procedure TfrmSurrogateEdit.setValues(aName, aStart, aStop, aMin, aMax,
  aData: String; bLast: Boolean = false);
var
  dtStart, dtStop, dtMin, dtMax: TDateTime;
  sTarget: String;

begin
  fSurrogate := aName;
  fData := aData;

  fStart := aStart;
  fStop := aStop;
  fMin := aMin;
  fMax := aMax;

  dtStart := StrDateToDate(aStart);
  dtStop := StrDateToDate(aStop);
  dtMin := StrDateToDate(aMin);
  dtMax := StrDateToDate(aMax);

  ordtbStart.DateRange.MinDate := dtMin;
  if dtStart < dtMin then  // 20170106
    ordtbStart.DateSelected := dtMin      // v32 Issue Tracker 101, SDS 4/4/2017
  else
    ordtbStart.DateSelected := dtStart;
  ordtbStart.Text := FormatDateTime(fmtListDateTime, ordtbStart.DateSelected);

  ordtbStart.DateRange.MaxDate := dtStop;

  ordtbStop.DateRange.MinDate := dtStop;
  ordtbStop.DateRange.MaxDate := dtMax;

  if bLast then
  begin
    ordtbStop.DateSelected := dtStart + defaultSurrPeriod;
    if ordtbStop.DateSelected > dtStop then
      ordtbStop.DateSelected := dtStop;
  end
  else
    ordtbStop.DateSelected := dtStop;

  ordtbStop.Text := FormatDateTime(fmtListDateTime, ordtbStop.DateSelected);

  sTarget := piece(aName, ',', 1);
  if length(sTarget) > 1 then
    sTarget := copy(sTarget, 1, length(sTarget) - 1);
  if aData = '' then // v32 Defect 101, SDS 4/10/2017 - avoid dropdown list trying to select "dbl..."
    cboSurrogate.InitLongList('')
  else
    cboSurrogate.InitLongList(sTarget);
  cboSurrogate.SelectByID(aData);

  // V32 Issue 101, SDS 4/4/2017
  stxtAllowedRange.Caption := format(
    'Start and stop times must be within this range...: %s (earliest) %s (latest)',
    [CRLF + aMin, CRLF + aMax]);

  setButtonStatus;

  ordtbStopDateDialogClosed(nil);
  ordtbStartDateDialogClosed(nil);
end;

procedure TfrmSurrogateEdit.ActionResetExecute(Sender: TObject);
// V32 defect 101 SDS
begin
  inherited;
  setValues(fSurrogate, fStart, fStop, fMin, fMax, fData);
end;

procedure TfrmSurrogateEdit.ALMainUpdate(Action: TBasicAction; var Handled: Boolean);
// V32 defect 101 SDS
// the action list onUpdate method is called when the Delphi event loop is idle
begin
  inherited;
  actionReset.Enabled := (piece(cboSurrogate.Text, ' ', 1) <> fSurrogate) or
    (ordtbStop.DateSelected <> StrDateToDate(fStop)) or
    (ordtbStart.DateSelected <> StrDateToDate(fStart));
end;

procedure TfrmSurrogateEdit.cboSurrogateChange(Sender: TObject);
begin
  inherited;
  setButtonStatus;
end;

procedure TfrmSurrogateEdit.cboSurrogateNeedData(Sender: TObject;
  const StartFrom: string; Direction, InsertAt: Integer);
begin
  inherited;
  cboSurrogate.ForDataUse(SubSetOfPersons(StartFrom, Direction));
end;

procedure TfrmSurrogateEdit.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
var
  msg: String;

  function DateTimeErrorReport(aSelector: TORDateBox): String;
  var
    FMDT: TFMDateTime;
  begin
    if not assigned(aSelector) then
      Result := 'Selector was not defined'
    else
    begin
      FMDT := ServerParseFMDate(aSelector.Text);
      if FMDT < 0 then
        Result := 'Invalid Date/time: ' + aSelector.Text + CRLF
      else
      begin
        Result := '';
        aSelector.DateSelected := FmDateTimeToDateTime(FMDT);
        aSelector.Validate(Result);
        if Result <> '' then
          Result := Result + CRLF;
      end;
    end;
  end;

begin
  inherited;
  if modalResult = mrCancel then // V32 Defect 101 SDS use modal result instead of FCancel, so X at top right works
    exit;

  msg := msg + DateTimeErrorReport(ordtbStart);
  msg := msg + DateTimeErrorReport(ordtbStop);

  if (cboSurrogate.Text = '') or (cboSurrogate.ItemID = 0) then
    msg := 'Please Select the Surrogate Name' + CRLF;
  if ordtbStop.DateSelected < ordtbStart.DateSelected then
    msg := msg + 'Start Date can''t be greater than Stop date' + CRLF;

  CanClose := msg = '';
  if not CanClose then
    MessageDLG(msg, mtError, [mbOK], 0);
end;

procedure TfrmSurrogateEdit.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  inherited;
  if Key = VK_ESCAPE then
    ModalResult := mrCancel;
end;

procedure TfrmSurrogateEdit.FormShow(Sender: TObject);
var
  i: LongInt;
begin
  inherited;
  // cboSurrogate.InitLongList(fSurrogate); V32 Defect 101  SDS - redundant, done in setupData
  i := StrToIntDef(fData, -1);
  if i > -1 then
    cboSurrogate.SelectByIEN(i);
  cboSurrogate.SetFocus;
{$IFDEF DEBUG_AA}
  borderStyle := bsSizeable;
  Caption :=
    FormatDateTime('yyyy-mm-dd',ordtbStart.DateRange.MinDate) + '..' +
    FormatDateTime('yyyy-mm-dd',ordtbStart.DateRange.MaxDate) + '   ' +
    FormatDateTime('yyyy-mm-dd',ordtbStop.DateRange.MinDate) + '..' +
    FormatDateTime('yyyy-mm-dd',ordtbStop.DateRange.MaxDate);
{$ENDIF}
end;

procedure TfrmSurrogateEdit.ordtbStartDateDialogClosed(Sender: TObject);
var
  dt: TDateTime;
begin
  inherited;
  dt := ordtbStart.DateSelected;
  if dt = 0 then
    exit;
  ordtbStop.DateRange.MinDate := dt;
end;

procedure TfrmSurrogateEdit.ordtbStopDateDialogClosed(Sender: TObject);
var
  dt: TDateTime;
begin
  inherited;
  dt := ordtbStop.DateSelected;
  if dt = 0 then
    exit;
  ordtbStart.DateRange.MaxDate := dt;
end;

procedure TfrmSurrogateEdit.pnlSurrogateToolsDblClick(Sender: TObject);
begin
  inherited;
{$IFDEF DEBUG}
  Caption :=
    FormatDateTime('yyyy-mm-dd',ordtbStart.DateRange.MinDate) + '..' +
    FormatDateTime('yyyy-mm-dd',ordtbStart.DateRange.MaxDate) + '   ' +
    FormatDateTime('yyyy-mm-dd',ordtbStop.DateRange.MinDate) + '..' +
    FormatDateTime('yyyy-mm-dd',ordtbStop.DateRange.MaxDate);
{$ENDIF}
end;

procedure TfrmSurrogateEdit.setButtonStatus;
begin
  btnOK.Enabled := cboSurrogate.Text <> '';
end;

end.
