﻿//*******************************************************************************************************************************
// Transaction Logging: Reusable class to be able to add logging to any solution easily.
//                      This class is simple in design on purpose. It is very easy to implement so keep it that way
// Created by: Allen Barge
// Date: 03/15/2011
// Version: 1.0.0
// Organization: Microsoft Corporation
//
// Source code is provided as unsupported and resricted to use by employees of Microsoft Corporation
//*******************************************************************************************************************************

using System;
using System.IO;
using System.Runtime.InteropServices;

namespace Microsoft.MCS.Logging
{
    public class Logger : IDisposable
    {
        private string LoggingFolderPath { get; set; }
        private DirectoryInfo LoggingFolderInfo { get; set; }
        private string LoggingFilename { get; set; }
        private string LoggingFullpath { get; set; }
        private Boolean LogToConsole { get; set; }
        private Boolean LogToFile { get; set; }
        private StreamWriter _streamToFile;
        private string LoggingApplication { get; set; }
        
        /// <summary>
        /// Default Instantiation without parameters of the Class. This establishes the class and any defaults that are required.
        /// </summary>
        public Logger()
        {
            // Use defaults for all parameters
            LoggingFolderPath = string.Concat(Directory.GetCurrentDirectory(), "\\logs");
            LoggingApplication = "TractionLogging";

            LoggingFilename = FormatedDateString(this.LoggingApplication);
            InitializeLogging();
        }

        /// <summary>
        /// Instantiation of the class by only supplying the target log file folder path
        /// </summary>
        /// <param name="loggingFolderPath"></param>
        public Logger(string loggingFolderPath)
        {
            // Use defaults for only the application parameter
            LoggingApplication = "TractionLogging";

            this.LoggingFolderPath = loggingFolderPath;
            LoggingFilename = FormatedDateString(this.LoggingApplication);
            InitializeLogging();
        }

        /// <summary>
        /// Instantiation of the class by suppying a target log file folder path and an application name to be appended to the file name for easier identification of your log files.
        /// </summary>
        /// <param name="loggingFolderPath"></param>
        /// <param name="application"></param>
        public Logger(string loggingFolderPath, string application)
        {
            LoggingApplication = application;
            this.LoggingFolderPath = loggingFolderPath;
            LoggingFilename = FormatedDateString(this.LoggingApplication);
            InitializeLogging();
        }

        /// <summary>
        /// Initializes the log file and establishes any naming
        /// </summary>
        private void InitializeLogging()
        {
            LogToFile = true;
            LogToConsole = true;
            // Check to see if the logging file exists
            if (!(Directory.Exists(LoggingFolderPath)))
            {
                LoggingFolderInfo = Directory.CreateDirectory(LoggingFolderPath);
            }
            else
            {
                LoggingFolderInfo = new DirectoryInfo(LoggingFolderPath);
            }

            // Stamp the logging file name so that each instance of the logging file is a separate file
            LoggingFullpath = string.Concat(LoggingFolderPath, "\\", LoggingFilename);

            _streamToFile = new StreamWriter(LoggingFullpath, true);
        }

        public string FormatedDateString(string application)
        {
            DateTime Value = DateTime.Now;
            return string.Format("{0}-{1:MMddyyyyhhmmsstt}.log", application, DateTime.Now);
        }

        public void WriteLine(string text)
        {
            if (LogToConsole == true)
            {
                Console.WriteLine(text);
            }

            if (LogToFile == true)
            {
                try
                {
                    string logLine = string.Format("{0:G}: {1}.", System.DateTime.Now, text);
                    _streamToFile.WriteLine(logLine);
                }
                catch (IOException iox)
                {
                    Console.WriteLine("File Error : Message Text - " + iox.Message.ToString());
                }

            }
        }

        public void BlankLine()
        {
            if (LogToConsole == true)
            {
                Console.WriteLine("");
            }

            if (LogToFile == true)
            {
                try
                {
                    _streamToFile.WriteLine("");
                }
                catch (IOException iox)
                {
                    Console.WriteLine("File Error : Message Text - " + iox.Message.ToString());
                }

            }
        }

        public void Close()
        {
            _streamToFile.Close();
        }

        protected virtual void Dispose(bool disposing)
        {
            if (disposing)
            {
                // dispose managed resources
                _streamToFile.Close();
            }
            // free native resources
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

    }
}
