﻿// **********************************************************************************************************
// OrganizationHelpers.cs
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 11/12/2011
// This class is designed to facilitate accessing components within the Organization
// This class uses the Discovery service to retrieve Organization Messages
//
// MetaDataPair is a custom class made up of 2 properties:  (Guid)MetaDataId
//                                                          (string)SchemaName
//
// Methods:
// GetOrganizationNamesList - returns a List<MetaDataPair> of the organization names of the discovery service in the
//                      Server Configuration object either passed in on call or established when instantiating
//                      the class
//
// GetOrganizationsDetailList - returns a List<EntityMetadata> object with the OrganizationDetails Objects for the Server
//                      Connection Configuration Object either passed in or already set
//
// **********************************************************************************************************
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data;
using System.ServiceModel;
using System.ServiceModel.Description;
using System.Diagnostics;

using Microsoft.Crm.Sdk;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using Microsoft.Xrm.Sdk.Discovery;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Metadata;

using Microsoft.MCS.CustomHelpers;
using Microsoft.MCS.Logging;

namespace Microsoft.MCS.CRMDiffingTool
{
    /// <summary>
    /// 
    /// </summary>
    class OrganizationHelpers
    {
        DiscoveryServiceProxy _orgDiscoveryServiceProxy;
        public ServerConnection.Configuration _serverConfig;
        
        /// <summary>
        /// 
        /// </summary>
        public OrganizationHelpers()
        {
        }

        /// <summary>
        /// Instantiate the class
        /// </summary>
        /// <param name="ServerConfiguration"></param>
        public OrganizationHelpers(ServerConnection.Configuration ServerConfiguration)
        {
            _serverConfig = ServerConfiguration;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="SourceMetadata"></param>
        /// <param name="TargetMetadata"></param>
        /// <returns></returns>
        public Boolean CompareOrganizationInformation(OrganizationDetail SourceOrganizationDetail, OrganizationDetail TargetOrganizationDetail, TransactionLogging LoggingConfig)
        {
            Boolean IsValid = true;

            if (SourceOrganizationDetail.OrganizationVersion != null && SourceOrganizationDetail.OrganizationVersion != TargetOrganizationDetail.OrganizationVersion)
            {
                LoggingConfig.WriteLine("*** Organization Metadata OrganizationVersion Failed Validation...");
                IsValid = false;
            }

            return IsValid;
        }

        /// <summary>
        /// Overload for GetOrganizationNamesList in order to allow the call to be made
        /// without passing in the configuration. It will use the existing configuration
        /// for the current instance of this class
        /// 
        /// Retrieves a list of the Organization names for the given server configuration object.
        /// </summary>
        /// <returns></returns>
        public List<OrganizationDetail> GetOrganizationDetailsList(TransactionLogging LoggingConfig)
        {
            ServerConnection.Configuration ServerConfiguration = _serverConfig;
            return GetOrganizationDetailsList(_serverConfig, LoggingConfig);
        }

        /// <summary>
        /// Retrieves a list of the organization names for the given server configuration object.
        /// </summary>
        /// <param name="ServerConfig"></param>
        /// <returns></returns>
        public List<OrganizationDetail> GetOrganizationDetailsList(ServerConnection.Configuration ServerConfiguration, TransactionLogging LoggingConfig)
        {
            try
            {
                // Connect to the Discovery service. 
                // The using statement assures that the service proxy will be properly disposed.
                using (_orgDiscoveryServiceProxy = new DiscoveryServiceProxy(ServerConfiguration.DiscoveryUri,
                                                                            ServerConfiguration.HomeRealmUri,
                                                                            ServerConfiguration.Credentials,
                                                                            ServerConfiguration.DeviceCredentials))
                {
                    List<OrganizationDetail> OrganizationDetailsList = new List<OrganizationDetail>();

                    IDiscoveryService service = _orgDiscoveryServiceProxy as IDiscoveryService;

                    RetrieveOrganizationsRequest request = new RetrieveOrganizationsRequest()
                    {
                        AccessType = EndpointAccessType.Default,
                        Release = OrganizationRelease.Current
                    };

                    // Retrieve the Organization Data.
                    RetrieveOrganizationsResponse organizationsResponse = (RetrieveOrganizationsResponse)_orgDiscoveryServiceProxy.Execute(request);
                    //return OrganizationNamesList;

                    // Loop through the results and create the return array list for the OrganizationId and Uniquename.
                    foreach (OrganizationDetail em in organizationsResponse.Details)
                    {
                        OrganizationDetail orgRec = em;

                        OrganizationDetailsList.Add(orgRec);
                    }

                    return OrganizationDetailsList;
                }
            }
            catch (TimeoutException Timeex)
            {
                LoggingConfig.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.DiscoveryServiceFault> DisEx)
            {
                LoggingConfig.WriteLine("Discovery Service Fault Exception - " + DisEx.Message);
            }
            catch (Exception ex)
            {
                LoggingConfig.WriteLine("Organization Helpers Metadata Exception - " + ex.Message);
            }

            return new List<OrganizationDetail>();
        }

        /// <summary>
        /// Overload for GetOrganizationNamesList in order to allow the call to be made
        /// without passing in the configuration. It will use the existing configuration
        /// for the current instance of this class
        /// 
        /// Retrieves a list of the Organization names for the given server configuration object.
        /// </summary>
        /// <returns></returns>
        public List<MetaDataPair> GetOrganizationNamesList(TransactionLogging LoggingConfig)
        {
            ServerConnection.Configuration ServerConfiguration = _serverConfig;
            return GetOrganizationNamesList(_serverConfig, LoggingConfig);
        }

        /// <summary>
        /// Retrieves a list of the organization names for the given server configuration object.
        /// </summary>
        /// <param name="ServerConfig"></param>
        /// <returns></returns>
        public List<MetaDataPair> GetOrganizationNamesList(ServerConnection.Configuration ServerConfiguration, TransactionLogging LoggingConfig)
        {
            try
            {
                // Connect to the Discovery service. 
                // The using statement assures that the service proxy will be properly disposed.
                using (_orgDiscoveryServiceProxy = new DiscoveryServiceProxy(ServerConfiguration.DiscoveryUri,
                                                                            ServerConfiguration.HomeRealmUri,
                                                                            ServerConfiguration.Credentials,
                                                                            ServerConfiguration.DeviceCredentials))
                {
                    List<MetaDataPair> OrganizationList = new List<MetaDataPair>();

                    IDiscoveryService service = _orgDiscoveryServiceProxy as IDiscoveryService;

                    RetrieveOrganizationsRequest request = new RetrieveOrganizationsRequest()
                        {
                            AccessType = EndpointAccessType.Default,
                            Release = OrganizationRelease.Current
                        };

                    // Retrieve the Organization Data.
                    RetrieveOrganizationsResponse organizationsResponse = (RetrieveOrganizationsResponse)_orgDiscoveryServiceProxy.Execute(request);
                    //return OrganizationNamesList;

                    // Loop through the results and create the return array list for the OrganizationId and Uniquename.
                    foreach (OrganizationDetail em in organizationsResponse.Details)
                    {
                        MetaDataPair orgRec = new MetaDataPair();
                        orgRec.MetaDataId = new Guid(em.OrganizationId.ToString());
                        orgRec.SchemaName = em.UniqueName;
                        OrganizationList.Add(orgRec);
                    }

                    return OrganizationList;
                }
            }
            catch (TimeoutException Timeex)
            {
                LoggingConfig.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.DiscoveryServiceFault> DisEx)
            {
                LoggingConfig.WriteLine("Discovery Service Fault Exception - " + DisEx.Message);
            }
            catch (Exception ex)
            {
                LoggingConfig.WriteLine("Organization Helpers Metadata Exception - " + ex.Message);
            }

            return new List<MetaDataPair>();
        }


        public void NotImplementedWindowsLiveIDConnection()
        {
            //var uri = System.Configuration.ConfigurationManager.AppSettings["CRMURI"];

            //ClientCredentials credentials = new ClientCredentials();
            //var dinfo = ServiceConfigurationFactory.CreateConfiguration<IDiscoveryService>(new Uri(uri));
            //var dsp = new DiscoveryServiceProxy(dinfo, credentials);

            //dsp.Authenticate();

            //var retrieveOrganizationsRequest = new RetrieveOrganizationsRequest();
            //var retrieveOrganizationsResponse = dsp.Execute(retrieveOrganizationsRequest) as RetrieveOrganizationsResponse;

            //if (retrieveOrganizationsResponse.Details.Count == 1)
            //{
            //    var organizationDetail = retrieveOrganizationsResponse.Details[0];

            //    Uri orgServiceUri = new Uri(organizationDetail.Endpoints[EndpointType.OrganizationService]);

            //    IServiceConfiguration<IOrganizationService> orgConfigInfo = ServiceConfigurationFactory.CreateConfiguration<IOrganizationService>(orgServiceUri);

            //    var creds = new ClientCredentials();
            //    IOrganizationService organizationService = new OrganizationServiceProxy(orgConfigInfo, creds);

            //    Guid userid = ((WhoAmIResponse)organizationService.Execute(new WhoAmIRequest())).UserId;
            //    var systemUser = organizationService.Retrieve("systemuser", userid, new ColumnSet(new string[] { "firstname", "lastname" }));

            //    // Retrieve the version of Microsoft Dynamics CRM.
            //    var versionRequest = new RetrieveVersionRequest();
            //    var versionResponse = (RetrieveVersionResponse)organizationService.Execute(versionRequest);

            //    ViewBag.FirstName = systemUser.GetAttributeValue<string>("firstname");
            //    ViewBag.LastName = systemUser.GetAttributeValue<string>("lastname");
            //    ViewBag.Version = versionResponse.Version;
            //}

        }

    }
}
