﻿// **********************************************************************************************************
// EntityHelpers.cs
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 10/26/2011
// This class is designed to facilitate accessing components within the Entity
//
// MetaDataPair is a custom class made up of 2 properties:  (Guid)MetaDataId
//                                                          (string)SchemaName
//
// Methods:
// GetEntityNamesList - returns a List<MetaDataPair> of the entity names of the Organization identified in the
//                      Server Configuration object either passed in on call or established when instantiating
//                      the class
//
// GetEntityEntityList - returns a List<EntityMetadata> object with the Entity Objects for the Server
//                      Connection Configuration Object either passed in or already set
//
// GetAttributeNamesList -  - returns a List<MetaDataPair> of the entity names of the Organization identified in the
//                          Server Configuration object either passed in on call or established when
//                          instantiating the class
//
// GetAttributeDetailsList - returns a List<AttributeMetadata> object with the Attribute Objects for the
//                          Server Connection Configuration Object either passed in or already set
//
// **********************************************************************************************************
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data;
using System.ServiceModel;
using System.ServiceModel.Description;
using System.Diagnostics;

using Microsoft.Crm.Sdk;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using Microsoft.Xrm.Sdk.Discovery;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Metadata;

using Microsoft.MCS.CustomHelpers;
using Microsoft.MCS.Logging;

namespace Microsoft.MCS.CRMDiffingTool
{
    /// <summary>
    /// 
    /// </summary>
    class EntityHelpers
    {
        OrganizationServiceProxy _orgServiceProxy;
        public ServerConnection.Configuration _serverConfig;

        /// <summary>
        /// 
        /// </summary>
        public EntityHelpers()
        {
        }

        /// <summary>
        /// Instantiate the class
        /// </summary>
        /// <param name="ServerConfiguration"></param>
        public EntityHelpers(ServerConnection.Configuration ServerConfiguration)
        {
            _serverConfig = ServerConfiguration;
        }

        /// <summary>
        /// Instantiate the class
        /// </summary>
        /// <param name="ServerConfiguration"></param>
        public EntityHelpers(ServerConnection.Configuration ServerConfiguration, TransactionLogging LoggingConfig)
        {
            _serverConfig = ServerConfiguration;
        }

        /// <summary>
        /// Overload for GetEntityListByMetaDataID in order to allow the call to be made without passing in the configuration. It will use the existing configuration
        /// for the current instance of this class
        /// 
        /// Retrieves a list of the entity names for the given server configuration object. It will then be sorted by Entity MetadataID and returned
        /// </summary>
        /// <returns></returns>
        public List<MetaDataPair> GetEntityNamesList(TransactionLogging LoggingConfig)
        {
            ServerConnection.Configuration ServerConfiguration = _serverConfig;
            return GetEntityNamesList(_serverConfig, LoggingConfig);
        }

        /// <summary>
        /// Retrieves a list of the entity names for the given server configuration object. It will then be sorted by Entity MetaData ID and returned
        /// </summary>
        /// <param name="ServerConfig"></param>
        /// <returns></returns>
        public List<MetaDataPair> GetEntityNamesList(ServerConnection.Configuration ServerConfiguration, TransactionLogging LoggingConfig)
        {
            try
            {
                // Connect to the Organization service. 
                // The using statement assures that the service proxy will be properly disposed.
                using (_orgServiceProxy = new OrganizationServiceProxy(ServerConfiguration.OrganizationUri,
                                                                        ServerConfiguration.HomeRealmUri,
                                                                        ServerConfiguration.Credentials,
                                                                        ServerConfiguration.DeviceCredentials))
                {
                    List<MetaDataPair> EntityList = new List<MetaDataPair>();

                    RetrieveAllEntitiesRequest request = new RetrieveAllEntitiesRequest()
                    {
                        EntityFilters = EntityFilters.Entity,
                        RetrieveAsIfPublished = true
                    };

                    // Retrieve the Entity Data.
                    RetrieveAllEntitiesResponse allEntitiesResponse = (RetrieveAllEntitiesResponse)_orgServiceProxy.Execute(request);
                    //return EntityNamesList;

                    // Loop through the results and create the return array list for the EntityId and SchemaName.
                    foreach (EntityMetadata em in allEntitiesResponse.EntityMetadata)
                    {
                        MetaDataPair entRec = new MetaDataPair();
                        entRec.MetaDataId = new Guid(em.MetadataId.ToString());
                        entRec.SchemaName = em.SchemaName;
                        EntityList.Add(entRec);
                    }

                    return EntityList;
                }
            }
            catch (TimeoutException Timeex)
            {
                LoggingConfig.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                LoggingConfig.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                LoggingConfig.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new List<MetaDataPair>();
        }

        /// <summary>
        /// Overload for GetEntityListByMetaDataID in order to allow the call to be made without passing in the configuration. It will use the existing configuration
        /// for the current instance of this class
        /// 
        /// Retrieves a list of the entity names for the given server configuration object. It will then be sorted by Entity MetadataID and returned
        /// </summary>
        /// <returns></returns>
        public List<EntityMetadata> GetEntityEntityList(TransactionLogging LoggingConfig)
        {
            ServerConnection.Configuration ServerConfiguration = _serverConfig;
            return GetEntityEntityList(_serverConfig, LoggingConfig);
        }

        /// <summary>
        /// Retrieves a list of the entity names for the given server configuration object. It will then be sorted by Entity MetaData ID and returned
        /// </summary>
        /// <param name="ServerConfig"></param>
        /// <returns></returns>
        public List<EntityMetadata> GetEntityEntityList(ServerConnection.Configuration ServerConfiguration, TransactionLogging LoggingConfig)
        {
            try
            {
                // Connect to the Organization service. 
                // The using statement assures that the service proxy will be properly disposed.
                using (_orgServiceProxy = new OrganizationServiceProxy(ServerConfiguration.OrganizationUri,
                                                                        ServerConfiguration.HomeRealmUri,
                                                                        ServerConfiguration.Credentials,
                                                                        ServerConfiguration.DeviceCredentials))
                {
                    List<EntityMetadata> EntityList = new List<EntityMetadata>();
                    
                    RetrieveAllEntitiesRequest request = new RetrieveAllEntitiesRequest()
                    {
                        EntityFilters = EntityFilters.Entity,
                        RetrieveAsIfPublished = true
                    };

                    // Retrieve the MetaData.
                    RetrieveAllEntitiesResponse allEntitiesResponse = (RetrieveAllEntitiesResponse)_orgServiceProxy.Execute(request);
                    //return EntityInfoList;
                    foreach (EntityMetadata em in allEntitiesResponse.EntityMetadata)
                    {
                        EntityMetadata entityRec = em;
                        EntityList.Add(entityRec);
                    }

                    return EntityList;
                }
            }
            catch (TimeoutException Timeex)
            {
                LoggingConfig.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                LoggingConfig.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                LoggingConfig.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new List<EntityMetadata>();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public List<MetaDataPair> GetAttributeNamesList(Guid EntityId, TransactionLogging LoggingConfig)
        {
            ServerConnection.Configuration ServerConfiguration = _serverConfig;
            return GetAttributeNamesList(ServerConfiguration, EntityId, LoggingConfig);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public List<AttributeMetadata> GetAttributeDetailsList(Guid EntityId, TransactionLogging LoggingConfig)
        {
            ServerConnection.Configuration ServerConfiguration = _serverConfig;
            return GetAttributeDetailsList(ServerConfiguration, EntityId, LoggingConfig);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="ServerConfiguration"></param>
        /// <param name="EntityId"></param>
        /// <returns></returns>
        public List<AttributeMetadata> GetAttributeDetailsList(ServerConnection.Configuration ServerConfiguration, Guid EntityId, TransactionLogging LoggingConfig)
        {
            try
            {
                // Connect to the Organization service. 
                // The using statement assures that the service proxy will be properly disposed.
                using (_orgServiceProxy = new OrganizationServiceProxy(ServerConfiguration.OrganizationUri,
                                                                        ServerConfiguration.HomeRealmUri,
                                                                        ServerConfiguration.Credentials,
                                                                        ServerConfiguration.DeviceCredentials))
                {
                    List<AttributeMetadata> AttributeList = new List<AttributeMetadata>();

                    RetrieveEntityRequest request = new RetrieveEntityRequest()
                    {
                        EntityFilters = EntityFilters.Attributes,
                        MetadataId = EntityId,
                        RetrieveAsIfPublished = true
                    };

                    // Retrieve the MetaData.
                    RetrieveEntityResponse entityResponse = (RetrieveEntityResponse)_orgServiceProxy.Execute(request);
                    //return EntityInfoList;

                    // Loop through the results and create the return array list for the EntityId and SchemaName.
                    foreach (AttributeMetadata em in entityResponse.EntityMetadata.Attributes)
                    {
                        AttributeMetadata attributeRec = em;
                        
                        AttributeList.Add(attributeRec);
                    }

                    return AttributeList;
                }
            }
            catch (TimeoutException Timeex)
            {
                LoggingConfig.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                LoggingConfig.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                LoggingConfig.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new List<AttributeMetadata>();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="ServerConfiguration"></param>
        /// <param name="EntityId"></param>
        /// <returns></returns>
        public List<MetaDataPair> GetAttributeNamesList(ServerConnection.Configuration ServerConfiguration, Guid EntityId, TransactionLogging LoggingConfig)
        {
            try
            {
                // Connect to the Organization service. 
                // The using statement assures that the service proxy will be properly disposed.
                using (_orgServiceProxy = new OrganizationServiceProxy(ServerConfiguration.OrganizationUri,
                                                                        ServerConfiguration.HomeRealmUri,
                                                                        ServerConfiguration.Credentials,
                                                                        ServerConfiguration.DeviceCredentials))
                {
                    List<MetaDataPair> AttributeList = new List<MetaDataPair>();

                    RetrieveEntityRequest request = new RetrieveEntityRequest()
                    {
                        EntityFilters = EntityFilters.Attributes,
                        MetadataId = EntityId,
                        RetrieveAsIfPublished = true
                    };

                    // Retrieve the MetaData.
                    RetrieveEntityResponse entityResponse = (RetrieveEntityResponse)_orgServiceProxy.Execute(request);
                    //return EntityInfoList;

                    // Loop through the results and create the return array list for the EntityId and SchemaName.
                    foreach (AttributeMetadata em in entityResponse.EntityMetadata.Attributes)
                    {
                        MetaDataPair attributeRec = new MetaDataPair();
                        attributeRec.MetaDataId = new Guid(em.MetadataId.ToString());
                        attributeRec.SchemaName = em.SchemaName;
                        AttributeList.Add(attributeRec);
                    }

                    return AttributeList;
                }
            }
            catch (TimeoutException Timeex)
            {
                LoggingConfig.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                LoggingConfig.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                LoggingConfig.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new List<MetaDataPair>();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="SourceMetadata"></param>
        /// <param name="TargetMetadata"></param>
        /// <returns></returns>
        public Boolean CompareEntityInformation(EntityMetadata SourceMetadata, EntityMetadata TargetMetadata, TransactionLogging LoggingConfig)
        {
            Boolean IsValid = true;
            
            if (SourceMetadata.ActivityTypeMask != null && SourceMetadata.ActivityTypeMask != TargetMetadata.ActivityTypeMask)
            {
                LoggingConfig.WriteLine("*** Entity Metadata ActivityTypeMask Failed Validation...");
                IsValid = false;
            }
            // SourceMetadata.Attributes Not Validated
            if (SourceMetadata.AutoRouteToOwnerQueue != null && SourceMetadata.AutoRouteToOwnerQueue.Value != TargetMetadata.AutoRouteToOwnerQueue.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata AutoRouteToOwnerQueue Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.CanBeInManyToMany != null && SourceMetadata.CanBeInManyToMany.Value != TargetMetadata.CanBeInManyToMany.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata CanBeInManyToMany Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.CanBePrimaryEntityInRelationship != null && SourceMetadata.CanBePrimaryEntityInRelationship.Value != TargetMetadata.CanBePrimaryEntityInRelationship.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata CanBePrimaryEntityInRelationship Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.CanBeRelatedEntityInRelationship != null && SourceMetadata.CanBeRelatedEntityInRelationship.Value != TargetMetadata.CanBeRelatedEntityInRelationship.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata CanBeRelatedEntityInRelationship Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.CanCreateAttributes != null && SourceMetadata.CanCreateAttributes.Value != TargetMetadata.CanCreateAttributes.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata CanCreateAttributes Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.CanCreateCharts != null && SourceMetadata.CanCreateCharts.Value != TargetMetadata.CanCreateCharts.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata CanCreateCharts Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.CanCreateForms != null && SourceMetadata.CanCreateForms.Value != TargetMetadata.CanCreateForms.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata CanCreateForms Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.CanCreateViews != null && SourceMetadata.CanCreateViews.Value != TargetMetadata.CanCreateViews.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata CanCreateViews Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.CanModifyAdditionalSettings != null && SourceMetadata.CanModifyAdditionalSettings.Value != TargetMetadata.CanModifyAdditionalSettings.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata CanModifyAdditionalSettings Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.CanTriggerWorkflow != null && SourceMetadata.CanTriggerWorkflow.Value != TargetMetadata.CanTriggerWorkflow.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata CanTriggerWorkflow Failed Validation...");
                IsValid = false;
            }
            if ((SourceMetadata.Description.UserLocalizedLabel != null && TargetMetadata.Description.UserLocalizedLabel != null) &&
                SourceMetadata.Description.UserLocalizedLabel.Label.ToString() != TargetMetadata.Description.UserLocalizedLabel.Label.ToString())
            {
                LoggingConfig.WriteLine("*** Entity Metadata Description Failed Validation...");
                IsValid = false;
            }
            if ((SourceMetadata.DisplayCollectionName.UserLocalizedLabel != null && TargetMetadata.DisplayCollectionName.UserLocalizedLabel != null) &&
                SourceMetadata.DisplayCollectionName.UserLocalizedLabel.Label.ToString() != TargetMetadata.DisplayCollectionName.UserLocalizedLabel.Label.ToString()) 
            {
                LoggingConfig.WriteLine("*** Entity Metadata DisplayCollectionName Failed Validation...");
                IsValid = false;
            }
            if ((SourceMetadata.DisplayName.UserLocalizedLabel != null && TargetMetadata.DisplayName.UserLocalizedLabel != null) &&
                SourceMetadata.DisplayName.UserLocalizedLabel.Label.ToString() != TargetMetadata.DisplayName.UserLocalizedLabel.Label.ToString())
            {
                LoggingConfig.WriteLine("*** Entity Metadata DisplayName Failed Validation...");
                IsValid = false;
            }
            // SourceMetadata.ExtensionData Not Validated
            if (SourceMetadata.IconLargeName != null && SourceMetadata.IconLargeName != TargetMetadata.IconLargeName)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IconLargeName Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IconMediumName != null && SourceMetadata.IconMediumName != TargetMetadata.IconMediumName)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IconMediumName Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IconSmallName != null && SourceMetadata.IconSmallName != TargetMetadata.IconSmallName)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IconSmallName Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsActivity != null && SourceMetadata.IsActivity.Value != TargetMetadata.IsActivity.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsActivity Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsActivityParty != null && SourceMetadata.IsActivityParty.Value != TargetMetadata.IsActivityParty.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsActivityParty Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsAuditEnabled != null && SourceMetadata.IsAuditEnabled.Value != TargetMetadata.IsAuditEnabled.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsAuditEnabled Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsAvailableOffline != null && SourceMetadata.IsAvailableOffline.Value != TargetMetadata.IsAvailableOffline.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsAvailableOffline Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsChildEntity != null && SourceMetadata.IsChildEntity.Value != TargetMetadata.IsChildEntity.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsChildEntity Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsConnectionsEnabled != null && SourceMetadata.IsConnectionsEnabled.Value != TargetMetadata.IsConnectionsEnabled.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsConnectionsEnabled Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsCustomEntity != null && SourceMetadata.IsCustomEntity.Value != TargetMetadata.IsCustomEntity.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsCustomEntity Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsCustomizable != null && SourceMetadata.IsCustomizable.Value != TargetMetadata.IsCustomizable.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsCustomizable Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsDocumentManagementEnabled != null && SourceMetadata.IsDocumentManagementEnabled.Value != TargetMetadata.IsDocumentManagementEnabled.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsDocumentManagementEnabled Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsDuplicateDetectionEnabled != null && SourceMetadata.IsDuplicateDetectionEnabled.Value != TargetMetadata.IsDuplicateDetectionEnabled.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsDuplicateDetectionEnabled Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsEnabledForCharts != null && SourceMetadata.IsEnabledForCharts.Value != TargetMetadata.IsEnabledForCharts.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsEnabledForCharts Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsImportable != null && SourceMetadata.IsImportable.Value != TargetMetadata.IsImportable.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsImportable Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsIntersect != null && SourceMetadata.IsIntersect.Value != TargetMetadata.IsIntersect.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsIntersect Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsMailMergeEnabled != null && SourceMetadata.IsMailMergeEnabled.Value != TargetMetadata.IsMailMergeEnabled.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsMailMergeEnabled Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsManaged != null && SourceMetadata.IsManaged.Value != TargetMetadata.IsManaged.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsManaged Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsMappable != null && SourceMetadata.IsMappable.Value != TargetMetadata.IsMappable.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsMappable Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsReadingPaneEnabled != null && SourceMetadata.IsReadingPaneEnabled.Value != TargetMetadata.IsReadingPaneEnabled.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsReadingPaneEnabled Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsRenameable != null && SourceMetadata.IsRenameable.Value != TargetMetadata.IsRenameable.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsRenameable Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsValidForAdvancedFind != null && SourceMetadata.IsValidForAdvancedFind.Value != TargetMetadata.IsValidForAdvancedFind.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsValidForAdvancedFind Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsValidForQueue != null && SourceMetadata.IsValidForQueue.Value != TargetMetadata.IsValidForQueue.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsValidForQueue Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.IsVisibleInMobile != null && SourceMetadata.IsVisibleInMobile.Value != TargetMetadata.IsVisibleInMobile.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata IsVisibleInMobile Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.LogicalName != null && SourceMetadata.LogicalName != TargetMetadata.LogicalName)
            {
                LoggingConfig.WriteLine("*** Entity Metadata LogicalName Failed Validation...");
                IsValid = false;
            }
            // SourceMetadata.ManyToManyRelationships Not Validated
            // SourceMetadata.ManyToOneRelationships Not Validated
            
            // Metadata ID should likely not be the same
            //if (SourceMetadata.MetadataId != null && SourceMetadata.MetadataId.Value != TargetMetadata.MetadataId.Value)
            //{
            //    LoggingConfig.WriteLine("*** Entity Metadata MetadataId Failed Validation...");
            //    IsValid = false;
            //}
            // Ojbect Type Codes Validation removed as they will likely never be the same
            //if (SourceMetadata.ObjectTypeCode != null && SourceMetadata.ObjectTypeCode.Value != TargetMetadata.ObjectTypeCode.Value)
            //{
            //    LoggingConfig.WriteLine("*** Entity Metadata ObjectTypeCode Failed Validation...");
            //    IsValid = false;
            //}
            // SourceMetadata.OneToManyRelationships Not Validated
            if (SourceMetadata.OwnershipType != null && SourceMetadata.OwnershipType.Value != TargetMetadata.OwnershipType.Value)
            {
                LoggingConfig.WriteLine("*** Entity Metadata OwnershipType Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.PrimaryIdAttribute != null && SourceMetadata.PrimaryIdAttribute != TargetMetadata.PrimaryIdAttribute)
            {
                LoggingConfig.WriteLine("*** Entity Metadata PrimaryIdAttribute Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.PrimaryNameAttribute != null && SourceMetadata.PrimaryNameAttribute != TargetMetadata.PrimaryNameAttribute)
            {
                LoggingConfig.WriteLine("*** Entity Metadata PrimaryNameAttribute Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.Privileges != null && SourceMetadata.Privileges != TargetMetadata.Privileges)
            {
                LoggingConfig.WriteLine("*** Entity Metadata Privileges Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.RecurrenceBaseEntityLogicalName != null && SourceMetadata.RecurrenceBaseEntityLogicalName != TargetMetadata.RecurrenceBaseEntityLogicalName)
            {
                LoggingConfig.WriteLine("*** Entity Metadata RecurrenceBaseEntityLogicalName Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.ReportViewName != null && SourceMetadata.ReportViewName != TargetMetadata.ReportViewName)
            {
                LoggingConfig.WriteLine("*** Entity Metadata ReportViewName Failed Validation...");
                IsValid = false;
            }
            if (SourceMetadata.SchemaName != null && SourceMetadata.SchemaName != TargetMetadata.SchemaName)
            {
                LoggingConfig.WriteLine("*** Entity Metadata SchemaName Failed Validation...");
                IsValid = false;
            }

            return IsValid;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="SourceMetadata"></param>
        /// <param name="TargetMetadata"></param>
        /// <returns></returns>
        public Boolean CompareAttributeInformation(AttributeMetadata SourceMetadata, AttributeMetadata TargetMetadata, TransactionLogging LoggingConfig)
        {
            Boolean IsMatch = true;

            if (SourceMetadata != null && TargetMetadata != null)
            {
                if (SourceMetadata.AttributeOf != TargetMetadata.AttributeOf)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata AttributeOf Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.AttributeType.Value != TargetMetadata.AttributeType.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata AttributeType Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.CanBeSecuredForCreate.Value != TargetMetadata.CanBeSecuredForCreate.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata CanBeSecuredForCreate Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.CanBeSecuredForRead.Value != TargetMetadata.CanBeSecuredForRead.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata CanBeSecuredForRead Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.CanBeSecuredForUpdate.Value != TargetMetadata.CanBeSecuredForUpdate.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata CanBeSecuredForUpdate Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.CanModifyAdditionalSettings.Value != TargetMetadata.CanModifyAdditionalSettings.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata CanModifyAdditionalSettings Failed Validation...");
                    IsMatch = false;
                }
                //if (SourceMetadata.ColumnNumber.Value != TargetMetadata.ColumnNumber.Value)
                //{
                //    LoggingConfig.WriteLine("*** Attribute Metadata ColumnNumber Failed Validation...");
                //    IsMatch = false;
                //}
                if (SourceMetadata.DeprecatedVersion != TargetMetadata.DeprecatedVersion)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata DeprecatedVersion Failed Validation...");
                    IsMatch = false;
                }
                if ((SourceMetadata.Description.UserLocalizedLabel != null && TargetMetadata.Description.UserLocalizedLabel != null) && SourceMetadata.Description.UserLocalizedLabel.Label.ToString() != TargetMetadata.Description.UserLocalizedLabel.Label.ToString())
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata Description Failed Validation...");
                    IsMatch = false;
                }
                if ((SourceMetadata.DisplayName.UserLocalizedLabel != null && TargetMetadata.DisplayName.UserLocalizedLabel != null) && SourceMetadata.DisplayName.UserLocalizedLabel.Label.ToString() != TargetMetadata.DisplayName.UserLocalizedLabel.Label.ToString())
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata DisplayName Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.EntityLogicalName != TargetMetadata.EntityLogicalName)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata EntityLogicalName Failed Validation...");
                    IsMatch = false;
                }
                //if (SourceMetadata.ExtensionData != TargetMetadata.ExtensionData)
                //{
                //    LoggingConfig.WriteLine("*** Attribute Metadata ExtensionData Failed Validation...");
                //    IsMatch = false;
                //}
                if (SourceMetadata.IsAuditEnabled.Value != TargetMetadata.IsAuditEnabled.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsAuditEnabled Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsCustomAttribute.Value != TargetMetadata.IsCustomAttribute.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsCustomAttribute Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsCustomizable.Value != TargetMetadata.IsCustomizable.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsCustomizable Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsManaged.Value != TargetMetadata.IsManaged.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsManaged Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsPrimaryId.Value != TargetMetadata.IsPrimaryId.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsPrimaryId Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsPrimaryName.Value != TargetMetadata.IsPrimaryName.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsPrimaryName Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsRenameable.Value != TargetMetadata.IsRenameable.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsRenameable Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsSecured.Value != TargetMetadata.IsSecured.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsSecured Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsValidForAdvancedFind.Value != TargetMetadata.IsValidForAdvancedFind.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsValidForAdvancedFind Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsValidForCreate.Value != TargetMetadata.IsValidForCreate.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsValidForCreate Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsValidForRead.Value != TargetMetadata.IsValidForRead.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsValidForRead Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.IsValidForUpdate.Value != TargetMetadata.IsValidForUpdate.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata IsValidForUpdate Failed Validation...");
                    IsMatch = false;
                }
                if ((SourceMetadata.LinkedAttributeId != null && TargetMetadata.LinkedAttributeId != null) && SourceMetadata.LinkedAttributeId.Value != TargetMetadata.LinkedAttributeId.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata LinkedAttributeId Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.LogicalName != TargetMetadata.LogicalName)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata LogicalName Failed Validation...");
                    IsMatch = false;
                }
                //if (SourceMetadata.MetadataId.Value != TargetMetadata.MetadataId.Value)
                //{
                //    LoggingConfig.WriteLine("*** Attribute Metadata MetadataId Failed Validation...");
                //    IsMatch = false;
                //}
                if (SourceMetadata.RequiredLevel.Value != TargetMetadata.RequiredLevel.Value)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata RequiredLevel Failed Validation...");
                    IsMatch = false;
                }
                if (SourceMetadata.SchemaName != TargetMetadata.SchemaName)
                {
                    LoggingConfig.WriteLine("*** Attribute Metadata SchemaName Failed Validation...");
                    IsMatch = false;
                }
            }
            else
            {
                if (!(SourceMetadata == null && TargetMetadata == null))
                    IsMatch = false;
            }


            return IsMatch;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="EntityMetadataArray"></param>
        /// <param name="EntityId"></param>
        /// <returns></returns>
        public EntityMetadata GetEntityMetadata(EntityMetadata [] EntityMetadataArray, Guid EntityId)
        {
            foreach(EntityMetadata emd in EntityMetadataArray)
            {
                if (emd.MetadataId == EntityId)
                    return emd;
            }

            return new EntityMetadata();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="AttributeMetadataArray"></param>
        /// <param name="AttributeId"></param>
        /// <returns></returns>
        public AttributeMetadata GetAttributeMetadata(AttributeMetadata[] AttributeMetadataArray, Guid AttributeId)
        {
            foreach (AttributeMetadata amd in AttributeMetadataArray)
            {
                if (amd.MetadataId == AttributeId)
                    return amd;
            }

            return new AttributeMetadata();
        }
     }
}

