﻿// **********************************************************************************************************
// Program.cs
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 10/26/2011
// This is the main method class is designed to facilitate accessing components within the Entity
//
// This file also contains the following Classes
// OrgParameters - used to hold the user parameters passed in on execution of the console application
// MetaDataPair - used to hold a key value pair of Guid and string.
//
// Methods:
// Main - console startup method
// ValidateParameters - uses various checks to validate the paramters met the expected criteria
//
// Call this application by executing the CRMDiffingTool.exe with a minimum of two parameters
// -sc:<source connection string>
// -tc:<target connection string>
//
// Additional Parameters
// -su:<source user> if only the username is supplied must also supply a -sd parameter to supply the login domain.
// -tu:<target user> if only the username is supplied must also supply a -td parameter to supply the login domain.
// -sp:<source user password> Must be supplied if you supply an -su paramater
// -tp:<target user password> Must be supplied if you supply an -tu paramater
// -sd:<source user login domain> Must be supplied if the source user name does not contain the <domain\user> format
// -td:<target user login domain> Must be supplied if the target user name does not contain the <domain\user> format
//
// To add additional parameters code them as key/Value pairs -<key>:<value> as the format of the parameter where
// -<key> is the key and <value> is the value you wish to act on.
//
// Additionally make sure that you make changes to the ValidateParameters method to allow your parameters
// to pass the validation test as well as not to fail the validataion test with the presence of your parmaters.
// **********************************************************************************************************

using System;

using Microsoft.MCS.Logging;
using Microsoft.MCS.CRM.ConnectionHelper;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Client;
using Microsoft.MCS.CRM2011OrgCompare;
using Microsoft.Xrm.Sdk.Discovery;
using System.Collections.Generic;
using Microsoft.Xrm.Sdk.Metadata;

namespace Microsoft.MCS
{
    /// <summary>
    /// 
    /// </summary>
    class OrgParameters
    {
        // Store the connection strings with or without the user info.
        public string CrmConnectionString { get; set; }

        // Store the user credentials
        public string Domain { get; set; }
        public string User { get; set; }
        public string Password { get; set; }

        public enum CrmVersion
        {
            DynamicsCrm4,
            DynamicsCrm2011,
            DynamicsCrm2013
        };
        public CrmVersion ApplicationVersion { get; set; }
    }

    class MetaDataPair
    {
        public Guid MetaDataId { get; set; }
        public string SchemaName { get; set; }
    }
    /// <summary>
    /// 
    /// </summary>
    static class Program
    {
        private static OrganizationServiceProxy sourceServiceProxy = null;
        private static OrganizationServiceProxy targetServiceProxy = null;


        /// <summary>
        /// 
        /// </summary>
        /// <param name="args"></param>
        /// 
        static void Main(string[] args)
        {
            Environment.ExitCode = 0; // Start with the assumption that they will all pass so if any step failes set the exit code to -1 so the run fails.
            Logger tLog = new Logger();
            OrgParameters sorgParam = new OrgParameters();
            OrgParameters torgParam = new OrgParameters();

            ServerConnection sourceConnect = new ServerConnection();
            ServerConnection targetConnect = new ServerConnection();

            ServerConnection.Configuration sourceConfig = sourceConnect.GetServerConfiguration();
            ServerConnection.Configuration targetConfig = targetConnect.GetServerConfiguration();

            if(sourceConfig == null || sourceConfig.OrganizationName == null)
            {


            ParseParameter.Arguments param = new ParseParameter.Arguments(args);
            if (ValidateParameters(param) != true)
            {
                tLog.WriteLine("Invalid Parameters used in execution : Error Code 20");
                Environment.Exit(-20);
            }

            foreach (System.Collections.DictionaryEntry arg in param.Parameters)
            {

                switch (arg.Key.ToString())
                {
                    case "sc":
                        sorgParam.CrmConnectionString = arg.Value.ToString();
                        Console.WriteLine("Source Connection String Parameter {0}.", arg.Value.ToString());
                        break;
                    case "tc":
                        torgParam.CrmConnectionString = arg.Value.ToString();
                        Console.WriteLine("Target Connection String Parameter {0}.", arg.Value.ToString());
                        break;
                    case "su":
                        sorgParam.User = arg.Value.ToString();
                        Console.WriteLine("Source User Parameter {0}.", arg.Value.ToString());
                        break;
                    case "tu":
                        torgParam.User = arg.Value.ToString();
                        Console.WriteLine("Target User Parameter {0}.", arg.Value.ToString());
                        break;
                    case "sd":
                        sorgParam.Domain = arg.Value.ToString();
                        Console.WriteLine("Source Domain Parameter {0}.", arg.Value.ToString());
                        break;
                    case "td":
                        torgParam.Domain = arg.Value.ToString();
                        Console.WriteLine("Target Domain Parameter {0}.", arg.Value.ToString());
                        break;
                    case "sp":
                        sorgParam.Password = arg.Value.ToString();
                        break;
                    case "tp":
                        torgParam.Password = arg.Value.ToString();
                        break;
                }
            }

            }

            sourceServiceProxy = new OrganizationServiceProxy(sourceConfig.OrganizationUri, sourceConfig.HomeRealmUri, sourceConfig.Credentials, sourceConfig.DeviceCredentials);
            targetServiceProxy = new OrganizationServiceProxy(targetConfig.OrganizationUri, targetConfig.HomeRealmUri, targetConfig.Credentials, targetConfig.DeviceCredentials);

            RetrieveVersionRequest sversionRequest = new RetrieveVersionRequest();
            RetrieveVersionResponse sversionResponse =
                (RetrieveVersionResponse)sourceServiceProxy.Execute(sversionRequest);
            Console.WriteLine("Microsoft Dynamics CRM version {0}.", sversionResponse.Version);

            RetrieveVersionRequest tversionRequest = new RetrieveVersionRequest();
            RetrieveVersionResponse tversionResponse =
                (RetrieveVersionResponse)targetServiceProxy.Execute(tversionRequest);
            Console.WriteLine("Microsoft Dynamics CRM version {0}.", tversionResponse.Version);

            tLog.WriteLine("Source Organization: " + sourceConfig.OrganizationName);
            tLog.WriteLine("Target Organization: " + targetConfig.OrganizationName);
            tLog.BlankLine();

            Boolean IsValidOrgInfo = ValidateOrganizations(sourceConfig, targetConfig, tLog);
            if (!IsValidOrgInfo)
            {
                Environment.ExitCode = -1;
            }

            Boolean IsValidEntityDetail = ValidateEntityDetails(sourceConfig, targetConfig, tLog);
            if (!IsValidEntityDetail)
            {
                Environment.ExitCode = -1;
            }

            Boolean IsValidAttributeDetails = ValidateAttributeDetails(sourceConfig, targetConfig, tLog);
            if (!IsValidAttributeDetails)
            {
                Environment.ExitCode = -1;
            }

            tLog.WriteLine("Validation Completed with Exit Code " + Environment.ExitCode.ToString());
            tLog.Close();



        }

        public static Boolean ValidateOrganizations(ServerConnection.Configuration SourceConfiguration, ServerConnection.Configuration TargetConfiguration, Logger LoggingConfig)
        {
            LoggingConfig.WriteLine("Validating Organization Versions..........");
            LoggingConfig.BlankLine();

            // Set up the working return
            Boolean IsValidOrganizations = true;

            // Create an entity validations object
            OrganizationHelpers SourceOrg = new OrganizationHelpers(SourceConfiguration);
            OrganizationHelpers TargetOrg = new OrganizationHelpers(TargetConfiguration);

            OrganizationDetail SourceDetail = new OrganizationDetail();
            OrganizationDetail TargetDetail = new OrganizationDetail();


            // Get the Organizations from the source and targets that will be used in the validation
            LoggingConfig.WriteLine("Obtaining Organization List from Discovery Service of Source Organization " + SourceOrg._serverConfig.OrganizationName + "..........");
            List<OrganizationDetail> sourceOrgList = SourceOrg.GetOrganizationDetailsList(SourceOrg._serverConfig, LoggingConfig);

            LoggingConfig.WriteLine("Obtaining Organization List from Discovery Service of Target Organization " + TargetOrg._serverConfig.OrganizationName + "..........");
            List<OrganizationDetail> targetOrgList = TargetOrg.GetOrganizationDetailsList(TargetOrg._serverConfig, LoggingConfig);

            // Get the Source Org Detail and assign it
            foreach (OrganizationDetail sourceOrgDet in sourceOrgList)
            {
                if (sourceOrgDet.UniqueName.ToUpper() == SourceOrg._serverConfig.OrganizationName.ToUpper())
                {
                    SourceDetail = sourceOrgDet;
                }
            }

            // Get the Target Org Detail and assign it
            foreach (OrganizationDetail targetOrgDet in targetOrgList)
            {
                if (targetOrgDet.UniqueName.ToUpper() == TargetOrg._serverConfig.OrganizationName.ToUpper())
                {
                    TargetDetail = targetOrgDet;
                }
            }

            if (SourceOrg.CompareOrganizationInformation(SourceDetail, TargetDetail, LoggingConfig) == true)
            {
                IsValidOrganizations = true;
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Organization Versions " + SourceDetail.OrganizationVersion.ToString() + " Match...");
                LoggingConfig.BlankLine();
            }
            else
            {
                IsValidOrganizations = false;
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Source Organization Version " + SourceDetail.OrganizationVersion.ToString() + " Does not Match target Organization version of " + TargetDetail.OrganizationVersion.ToString() + "...");
                LoggingConfig.BlankLine();
            }

            return IsValidOrganizations;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="SourceOrganizationUri"></param>
        /// <param name="TargetOrganizationUri"></param>
        /// <returns></returns>
        public static Boolean ValidateEntityDetails(ServerConnection.Configuration sourceServerConfig, ServerConnection.Configuration targetServerConfig, Logger LoggingConfig)
        {
            LoggingConfig.WriteLine("Validating Entity Detail Settings Information..........");
            LoggingConfig.BlankLine();

            // Set up the working return
            Boolean isValidateEntityDetails = true;

            // Create an entity validations object and preload the connection info
            EntityHelpers SourceEntity = new EntityHelpers();
            EntityHelpers TargetEntity = new EntityHelpers();

            // Get the entities list from the source and targets that will be used in the validation
            LoggingConfig.WriteLine("Obtaining Entity List from Source Organization " + sourceServerConfig.OrganizationName + "..........");
            List<EntityMetadata> semdid = SourceEntity.GetEntityEntityList(sourceServerConfig, LoggingConfig);

            LoggingConfig.WriteLine("Obtaining Entity List from Target Organization " + targetServerConfig.OrganizationName + "..........");
            List<EntityMetadata> temdid = TargetEntity.GetEntityEntityList(targetServerConfig, LoggingConfig);

            // Roll through the source get the associated Metadata ID from the target to use to get the EntityMetadata
            // then compare the detail of both.
            foreach (EntityMetadata SourceEmd in semdid)
            {
                EntityMetadata TargetEmd = temdid.Find(
                   delegate (EntityMetadata mdp)
                   {
                       return mdp.SchemaName == SourceEmd.SchemaName;
                   }
                   );

                if (TargetEmd != null)
                {
                    // go run the compare of the two entity metadata ojbects
                    if (!(SourceEntity.CompareEntityInformation(SourceEmd, TargetEmd, LoggingConfig) == true))
                    {
                        isValidateEntityDetails = false;
                        LoggingConfig.WriteLine("***** Entity comparison for source entity " + SourceEmd.SchemaName.ToString() + " Failed");
                    }
                    else
                    {
                        //LoggingConfig.WriteLine("Entity " + SourceEmd.SchemaName.ToString() + " was compared");
                    }
                }
                else
                {
                    isValidateEntityDetails = false;
                    LoggingConfig.WriteLine("*** Source Entity " + SourceEmd.SchemaName.ToString() + " Not Found in the Target Organization ");
                }
            }

            // Reverse the process to identify any missing entities
            foreach (EntityMetadata TargetEmd in temdid)
            {
                EntityMetadata SourceEmd = semdid.Find(
                   delegate (EntityMetadata mdp)
                   {
                       return mdp.SchemaName == TargetEmd.SchemaName;
                   }
                   );


                if (SourceEmd == null)
                {
                    isValidateEntityDetails = false;
                    LoggingConfig.WriteLine("***** Target Entity " + TargetEmd.SchemaName.ToString() + " Not Found in the Source Organization ");
                }
            }

            if (isValidateEntityDetails)
            {
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Entity Information Comparison PASSED *****");
                LoggingConfig.BlankLine();
            }
            else
            {
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Entity Information Comparison FAILED *****");
                LoggingConfig.BlankLine();
            }
            return isValidateEntityDetails;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="SourceOrganizationUri"></param>
        /// <param name="TargetOrganizationUri"></param>
        /// <returns></returns>
        public static Boolean ValidateAttributeDetails(ServerConnection.Configuration sourceServerConfig, ServerConnection.Configuration targetServerConfig, Logger LoggingConfig)
        {
            LoggingConfig.WriteLine("Validating Attribute Detail Information..........");
            LoggingConfig.BlankLine();

            // Set up the working return
            Boolean isValidAttributeDetails = true;

            // Create an entity validations object
            EntityHelpers SourceEntity = new EntityHelpers();
            EntityHelpers TargetEntity = new EntityHelpers();

            // Get the entities list from the source and targets that will be used in the validation
            LoggingConfig.WriteLine("Obtaining Entity List from Source Organization " + sourceServerConfig.OrganizationName + "..........");
            List<MetaDataPair> sourceEntityList = SourceEntity.GetEntityNamesList(sourceServerConfig, LoggingConfig);

            LoggingConfig.WriteLine("Obtaining Entity List from Target Organization " + targetServerConfig.OrganizationName + "..........");
            List<MetaDataPair> targetEntityList = TargetEntity.GetEntityNamesList(targetServerConfig, LoggingConfig);

            // Roll through the source and check the target in detail
            foreach (MetaDataPair sourceEntityPair in sourceEntityList)
            {
                MetaDataPair targetEntityPair = targetEntityList.Find(
                    delegate (MetaDataPair mdp)
                    {
                        return mdp.SchemaName == sourceEntityPair.SchemaName;
                    }
                    );

                if (targetEntityPair != null)
                {
                    //LoggingConfig.WriteLine("Processing attributes in Entity " + sourceEntityPair.SchemaName.ToString() + "...");

                    // get the Attribues and validate them here
                    List<AttributeMetadata> sourceAttributeList = SourceEntity.GetAttributeDetailsList(sourceServerConfig, sourceEntityPair.MetaDataId, LoggingConfig);
                    List<AttributeMetadata> targetAttributeList = TargetEntity.GetAttributeDetailsList(targetServerConfig, targetEntityPair.MetaDataId, LoggingConfig);

                    // Foreach loop attributes source to target
                    foreach (AttributeMetadata sourceAttribute in sourceAttributeList)
                    {
                        AttributeMetadata targetAttribute = targetAttributeList.Find(
                           delegate (AttributeMetadata mdp)
                           {
                               return mdp.SchemaName == sourceAttribute.SchemaName;
                           }
                           );

                        if (targetAttribute != null)
                        {
                            if (!(SourceEntity.CompareAttributeInformation(sourceAttribute, targetAttribute, LoggingConfig)))
                            {
                                isValidAttributeDetails = false;
                                LoggingConfig.WriteLine("***** Attribute comparison for Source Attribute " + sourceAttribute.SchemaName.ToString() + " in Entity " + sourceEntityPair.SchemaName.ToString() + " Failed");
                            }
                        }
                        else
                        {
                            LoggingConfig.WriteLine("***** Attribute " + sourceAttribute.SchemaName.ToString() + " was Not Found in the target entity " + sourceEntityPair.SchemaName.ToString());
                            isValidAttributeDetails = false;
                        }
                    }

                    // Foreach loop attributes target to source
                    foreach (AttributeMetadata targetAttribute in targetAttributeList)
                    {
                        AttributeMetadata sourceAttribute = sourceAttributeList.Find(
                            delegate (AttributeMetadata rmdp)
                            {
                                return rmdp.SchemaName == targetAttribute.SchemaName;
                            }
                        );
                        if (sourceAttribute != null)
                        {
                            // Do nothing. The detail has already been compared in the other foreach.
                        }
                        else
                        {
                            LoggingConfig.WriteLine("*****  Attribute " + targetAttribute.SchemaName.ToString() + " was Not Found in the source entity " + sourceEntityPair.SchemaName.ToString());
                            isValidAttributeDetails = false;
                        }
                        // LoggingConfig.WriteLine("Target Attribute " + targetAttribute.SchemaName);
                    }
                }
                else
                {
                    LoggingConfig.WriteLine("***** Entity " + sourceEntityPair.SchemaName.ToString() + "  Skipped as it was Not Found in the target...");
                    isValidAttributeDetails = false;
                }
            }

            // flip the logic and check the other way for missing entities. It is probably best to at least report them
            // Any entities found will have had the attributes checked both ways in the prior foreach loop.
            foreach (MetaDataPair sourceEntityPair in sourceEntityList)
            {
                MetaDataPair targetEntityPair = targetEntityList.Find(
                    delegate (MetaDataPair mdp)
                    {
                        return mdp.SchemaName == sourceEntityPair.SchemaName;
                    }
                    );

                if (targetEntityPair == null)
                {
                    LoggingConfig.WriteLine("*****  Entity " + sourceEntityPair.SchemaName.ToString() + " was Not Found in the target");
                    isValidAttributeDetails = false;
                }
            }

            if (isValidAttributeDetails)
            {
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Attribute Schema Name Comparison PASSED *****");
                LoggingConfig.BlankLine();
            }
            else
            {
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Attribute Schema Name Comparison FAILED *****");
                LoggingConfig.BlankLine();
            }
            return isValidAttributeDetails;
        }

        private static Boolean ValidateParameters(ParseParameter.Arguments param)
        {            
            //var sourceParameter = false;
            //var targetParameter = false;
            //// Stop the application if there were not any parameters passed in at all
            //if (param.Parameters.Count <= 0)
            //{
            //    return false;
            //}

            //// Stop the application if -s and -t were not both passed in
            //foreach (System.Collections.DictionaryEntry args in param.Parameters)
            //{
            //    // Check for invalid parameters. If you add a parameter make sure to put it in this switch as a case
            //    // Deadman switch (If it hits a value that is not a case the app will bail with exit code -1
            //    switch (args.Key.ToString())
            //    {
            //        case "sc":
            //            sourceParameter = true;
            //            break;
            //        case "tc":
            //            targetParameter = true;
            //            break;
            //        case "su":
            //            break;
            //        case "sp":
            //            break;
            //        case "sd":
            //            break;
            //        case "tu":
            //            break;
            //        case "tp":
            //            break;
            //        case "td":
            //            break;
            //        default:
            //            return false;
            //    }
            //}

            //// Validates that both the s and the t were passed in
            //// Make sure that both an s and a t parameter were passed in
            //if (!(sourceParameter && targetParameter))
            //{
            //    return false;
            //}

            ////// In order to make sure that a user does not pass in more than one s or t parameter.
            ////// This also mandates that they MUST pass in an s and a t parameter
            ////// validate that only 1 s and 1 t parameters were passed in
            ////if( sCount != 1 || tCount != 1 )
            ////{
            ////    return false;
            ////}

            //// passed all validation checks
            return true;
        }
    }
}
