﻿// **********************************************************************************************************
// Program.cs
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 10/26/2011
// This is the main method class is designed to facilitate accessing components within the Entity
//
// This file also contains the following Classes
// OrgParameters - used to hold the user parameters passed in on execution of the console application
// MetaDataPair - used to hold a key value pair of Guid and string.
//
// Methods:
// Main - console startup method
// ValidateParameters - uses various checks to validate the paramters meet the expected criteria
//
// Call this application by executing the CRMDiffingTool.exe with a minimum of two parameters
// -s:<source organization uri>
// -t:<target organizatino uri>
//
// Additional Parameters
// -su:<source user> if only the username is supplied must also supply a -sd parameter to supply the login domain.
// -tu:<target user> if only the username is supplied must also supply a -td parameter to supply the login domain.
// -sp:<source user password> Must be supplied if you supply an -su paramater
// -tp:<target user password> Must be supplied if you supply an -tu paramater
// -sd:<source user login domain> Must be supplied if the source user name does not contain the <domain\user> format
// -td:<target user login domain> Must be supplied if the target user name does not contain the <domain\user> format
//
// To add additional parameters code them as key/Value pairs -<key>:<value> as the format of the parameter where
// -<key> is the key and <value> is the value you wish to act on.
//
// Additionally make sure that you make changes to the ValidateParameters method to allow your parameters
// to pass the validation test as well as not to fail the validataion test with the presence of your parmaters.
// **********************************************************************************************************
using System;
using System.Collections.Generic;
using Microsoft.MCS.Logging;
using Microsoft.Xrm.Sdk.Metadata;
using Microsoft.Xrm.Sdk.Client;
using Microsoft.Xrm.Sdk.Discovery;
using CRM2011OrgCompare.Helpers;

namespace Microsoft.MCS
{
    /// <summary>
    /// 
    /// </summary>
    class OrgParameters
    {
        // Store the Uri
        public Uri SourceOrg { get; set; }
        public Uri TargetOrg { get; set; }
        // Store the Server names
        public string SourceServer { get; set; }
        public string TargetServer { get; set; }
        // Store the tenant Unique Name
        public string SourceUniqueName { get; set; }
        public string TargetUniqueName { get; set; }
        // Store the Uri SSL state
        public Boolean SourceIsSSL { get; set; }
        public Boolean TargetIsSSL { get; set; }
        // Store the Discovery Uri
        public Uri SourceDiscoveryUri { get; set; }
        public Uri TargetDiscoveryUri { get; set; }
        // Store the source user credentials
        public string SourceUser { get; set; }
        public string SourcePassword { get; set; }
        // Store the target user credentials
        public string TargetUser { get; set; }
        public string TargetPassword { get; set; }
        // Store the source domain
        public string SourceDomain { get; set; }
        // Store the target domain
        public string TargetDomain { get; set; }

        public enum CRMVersion{    DynamicsCrm4,
                                DynamicsCrm2011,
                                DynamicsCrmPolaris};
        public CRMVersion ApplicationVersion { get; set; }

        public OrganizationServiceProxy _sourceServiceProxy { get; set; }
        public OrganizationServiceProxy _targetServiceProxy { get; set; }
    }

    class MetaDataPair
    {
        public Guid MetaDataId { get; set; }
        public string SchemaName { get; set; }
    }
    /// <summary>
    /// 
    /// </summary>
    class Program
    {
        /// <summary>
        /// 
        /// </summary>
        /// <param name="args"></param>
        static void Main(string[] args)
        {
            Environment.ExitCode = 0; // Start with the assumption that they will all pass so if any step failes the whole process failes. At that point the exit code will be changed to -1
            OrgParameters oups = new OrgParameters();
            ServerConnection sourceConn = new ServerConnection();
            ServerConnection targetConn = new ServerConnection();
            Logger tLog = new Logger();

            ParseParameter.Arguments param = new ParseParameter.Arguments(args);
            if (!(ValidateParameters(param) == true))
            {
                tLog.WriteLine("Invalid Parameters used in execution : Error Code 20");
                Environment.Exit(-20);
            }

            foreach (System.Collections.DictionaryEntry arg in param.Parameters)
            {

                switch (arg.Key.ToString())
                {
                    case "s":
                        oups.SourceOrg = new Uri(arg.Value.ToString());
                        break;
                    case "t":
                        oups.TargetOrg = new Uri(arg.Value.ToString());
                        break;
                    case "su":
                        oups.SourceUser = arg.Value.ToString();
                        break;
                    case "sp":
                        oups.SourcePassword = arg.Value.ToString();
                        break;
                    case "sd":
                        oups.SourceDomain = arg.Value.ToString();
                        break;
                    case "tu":
                        oups.TargetUser = arg.Value.ToString();
                        break;
                    case "tp":
                        oups.TargetPassword = arg.Value.ToString();
                        break;
                    case "td":
                        oups.TargetDomain = arg.Value.ToString();
                        break;
                }
            }

            tLog.WriteLine("Source Organization: " + oups.SourceOrg);
            tLog.WriteLine("Target Organization: " + oups.TargetOrg);
            tLog.BlankLine();  // Skip a line to make it easier to read

            //oups.SourceServer = sourceConn.GetServerFromUri(oups.SourceOrg);
            //oups.TargetServer = targetConn.GetServerFromUri(oups.TargetOrg);

            //oups.SourceUniqueName = sourceConn.GetUniqueNameFromUri(oups.SourceOrg, CRMVersion.DynamicsCrm2011);
            //oups.TargetUniqueName = targetConn.GetUniqueNameFromUri(oups.TargetOrg, CRMVersion.DynamicsCrm2011);

            //oups.SourceIsSSL = sourceConn.SSLStateFromUri(oups.SourceOrg);
            //oups.TargetIsSSL = targetConn.SSLStateFromUri(oups.TargetOrg);

            //oups.SourceDiscoveryUri = sourceConn.ConcatenateDiscoveryUri(oups.SourceServer, oups.SourceIsSSL);
            //oups.TargetDiscoveryUri = targetConn.ConcatenateDiscoveryUri(oups.TargetServer, oups.TargetIsSSL);

            //oups._sourceServiceProxy = GetConnection();
            //oups._targetServiceProxy = GetConnection();

            Boolean IsValidOrgInfo = ValidateOrganizations(oups, tLog);
            if (!IsValidOrgInfo)
            {
                Environment.ExitCode = -1;
            }

            Boolean IsValidEntityDetail = ValidateEntityDetails(oups, tLog);
            if (!IsValidEntityDetail)
            {
                Environment.ExitCode = -1;
            }

            Boolean IsValidAttributeDetails = ValidateAttributeDetails(oups, tLog);
            if (!IsValidAttributeDetails)
            {
                Environment.ExitCode = -1;
            }

            tLog.WriteLine("Validation Completed with Exit Code " + Environment.ExitCode.ToString());
            tLog.Close();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="SourceOrganizationUri"></param>
        /// <param name="TargetOrganizationUri"></param>
        /// <returns></returns>
        public static Boolean ValidateOrganizations(OrgParameters ValidateOrgParams, Logger LoggingConfig)
        {
            LoggingConfig.WriteLine("Validating Organization Versions..........");
            LoggingConfig.BlankLine();

            // Set up the working return
            Boolean IsValidOrganizations = true;

            // Establish connection objects to be able to use the helper functions.
            ServerConnection sourceServerConn = new ServerConnection();
            ServerConnection targetServerConn = new ServerConnection();

            // Create a source config
            ServerConnection.Configuration sourceServerConfig = new ServerConnection.Configuration();
            if (ValidateOrgParams.TargetUser != string.Empty && ValidateOrgParams.TargetPassword != string.Empty)
            {
                //sourceServerConfig = sourceServerConn.CreateServerConfiguration(ValidateOrgParams.SourceOrg, ValidateOrgParams.SourceUser, ValidateOrgParams.SourceDomain, ValidateOrgParams.SourcePassword);
            }
            else
            {
                //sourceServerConfig = sourceServerConn.CreateServerConfiguration(ValidateOrgParams.SourceOrg);
            }
            // Create a target config
            ServerConnection.Configuration targetServerConfig = new ServerConnection.Configuration();
            if (ValidateOrgParams.TargetUser != string.Empty && ValidateOrgParams.TargetPassword != string.Empty)
            {
                //targetServerConfig = targetServerConn.CreateServerConfiguration(ValidateOrgParams.TargetOrg, ValidateOrgParams.TargetUser, ValidateOrgParams.TargetDomain, ValidateOrgParams.TargetPassword);
            }
            else
            {
                //targetServerConfig = targetServerConn.CreateServerConfiguration(ValidateOrgParams.TargetOrg);
            }

            // Create an entity validations object
            OrganizationHelpers SourceOrg = new OrganizationHelpers();
            OrganizationHelpers TargetOrg = new OrganizationHelpers();

            OrganizationDetail SourceDetail = new OrganizationDetail();
            OrganizationDetail TargetDetail = new OrganizationDetail();


            // Get the Organizations from the source and targets that will be used in the validation
            LoggingConfig.WriteLine("Obtaining Organization List from Discovery Service of Source Organization " + sourceServerConfig.OrganizationName + "..........");
            //List<OrganizationDetail> sourceOrgList = SourceOrg.GetOrganizationDetailsList(sourceServerConfig, LoggingConfig);

            LoggingConfig.WriteLine("Obtaining Organization List from Discovery Service of Target Organization " + targetServerConfig.OrganizationName + "..........");
            //List<OrganizationDetail> targetOrgList = TargetOrg.GetOrganizationDetailsList(targetServerConfig, LoggingConfig);

            // Get the Source Org Detail and assign it
            //foreach (OrganizationDetail sourceOrgDet in sourceOrgList)
            //{
            //    if (sourceOrgDet.UniqueName.ToUpper() == sourceServerConfig.OrganizationName.ToUpper())
            //    {
            //        SourceDetail = sourceOrgDet;
            //    }
            //}

            //// Get the Target Org Detail and assign it
            //foreach (OrganizationDetail targetOrgDet in targetOrgList)
            //{
            //    if (targetOrgDet.UniqueName.ToUpper() == targetServerConfig.OrganizationName.ToUpper())
            //    {
            //        TargetDetail = targetOrgDet;
            //    }
            //}

            if (SourceOrg.CompareOrganizationInformation(SourceDetail, TargetDetail, LoggingConfig) == true)
            {
                IsValidOrganizations = true;
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Organization Versions " + SourceDetail.OrganizationVersion.ToString() + " Match...");
                LoggingConfig.BlankLine();
            }
            else
            {
                IsValidOrganizations = false;
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Source Organization Version " + SourceDetail.OrganizationVersion.ToString() + " Does not Match target Organization version of " + TargetDetail.OrganizationVersion.ToString() + "...");
                LoggingConfig.BlankLine();
            }

            return IsValidOrganizations;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="SourceOrganizationUri"></param>
        /// <param name="TargetOrganizationUri"></param>
        /// <returns></returns>
        public static Boolean ValidateAttributeDetails(OrgParameters ValidateAttribParams, Logger LoggingConfig)
        {
            LoggingConfig.WriteLine("Validating Attribute Detail Information..........");
            LoggingConfig.BlankLine();

            // Set up the working return
            Boolean isValidAttributeDetails = true;

            // Establish connection objects to be able to use the helper functions.
            ServerConnection sourceServerConn = new ServerConnection();
            ServerConnection targetServerConn = new ServerConnection();

            // Create a source config
            ServerConnection.Configuration sourceServerConfig = new ServerConnection.Configuration();
            if (ValidateAttribParams.SourceUser != string.Empty && ValidateAttribParams.SourcePassword != string.Empty)
            {
                //sourceServerConfig = sourceServerConn.CreateServerConfiguration(ValidateAttribParams.SourceOrg, ValidateAttribParams.SourceUser, ValidateAttribParams.SourceDomain, ValidateAttribParams.SourcePassword);
            }
            else
            {
                //sourceServerConfig = sourceServerConn.CreateServerConfiguration(ValidateAttribParams.SourceOrg);
            }
            // Create a target config
            ServerConnection.Configuration targetServerConfig = new ServerConnection.Configuration();
            if (ValidateAttribParams.TargetUser != string.Empty && ValidateAttribParams.TargetPassword != string.Empty)
            {
                //targetServerConfig = targetServerConn.CreateServerConfiguration(ValidateAttribParams.TargetOrg, ValidateAttribParams.TargetUser, ValidateAttribParams.TargetDomain, ValidateAttribParams.TargetPassword);
            }
            else
            {
                //targetServerConfig = targetServerConn.CreateServerConfiguration(ValidateAttribParams.TargetOrg);
            }
            // Create an entity validations object
            EntityHelpers SourceEntity = new EntityHelpers();
            EntityHelpers TargetEntity = new EntityHelpers();

            // Get the entities list from the source and targets that will be used in the validation
            LoggingConfig.WriteLine("Obtaining Entity List from Source Organization " + sourceServerConfig.OrganizationName + "..........");
            //List<MetaDataPair> sourceEntityList = SourceEntity.GetEntityNamesList(sourceServerConfig, LoggingConfig);

            LoggingConfig.WriteLine("Obtaining Entity List from Target Organization " + targetServerConfig.OrganizationName + "..........");
            //List<MetaDataPair> targetEntityList = TargetEntity.GetEntityNamesList(targetServerConfig, LoggingConfig);

            // Roll through the source and check the target in detail
            //foreach (MetaDataPair sourceEntityPair in sourceEntityList)
            //{
            //    MetaDataPair targetEntityPair = targetEntityList.Find(
            //        delegate(MetaDataPair mdp)
            //        {
            //            return mdp.SchemaName == sourceEntityPair.SchemaName;
            //        }
            //        );

            //    if (targetEntityPair != null)
            //    {
            //        //LoggingConfig.WriteLine("Processing attributes in Entity " + sourceEntityPair.SchemaName.ToString() + "...");

            //        // get the Attribues and validate them here
            //        List<AttributeMetadata> sourceAttributeList = SourceEntity.GetAttributeDetailsList(sourceServerConfig, sourceEntityPair.MetaDataId, LoggingConfig);
            //        List<AttributeMetadata> targetAttributeList = TargetEntity.GetAttributeDetailsList(targetServerConfig, targetEntityPair.MetaDataId, LoggingConfig);

            //        // Foreach loop attributes source to target
            //        foreach (AttributeMetadata sourceAttribute in sourceAttributeList)
            //        {
            //            AttributeMetadata targetAttribute = targetAttributeList.Find(
            //               delegate(AttributeMetadata mdp)
            //               {
            //                   return mdp.SchemaName == sourceAttribute.SchemaName;
            //               }
            //               );

            //            if (targetAttribute != null)
            //            {
            //                if (!(SourceEntity.CompareAttributeInformation(sourceAttribute, targetAttribute, LoggingConfig)))
            //                {
            //                    isValidAttributeDetails = false;
            //                    LoggingConfig.WriteLine("***** Attribute comparison for Source Attribute " + sourceAttribute.SchemaName.ToString() + " in Entity " + sourceEntityPair.SchemaName.ToString() + " Failed");
            //                }
            //            }
            //            else
            //            {
            //                LoggingConfig.WriteLine("***** Attribute " + sourceAttribute.SchemaName.ToString() + " was Not Found in the target entity " + sourceEntityPair.SchemaName.ToString());
            //                isValidAttributeDetails = false;
            //            }
            //        }

            //        // Foreach loop attributes target to source
            //        foreach (AttributeMetadata targetAttribute in targetAttributeList)
            //        {
            //            AttributeMetadata sourceAttribute = sourceAttributeList.Find(
            //                delegate(AttributeMetadata rmdp)
            //                {
            //                    return rmdp.SchemaName == targetAttribute.SchemaName;
            //                }
            //            );
            //            if (sourceAttribute != null)
            //            {
            //                // Do nothing. The detail has already been compared in the other foreach.
            //            }
            //            else
            //            {
            //                LoggingConfig.WriteLine("*****  Attribute " + targetAttribute.SchemaName.ToString() + " was Not Found in the source entity " + sourceEntityPair.SchemaName.ToString());
            //                isValidAttributeDetails = false;
            //            }
            //            // LoggingConfig.WriteLine("Target Attribute " + targetAttribute.SchemaName);
            //        }
            //    }
            //    else
            //    {
            //        LoggingConfig.WriteLine("***** Entity " + sourceEntityPair.SchemaName.ToString() + "  Skipped as it was Not Found in the target...");
            //        isValidAttributeDetails = false;
            //    }
            //}

            //// flip the logic and check the other way for missing entities. It is probably best to at least report them
            //// Any entities found will have had the attributes checked both ways in the prior foreach loop.
            //foreach (MetaDataPair sourceEntityPair in sourceEntityList)
            //{
            //    MetaDataPair targetEntityPair = targetEntityList.Find(
            //        delegate(MetaDataPair mdp)
            //        {
            //            return mdp.SchemaName == sourceEntityPair.SchemaName;
            //        }
            //        );

            //    if (targetEntityPair == null)
            //    {
            //        LoggingConfig.WriteLine("*****  Entity " + sourceEntityPair.SchemaName.ToString() + " was Not Found in the target");
            //        isValidAttributeDetails = false;
            //    }
            //}

            if (isValidAttributeDetails)
            {
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Attribute Schema Name Comparison PASSED *****");
                LoggingConfig.BlankLine();
            }
            else
            {
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Attribute Schema Name Comparison FAILED *****");
                LoggingConfig.BlankLine();
            }
            return isValidAttributeDetails;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="SourceOrganizationUri"></param>
        /// <param name="TargetOrganizationUri"></param>
        /// <returns></returns>
        public static Boolean ValidateEntityDetails(OrgParameters ValidateEntDetParams, Logger LoggingConfig)
        {
            LoggingConfig.WriteLine("Validating Entity Detail Settings Information..........");
            LoggingConfig.BlankLine();

            // Set up the working return
            Boolean isValidateEntityDetails = true;

            // Establish connection objects to be able to use the helper functions.
            ServerConnection sourceServerConn = new ServerConnection();
            ServerConnection targetServerConn = new ServerConnection();

            // Create a source config
            ServerConnection.Configuration sourceServerConfig = new ServerConnection.Configuration();
            if (ValidateEntDetParams.SourceUser != string.Empty && ValidateEntDetParams.SourcePassword != string.Empty)
            {
                //sourceServerConfig = sourceServerConn.CreateServerConfiguration(ValidateEntDetParams.SourceOrg, ValidateEntDetParams.SourceUser, ValidateEntDetParams.SourceDomain, ValidateEntDetParams.SourcePassword);
            }
            else
            {
                //sourceServerConfig = sourceServerConn.CreateServerConfiguration(ValidateEntDetParams.SourceOrg);
            }
            // Create a target config
            ServerConnection.Configuration targetServerConfig = new ServerConnection.Configuration();
            if (ValidateEntDetParams.TargetUser != string.Empty && ValidateEntDetParams.TargetPassword != string.Empty)
            {
                //targetServerConfig = targetServerConn.CreateServerConfiguration(ValidateEntDetParams.TargetOrg, ValidateEntDetParams.TargetUser, ValidateEntDetParams.TargetDomain, ValidateEntDetParams.TargetPassword);
            }
            else
            {
                //targetServerConfig = targetServerConn.CreateServerConfiguration(ValidateEntDetParams.TargetOrg);
            }
            // Create an entity validations object and preload the connection info
            EntityHelpers SourceEntity = new EntityHelpers();
            EntityHelpers TargetEntity = new EntityHelpers();

            // Get the entities list from the source and targets that will be used in the validation
            LoggingConfig.WriteLine("Obtaining Entity List from Source Organization " + sourceServerConfig.OrganizationName + "..........");
            //List<EntityMetadata> semdid = SourceEntity.GetEntityEntityList(sourceServerConfig, LoggingConfig);

            LoggingConfig.WriteLine("Obtaining Entity List from Target Organization " + targetServerConfig.OrganizationName + "..........");
            //List<EntityMetadata> temdid = TargetEntity.GetEntityEntityList(targetServerConfig, LoggingConfig);

            // Roll through the source get the associated Metadata ID from the target to use to get the EntityMetadata
            // then compare the detail of both.
            //foreach (EntityMetadata SourceEmd in semdid)
            //{
            //    EntityMetadata TargetEmd = temdid.Find(
            //       delegate(EntityMetadata mdp)
            //       {
            //           return mdp.SchemaName == SourceEmd.SchemaName;
            //       }
            //       );

            //    if (TargetEmd != null)
            //    {
            //        // go run the compare of the two entity metadata ojbects
            //        if (!(SourceEntity.CompareEntityInformation(SourceEmd, TargetEmd, LoggingConfig) == true))
            //        {
            //            isValidateEntityDetails = false;
            //            LoggingConfig.WriteLine("***** Entity comparison for source entity " + SourceEmd.SchemaName.ToString() + " Failed");
            //        }
            //        else
            //        {
            //            //LoggingConfig.WriteLine("Entity " + SourceEmd.SchemaName.ToString() + " was compared");
            //        }
            //    }
            //    else
            //    {
            //        isValidateEntityDetails = false;
            //        LoggingConfig.WriteLine("*** Source Entity " + SourceEmd.SchemaName.ToString() + " Not Found in the Target Organization ");
            //    }
            //}

            // Reverse the process to identify any missing entities
            //foreach (EntityMetadata TargetEmd in temdid)
            //{
            //    EntityMetadata SourceEmd = semdid.Find(
            //       delegate(EntityMetadata mdp)
            //       {
            //           return mdp.SchemaName == TargetEmd.SchemaName;
            //       }
            //       );


            //    if (SourceEmd == null)
            //    {
            //        isValidateEntityDetails = false;
            //        LoggingConfig.WriteLine("***** Target Entity " + TargetEmd.SchemaName.ToString() + " Not Found in the Source Organization ");
            //    }
            //}

            if (isValidateEntityDetails)
            {
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Entity Information Comparison PASSED *****");
                LoggingConfig.BlankLine();
            }
            else
            {
                LoggingConfig.BlankLine();
                LoggingConfig.WriteLine("***** Entity Information Comparison FAILED *****");
                LoggingConfig.BlankLine();
            }
            return isValidateEntityDetails;
        }

        public static Boolean ValidateParameters(ParseParameter.Arguments param)
        {
            Boolean SourceParameter = false;
            Boolean TargetParameter = false;
            int sCount = 0;
            int tCount = 0;
            // Stop the application if there were not any parameters passed in at all
            if (param.Parameters.Count <= 0)
            {
                return false;
            }

            // Stop the application if -s and -t were not both passed in
            foreach (System.Collections.DictionaryEntry args in param.Parameters)
            {
                // Check for invalid parameters. If you add a parameter make sure to put it in this switch as a case
                // Deadman switch (If it hits a value that is not a case the app will bail with exit code -1
                switch (args.Key.ToString())
                {
                    case "s":
                        sCount++;
                        SourceParameter = true;
                        break;
                    case "t":
                        tCount++;
                        TargetParameter = true;
                        break;
                    case "su":
                        break;
                    case "sp":
                        break;
                    case "sd":
                        break;
                    case "tu":
                        break;
                    case "tp":
                        break;
                    case "td":
                        break;
                    default:
                        return false;
                }
            }

            // Validates that both the s and the t were passed in
            // Make sure that both an s and a t parameter were passed in
            if (!(SourceParameter && TargetParameter))
            {
                return false;
            }

            //// In order to make sure that a user does not pass in more than one s or t parameter.
            //// This also mandates that they MUST pass in an s and a t parameter
            //// validate that only 1 s and 1 t parameters were passed in
            //if( sCount != 1 || tCount != 1 )
            //{
            //    return false;
            //}

            // passed all validation checks
            return true;
        }
    }
}
