﻿// **********************************************************************************************************
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 01/15/2013
// CmdLet: Create-CRMTeam
// This is a Custom Cmdlet that will allow you to create a Team with a specific Guid. This is handy when deploying to new environments.
// This is not intened to fully populate all fields for a team. You would need to update the created record with the expected value as
// part of a diffferent process. This is designed to only create a placeholder record.
// Required Parameters
//
// - Guid <the assigned guid that you want the team name to be saved with>
// - TeamName <the schema name of the entity to remove all records in>
// **********************************************************************************************************
using System;
using System.Management.Automation;
using System.Web.Services.Protocols;
using System.ServiceModel;

using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using Microsoft.Xrm.Sdk.Messages;

namespace Microsoft.MCS.Commandlets
{
    [Cmdlet(VerbsCommon.New, "CRMTeam", SupportsShouldProcess = true)]

    public class CreateTeamRecord : CrmServiceCommand
    {
        [Parameter(Mandatory = false)]
        public Guid TeamGuid { get; set; }

        [Parameter(Mandatory = false)]
        public string TeamName { get; set; }

        //[Parameter(Mandatory = false)]
        //public string BusinessUnitName { get; set; }

        [Parameter(Mandatory = false)]
        public string BusinessUnitGuid { get; set; }

        //[Parameter(Mandatory = false)]
        //public string AdministratorName { get; set; }

        [Parameter(Mandatory = false)]
        public string AdministratorGuid { get; set; }

        private Entity Record { get; set; }

        protected override void ProcessRecord()
        {
            try
            {
                IOrganizationService os = Service;

                Guid buGuid = Guid.Empty;
                Guid adminGuid = Guid.Empty;

                buGuid = new Guid(BusinessUnitGuid.ToString());
                adminGuid = new Guid(AdministratorGuid);

                // Then add the new team record
                Entity newTeam = new Entity()
                {
                    LogicalName = "team",
                    Id = TeamGuid,
                };

                newTeam["administratorid"] = new EntityReference("systemuser", adminGuid);
                newTeam["businessunitid"] = new EntityReference("businessunit", buGuid);
                newTeam["teamtype"] = new OptionSetValue(0);
                newTeam["name"] = TeamName;

                _service.Create(newTeam);
                Console.WriteLine("Team - " + TeamName.ToString() + "... Created Team record with GUID = " + TeamGuid.ToString());
            }
            catch (SoapException exception)
            {
                throw new Exception(GetErrorMessageFromSoapException(exception) + "Stack Trace: " + exception.StackTrace.ToString());
            }
        }

        public Guid GetAdministratorGuid(string adminName, IOrganizationService Service)
        {
            Guid AdminGuid = new Guid();
            // Obtain the query from context
            QueryExpression objQE = new QueryExpression()
            {
                EntityName = "systemuser",
                ColumnSet = new ColumnSet() { AllColumns = true }
            };

            // Modify the query by filtering the lookup where the nor_callcenterdepartment the current users business unit (which is named the same as the department) are the same
            ConditionExpression nameFilter = new ConditionExpression()
            {
                AttributeName = "fullname",
                Operator = ConditionOperator.Equal,
                Values = { adminName }
            };

            FilterExpression objFE = new FilterExpression()
            {
                FilterOperator = LogicalOperator.And,
                Conditions = { nameFilter }
            };

            objQE.Criteria.AddFilter(objFE);

            EntityCollection ec = (EntityCollection)Service.RetrieveMultiple(objQE);
            if (ec != null && ec.Entities.Count == 1)
            {
                AdminGuid = ((EntityReference)((Entity)ec.Entities[0]).Attributes["systemuserid"]).Id;
            }
            else
            {
                return new Guid();
            }

            return AdminGuid;
        }

        public Guid GetBusinessUnitGuidFromName(string businessUnitName, IOrganizationService Service)
        {
            Console.WriteLine("Start of GetBusinessUnitGuidFromName");
            Guid buGuid = new Guid();
            // Obtain the query from context

            QueryExpression objQE = new QueryExpression()
            {
                EntityName = "businessunit",
                ColumnSet = new ColumnSet() { AllColumns = true }
            };

            // Modify the query by filtering the lookup where the nor_callcenterdepartment the current users business unit (which is named the same as the department) are the same
            ConditionExpression nameFilter = new ConditionExpression()
            {
                AttributeName = "name",
                Operator = ConditionOperator.Contains,
                Values = { businessUnitName.ToString() }
            };

            FilterExpression objFE = new FilterExpression()
            {
                FilterOperator = LogicalOperator.And,
                Conditions = { nameFilter }
            };

            Console.WriteLine("Add Filter to Expression");
            objQE.Criteria.AddFilter(objFE);

            RetrieveMultipleRequest rm = new RetrieveMultipleRequest()
            {
                Query = objQE
            };

            EntityCollection ec = ((RetrieveMultipleResponse)Service.Execute(rm)).EntityCollection;

            Console.WriteLine("Check to see if it returned data");
            if (ec != null && ec.Entities.Count == 1)
            {
                Console.WriteLine("Obtain Guid from entity collection");

                buGuid = ((EntityReference)((Entity)ec.Entities[0]).Attributes["businessunitid"]).Id;
            }
            else
            {
                return new Guid();
            }

            Console.WriteLine("End of GetBusinessUnitGuidFromName. Returning Guid - " + buGuid.ToString());

            return buGuid;

        }

        public bool CreateRecord(Entity Record)
        {
            try
            {
                _service.Create(Record);
                return true;
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }
            return false;
        }

        public Entity GetEntityRecord(Guid EntityId)
        {
            try
            {
                Entity EntityRecord = new Entity();

                RetrieveRequest Request = new RetrieveRequest()
                {
                    ColumnSet = new ColumnSet() { AllColumns = true }
                };

                EntityRecord = Service.Retrieve("team", EntityId, new ColumnSet() { AllColumns = true });

                return EntityRecord;
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new Entity();
        }
    }
}
