﻿// **********************************************************************************************************
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 01/15/2013
// CmdLet: Import-Customizations
// This is a Custom Cmdlet that will execute a publish all against the CRM organization supplied in the -OrganizationUri parameter
//
// No additional parameters over and above the base classes required parameters are necessary. If you do not supply
// a filename it will assume the filename of default.zip. If you do not specify the solution folder it will use the
// current folder as the working folder and look for the customizations file there.
//
// Optional Parameters
// -SolutionFolder <folder path to look for the customization file>
// -FileName <customization file name>
// -FullFilePath <the full file path to the customization file>
//
// **********************************************************************************************************
using System;
using System.IO;
using System.Management.Automation;
using System.Web.Services.Protocols;
using Microsoft.Crm.Sdk.Messages;

namespace Microsoft.MCS.Commandlets
{
    [Cmdlet(VerbsData.Import, "CRMCustomizations", SupportsShouldProcess = true)]

    public class ImportCRMCustomizations : CrmServiceCommand
    {
        [Parameter(Mandatory = false)]
        public string SolutionFolder { get; set; }

        [Parameter(Mandatory = false)]
        public string FileName { get; set; }

        [Parameter(Mandatory = false)]
        public string FullFilePath { get; set; }

        protected override void ProcessRecord()
        {
            try
            {
                WriteVerbose("Importing Customizations");

                if (FullFilePath == null || FullFilePath.Length == 0)
                {
                    if (FileName == null || FileName.Length == 0)
                    {
                        FileName = "default.zip";
                    }

                    // If the solution directory is null or  blank use the current directory
                    if (SolutionFolder == null || SolutionFolder.Length == 0)
                    {
                        SolutionFolder = Directory.GetCurrentDirectory().ToString();
                    }

                    // Make sure that the directory entry ends with a backslash character so that the resulting url is valid
                    if (!(SolutionFolder.EndsWith("\\")))
                    {
                        SolutionFolder = String.Concat(SolutionFolder.ToString(), ("\\"));
                    }

                    FullFilePath = String.Concat(SolutionFolder, FileName);

                }

                //Import a solution
                byte[] fileBytes = File.ReadAllBytes(FullFilePath);

                var importSolutionRequest = new ImportSolutionRequest()
                {
                    CustomizationFile = fileBytes,
                    PublishWorkflows = true,
                    OverwriteUnmanagedCustomizations = true
                };

                Service.Execute(importSolutionRequest);
            }
            catch (SoapException exception)
            {
                WriteVerbose("Import Failed: " + GetErrorMessageFromSoapException(exception));
                throw new Exception(GetErrorMessageFromSoapException(exception));
            }
        }
    }
}