﻿// **********************************************************************************************************
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 01/15/2013
// CmdLet: Edit-BusinessUnitName
// This is a Custom Cmdlet that will allow you to change the name of the business unit specified.
// If you supply the guid of the business unit id only that name will be changed
// If you supply only the name of the business unit and multiple business units exist with the same name, all will be changed.
//
// Required Parameters
// -ChangeTo
//
// Optional Parameters (Only one of these needs to be supplied for it to find a record. The Cmdlet will not produce an error
//                      if you do not supply either one of them however it will also not be able to make any changes)
// -BusinessUnitId
// -ChangeFrom
//
// **********************************************************************************************************
using System;
using System.IO;
using System.Collections.Generic;
using System.Management.Automation;
using System.Web.Services.Protocols;
using System.ServiceModel;

using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using Microsoft.Xrm.Sdk.Messages;

namespace Microsoft.MCS.Commandlets
{
    [Cmdlet(VerbsCommon.Rename, "CRMBusinessUnitName", SupportsShouldProcess = true)]

    public class RenameBusinessEntity : CrmServiceCommand
    {

        private Entity Record { get; set; }

        [Parameter(Mandatory = false)]
        public string ChangeFrom { get; set; }

        [Parameter(Mandatory = false)]
        public Guid BusinessUnitId { get; set; }

        [Parameter(Mandatory = true)]
        public string ChangeTo { get; set; }

        protected override void ProcessRecord()
        {
            try
            {
                IOrganizationService os = _service;

                WriteVerbose("Changing the business entity name of " + ChangeFrom.ToString() + "To ..." + ChangeTo.ToString());
                Boolean Continue = true;

                // if the BU guid is passed in then no need to get the list and run through the names.
                if (BusinessUnitId.ToString() != string.Empty && BusinessUnitId.ToString() != Guid.Empty.ToString())
                {
                    Record = GetEntityRecord(BusinessUnitId);
                    if (Record != null)
                    {
                        Record.Attributes["name"] = ChangeTo.ToString();
                        UpdateRecord(Record);
                    }
                }
                else
                {
                    // This is in a while loop to overcome the 5000 max record cound limit on retrieve multiple
                    while (Continue == true)
                    {
                        // Set the escape clause to avoid infinit loop
                        Continue = false;
                        // Get a list of the record Id's Then run through them and see if the one you are looking for is in the list.
                        List<Entity> EntityRecordList = GetEntityRecords();

                        // Only if the record count maxes out do you continue on... Max is 5000 records
                        if (EntityRecordList.Count == 5000)
                        {
                            Continue = true;
                        }

                        // Run throught the list and find the record you are looking for and send it on to get the name changed.
                        foreach (Entity Record in EntityRecordList)
                        {
                            // if you find the record then set the Guid
                            if (Record.Attributes["name"].ToString() == ChangeFrom.ToString())
                            {
                                Record.Attributes["name"] = ChangeTo.ToString();
                                UpdateRecord(Record);
                            }
                        }
                    }
                }
                Console.WriteLine("Processing Completed...");
            }
            catch (SoapException exception)
            {
                throw new Exception(GetErrorMessageFromSoapException(exception));
            }
        }

        public bool UpdateRecord(Entity Record)
        {
            try
            {
                _service.Update(Record);
                return true;
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }
            return false;
        }

        public List<Entity> GetEntityRecords()
        {
            try
            {
                List<Entity> EntityRecordIDList = new List<Entity>();

                //Create a query expression specifying the link entity alias and the columns of the link entity that you want to return
                QueryExpression qe = new QueryExpression()
                {
                    EntityName = "businessunit",
                    ColumnSet = new ColumnSet() { AllColumns = true }
                };

                RetrieveMultipleRequest request = new RetrieveMultipleRequest()
                {
                    Query = qe
                };

                EntityCollection ec = ((RetrieveMultipleResponse)Service.Execute(request)).EntityCollection;

                // Loop through the results and create the return array list for the EntityId and SchemaName.
                for (int i = 0; i < ec.Entities.Count; i++)
                {
                    EntityRecordIDList.Add(ec[i]);
                }

                return EntityRecordIDList;
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new List<Entity>();
        }


        public Entity GetEntityRecord(Guid EntityId)
        {
            try
            {
                Entity EntityRecord = new Entity();

                RetrieveRequest Request = new RetrieveRequest()
                {
                    ColumnSet = new ColumnSet() { AllColumns = true }
                };

                EntityRecord = Service.Retrieve("businessunit", EntityId, new ColumnSet() { AllColumns = true });

                return EntityRecord;
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new Entity();
        }
    }
}
