﻿// **********************************************************************************************************
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 01/15/2013
// CmdLet: Edit-BusinessUnitName
// This is a Custom Cmdlet that will allow you to change the name of the business unit specified.
// If you supply the guid of the business unit id only that name will be changed
// If you supply only the name of the business unit and multiple business units exist with the same name, all will be changed.
//
// Required Parameters
// -ChangeTo
//
// Optional Parameters (Only one of these needs to be supplied for it to find a record. The Cmdlet will not produce an error
//                      if you do not supply either one of them however it will also not be able to make any changes)
// -BusinessUnitId
// -ChangeFrom
//
// **********************************************************************************************************
using System;
using System.IO;
using System.Collections.Generic;
using System.Management.Automation;
using System.Web.Services.Protocols;
using System.ServiceModel;

using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using Microsoft.Xrm.Sdk.Messages;

namespace Microsoft.MCS.Commandlets
{
    [Cmdlet(VerbsCommon.New, "CRMBusinessUnitRecord", SupportsShouldProcess = true)]

    public class CreateBusinessEntity : CrmServiceCommand
    {

        private Entity Record { get; set; }

        [Parameter(Mandatory = true)]
        public String Name { get; set; }

        [Parameter(Mandatory = false)]
        public Guid BusinessUnitId { get; set; }

        [Parameter(Mandatory = true)]
        public String ParentName { get; set; }

        [Parameter(Mandatory = false)]
        public Guid ParentId { get; set; }

        protected override void ProcessRecord()
        {
            try
            {
                IOrganizationService os = _service;

                Console.WriteLine("Checking to see if the parent business unit exists. If it exists it will return the Id. If it is a Duplicate the New Record will fail");

                // Get the Parent record from the Id if it was supplied
                if (ParentId != null && ParentId.ToString() != Guid.Empty.ToString())
                {
                    Entity er = GetParentRecordById(ParentId);

                    ParentId = er.Id;

                    Console.WriteLine("Parent Business Unit Id " + ParentId.ToString().ToUpper() + " Was retrieved from the target system.");
                }
                else
                {
                    if (String.IsNullOrEmpty(ParentName) != true)
                    {
                        Entity er = GetParentRecordByName(ParentName);
                        if(er != null && er.Id != null && er.Id.ToString() != Guid.Empty.ToString())
                        {
                            ParentId = er.Id;

                            Console.WriteLine("Parent Id " + ParentId.ToString().ToUpper() + " Was retrieved from the target system.");
                        }
                    }
                    else
                    {
                        Console.WriteLine("You Must supply a business unit parent name. You can only change the name for the Top Level Business Unit Record.");
                        return;
                    }
                }

                Record = new Entity();
                Record.LogicalName = "businessunit";

                if (BusinessUnitId != null && BusinessUnitId.ToString() != Guid.Empty.ToString())
                {
                    Record.Id = BusinessUnitId;
                }
                else
                {
                    Record.Id = Guid.NewGuid();
                }

                Record["name"] = Name;
                Record["parentbusinessunitid"] = new EntityReference("businessunit", ParentId);

                Guid result = Service.Create(Record);

                if (result == null || result.ToString() == Guid.Empty.ToString())
                {
                    Console.WriteLine("*** Error: Business Unit " + Name.ToString() + " was not added due to an unreported error.");
                }
                else
                {
                    Console.WriteLine("Business Unit " + Name.ToString() + " was added with Guid " + result.ToString().ToUpper());
                }
                Console.WriteLine("Processing Completed...");
            }
            catch (SoapException exception)
            {
                throw new Exception(GetErrorMessageFromSoapException(exception));
            }
        }

        public bool CreateRecord(Entity Record)
        {
            try
            {
                _service.Update(Record);
                return true;
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }
            return false;
        }

        public Entity GetParentRecordByName(String ParentName)
        {
            try
            {
                Console.WriteLine("Searching for Parent Record " + ParentName.ToString());

                List<Entity> EntityRecordIDList = new List<Entity>();

                //Create a query expression specifying the link entity alias and the columns of the link entity that you want to return
                QueryExpression qe = new QueryExpression()
                {
                    EntityName = "businessunit",
                    ColumnSet = new ColumnSet() { AllColumns = true }
                };

                RetrieveMultipleRequest request = new RetrieveMultipleRequest()
                {
                    Query = qe
                };

                RetrieveMultipleResponse ec = (RetrieveMultipleResponse)Service.Execute(request);

                foreach(Entity ent in ec.EntityCollection.Entities)
                {
                    if (ent.Contains("name") && ent["name"].ToString().Equals(ParentName))
                    {
                        EntityRecordIDList.Add(ent);
                    }
                }

                if(EntityRecordIDList.Count != 1)
                {
                    return new Entity();
                }
                else
                {
                    return EntityRecordIDList[0];
                }

            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new Entity();
        }


        public Entity GetParentRecordById(Guid ParentId)
        {
            try
            {

                Entity ec = Service.Retrieve("businessunit", ParentId, new ColumnSet(true));

                if (ec != null && ec.Id.ToString() != Guid.Empty.ToString())
                {
                    return ec;
                }
                else
                {
                    return new Entity();
                }
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new Entity();
        }
    }
}
