﻿// **********************************************************************************************************
// Microsoft MCS Enterprise Services
// Author: Allen Barge
// Date: 01/15/2013
// CmdLet: Remove-CRMRecords
// This is a Custom Cmdlet that will allow you to delete all records in the supplied entity. Records will have to be
// eligible for delete or the record will error out and a report of the error will be echoed to the cmd window. It will continue to try to delete the other records.
//
// Required Parameters
// -EntityName <the schema name of the entity to remove all records in>
//
// Optional Parameters - if the primary key attribute is not the ENTITYNAME + id then you will have to supply the primary key for the entity.
// -PrimaryKey <attribute name used as the primary key> (For activities you will have to supply the attribute name "activityid")
//
// **********************************************************************************************************
using System;
using System.IO;
using System.Collections.Generic;
using System.Management.Automation;
using System.Web.Services.Protocols;
using System.ServiceModel;

using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using Microsoft.Xrm.Sdk.Messages;

namespace Microsoft.MCS.Commandlets
{
    [Cmdlet(VerbsCommon.Get, "CRMRecordCount", SupportsShouldProcess = true)]

    public class CountEntityRecords : CrmServiceCommand
    {
        [Parameter(Mandatory = true, HelpMessage = "Provide the entity name you wish to count all of the records in.")]
        public string EntityName { get; set; }

        [Parameter(Mandatory = false, HelpMessage = "If the default primary key field name is not name of the entity + id then you must supply the primary key...eg. Activityid for activity records would need to be supplied")]
        public string PrimaryKey { get; set; }

        protected override void ProcessRecord()
        {
            try
            {
                IOrganizationService os = _service;
                int countValue = 0;

                // Establish the primary key name
                string pkID = string.Empty;
                // Need to add functionality to get this from the metadata
                if (PrimaryKey != null && PrimaryKey.Length > 0 && PrimaryKey != string.Empty)
                {
                    pkID = PrimaryKey;
                }
                else
                {
                    pkID = String.Concat(EntityName.ToString(), "id");
                }

                WriteVerbose("Counting All Records from the " + EntityName.ToString() + "Entity...");
                Boolean Continue = true;
                while (Continue == true)
                {
                    // Set the escape clause to avoid infinit loop
                    Continue = false;
                    // Get a list of the record Id's
                    List<Guid> EntityRecordList = GetEntityRecordsList(pkID);

                    // Only if the record count maxes out do you continue on... Max is 5000 records
                    if (EntityRecordList.Count == 5000)
                    {
                        Continue = true;
                    }

                    countValue = countValue + EntityRecordList.Count;
                }

                Console.WriteLine(EntityName.ToString() + " - " + countValue.ToString() + " Records");
            }
            catch (SoapException exception)
            {
                throw new Exception(GetErrorMessageFromSoapException(exception));
            }
        }
        public List<Guid> GetEntityRecordsList(string PrimaryKeyId)
        {
            try
            {

                MetaDataPair mdp = new MetaDataPair();
                List<Guid> EntityRecordIDList = new List<Guid>();

                //Create a query expression specifying the link entity alias and the columns of the link entity that you want to return
                QueryExpression qe = new QueryExpression()
                {
                    EntityName = EntityName.ToString(),
                    ColumnSet = new ColumnSet(PrimaryKeyId),
                };

                RetrieveMultipleRequest request = new RetrieveMultipleRequest()
                {
                    Query = qe
                };

                EntityCollection ec = ((RetrieveMultipleResponse)Service.Execute(request)).EntityCollection;

                // Loop through the results and create the return array list for the EntityId and SchemaName.
                for (int i = 0; i < ec.Entities.Count; i++)
                {
                    EntityRecordIDList.Add(ec[i].Id);
                }

                return EntityRecordIDList;
            }
            catch (TimeoutException Timeex)
            {
                Console.WriteLine("Timeout Exception - " + Timeex.Message);
            }
            catch (FaultException<Microsoft.Xrm.Sdk.OrganizationServiceFault> Orgex)
            {
                Console.WriteLine("Organization Service Fault Exception - " + Orgex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Entity Metadata Exception - " + ex.Message);
            }

            return new List<Guid>();
        }
    }
}
